/* 
 * Copyright (C) 1999 Robert Wilhelm
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <ctype.h>
#include <sys/time.h>
#include <stdarg.h>
#include <signal.h>

#include "makros.h"
#include "child.h"
#include "notation.h"
#include "movlist.h"
#include "position.h"
#include "board.h"
#include "engine.h"

/* Class signals */
enum {
	GAME_SIGNAL,
	MOVE_SIGNAL,
	UNMOVE_SIGNAL,
	INFO_SIGNAL,
	LAST_SIGNAL
};
static gint engine_signals[LAST_SIGNAL] = { 0 };

/* Call back set up */
typedef void (*EngineSignalFunc1) (GtkObject *, gint, gint, gpointer);
typedef void (*EngineSignalFunc2) (GtkObject *, GameInfo *, gpointer);
typedef void (*EngineSignalFunc3) (GtkObject *, gint, Position *, gpointer);

static void engine_marshal_1(GtkObject *object,
			     GtkSignalFunc func,
			     gpointer func_data,
			     GtkArg *args)
{
        EngineSignalFunc1 rfunc;
               
        rfunc = (EngineSignalFunc1)func;
        (*rfunc)(object,
		 GTK_VALUE_INT(args[0]),
		 GTK_VALUE_INT(args[1]),
		 func_data);
}

static void engine_marshal_2(GtkObject *object,
			     GtkSignalFunc func,
			     gpointer func_data,
			     GtkArg *args)
{
        EngineSignalFunc2 rfunc;
               
        rfunc = (EngineSignalFunc2)func;
        (*rfunc)(object,
		 GTK_VALUE_POINTER(args[0]),
		 func_data);
}

static void engine_marshal_3(GtkObject *object,
			     GtkSignalFunc func,
			     gpointer func_data,
			     GtkArg *args)
{
        EngineSignalFunc3 rfunc;
               
        rfunc = (EngineSignalFunc3)func;
        (*rfunc)(object,
		 GTK_VALUE_INT(args[0]),
		 GTK_VALUE_POINTER(args[1]),
		 func_data);
}

/* Prototypes */
static void engine_class_init (EngineClass *klass);
static void engine_init (Engine *engine);

GtkType engine_get_type()
{
	static guint engine_type = 0;

	if (!engine_type) {
           GtkTypeInfo engine_info = {
               "Engine",
               sizeof (Engine),
               sizeof (EngineClass),
               (GtkClassInitFunc) engine_class_init,
               (GtkObjectInitFunc) engine_init,
               (GtkArgSetFunc) NULL,
               (GtkArgGetFunc) NULL
             };
             engine_type = gtk_type_unique (gtk_object_get_type (),
					    &engine_info);
           }

         return engine_type;
}

static void
engine_class_init (EngineClass *klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass*) klass;

	engine_signals[GAME_SIGNAL] =
		gtk_signal_new ("game",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (EngineClass, game),
				engine_marshal_3, GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_POINTER);

		
	engine_signals[MOVE_SIGNAL] =
		gtk_signal_new ("move",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (EngineClass, move),
				engine_marshal_1, GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_INT);

	engine_signals[UNMOVE_SIGNAL] =
		gtk_signal_new ("unmove",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (EngineClass, unmove),
				gtk_marshal_NONE__INT, GTK_TYPE_NONE,
				1, GTK_TYPE_INT);
	
	engine_signals[INFO_SIGNAL] =
		gtk_signal_new ("info",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (EngineClass, info),
				engine_marshal_2, GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	
	gtk_object_class_add_signals (object_class,
				      engine_signals,
				      LAST_SIGNAL);
	
	/* Init the signal stuff */
	klass->game = NULL;
	klass->move = NULL;
	klass->unmove = NULL;
	klass->info = NULL;
}

static void 
engine_init (Engine *engine) {
	engine->board = NULL;
}

GtkObject *
engine_new () 
{
	return GTK_OBJECT(gtk_type_new(engine_get_type()));
}

Board *
engine_get_board (Engine *engine) {
	return BOARD (engine->board);
}

void 
engine_set_board (Engine *engine, Board *board) 
{
	engine->board = GTK_WIDGET(board);
}


void 
engine_fill_menu (Engine *engine, GtkMenuShell *shell, gint pos)
{
	EngineClass *klass = ENGINE_CLASS (GTK_OBJECT (engine)->klass);
	
	if (klass->engine_fill_menu)
		klass->engine_fill_menu (engine, shell, pos);
}

void
engine_move (Engine *engine, gint from, gint to)
{
	EngineClass *klass = ENGINE_CLASS (GTK_OBJECT (engine)->klass);
	
	if (klass->engine_move)
		klass->engine_move (engine, from, to);
}

