/* 
 * Copyright (C) 1999 Robert Wilhelm
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "gnome_chess.h"

#include "makros.h"
#include "position.h"
#include "pro.h"
#include "movlist.h"
#include "disk.h"
#include "cho1.h"

#include <zvt/zvtterm.h>

static char db[120] = { [0 ... 119]= -1 };
static void * db_image[120] = { [0 ... 119]= NULL };

static void gnomechess_board_draw(void);

GtkWidget * gnomechess_board_new()
{
  GtkWidget *table;
  GtkWidget *vbox, *hbox;
  GtkWidget *scrollbar;

  board.flip = FALSE;
  board.mode = COMPUTER_BLACK;

  table = gtk_table_new (3,2,FALSE);
  gtk_container_border_width (GTK_CONTAINER (table), 4);

  /* Message area */
  hbox = gtk_hbox_new(FALSE, FALSE);
  board.messages = zvt_term_new_with_size(80, 10);
  zvt_term_set_blink(ZVT_TERM(board.messages), FALSE);
  scrollbar = gtk_vscrollbar_new (ZVT_TERM(board.messages)->adjustment);
  gtk_vscrollbar_new (GTK_ADJUSTMENT(ZVT_TERM(board.messages)->adjustment));
  GTK_WIDGET_UNSET_FLAGS(scrollbar, GTK_CAN_FOCUS);
  gtk_container_add(GTK_CONTAINER(hbox), board.messages);
  gtk_container_add(GTK_CONTAINER(hbox), scrollbar);
  gtk_widget_show(scrollbar);
  gtk_widget_show (board.messages);
  gtk_widget_show(hbox);
  gtk_table_attach_defaults (GTK_TABLE (table), hbox, 0, 2, 1, 2);

  /* Command area */
  board.command = gtk_entry_new_with_max_length (255);
  gtk_signal_connect(GTK_OBJECT(board.command), "activate", GTK_SIGNAL_FUNC(command_cb), NULL);
  gtk_table_attach_defaults (GTK_TABLE (table), board.command,0,2,2,3);
  gtk_widget_show (board.command);

  /* Move List area */
  board.movelist = gnomechess_movelist_new();     
  gtk_table_attach_defaults (GTK_TABLE (table), 
			gnomechess_movelist_getview(board.movelist), 1, 2, 0, 1);
  gtk_widget_show (gnomechess_movelist_getview(board.movelist));

  vbox = gtk_vbox_new(0,0);

  board.canvas = gnome_canvas_new ();
  gtk_widget_set_usize (board.canvas, PIECE_SIZE * 8 + 1, PIECE_SIZE * 8 + 1);
  gnome_canvas_set_scroll_region (GNOME_CANVAS (board.canvas), 0, 0, PIECE_SIZE * 8 + 1, PIECE_SIZE * 8 + 1);
  gnomechess_board_draw();
  gnomechess_board_update();
  gtk_container_add(GTK_CONTAINER(vbox),board.canvas);

  hbox = gtk_hbox_new(0,0);
  board.whitename = gtk_label_new(_("White"));
  board.blackname = gtk_label_new(_("Black"));
  board.whitetime = gnomechess_timer_new_time(0);
  board.blacktime = gnomechess_timer_new_time(0);
  gtk_container_add(GTK_CONTAINER(hbox),board.whitename);
  gtk_container_add(GTK_CONTAINER(hbox),gnomechess_timer_getview(board.whitetime));
  gtk_container_add(GTK_CONTAINER(hbox),board.blackname);
  gtk_container_add(GTK_CONTAINER(hbox),gnomechess_timer_getview(board.blacktime));
  gtk_container_add(GTK_CONTAINER(vbox),hbox);
  gtk_table_attach_defaults (GTK_TABLE (table), vbox, 0, 1 , 0, 1);

  gtk_widget_show (board.whitename);
  gtk_widget_show (board.blackname);
  gtk_widget_show (gnomechess_timer_getview(board.whitetime));
  gtk_widget_show (gnomechess_timer_getview(board.blacktime));
  gtk_widget_show (hbox);
  gtk_widget_show (board.canvas);
  gtk_widget_show (vbox);
  gtk_widget_show (table);

  return table;
}

void gnomechess_board_set_flip(gboolean f) {
	board.flip = f;
}

void gnomechess_board_flip() {
	if (board.flip)
		board.flip = FALSE;
	else
		board.flip = TRUE;
}

static int gnomechess_board_press_cb(GnomeCanvasItem *item, GdkEvent *event, gpointer data)
{
  static double  x,y; 
  static int from, to;
  static int xi,yi;
  int xt,yt;
  double new_x, new_y;

  if (data) {
      item = db_image[(unsigned long)data];
      if ( item == NULL) return 0;
    }

  switch (event->type)
    {
	case GDK_BUTTON_PRESS:
		switch (event->button.button) 
		  {
		  case 1:
		    xi = x = event->button.x;
		    yi = y = event->button.y;
		    gnome_canvas_item_raise_to_top (item);
		  
		    xt = x / PIECE_SIZE;
		    yt = y / PIECE_SIZE;

			if (!board.flip)
			    from = A1 + xt  + 10 * (7-yt);
			else
				from = H8 - xt -10 *(7 - yt);
		    to = from;
		    break;

		  default:
		    break;
		  }
		
		break;

    case GDK_MOTION_NOTIFY:
      if ((event->motion.state & GDK_BUTTON1_MASK))
	{
	  new_x = event->motion.x;
	  new_y = event->motion.y;

	  gnome_canvas_item_move (item, new_x - x, new_y - y);
	  x = new_x;
	  y = new_y;
	}
      break;

    case GDK_BUTTON_RELEASE:
      xt = x / PIECE_SIZE;
      yt = y / PIECE_SIZE;

	if (!board.flip)
		to  = A1 + xt  + 10 * (7-yt);
	else
		to = H8 - xt - 10 *(7 - yt);
	to=reg_zug(from, to);

	if (to) {
		int p = currPositionPtr->square[from];
		gnomechess_position_make_move(currPositionPtr,from,to);
		if (!(board.mode == HUMANS))
			engine_write_move(from,to);
		gnomechess_movelist_add (board.movelist,p,from,to);
		gnomechess_board_update();
	} else 
		gnome_canvas_item_move (item, xi - x, yi - y);
	break;
    default:
    }

  return 0;
}

static void * gnomechess_board_draw_piece(int sq, int piece)
{
	GdkImlibImage *im;
	GnomeCanvasItem *image;
        GnomeCanvasGroup *root;
	double x,y;

        int i,j;

	char *s;

	root = GNOME_CANVAS_GROUP (gnome_canvas_root (GNOME_CANVAS (board.canvas)));

	i= 9 - sq / 10 ;
	j= sq% 10 - 1;
 
	x = j * PIECE_SIZE;
	y = i * PIECE_SIZE;


	switch (piece)  {
	  case EMPTY:  s= "";        break;
	  case WP  :  s = "P.png";  break;
	  case WN  :  s = "N.png";  break;
	  case WB  :  s = "B.png";  break;
	  case WR  :  s = "R.png";  break;
	  case WQ  :  s = "Q.png";  break;
	  case WK  :  s = "K.png";  break;
	  case BP  :  s = "p.png";  break;
	  case BN  :  s = "n.png";  break;
	  case BB  :  s = "b.png";  break;
	  case BR  :  s = "r.png";  break;
	  case BQ  :  s = "q.png";  break;
	  case BK  :  s = "k.png";  break;
	  default  :  abort();
	  }

	if (*s)
	  {
	    /* first try gnome_unconditional_pixmap_file */

	    char *tmp = g_strconcat("gnome-chess/",s,NULL);
	    char *fname = gnome_unconditional_pixmap_file (tmp);
	    im = gdk_imlib_load_image (fname);
	    g_free(fname);  
	    g_free(tmp);

	    /* if not found (e.g. user installed in different dir than
	     *  rest of gnome, try there.
	     */
	    if(!im) 
	      {
		fname = g_strconcat(GNOMEDATADIR,"/pixmaps/gnome-chess/",s,NULL);
		im = gdk_imlib_load_image (fname);
		g_free(fname);
	      }  

	    /* if still not found (e.g. user did not install at all),
	     *  try a little bit harder.
	     */

	    if(!im) 
	      {
		fname = g_strconcat("../Sets/",s,NULL);
		im = gdk_imlib_load_image (fname);
		g_free(fname);
	      }  

	    /* now we surrender. */

	    if(!im) 
	      {
		fprintf(stderr,_("pic %s not found\n"),s);
		abort();
	      }

	    image = gnome_canvas_item_new (root,
				       gnome_canvas_image_get_type (),
				       "image", im,
				       "x", x,
				       "y", y,
				       "width", (double) im->rgb_width,
				       "height", (double) im->rgb_height,
				       "anchor", GTK_ANCHOR_NW,
				       NULL);

	    gtk_signal_connect (GTK_OBJECT (image), "event",
				(GtkSignalFunc) gnomechess_board_press_cb,
				NULL);

	    return image;
	  }

	return 0;
}

static void gnomechess_board_draw_background(int f)
{
	GnomeCanvasItem *image;
        GnomeCanvasGroup *root;
	double x,y;

        int i,j;

	root = GNOME_CANVAS_GROUP (gnome_canvas_root (GNOME_CANVAS (board.canvas)));

        i= 9 - f / 10 ;
	j= f % 10 - 1;
 
        x = j * PIECE_SIZE;
	y = i * PIECE_SIZE;

	image = gnome_canvas_item_new (root,
				       gnome_canvas_rect_get_type (),
				       "x1", x,
				       "y1", y,
				       "x2", x + (double) PIECE_SIZE,
				       "y2", y + (double) PIECE_SIZE,
				       "fill_color", (i+j) & 1 ? "darkgreen":"lightgreen",
              			       "outline_color", "black",
				       "width_pixels", 0,
				       NULL);

	gtk_signal_connect (GTK_OBJECT (image), "event",
			    (GtkSignalFunc) gnomechess_board_press_cb,
			    (void *)(unsigned long) f);

}

static void gnomechess_board_draw()
{
  int i,j;

  for ( i=0;i<8;i++)
    for ( j=0;j<8;j++)
      gnomechess_board_draw_background(H8 - i * 10 - j);
}

void gnomechess_board_update(void) {
  int n, n2;
  int i,j;

	for ( i=0;i<8;i++)
		for ( j=0;j<8;j++) {
			n  = H8 - i * 10 - j;
			if (!board.flip) 
				n2 = n;
			else
				n2 = A1 + i*10 + j;

			if (currPositionPtr->square[n2] != db[n]) 	{
				if (db_image[n])
					gtk_object_destroy (GTK_OBJECT (db_image[n]));
				db_image[n] = gnomechess_board_draw_piece(n, currPositionPtr->square[n2]);
				db[n]=currPositionPtr->square[n2];
			}
		}
	gtk_widget_show (board.canvas);
}

