/**
 * gnome-braille, a braille encoding and output system for GNOME, 
 *                and Unix-like operating environments.
 *
 * Copyright Sun Microsystems Inc. 2004
 *
 * Copyright Sun Microsystems Inc. 2004
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 **/

#include "braille-table-encoder.h"

typedef struct {
    gchar *out_string;
    guint *offsets;
    guint size;
    guint len;
    BrailleContext *context;
} BrailleOutputBuffer;

typedef enum {
    TABLE_LINE_FORMAT_UNICODE,
    TABLE_LINE_FORMAT_UTF8
} TableLineFormat;

typedef struct {
    const gchar *string;
    BrailleContext *context;
} MatchStringKey;

typedef struct {
    const gchar *string;
    guint match_len;
} MatchStringValue;

/* BrailleTableEncoder implementations */
static void braille_table_encoder_real_set_table (BrailleTableEncoder *encoder, BrailleTable *table);
static gchar *braille_table_decompose_and_lookup_char (BrailleTable *table, gunichar ucs, BrailleContext **context);
static gchar *braille_table_lookup_char (BrailleTable *table, gunichar ucs, BrailleContext **context);

/* BrailleEncoder implementations */
static const gchar *braille_table_encoder_translate_string (BrailleEncoder *encoder, const gchar *string, guint **offsets);
static void braille_table_encoder_set_alternate  (BrailleEncoder *encoder, BrailleEncoder *alternate);

static void braille_table_check_suffix (BrailleTable *table, const gchar *string, BrailleOutputBuffer *outbuf);

static BrailleContext** _braille_contexts = NULL;

static BrailleContext* BRAILLE_CONTEXT_ALL;
static BrailleContext* BRAILLE_CONTEXT_ALPHA;

static BrailleContext*
braille_context_add_new (gchar *name)
{
    BrailleContext *context;
    gint n_contexts = 0;
    if (_braille_contexts == NULL)
    {
	context = g_new (BrailleContext, 1);
	context->name = name;
	n_contexts = 1;
	_braille_contexts = g_new (BrailleContext *, n_contexts + 1);
    }
    else
    {
	while (_braille_contexts[n_contexts]) 
	{
	    if (!strcmp (_braille_contexts[n_contexts]->name, name))
	    {
		return _braille_contexts[n_contexts];
	    }
	    n_contexts++;
	}
	n_contexts++;
	_braille_contexts = g_renew (BrailleContext *, _braille_contexts, n_contexts + 1);
    }
    context = g_new (BrailleContext, 1);
    context->name = name;
    _braille_contexts[n_contexts - 1] = context;
    _braille_contexts[n_contexts] = NULL;

    return context;
}

static BrailleContext**
braille_get_contexts (void)
{
    if (_braille_contexts == NULL)
    {
	BRAILLE_CONTEXT_ALL = braille_context_add_new (g_strdup ("all"));
	BRAILLE_CONTEXT_ALPHA = braille_context_add_new (g_strdup ("alpha"));
    }
    return _braille_contexts;
}

static gboolean
braille_context_match (BrailleContext *a, BrailleContext *b)
{
    return ((!a && !b) || (a == b) || (a && b && ((a->name == b->name) || !strcmp (a->name, b->name))));
}

static void
braille_encoder_interface_init (BrailleEncoderInterface *iface)
{
    g_return_if_fail (iface != NULL);

    iface->translate_string = braille_table_encoder_translate_string;
    iface->set_alternate = braille_table_encoder_set_alternate;
}

static void
braille_table_encoder_init (BrailleTableEncoder *encoder)
{
    encoder->table = NULL;
    encoder->alternate = NULL;
}

static void
braille_table_encoder_class_init (BrailleTableEncoderClass *klass)
{
    klass->set_table = braille_table_encoder_real_set_table;
}

GType
braille_table_encoder_get_type (void)
{
    static GType type = 0;
    if (!type)
    {
	static const GTypeInfo info =
	    {
		sizeof (BrailleTableEncoderClass),
		(GBaseInitFunc) NULL,
		(GBaseFinalizeFunc) NULL,
		(GClassInitFunc) braille_table_encoder_class_init,
		(GClassFinalizeFunc) NULL,
		NULL,
		sizeof (BrailleTableEncoder),
		0,
		(GInstanceInitFunc) braille_table_encoder_init,
	    };

	static const GInterfaceInfo braille_encoder_info =
	    {
		(GInterfaceInitFunc) braille_encoder_interface_init,
		(GInterfaceFinalizeFunc) NULL,
		NULL
	    };

	type = g_type_register_static (G_TYPE_OBJECT, "BrailleTableEncoder", &info, 0);

	g_type_add_interface_static (type, BRAILLE_TYPE_ENCODER, &braille_encoder_info);

    }
    return type;
}

/**
 * braille_table_encoder_set_table:
 * @encoder: the #BrailleTableEncoder instance on which to operate.
 * @table: the #BrailleTable which should serve as the base for this encoder.
 *
 * Sets the #BrailleTable instance which this encoder should use for its 
 * dot or character conversions, freeing any previous #BrailleTable which
 * this encoder was using.
 **/
void braille_table_encoder_set_table (BrailleTableEncoder *encoder, BrailleTable *table)
{
    BrailleTableEncoderClass *klass;
    g_return_if_fail (BRAILLE_IS_TABLE_ENCODER (encoder));
    klass = BRAILLE_TABLE_ENCODER_GET_CLASS (encoder);
    if (klass->set_table)
	(klass->set_table) (encoder, table);
}

/* implementations of virtual funcs for BrailleTableEncoder */
static void
braille_table_encoder_real_set_table (BrailleTableEncoder *encoder, BrailleTable *table)
{
    g_return_if_fail (BRAILLE_IS_TABLE_ENCODER (encoder));

    if (encoder->table) 
	g_free (encoder->table);
    encoder->table = table;
}

/* private utility functions */
static BrailleOutputBuffer *
braille_output_buffer_new (void)
{
    BrailleOutputBuffer *buf = g_new (BrailleOutputBuffer, 1);
    buf->out_string = g_malloc0 (1024);
    buf->offsets = g_malloc0 (1024 * sizeof (guint));
    buf->size = 1024;
    buf->len = 0;
    buf->context = BRAILLE_CONTEXT_ALPHA;
    return buf;
}

static gchar *
braille_table_decompose_and_lookup_char (BrailleTable *table, gunichar ucs, 
					 BrailleContext **context)
{
    /* XXX FIXME!  This leaks like a sieve! XXX */
    unsigned int res_len;
    gunichar *chars = g_unicode_canonical_decomposition (ucs, &res_len);
    if (res_len > 1)
    {
	int i;
	gchar *result = NULL;
	for (i = 0; i < res_len; ++i)
	{
	    char cbuf[6];
	    char sbuf[201];
	    gchar *lookup = braille_table_lookup_char (table, 
						       chars[i], 
						       context);
	    if (result)
	    {
		gchar *tmp = result;
		result = g_strconcat (result, lookup, NULL);
	        g_free (tmp);
	    }
	    else
	    {
		result = g_strdup (lookup);
	    }
	}
	return result;
    }
    return NULL;
}

static gchar *
braille_table_lookup_char (BrailleTable *table, gunichar ucs, BrailleContext **context)
{
    BrailleTable *tbl = table;
    GUnicodeBreakType type = g_unichar_break_type (ucs);
    if (type == G_UNICODE_BREAK_CARRIAGE_RETURN || type == G_UNICODE_BREAK_LINE_FEED) 
    {
	*context = BRAILLE_CONTEXT_ALPHA; /* should we always reset this way? */
	return "\n";
    }
    if (ucs >= 0x0000 && ucs <= 0x0000) /* pass-through any prebrailled characters */
    {
	return g_ucs4_to_utf8 (&ucs, 1, NULL, NULL, NULL);
    }
    do {
	gchar *decomposition = NULL;
	gint i = 0;
	while (i < tbl->n_blocks)
	{
	    UCSBlock *block = tbl->blocks[i];
	    if (ucs >= block->min && ucs <= block->max && 
		((block->context == BRAILLE_CONTEXT_ALL) || 
		 braille_context_match (block->context, *context)))
	    {
		if (block->out_context != BRAILLE_CONTEXT_ALL) 
		{
		    *context = block->out_context;
		}
		return block->data[ucs - block->min];
	    }
	    ++i;
	}
	/* try decomposing the character and then looking it up */
	if (decomposition = braille_table_decompose_and_lookup_char (table, ucs, context))
	{
	    /* FIXME: this introduces a memory management problem... we leak this string */
	    return decomposition;
	}
    } while ((tbl = tbl->alternate) != NULL);


    return table->out_of_range; /* we fell through and didn't find a match */
}

gboolean
match_string_func (gpointer kp, gpointer vp, gpointer mp)
{
    MatchStringKey *key = kp;
    MatchStringKey *match = mp;
    g_assert (match);
    g_assert (match->context);
    g_assert (match->context->name);

    return (g_str_has_prefix (match->string, key->string) && !strcmp (key->context->name, match->context->name));
}

static const gchar*
braille_table_match_string (BrailleTable *table, BrailleContext *context, const gchar **char_ptr, gint *char_index)
{
    /* TODO */
    MatchStringKey key = {*char_ptr, context};
    MatchStringValue *value = NULL;
    if (table->strings && (value = g_hash_table_find (table->strings, match_string_func, &key)))
    {
	*char_index += value->match_len;
	return value->string;
    }
    else
    {
	return NULL;
    }
}

static gboolean
braille_table_encode_to_buffer (BrailleTable *table, BrailleOutputBuffer *outbuf, const gchar **char_ptr, gint *char_index)
{
    static gint chunk_size = 1024;
    const gchar *cells = NULL;
    gunichar ucs;

    g_assert (char_index != NULL);
    g_assert (char_ptr != NULL);

    /* first we do substring/context matching, then try character lookup */
    if (!(cells = braille_table_match_string (table, outbuf->context, char_ptr, char_index)))
    {
	if (ucs = g_utf8_get_char (*char_ptr))
	{
	    *char_ptr = g_utf8_next_char (*char_ptr);
	    /* detect suffix and modify context if needed: usually a 1-char lookahead will do */
	    braille_table_check_suffix (table, (const gchar *) *char_ptr, outbuf);
	    cells = braille_table_lookup_char (table, ucs, &outbuf->context);
	    ++(* char_index);
	}
	else
	{
	    *char_ptr = NULL; /* we are at the end of the string, or hit some invalid UTF-8 */
	}
    }
    if (cells) /* we successfully converted one or more chars */
    {
	gint i, len;
	len = g_utf8_strlen (cells, -1);
	
	/* we're running out of room, and must reallocate */
	if ((outbuf->len + len) >= outbuf->size)
	{
	    g_message ("REALLOC");
	    outbuf->size += chunk_size;
	    outbuf->out_string = g_realloc (outbuf->out_string, outbuf->size);
	    outbuf->offsets = g_realloc (outbuf->offsets, sizeof (guint) * outbuf->size);
	}
	g_strlcat (outbuf->out_string, cells, outbuf->size);
	i = outbuf->len;
	outbuf->len += len;
	for (; i < outbuf->len; ++i) 
	    outbuf->offsets[i] = *char_index;
	return TRUE;
    }
    return FALSE;
}

static BrailleContext *
tok_to_context (gchar *tok)
{
    if (strlen (tok) < 1) 
    {
	return BRAILLE_CONTEXT_ALL;
    }
    else 
    {
	BrailleContext **contexts = braille_get_contexts ();
	gint i = 0;
	while (contexts[i]) 
	{
	    if (!strcmp (tok, contexts[i]->name)) {
		return contexts[i];
	    }
	    ++i;
	}
    }
    return braille_context_add_new (g_strdup (tok));
}

static gulong
tok_to_int (gchar *tok)
{
    guint base = 10;
    gchar *cp = tok;
    
    if (*tok == 'U' && (strlen (tok) > 2))
    {
	base = 16;
	cp = tok+2;
    }
    else if (!strncmp (tok, "0x", 2) && (strlen (tok) > 2))
    {
	base = 16;
	cp = tok+2;
    }
    else if (*tok == '\\')
    {
	base = 8;
	cp = tok+1;
    }
    else
    {
	base = 10;
    }
    
    return g_ascii_strtoull (cp, NULL, base);
}

static UCSBlock*
ucs_block_new (gulong min, gulong max, BrailleContext *in_context, BrailleContext *out_context)
{
    UCSBlock *block = g_new0 (UCSBlock, 1);
    block->min = min;
    block->max = max;
    block->data = g_malloc (sizeof (gchar *) * (max - min + 1));
    block->context = in_context;
    block->out_context = out_context;
    g_assert (in_context);
    g_assert (out_context);
    g_assert (in_context->name);
    g_assert (out_context->name);
    return block;
}

static void
braille_table_add_block (BrailleTable *table, UCSBlock *block)
{
    g_return_if_fail (table);

    table->n_blocks++;

    if (table->blocks)
	table->blocks = g_renew (UCSBlock *, table->blocks, table->n_blocks);
    else
	table->blocks = g_new (UCSBlock *, table->n_blocks);

    table->blocks[table->n_blocks-1] = block;
}

static void
braille_table_add_string (BrailleTable *table, gchar *in_string, gchar *out_string, BrailleContext *context)
{
    MatchStringKey *key;
    MatchStringValue *value;

    g_return_if_fail (table);

    if (table->strings)
    {
	table->strings = g_hash_table_new (NULL, NULL);
    }

    key = g_new0 (MatchStringKey, 1);
    value = g_new0 (MatchStringValue, 1);

    key->string = g_strdup (in_string);
    key->context = context;

    value->string = g_strdup (out_string);
    value->match_len = g_utf8_strlen (in_string, -1);

    g_hash_table_insert (table->strings, key, value);
}


static void
braille_table_add_suffix (BrailleTable *table, gchar *suffix_string, gchar *suffix_name)
{
    gint n_suffixes = 0;
    BrSuffix *suffix = g_new0 (BrSuffix, 1);

    g_return_if_fail (table);

    if (table->suffixes)
    {
	while (table->suffixes[n_suffixes]) n_suffixes++;
	table->suffixes = g_renew (BrSuffix *, table->suffixes, n_suffixes + 1);
    }
    else {
	n_suffixes = 1;
	table->suffixes = g_new0 (BrSuffix *, n_suffixes + 1);
    }

    suffix->s = suffix_string;
    suffix->name = suffix_name;
    suffix->context = tok_to_context (suffix_name);
    table->suffixes[n_suffixes - 1] = suffix;
    table->suffixes[n_suffixes] = NULL;
}

static void
braille_table_check_suffix (BrailleTable *table, const gchar *string, BrailleOutputBuffer *outbuf)
{
    if (table->suffixes) 
    {
	int i = 0;
	while (table->suffixes[i] && strlen (table->suffixes[i]->s))
	{
	    if (g_str_has_prefix (string, table->suffixes[i]->s))
	    {
		outbuf->context = table->suffixes[i]->context;
		g_message ("detected %s suffix\n", table->suffixes[i]->name);
		return;
	    }
	    ++i;
	}
    }

    return;
}


static void
string_list_add (gchar ***list, gchar *addstring)
{
    gint n_strings = 0;

    if (*list)
    {
	while ((*list)[n_strings]) n_strings++;
    }

    if (*list)
	*list = g_renew (gchar*, *list, n_strings + 2);
    else
	*list = g_new0 (gchar*, 2);

    (*list)[n_strings] = addstring;
    (*list)[n_strings+1] = NULL;
}

/* implementations of virtual funcs */
static void
braille_table_encoder_set_alternate (BrailleEncoder *encoder, BrailleEncoder *alternate)
{
    g_return_if_fail (BRAILLE_IS_TABLE_ENCODER (encoder));

    if (BRAILLE_TABLE_ENCODER (encoder)->alternate) 
	g_object_unref (BRAILLE_TABLE_ENCODER (encoder)->alternate);

    if (G_IS_OBJECT (alternate))
	g_object_ref (alternate);

    if (alternate != NULL)
	g_assert (BRAILLE_IS_ENCODER (alternate));

    BRAILLE_TABLE_ENCODER (encoder)->alternate = alternate;
}

static const gchar*
braille_table_encoder_translate_string (BrailleEncoder *encoder, const gchar *string, guint **offsets)
{
    BrailleTableEncoder *table_encoder;
    const gchar *encoded;

    g_return_if_fail (BRAILLE_IS_TABLE_ENCODER (encoder));

    table_encoder = BRAILLE_TABLE_ENCODER (encoder);

    if (table_encoder->table) {
	guint char_index = 0;
	gboolean chars_missing = FALSE;
	BrailleOutputBuffer *outbuf;
	BrailleContext *context = BRAILLE_CONTEXT_ALPHA; /* stores the context state */

	if (!g_utf8_validate (string, -1, NULL)) 
	{
	    g_warning ("Braille Encoder passed invalid UTF-8.");
	    return NULL;
	}

	outbuf = braille_output_buffer_new ();

	while (string)
	{
	    if (!braille_table_encode_to_buffer (table_encoder->table, outbuf, &string, &char_index))
	    {
		chars_missing = TRUE; /* maybe our delegate can fill in the blanks */
	    }
	}

	/* TODO: should hand off to delegate when lookup fails, doing a look-ahead first.    */
	/* translate_string() call should allow an unknown cell pattern to be passed.        */
	/* the handoff below is technically chaining, not delegation.                        */
        /* if necessary, hand this to our delegate for further processing, and fixup offsets */
	if (chars_missing && table_encoder->alternate != NULL) 
	{
	    guint *delegate_offsets;
	    encoded = braille_encoder_translate_string (table_encoder->alternate, outbuf->out_string, 
							offsets != NULL ? &delegate_offsets : NULL);
	    g_free (outbuf->out_string);
	    if (offsets != NULL)
	    {
		*offsets = braille_encoder_chain_offsets (outbuf->offsets, outbuf->len, delegate_offsets, encoded ? g_utf8_strlen (encoded, -1) : 0);
	    }
	}
	else
	{
	    encoded = outbuf->out_string;
	    if (offsets) 
		*offsets = outbuf->offsets;
	}
	g_free (outbuf);

	if (encoded && !g_utf8_validate (encoded, -1, NULL)) 
	{
	    g_warning ("error encoding braille: invalid UTF-8 generated");
	    return NULL;
	}
	return encoded;
    }
    else
    {
	g_warning ("Attempting to encode braille without a valid braille table!\n");
	return g_strdup (string);
    }
}

/*
 ************************************ 
 *
 * Braille Table utility functions 
 *
 ************************************
 */

/**
 * braille_table_construct_from_file:
 * @filename: the name of a file containing a braille encoding table.  If this is not
 * a fully-qualified pathname, the default braille table path will be used
 * (on gnome platforms, usually /usr/share/gnome-braille/)
 *
 * Return value: a #BrailleTable instance, or NULL if no table could be initialized from the 
 *               input file.
 **/
BrailleTable *braille_table_construct_from_file (gchar *filename)
{
    GIOChannel *io;

    if (filename)
    { 
	if (g_file_test (filename, G_FILE_TEST_EXISTS))
	    filename = g_strdup (filename);
	else 
	{
	    gchar *tmp = filename;
	    filename = g_build_filename (GNOME_BRAILLE_DATADIR, filename, NULL);
	    if (!g_file_test (filename, G_FILE_TEST_EXISTS))
	    {
		g_free (filename);
		filename = g_build_filename ("..", "tables", tmp, NULL);
	    }
	}
	if (g_file_test (filename, G_FILE_TEST_EXISTS))
	    io = g_io_channel_new_file (filename, "r", NULL);
	else 
	    g_warning ("file %s not found", filename);
	g_free (filename);
	if (io)
	    return braille_table_construct_from_stream (io);
    }
    return NULL;
}

/**
 * braille_table_construct_from_stream:
 * @io: a #GIOChannel whose contents can be parsed into a braille table.
 *
 * Return value: a #BrailleTable instance, or NULL if no table could be initialized from the 
 *               input stream.
 **/
BrailleTable *braille_table_construct_from_stream (GIOChannel *io)
{
    GError *error = NULL;
    GIOStatus status;
    GString *string = g_string_new (NULL);
    UCSBlock *block = NULL;
    BrailleTable *table = NULL;
    TableLineFormat line_format;
    gchar *encoding = NULL;
    gchar **tokens = NULL;
    gchar cbuf[7];
    gboolean in_block = FALSE;
    gint line;
    gint n_blocks = 0;

    braille_get_contexts (); /* initializes */
    status = g_io_channel_read_line_string (io, string, NULL, &error);
    if (status != G_IO_STATUS_NORMAL) g_warning ("I/O status=%d reading braille table", status);
    g_return_val_if_fail ((status == G_IO_STATUS_NORMAL || status == G_IO_STATUS_AGAIN), NULL);

    tokens = g_strsplit (string->str, " ", 2);
    if (tokens[0] && !strcmp (tokens[0], "ENCODING"))
    {
	encoding = tokens[1];
    }

    g_message ("channel encoding=%s; content encoding=%s", g_io_channel_get_encoding (io), tokens[0] ? tokens[1] : NULL);

    if (encoding && strcmp (g_io_channel_get_encoding (io), encoding))
    {
	g_io_channel_seek_position (io, 0, G_SEEK_SET, &error);
	g_io_channel_set_encoding (io, encoding, &error);
	g_io_channel_read_line_string (io, string, NULL, &error);
    }
    g_strfreev (tokens);

    table = g_new0 (BrailleTable, 1);
    /* TODO: explicit init func instead of relying on zeroing */

    line = 2;
    while ((status = g_io_channel_read_line_string (io, string, NULL, &error)) != G_IO_STATUS_EOF)
    {
	if (status == G_IO_STATUS_AGAIN) 
	{
	    g_message ("wait on reading Braille Table...");
	    continue;
	}
	else if (status == G_IO_STATUS_ERROR)
	{
	    g_io_channel_unref (io);
	    g_string_free (string, TRUE);
	    return NULL;
	}
	if (string->str && *string->str != '#') 
	{
	    g_strdelimit (string->str, "\n\t\r", ' ');
	    tokens = g_strsplit (string->str, " ", 20);
	    if (!strcmp (tokens[0], "NAME"))
	    {
		gint j = 1;
		gchar **nametokens = g_strsplit (string->str, "\"", 20);
		if (!nametokens[0]) 
		{
		    g_warning ("No names parsed in NAME field\n.");
		    continue;
		}
		while (nametokens[j])
		{
		    g_strstrip (nametokens[j]);
		    if (strlen (nametokens[j]) > 0)
		    {
			string_list_add (&table->names, g_strdup (nametokens[j]));
		    }
		    ++j;
		}
		g_strfreev (nametokens);
	    }
	    else if (!strcmp (tokens[0], "LOCALES"))
	    {
		gint j = 1;
		while (tokens[j])
		{
		    string_list_add (&table->locales, g_strdup (tokens[j]));
		    ++j;
		}
	    }
	    else if (!strcmp (tokens[0], "UCS-BLOCK"))
	    {
		if (!strcmp (tokens[1], "START"))
		{
		    BrailleContext *in_context = BRAILLE_CONTEXT_ALL, *out_context = BRAILLE_CONTEXT_ALL;
		    in_block = TRUE;
		    if (tokens[5]) 
		    {
			in_context = tok_to_context (tokens[5]);
			if (tokens[6]) 
			{
			    out_context = tok_to_context (tokens[6]);
			}
		    }
		    block = ucs_block_new (tok_to_int (tokens[2]), tok_to_int (tokens[3]), in_context, out_context);
		    braille_table_add_block (table, block);
		    if (!strcmp (tokens[4], "FORMAT-UNICODE"))
		    {
			line_format = TABLE_LINE_FORMAT_UNICODE;
		    }
		    else
		    {
			line_format = TABLE_LINE_FORMAT_UTF8;
		    }
		}
		else if (!strcmp (tokens[1], "END"))
		{
		    in_block = FALSE;
		}
	    }
	    else if (!strcmp (tokens[0], "UNICODE-CHAR"))
	    {
		/* create a one-character block */
		BrailleContext *in_context = BRAILLE_CONTEXT_ALL, *out_context = BRAILLE_CONTEXT_ALL;
		gulong ucs = tok_to_int (tokens[1]);
		if (tokens[3]) 
		{
		    in_context = tok_to_context (tokens[3]);
		    if (tokens[4]) 
		    {
			out_context = tok_to_context (tokens[4]);
		    }
		}
		block = ucs_block_new (ucs, ucs, in_context, out_context);
		cbuf[g_unichar_to_utf8 (tok_to_int (tokens[2]), cbuf)] = '\0';
		block->data[0] = g_strdup (cbuf);
		braille_table_add_block (table, block);
	    }
	    else if (!strcmp (tokens[0], "UCS-CHAR"))
	    {
		/* create a one-character block */
		BrailleContext *in_context = BRAILLE_CONTEXT_ALL, *out_context = BRAILLE_CONTEXT_ALL;
		gulong ucs = g_utf8_get_char (tokens[1]);
		if (tokens[3]) 
		{
		    in_context = tok_to_context (tokens[3]);
		    if (tokens[4]) 
		    {
			out_context = tok_to_context (tokens[4]);
		    }
		}
		block = ucs_block_new (ucs, ucs, in_context, out_context);
		block->data[0] = g_strdup (tokens[2]);
		braille_table_add_block (table, block);
	    }
	    else if (!strcmp (tokens[0], "UNKNOWN-CHAR"))
	    {
		table->out_of_range = g_strdup (tokens[1]);
	    }
	    else if (!strcmp (tokens[0], "UCS-SUFFIX"))
	    {
		if (tokens[1] && tokens[2])
		    braille_table_add_suffix (table, g_strdup (tokens[1]), g_strdup (tokens[2]));
	    }
	    else if (!strcmp (tokens[0], "UTF8-STRING"))
	    {
		BrailleContext *context = BRAILLE_CONTEXT_ALL;
		if (tokens[1] && tokens[2])
		{
		    if (tokens[3]) 
			context = tok_to_context (tokens[3]);
		    braille_table_add_string (table, g_strdup (tokens[1]), g_strdup (tokens[2]), context);
		}
	    }
	    else if (!strcmp (tokens[0], "DELEGATE"))
	    {
		/* FIXME: shouldn't use paths here */
		if (!strcmp (tokens[1], "FILE"))
		    table->alternate = braille_table_construct_from_file (tokens[2]);
	    }	    
	    else /* default assumption: line is utf8/utf8 pair within a code block */
	    {
		gint ucs, n;
		gchar *cells;
		gchar **celltokens;
		
		if (!in_block) g_warning ("Error reading braille table, line %d", line);
		g_return_val_if_fail (in_block, NULL);

		switch (line_format)
		{
		    case TABLE_LINE_FORMAT_UNICODE:
			/* read a comma-delimited list of unicode points, and concatenate to utf-8 */
			ucs = tok_to_int (tokens[0]);
			celltokens = g_strsplit (tokens[1], ",", 10);
			n = 0;
			cells = g_strdup ("");
			while (celltokens[n])
			{
			    cbuf[g_unichar_to_utf8 (tok_to_int (tokens[2]), cbuf)] = '\0';
			    gchar *tmp = g_strconcat (cells, cbuf, NULL);
			    g_free (cells);
			    cells = tmp;
			}
			g_strfreev (celltokens);
			break;
		    case TABLE_LINE_FORMAT_UTF8:
		    default:
			ucs = g_utf8_get_char (tokens[0]);
			cells = g_strdup (tokens[1]);
			break;
		}

		if (ucs <= block->max) {
		    block->data[ucs - block->min] = cells;
		}
	    }
	    g_strfreev (tokens);
	}
	else {
	    g_message ("comment on line %d %s", line, string->str);
	}
	++line;
    }

    g_string_free (string, TRUE);

    return table;
}
