/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include <gmf.h>
#include <libgmf/gnome-genericfactory.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>

typedef struct {
  char *name, *file;
  int fd;
  GSList *source_filters;
  gint fd_tag;
} DevInfo;

typedef struct {
  CORBA_ORB orb;
  CORBA_Environment *ev;
  GHashTable *devices;
  GMFTimeReference *timeref;
  GnomeGenericFactory *factory;
  char *default_device;
} RawMIDIInfo;

typedef struct {
  RawMIDIInfo *ai;
  char *devname;
  GMFFilter *filter;
} FiltInfo;

static void raw_midi_open_devices(RawMIDIInfo *ai, poptContext ctx);

static void raw_midi_create_filter(GnomeGenericFactory *factory,
				   const char *goad_id, GNOME_stringlist *params,
				   CORBA_Object *new_object, RawMIDIInfo *ai);

static void raw_midi_in_get_pipe(GMFFilter *filter,
				 GMF_Direction pDirection,
				 GMFPipe **pipe,
				 FiltInfo *fi);
static void raw_midi_out_get_pipe(GMFFilter *filter,
				  GMF_Direction pDirection,
				  GMFPipe **pipe,
				  FiltInfo *fi);
static gint raw_fd_handle(DevInfo *di, gint source, GdkInputCondition cond);

int main(int argc, char *argv[])
{
  CORBA_Environment ev;
  poptContext ctx;
  RawMIDIInfo appinfo;

  CORBA_exception_init(&ev);
  appinfo.ev = &ev;
  appinfo.default_device = NULL;
  appinfo.orb = gnome_CORBA_init_with_popt_table("raw-midi", VERSION,
						 &argc, argv, NULL, 0,
						 &ctx, GNORBA_INIT_SERVER_FUNC,
						 &ev);

  appinfo.factory = GNOME_GENERIC_FACTORY(gnome_generic_factory_new("gmf-filter-raw-midi-factory"));
  appinfo.devices = g_hash_table_new(g_str_hash, g_str_equal);
  appinfo.timeref = GMF_TIME_REFERENCE(gmf_time_reference_new());

  raw_midi_open_devices(&appinfo, ctx);

  gtk_signal_connect(GTK_OBJECT(appinfo.factory), "create_object",
		     GTK_SIGNAL_FUNC(raw_midi_create_filter), &appinfo);

  gtk_main();

  return 0;
}

static void
raw_midi_open_devices(RawMIDIInfo *ai, poptContext ctx)
{
  char *curarg;

  while((curarg = poptGetArg(ctx))) {
    char **pieces;
    DevInfo *newdev;

    newdev = g_new0(DevInfo, 1);

    pieces = g_strsplit(curarg, ":", 2);

    g_assert(pieces && pieces[0] && pieces[1]);

    newdev->name = pieces[0];
    newdev->file = pieces[1];
    g_free(pieces);

    newdev->fd = open(newdev->file, O_RDWR|O_NONBLOCK);
    g_assert(newdev->fd >= 0);

    g_hash_table_insert(ai->devices, newdev->name, newdev);

    if(!ai->default_device)
      ai->default_device = g_strdup(newdev->name);

    newdev->fd_tag = gdk_input_add(newdev->fd, GDK_INPUT_READ,
				   (GdkInputFunction)raw_fd_handle,
				   newdev);
  }
}

static void
raw_midi_device_add_filter(RawMIDIInfo *ai,
			   const char *devname,
			   FiltInfo *fi)
{
  DevInfo *di;

  di = g_hash_table_lookup(ai->devices, devname);

  g_assert(di);

  di->source_filters = g_slist_append(di->source_filters, fi);
}

static int
raw_filter_get_fd(FiltInfo *fi)
{
  DevInfo *di;

  if(!fi->devname)
    fi->devname = g_strdup(fi->ai->default_device);

  di = g_hash_table_lookup(fi->ai->devices, fi->devname);
  g_return_val_if_fail(di, -1);
  g_return_val_if_fail(di->fd >= 0, -1);

  return di->fd;
}

static gint
gmf_midi_event_len(guchar *data, gulong data_len)
{
  guint retval = 0;
  guchar first_byte;

  g_return_val_if_fail(data_len > 0, -1);

  first_byte = data[0];

  switch(first_byte & 0xF0) {
  case 0xF0:
    switch(first_byte) {
    case 0xF7:
    case 0xF8:
    case 0xF9:
    case 0xFA:
    case 0xFB:
    case 0xFC:
    case 0xFE:
    case 0xFF:
      goto onebyteop;
    case 0xF1:
    case 0xF3:
      goto twobyteop;
    case 0xF0:
      { /* handle the dreadful sysex */
	int i;
	for(i = 1;
	    (i < data_len) && (data[i] & 0x80);
	    i++) /**/ ;
	
	if(data[i] & 0x80) {

	  if(data[i] == 0xF7)
	    i++; /* If they put an EOX byte in, then it
		    is part of the current message */

	  return i;
	} else
	  return -1;
      }
    break;
    default:
      break;
    }
  case 0x90:
  case 0x80:
  case 0xA0:
  case 0xB0:
  case 0xD0:
  case 0xE0:
    retval++;
  twobyteop:
  case 0xC0:
    retval++;
  onebyteop:
  default:
    retval++;
  }

  if(retval <= data_len)
    return retval;
  else
    return -1;
}

static gint
raw_fd_handle(DevInfo *di, gint source, GdkInputCondition cond)
{
  GSList *ltmp;
  FiltInfo *fi;
  GMF_Sample sample;
  static unsigned char inbuf[2048];
  static unsigned short inbuf_len = 0;
  RawMIDIInfo *ai;
  int nread, evlen, totalevlen;

  nread = read(source, inbuf + inbuf_len, sizeof(inbuf) - inbuf_len);

  g_return_val_if_fail(nread > 0, TRUE);

  if(!di->source_filters)
    return TRUE;

  inbuf_len += nread;

  for(totalevlen = evlen = 0; totalevlen < inbuf_len; totalevlen += evlen) {
    evlen = gmf_midi_event_len(inbuf + totalevlen, inbuf_len);
    if(evlen <= 0) break;
  }

  if(totalevlen > 0
     && totalevlen <= inbuf_len) {
    ai = ((FiltInfo *)di->source_filters->data)->ai;

    sample.sInfo.streamStartTime = sample.sInfo.streamEndTime =
      sample.sInfo.mediaStartTime = sample.sInfo.mediaEndTime =
      gmf_time_reference_current_time(ai->timeref);
    sample.sInfo.isSyncPoint = sample.sInfo.isPreroll = sample.sInfo.isDiscontinuity = CORBA_FALSE;
    sample.sInfo.actualDataLength = totalevlen;
    sample.sData._length = totalevlen;
    sample.sData._buffer = inbuf;

    for(ltmp = di->source_filters; ltmp; ltmp = g_slist_next(ltmp)) {
      fi = ltmp->data;
      
      gmf_filter_send_sample(fi->filter, &sample);
    }

    memmove(inbuf, inbuf+totalevlen, inbuf_len - totalevlen);
    inbuf_len = inbuf_len - totalevlen;
  }

  return TRUE;
}

static void
raw_midi_device_remove_filter(RawMIDIInfo *ai,
			      const char *devname,
			      FiltInfo *fi)
{
  DevInfo *di;

  di = g_hash_table_lookup(ai->devices, devname);

  g_assert(di);

  di->source_filters = g_slist_remove(di->source_filters, fi);
}


static void
raw_midi_create_filter(GnomeGenericFactory *factory,
		       const char *goad_id, GNOME_stringlist *params,
		       CORBA_Object *new_object, RawMIDIInfo *ai)
{
  GMFFilter *new_filter;
  GMF_Filter_Type ftype;
  FiltInfo *fi;

  if(!strcmp(goad_id, "gmf-filter-raw-midi-in")) {
    ftype = GMF_Filter_SOURCE;
  } else if(!strcmp(goad_id, "gmf-filter-raw-midi-out")) {
    ftype = GMF_Filter_RENDERER;
  } else {
    return; /* Can't activate anything else */
  }

  new_filter = GMF_FILTER(gmf_filter_new(ftype, goad_id));
  g_assert(new_filter);

  fi = g_new0(FiltInfo, 1);
  fi->ai = ai;
  if(!strcmp(goad_id, "gmf-filter-raw-midi-in")) {
    gtk_signal_connect(GTK_OBJECT(new_filter), "get_pipe",
		       GTK_SIGNAL_FUNC(raw_midi_in_get_pipe), fi);
    raw_filter_get_fd(fi); /* Do devname setup */
    raw_midi_device_add_filter(fi->ai, fi->devname, fi);
  } else if(!strcmp(goad_id, "gmf-filter-raw-midi-out")) {
    gtk_signal_connect(GTK_OBJECT(new_filter), "get_pipe",
		       GTK_SIGNAL_FUNC(raw_midi_out_get_pipe), fi);
  } else
    g_assert(!"Not reached!");

  fi->filter = new_filter;

  *new_object = CORBA_Object_duplicate(new_filter->corba_object, ai->ev);
}

static void raw_pipe_can_process(GMFPipe *pipe, GMF_MediaTypeInfo *type_info,
				 gboolean *retval)
{
  if(type_info->majorType != GMF_MEDIA_AUDIO_SCENE)
    return;

  if(strcmp(type_info->minorType, "audio/midi"))
    return;

  *retval = TRUE;
}

static void raw_pipe_processable_types(GMFPipe *pipe,
				       GMF_MediaTypeInfoList **out_typelist,
				       FiltInfo *fi)
{
  GMF_MediaTypeInfoList *list;
  GMF_MediaTypeInfo *ti;

  list = GMF_MediaTypeInfoList__alloc();
  list->_length = 1;
  list->_buffer = CORBA_sequence_GMF_MediaTypeInfo_allocbuf(list->_length);

  ti = &list->_buffer[0];
  {
    memset(ti, 0, sizeof(*ti));
    ti->majorType = GMF_MEDIA_AUDIO_SCENE;
    ti->minorType = CORBA_string_dup("audio/midi");
    ti->typeData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_null, fi->ai->ev);
    ti->formatData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_null, fi->ai->ev);
  }

  *out_typelist = list;
}

static void raw_pipe_out_handle_sample(GMFPipe *pipe,
				       GMF_Sample *sample,
				       gboolean must_copy,
				       FiltInfo *fi)
{
  int fd;

  fd = raw_filter_get_fd(fi);

  g_return_if_fail(fd >= 0);

  write(fd, sample->sData._buffer, sample->sData._length);
}

static void raw_midi_in_get_pipe(GMFFilter *filter,
				 GMF_Direction pDirection,
				 GMFPipe **pipe,
				 FiltInfo *fi)
{
  GMFPipe *newpipe;

  g_return_if_fail(pDirection == GMF_OUT); /* We only send data out */

  newpipe = GMF_PIPE(gmf_pipe_new(filter, pDirection,
				  GMF_Transport_UNIX_SOCKETS|GMF_Transport_CORBA));

  g_assert(newpipe);

  gtk_signal_connect(GTK_OBJECT(newpipe), "can_process_type",
		     GTK_SIGNAL_FUNC(raw_pipe_can_process), fi);
  gtk_signal_connect(GTK_OBJECT(newpipe), "get_processable_types",
		     GTK_SIGNAL_FUNC(raw_pipe_processable_types), fi);

  *pipe = newpipe;
}

static void raw_midi_out_get_pipe(GMFFilter *filter,
				  GMF_Direction pDirection,
				  GMFPipe **pipe,
				  FiltInfo *fi)
{
  GMFPipe *newpipe;

  g_return_if_fail(pDirection == GMF_IN); /* We only allow data to be fed in */

  newpipe = GMF_PIPE(gmf_pipe_new(filter, pDirection,
				  GMF_Transport_UNIX_SOCKETS|GMF_Transport_CORBA));

  g_assert(newpipe);

  gtk_signal_connect(GTK_OBJECT(newpipe), "can_process_type",
		     GTK_SIGNAL_FUNC(raw_pipe_can_process), fi);
  gtk_signal_connect(GTK_OBJECT(newpipe), "get_processable_types",
		     GTK_SIGNAL_FUNC(raw_pipe_processable_types), fi);

  gtk_signal_connect(GTK_OBJECT(newpipe), "receive_sample",
		     GTK_SIGNAL_FUNC(raw_pipe_out_handle_sample), fi);

  *pipe = newpipe;
}

