/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include <gmf.h>
#include <libgmf/gnome-genericfactory.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>

#define NNOTES 2

typedef struct {
  char *name, *file;
  int fd;
  GSList *source_filters;
  gint fd_tag;
} DevInfo;

typedef struct {
  CORBA_ORB orb;
  CORBA_Environment *ev;
  GHashTable *devices;
  GMFTimeReference *timeref;
  GnomeGenericFactory *factory;
  char *default_device;
} RandomMIDIInfo;

typedef struct {
  RandomMIDIInfo *ai;
  char *devname;
  GMFFilter *filter;
  guint timer_id;
  GList *noteons;
} FiltInfo;

static void random_midi_create_filter(GnomeGenericFactory *factory,
				      const char *goad_id, GNOME_stringlist *params,
				      CORBA_Object *new_object, RandomMIDIInfo *ai);

static void random_midi_in_get_pipe(GMFFilter *filter,
				    GMF_Direction pDirection,
				    GMFPipe **pipe,
				    FiltInfo *fi);
static void random_midi_out_get_pipe(GMFFilter *filter,
				     GMF_Direction pDirection,
				     GMFPipe **pipe,
				     FiltInfo *fi);
static gint random_midi_gen(FiltInfo *fi);

int main(int argc, char *argv[])
{
  CORBA_Environment ev;
  poptContext ctx;
  RandomMIDIInfo appinfo;

  CORBA_exception_init(&ev);
  appinfo.ev = &ev;
  appinfo.orb = gnome_CORBA_init_with_popt_table("random-midi", VERSION,
						 &argc, argv, NULL, 0,
						 &ctx, GNORBA_INIT_SERVER_FUNC,
						 &ev);

  appinfo.factory = GNOME_GENERIC_FACTORY(gnome_generic_factory_new("gmf-filter-random-midi-factory"));
  appinfo.devices = g_hash_table_new(g_str_hash, g_str_equal);
  appinfo.timeref = GMF_TIME_REFERENCE(gmf_time_reference_new());

  gtk_signal_connect(GTK_OBJECT(appinfo.factory), "create_object",
		     GTK_SIGNAL_FUNC(random_midi_create_filter), &appinfo);

  gtk_main();

  return 0;
}

static gint
gmf_midi_event_len(guchar *data, gulong data_len)
{
  guint retval = 0;
  guchar first_byte;

  g_return_val_if_fail(data_len > 0, -1);

  first_byte = data[0];

  switch(first_byte & 0xF0) {
  case 0xF0:
    switch(first_byte) {
    case 0xF7:
    case 0xF8:
    case 0xF9:
    case 0xFA:
    case 0xFB:
    case 0xFC:
    case 0xFE:
    case 0xFF:
      goto onebyteop;
    case 0xF1:
    case 0xF3:
      goto twobyteop;
    case 0xF0:
      { /* handle the dreadful sysex */
	int i;
	for(i = 1;
	    (i < data_len) && (data[i] & 0x80);
	    i++) /**/ ;
	
	if(data[i] & 0x80) {

	  if(data[i] == 0xF7)
	    i++; /* If they put an EOX byte in, then it
		    is part of the current message */

	  return i;
	} else
	  return -1;
      }
    break;
    default:
      break;
    }
  case 0x90:
  case 0x80:
  case 0xA0:
  case 0xB0:
  case 0xD0:
  case 0xE0:
    retval++;
  twobyteop:
  case 0xC0:
    retval++;
  onebyteop:
  default:
    retval++;
  }

  if(retval <= data_len)
    return retval;
  else
    return -1;
}

static void
random_midi_create_filter(GnomeGenericFactory *factory,
		       const char *goad_id, GNOME_stringlist *params,
		       CORBA_Object *new_object, RandomMIDIInfo *ai)
{
  GMFFilter *new_filter;
  GMF_Filter_Type ftype;
  FiltInfo *fi;

  if(!strcmp(goad_id, "gmf-filter-random-midi-in")) {
    ftype = GMF_Filter_SOURCE;
  } else if(!strcmp(goad_id, "gmf-filter-random-midi-out")) {
    ftype = GMF_Filter_RENDERER;
  } else {
    return; /* Can't activate anything else */
  }

  new_filter = GMF_FILTER(gmf_filter_new(ftype, goad_id));
  g_assert(new_filter);

  fi = g_new0(FiltInfo, 1);
  fi->ai = ai;
  if(!strcmp(goad_id, "gmf-filter-random-midi-in")) {
    gtk_signal_connect(GTK_OBJECT(new_filter), "get_pipe",
		       GTK_SIGNAL_FUNC(random_midi_in_get_pipe), fi);
    fi->timer_id = gtk_timeout_add(1000, (GtkFunction)random_midi_gen, fi);
  } else if(!strcmp(goad_id, "gmf-filter-random-midi-out")) {
    gtk_signal_connect(GTK_OBJECT(new_filter), "get_pipe",
		       GTK_SIGNAL_FUNC(random_midi_out_get_pipe), fi);
  } else
    g_assert(!"Not reached!");

  fi->filter = new_filter;

  *new_object = CORBA_Object_duplicate(new_filter->corba_object, ai->ev);
}

static void raw_pipe_can_process(GMFPipe *pipe, GMF_MediaTypeInfo *type_info,
				 gboolean *retval)
{
  if(type_info->majorType != GMF_MEDIA_AUDIO_SCENE)
    return;

  if(strcmp(type_info->minorType, "audio/midi"))
    return;

  *retval = TRUE;
}

static void raw_pipe_processable_types(GMFPipe *pipe,
				       GMF_MediaTypeInfoList **out_typelist,
				       FiltInfo *fi)
{
  GMF_MediaTypeInfoList *list;
  GMF_MediaTypeInfo *ti;

  list = GMF_MediaTypeInfoList__alloc();
  list->_length = 1;
  list->_buffer = CORBA_sequence_GMF_MediaTypeInfo_allocbuf(list->_length);

  ti = &list->_buffer[0];
  {
    memset(ti, 0, sizeof(*ti));
    ti->majorType = GMF_MEDIA_AUDIO_SCENE;
    ti->minorType = CORBA_string_dup("audio/midi");
    ti->typeData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_null, fi->ai->ev);
    ti->formatData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_null, fi->ai->ev);
  }

  *out_typelist = list;
}

static void raw_pipe_out_handle_sample(GMFPipe *pipe,
				       GMF_Sample *sample,
				       gboolean must_copy,
				       FiltInfo *fi)
{
  unsigned char *curptr, *endptr;
  unsigned int elen, i;

  curptr = sample->sData._buffer;
  endptr = curptr + sample->sData._length;

  g_print("Pipe %p:\n", pipe);

  while(curptr < endptr) {
    g_print("  ");
    elen = gmf_midi_event_len(curptr, endptr - curptr);

    for(i = 0; i < elen; i++)
      g_print("%.2x ", curptr[i]);

    g_print("\n");

    curptr += elen;
  }
}

static void random_midi_in_get_pipe(GMFFilter *filter,
				 GMF_Direction pDirection,
				 GMFPipe **pipe,
				 FiltInfo *fi)
{
  GMFPipe *newpipe;

  g_return_if_fail(pDirection == GMF_OUT); /* We only send data out */

  newpipe = GMF_PIPE(gmf_pipe_new(filter, pDirection,
				  GMF_Transport_UNIX_SOCKETS|GMF_Transport_CORBA));

  g_assert(newpipe);

  gtk_signal_connect(GTK_OBJECT(newpipe), "can_process_type",
		     GTK_SIGNAL_FUNC(raw_pipe_can_process), fi);
  gtk_signal_connect(GTK_OBJECT(newpipe), "get_processable_types",
		     GTK_SIGNAL_FUNC(raw_pipe_processable_types), fi);

  *pipe = newpipe;
}

static void random_midi_out_get_pipe(GMFFilter *filter,
				  GMF_Direction pDirection,
				  GMFPipe **pipe,
				  FiltInfo *fi)
{
  GMFPipe *newpipe;

  g_return_if_fail(pDirection == GMF_IN); /* We only allow data to be fed in */

  newpipe = GMF_PIPE(gmf_pipe_new(filter, pDirection,
				  GMF_Transport_UNIX_SOCKETS|GMF_Transport_CORBA));

  g_assert(newpipe);

  gtk_signal_connect(GTK_OBJECT(newpipe), "can_process_type",
		     GTK_SIGNAL_FUNC(raw_pipe_can_process), fi);
  gtk_signal_connect(GTK_OBJECT(newpipe), "get_processable_types",
		     GTK_SIGNAL_FUNC(raw_pipe_processable_types), fi);

  gtk_signal_connect(GTK_OBJECT(newpipe), "receive_sample",
		     GTK_SIGNAL_FUNC(raw_pipe_out_handle_sample), fi);

  *pipe = newpipe;
}

/* Every time we turn a note on, we add it to the list. */
static gint
random_midi_gen(FiltInfo *fi)
{
  int noteon;
  unsigned char noteon_data[6];
  GMF_Sample sample;

  sample.sInfo.streamStartTime = sample.sInfo.streamEndTime =
    sample.sInfo.mediaStartTime = sample.sInfo.mediaEndTime =
    gmf_time_reference_current_time(fi->ai->timeref);
  sample.sInfo.isSyncPoint = sample.sInfo.isPreroll = sample.sInfo.isDiscontinuity = CORBA_FALSE;
  sample.sData._length = 0;
  sample.sData._buffer = noteon_data;

  /* add new note */
  noteon = (rand()%64)+32;

  fi->noteons = g_list_append(fi->noteons, GUINT_TO_POINTER(noteon));

  noteon_data[sample.sData._length++] = 0x90;
  noteon_data[sample.sData._length++] = noteon;
  noteon_data[sample.sData._length++] = 64;

  if(g_list_length(fi->noteons) > NNOTES) {
    GList *ltmp;

    ltmp = fi->noteons;
    noteon = GPOINTER_TO_UINT(fi->noteons->data);
    fi->noteons = g_list_remove_link(fi->noteons, ltmp);

    g_list_free_1(ltmp);

    noteon_data[sample.sData._length++] = 0x80;
    noteon_data[sample.sData._length++] = noteon;
    noteon_data[sample.sData._length++] = 64;
  }

  sample.sInfo.actualDataLength = sample.sData._length;
  gmf_filter_send_sample(fi->filter, &sample);

  return TRUE;
}
