#include "gdseditor.h"

//#define GENEROUS_PARSE 1
#define EXTRA_PARSE 80

static void gds_editor_init(GdsEditor *gds_editor);
static void gds_editor_class_init(GdsEditorClass *klass);
GtkWidget *gds_editor_new();
static void gds_editor_destroy(GtkObject *object);
static gint key_press_cb(GtkWidget *widget, GdkEventKey *event, GdsEditor *gds_editor);
static void property_mark(GtkExText *text, GtkExTextProperty *prop);
gint property_text_remove(GtkExText *text, GtkExTextProperty *prop, gint start, gint end);
gint property_text_insert(GtkExText *text, GtkExTextProperty *prop, gint start, gint end);
void check_pattern(GtkExText *text, gint start, gint end, GtkExTextProperty *prev);
void check_embedded(GtkExText *text, gint start, gint end, GtkExTextProperty *prev);
gint get_syntax_end(GtkExText *text, gint pos, GdsEditorSyntaxEntry *table, GtkExTextMatch *m);
void check_syntax(GtkExText *text, gint i, gint end);
static void gds_editor_dnd_drop(GdsEditor *editor, GdkDragContext *context, gint x, gint y, GtkSelectionData *selection_data, guint info, guint time, gpointer data);

enum
{
   TARGET_COLOR
};
static GtkTargetEntry drop_types[] =
{
   { "application/x-color", 0, TARGET_COLOR }
};
static gint n_drop_types = sizeof(drop_types) / sizeof(drop_types[0]);

static GtkWidgetClass *parent_class = NULL;

GtkType gds_editor_get_type(void)
{
   static GtkType gds_editor_type=0;
   if(!gds_editor_type)
   {
      static const GtkTypeInfo gds_editor_info = 
      {	
         "GdsEditor",
         sizeof(GdsEditor),
         sizeof(GdsEditorClass),
         (GtkClassInitFunc) gds_editor_class_init,
         (GtkObjectInitFunc) gds_editor_init,
         NULL,
         NULL,
         (GtkClassInitFunc)NULL,
      };
      gds_editor_type = gtk_type_unique(GTK_TYPE_EXTEXT, &gds_editor_info);
   }
   return(gds_editor_type);
}

static void gds_editor_class_init(GdsEditorClass *klass)
{
   GtkObjectClass *object_class;
   object_class = (GtkObjectClass*)klass;
   parent_class = gtk_type_class(GTK_TYPE_EXTEXT);
   object_class->destroy = gds_editor_destroy;
}

static void gds_editor_init(GdsEditor *gds_editor)
{
   gds_editor->changed = FALSE;
   gds_editor->highlight = FALSE;
}

GtkWidget* gds_editor_new()
{
   GdsEditor *gds_editor;
   GtkObject *hadj;
   hadj = gtk_adjustment_new(0, 0, 256, 20, 1, 80);
   /*
      Commented here for switch to new GtkExText
      gds_editor = (GdsEditor *) gtk_widget_new(GDS_TYPE_EDITOR, "hadjustment", hadj, "vadjustment", NULL, NULL);
   */
   gds_editor = (GdsEditor *) gtk_widget_new(GDS_TYPE_EDITOR, NULL);
   gtk_signal_connect(GTK_OBJECT(gds_editor), "property_text_remove", GTK_SIGNAL_FUNC(property_text_remove), NULL);
   gtk_signal_connect(GTK_OBJECT(gds_editor), "property_text_insert", GTK_SIGNAL_FUNC(property_text_insert), NULL);
   gtk_signal_connect(GTK_OBJECT(gds_editor), "property_mark", GTK_SIGNAL_FUNC(property_mark), NULL);
   gtk_signal_connect(GTK_OBJECT(gds_editor), "key_press_event", GTK_SIGNAL_FUNC(key_press_cb), gds_editor);
   gtk_signal_connect(GTK_OBJECT(gds_editor), "changed", GTK_SIGNAL_FUNC(gds_editor_set_changed), 0);

   gtk_drag_dest_set(GTK_WIDGET(gds_editor), (GtkDestDefaults) (GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP), drop_types, n_drop_types, GDK_ACTION_COPY);
   gtk_signal_connect(GTK_OBJECT(gds_editor), "drag_data_received", GTK_SIGNAL_FUNC(gds_editor_dnd_drop), gds_editor);
   return GTK_WIDGET(gds_editor);
}

static void gds_editor_destroy(GtkObject *object)
{
   GdsEditor *gds_editor;
   g_return_if_fail(object != NULL);
   g_return_if_fail(GDS_IS_EDITOR(object));
   gds_editor = GDS_EDITOR(object);
	GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

void gds_editor_set_changed(GdsEditor *gds_editor, gpointer data)
{
   g_return_if_fail(gds_editor != NULL);
   g_return_if_fail(GDS_IS_EDITOR(gds_editor));
   if(GTK_EXTEXT(gds_editor)->editable)
      gds_editor->changed++;
   gtk_signal_emit_by_name(GTK_OBJECT(gds_editor), "move_to_column", 0, NULL);
}

gboolean gds_editor_changed(GdsEditor *gds_editor)
{
   g_return_val_if_fail(gds_editor != NULL, FALSE);
   g_return_val_if_fail(GDS_IS_EDITOR(gds_editor), FALSE);
   if(gds_editor->changed) return(TRUE);
   return(FALSE);
}

void gds_editor_set_style(GdsEditor *gds_editor, GtkStyle *style)
{
   g_return_if_fail(gds_editor != NULL);
   g_return_if_fail(GDS_IS_EDITOR(gds_editor));
   GTK_WIDGET(gds_editor)->style = style;
   gtk_extext_style_set(GTK_WIDGET(gds_editor), NULL);
   gtk_extext_style_insert(GTK_EXTEXT(gds_editor), "Default", NULL, &style->text[GTK_STATE_NORMAL], NULL, 0);
}

void gds_editor_set_highlight(GdsEditor *text, gboolean value)
{
   g_return_if_fail(text != NULL);
   g_return_if_fail(GDS_IS_EDITOR(text));
  
   text->highlight = value;
   gtk_extext_property_remove_all(GTK_EXTEXT(text), 0, GTK_EXTEXT(text)->length, NULL);
   if(text->highlight) check_syntax(GTK_EXTEXT(text), 0, GTK_EXTEXT(text)->length);
}

void gds_editor_highlight(GdsEditor *text)
{
   if(text->highlight)
   {
      gtk_extext_freeze(GTK_EXTEXT(text));
      gtk_extext_property_remove_all(GTK_EXTEXT(text), 0, GTK_EXTEXT(text)->length, NULL);
      check_syntax(GTK_EXTEXT(text), 0, GTK_EXTEXT(text)->length);
      gtk_extext_thaw(GTK_EXTEXT(text));
   }
}

static gint key_press_cb(GtkWidget *widget, GdkEventKey *event, GdsEditor *gds_editor)
{
   GtkExText *text;
   gint curpos;
   gboolean handled = FALSE;
   gboolean shift_state;
   gboolean control_state;
   gint key;
   
   shift_state = event->state & GDK_SHIFT_MASK;
   control_state = event->state & GDK_CONTROL_MASK;  
   key = event->keyval;
   
   text = (GtkExText *)widget;
   curpos = gtk_extext_get_position(text);

   if(event->keyval == 'x' && control_state && gtk_extext_get_editable(text))
   {
      gtk_extext_cut_clipboard(text);
      text->has_selection = FALSE;
      text->selection_start_pos = -1;
      text->selection_end_pos = -1;
      handled = TRUE;
   }
   else if(event->keyval == 'z' && control_state  && gtk_extext_get_editable(text))
   {
      text->has_selection = FALSE;
      text->selection_start_pos = -1;
      text->selection_end_pos = -1;
      if(!gtk_extext_undo_is_empty(text))
      {
         gtk_extext_undo(text);
         gds_editor->changed -= 2;
         gtk_signal_emit_by_name(GTK_OBJECT(gds_editor), "move_to_column", 0, NULL);
      }
      handled = TRUE;
   }
   else if(event->keyval == 'r' && control_state && gtk_extext_get_editable(text))
   {
      text->has_selection = FALSE;
      text->selection_start_pos = -1;
      text->selection_end_pos = -1;
      if(!gtk_extext_redo_is_empty(text))
      {
         gtk_extext_redo(text);
         gtk_signal_emit_by_name(GTK_OBJECT(gds_editor), "move_to_column", 0, NULL);
      }
      handled = TRUE;
   }
   if(handled) gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
   return(handled);
}

gboolean gds_editor_compile_regex(const gchar *pattern, Regex *regex)
{
   memset (&regex->buf, 0, sizeof(regex->buf));
   regex->buf.translate = NULL;
   regex->buf.fastmap = g_malloc (256);
   regex->buf.allocated = 0;
   regex->buf.buffer = NULL;
   regex->buf.can_be_null = 0;	/* so we wont allow that in patterns! */
   regex->buf.no_sub = 0;
   if(re_compile_pattern(pattern, strlen(pattern), &regex->buf) == 0)
   {
      if(re_compile_fastmap(&regex->buf) != 0)
      {
         g_warning("IMPORTANT REGEX FAILED TO CREATE FASTMAP\n");
         g_free (regex->buf.fastmap);
         regex->buf.fastmap = NULL;
      }
      return TRUE;
   }
   else
   {
      g_warning("IMPORTANT REGEX FAILED TO COMPILE\n");
      return FALSE;
   }
}

void property_mark(GtkExText *text,GtkExTextProperty *prop)
{
   if(!prop) return;
}

gint property_text_remove(GtkExText *text, GtkExTextProperty *prop, gint start, gint end)
{
   GtkExTextLineData *slinedataptr, *elinedataptr;
   gint diff, e;
   g_return_val_if_fail(text != NULL, FALSE);
   g_return_val_if_fail(GTK_IS_EXTEXT(text), FALSE);
   if(!GDS_EDITOR(text)->highlight) return FALSE;

   if(end > text->length)
   {
      gtk_extext_property_remove_all(text, text->length, end, NULL);
      end=text->length;
   }  
   diff = end - start;
   e = end;
   /* If the whole text chunk was inside a highlight, parse from prop->start to prop->end */
   if((prop = gtk_extext_property_get_at_pos(text, start, prop)) && ((gint)prop->startpos <= start) && ((gint)(prop->endpos-diff) > start))
   {
      start = prop->startpos;
      end = prop->endpos-diff > end ? prop->endpos-diff : end;
   }
   /* If the text chunk was after a SYNTAX_TABLE, parse from prop->start to line->end */
   else if((prop = gtk_extext_property_nearest_backward(text, start, prop)) && GPOINTER_TO_INT(prop->user_data) == SYNTAX_TABLE)
   {
      start = prop->startpos;
      elinedataptr = gtk_extext_get_line_by_char_pos(text, end, NULL);
      end = elinedataptr && elinedataptr->endpos > end ? elinedataptr->endpos : end;
      g_free(elinedataptr);
   }
   /* Else, parse from line1->start to line2->end*/
   else
   {
#ifdef GENEROUS_PARSE
      slinedataptr = gtk_extext_get_line_by_char_pos(text, start-EXTRA_PARSE > 0 ? start-EXTRA_PARSE : 0, NULL);
#else
      slinedataptr = gtk_extext_get_line_by_char_pos(text, start, NULL);
#endif
      start = slinedataptr && slinedataptr->startpos < start ? slinedataptr->startpos : start;
      g_free(slinedataptr);
#ifdef GENEROUS_PARSE
      elinedataptr = gtk_extext_get_line_by_char_pos(text, end+EXTRA_PARSE < text->length ? end+EXTRA_PARSE : text->length, NULL);
#else
      elinedataptr = gtk_extext_get_line_by_char_pos(text, end, NULL);
#endif
      end = elinedataptr && elinedataptr->endpos > end ? elinedataptr->endpos : end;
      g_free(elinedataptr);
   }
   //g_print("Remove Highlight - (%d, %d)\n", start, end);
   gtk_extext_property_move_all(text, e, -diff, prop);
   gtk_extext_property_remove_all(text, start, end, prop);
   check_syntax(text, start, end);
   return(TRUE);
}

gint property_text_insert(GtkExText *text, GtkExTextProperty *prop, gint start, gint end)
{
   gint diff, s;
   GtkExTextLineData *slinedataptr, *elinedataptr;
   g_return_val_if_fail(text != NULL, FALSE);
   g_return_val_if_fail(GDS_IS_EDITOR(text), FALSE);
   if(!GDS_EDITOR(text)->highlight) return(FALSE);

   s = start;
   diff = end - start;
   if((prop = gtk_extext_property_get_at_pos(text, start, prop)) && ((gint)prop->startpos < start) && ((gint)(prop->endpos+diff) >= end))
   {
      start = prop->startpos;
      end = prop->endpos+diff < text->length ? prop->endpos+diff : text->length;
   }
   else if((prop = gtk_extext_property_nearest_backward(text, start, prop)) && GPOINTER_TO_INT(prop->user_data) == SYNTAX_TABLE)
   {
      start = prop->startpos;
      elinedataptr = gtk_extext_get_line_by_char_pos(text, end, NULL);
      end = elinedataptr && elinedataptr->endpos > end ? elinedataptr->endpos : end;
      g_free(elinedataptr);
   }
   else
   {
#ifdef GENEROUS_PARSE
      slinedataptr = gtk_extext_get_line_by_char_pos(text, start-EXTRA_PARSE > 0 ? start-EXTRA_PARSE : 0, NULL);
#else
      slinedataptr = gtk_extext_get_line_by_char_pos(text, start, NULL);
#endif
      start = slinedataptr && slinedataptr->startpos < start ? slinedataptr->startpos : start;
      g_free(slinedataptr);
#ifdef GENEROUS_PARSE
      elinedataptr = gtk_extext_get_line_by_char_pos(text, end+EXTRA_PARSE < text->length ? end+EXTRA_PARSE : text->length, NULL);
#else
      elinedataptr = gtk_extext_get_line_by_char_pos(text, end, NULL);
#endif
      end = elinedataptr && elinedataptr->endpos > end ? elinedataptr->endpos : end;
      g_free(elinedataptr);
   }
   //g_print("Insert Highlight - (%d, %d)\n", start, end);
   gtk_extext_property_move_all(text, s, diff, prop);
   gtk_extext_property_remove_all(text, start, end, prop);
   check_syntax(text, start, end);
   return(TRUE);
}

void check_syntax(GtkExText *text, gint start, gint end)
{
   gint s, i, j;
   gboolean found = FALSE;
   GList *list;
   GtkExTextMatch m;
   GtkExTextProperty *prev;
   GdsEditorHighlightTable *table;
   GdsEditorSyntaxEntry *sentry;
   table = GDS_EDITOR(text)->syntax_table;
   if(!table)
   {
      check_embedded(text, start, end, NULL);
      return;
   }
   list = g_list_first(table->entries);
   i = start;
   prev = text->scroll_line_start->property_first;
   if(end > text->length)
   {
      g_warning("THIS IS BUGGY check_syntax end [%d] is more than text->length [%d]\n", end, text->length);
      end=text->length;
   }
   while(i < end)
   {
      s = gtk_extext_regex_search(text, i, &table->regex_all, TRUE, &m);
      if(s < 0 || s > end) break;
      if(i < s) check_embedded(text, i, s, prev);
      i = m.endpos;
      list = g_list_first(table->entries);
      while(list)
      {    
         sentry = (GdsEditorSyntaxEntry *)list->data;
         if(gtk_extext_regex_match(text, s, &sentry->regex_start) > 0 && GTK_EXTEXT_INDEX(text, s-1) != '\\')
         {
            if((j = get_syntax_end(text, i, sentry, &m)) > 0)
               i = m.endpos;
            else if(j == 0) continue;
            else i = text->length;
            prev = gtk_extext_property_insert(text, sentry->key, s, i, GINT_TO_POINTER(SYNTAX_TABLE), PROPERTY_REMOVEALL, prev);
            found = TRUE;
            break;
         }
         else if(GTK_EXTEXT_INDEX(text, s-1) == '\\') found = TRUE;
         list = g_list_next(list);
      }
      if(!found) i++;
   }
   if(i < end) check_embedded(text, i, end, prev);
}

void check_embedded(GtkExText *text, gint start, gint end, GtkExTextProperty *prev)
{
   gint i, j, pos;
   gint startpos = 0;
   gint endpos = 0;
   GList *list;
   GdsEditorHighlightTable *table;
   GdsEditorEmbeddedEntry *embedded;
   gboolean found = FALSE;

   table = GDS_EDITOR(text)->embedded_table;
   if(!table)
   {
   	check_pattern(text, start, end, prev);
   	return;
   }
   list = g_list_first(table->entries);
   if(end > text->length)
   {
      g_warning("THIS IS BUGGY! end [%d] is more than text->length [%d]\n", end, text->length);
      end = text->length;
   }
   prev = NULL;
   i = start;
   while(i < end)
   {
      list = g_list_first(table->entries);
      while(list)
      {
         found = FALSE;
         embedded = (GdsEditorEmbeddedEntry *)list->data;
         if((pos = gtk_extext_regex_match(text, i, &embedded->start) > 0))
         {
            startpos = endpos = i + pos;
            while(endpos < text->length)
            {
               if((pos = gtk_extext_regex_match(text, endpos, &embedded->end)) > 0)
               {
                  j = endpos + pos;
                  while(startpos < endpos)
                  {
                     if((pos = gtk_extext_regex_match(text, startpos, &embedded->regex)) > 0)
                     {
                        prev = gtk_extext_property_insert(text, embedded->key, startpos, startpos + pos, GINT_TO_POINTER(EMBEDDED_TABLE), PROPERTY_REMOVEALL, prev);
                        startpos += pos;
                     }
                     else startpos++;
                  }
                  break;
               }
               endpos++;
            }
         }
/*
         This is test code for use sometime in the future when I re-write the loop above.
         else if((gtk_extext_regex_search(text, i, &embedded->start, 0, &m) > 0))
         {
            //g_print("trying mode number 2...i is %d\n", i);
            startpos = endpos = i;
            if((gtk_extext_regex_search(text, startpos, &embedded->end, 1, &m) > 0) && m.endpos <= text->length)
            {
               endpos = m.startpos > end ? end : m.startpos;
               while(startpos < text->length && (gtk_extext_regex_search(text, startpos, &embedded->regex, 1, &m) > 0) && m.endpos <= endpos)
               {
                  prev = gtk_extext_property_insert(text, embedded->key, m.startpos, m.endpos, GINT_TO_POINTER(EMBEDDED_TABLE), PROPERTY_REMOVEALL, prev);
                  startpos = m.endpos;
                  found = TRUE;
               }
            }
            //else i++;
         }
*/
         list = g_list_next(list);
      }
      i++;
   }
   check_pattern(text, start, end, prev);
}

void check_pattern(GtkExText *text, gint start, gint end, GtkExTextProperty *prev)
{
   gint i;
   gint pos = 0;
   gboolean found;
   GList *list;
   GdsEditorHighlightTable *table;
   GdsEditorPatternEntry *pentry;
 
   table = GDS_EDITOR(text)->pattern_table;
   if(!table) return;
   list = g_list_first(table->entries);
   if(end > text->length)
   {
      g_warning("THIS IS BUGGY! end [%d] is more than text->length [%d]\n", end, text->length);
      end = text->length;
   }
   prev = NULL;
   i = start;
   while(i < end)
   {
      list = g_list_first(table->entries);
      found = FALSE;
      while(list)
      {
         pentry = (GdsEditorPatternEntry *)list->data;      
         if((pos = gtk_extext_regex_match(text, i, &pentry->regex) ) > 0 && end >= i + pos)
         {
            prev = gtk_extext_property_insert(text, pentry->key, i, i + pos, GINT_TO_POINTER(PATTERN_TABLE), PROPERTY_REMOVEALL,prev);
            i += pos;
            found = TRUE;
            break;
         }
         list = g_list_next(list);
      }
      if(!found) i++;
   }
   gtk_widget_queue_draw(GTK_WIDGET(text));
}

gint get_syntax_end(GtkExText *text, gint pos, GdsEditorSyntaxEntry *table, GtkExTextMatch *m)
{
   int ret;
   int nbesc;
   do {
      ret = gtk_extext_regex_search(text, m->endpos, &table->regex_end, TRUE, m);
      if(ret < 0) return(-1);
      for(nbesc = 0; GTK_EXTEXT_INDEX(text, m->endpos-2-nbesc) == '\\'; nbesc++);
   } while(m->endpos && nbesc&1);
   return(ret);
}

void gds_editor_install_table(GdsEditor *text, GdsEditorHighlightTable *table)
{
   GList *cur;
   GdsEditorSyntaxEntry *sentry;
   GdsEditorPatternEntry *pentry;
   GdsEditorEmbeddedEntry *eentry;
 
   g_return_if_fail(text != NULL);
   g_return_if_fail(GDS_IS_EDITOR(text));
   g_return_if_fail(table != NULL);
   
   if(table->type == SYNTAX_TABLE)
   {
      text->syntax_table = table;
      cur = g_list_first(table->entries);
      while(cur)
      {
         sentry = (GdsEditorSyntaxEntry *)cur->data;
         gtk_extext_style_insert(GTK_EXTEXT(text), sentry->key, sentry->font, sentry->color, sentry->bg, sentry->flags);
         cur = g_list_next(cur);
      }
   }
   else if(table->type == PATTERN_TABLE)
   {
      text->pattern_table = table;
      cur=g_list_first(table->entries);
      while(cur)
      {
         pentry = (GdsEditorPatternEntry*)cur->data;
         gtk_extext_style_insert(GTK_EXTEXT(text), pentry->key, pentry->font, pentry->color, pentry->bg, pentry->flags);
         cur = g_list_next(cur);
      }
   }
   else if(table->type == EMBEDDED_TABLE)
   {
      text->embedded_table = table;
      cur=g_list_first(table->entries);
      while(cur)
      {
         eentry = (GdsEditorEmbeddedEntry*)cur->data;
         gtk_extext_style_insert(GTK_EXTEXT(text), eentry->key, eentry->font, eentry->color, eentry->bg, eentry->flags);
         cur = g_list_next(cur);
      }
   }
   else
   {
      g_warning("Table could not be inserted.\n");
   }
}

GdsEditorHighlightTable *gds_editor_syntax_table_new(GList *entries)
{
   gchar *regs;
   GList *cur;
   GdsEditorHighlightTable *table;
   GdsEditorSyntaxEntry *syntax;
   g_return_val_if_fail(entries!=NULL,NULL);
   g_return_val_if_fail(g_list_length(entries),NULL);
 
   table = g_malloc0(sizeof(GdsEditorHighlightTable));
   table->type = SYNTAX_TABLE;    
   table->entries = entries;
   regs = g_malloc0(1024);
   cur = g_list_first(entries);
   while(cur)
   {
      syntax = (GdsEditorSyntaxEntry *)cur->data;
      strcat(regs,syntax->start);
      cur = g_list_next(cur);
      if(cur) strcat(regs,"\\|");    
   }
   gds_editor_compile_regex(regs,&table->regex_all);
   g_free(regs);
   return(table);
}

GdsEditorHighlightTable *gds_editor_pattern_table_new(GList *entries)
{
   GdsEditorHighlightTable *table;
   g_return_val_if_fail(entries!=NULL,NULL);
   g_return_val_if_fail(g_list_length(entries),NULL);

   table = g_malloc0(sizeof(GdsEditorHighlightTable));  
   table->type = PATTERN_TABLE;    
   table->entries = entries;
   return(table);
}

GdsEditorHighlightTable *gds_editor_embedded_table_new(GList *entries)
{
   GdsEditorHighlightTable *table;
   g_return_val_if_fail(entries != NULL, NULL);
   g_return_val_if_fail(g_list_length(entries), NULL);

   table = g_malloc0(sizeof(GdsEditorHighlightTable));  
   table->type = EMBEDDED_TABLE;    
   table->entries = entries;
   return(table);
}

GList *gds_editor_syntax_entry_new(GList *entries, gchar *name, gchar *start, gchar *end, GdkColor *fg, GdkColor *bg, GdkFont *font, guint16 flags, gint type)
{
   GdsEditorSyntaxEntry *syntax;
   syntax = g_malloc0(sizeof(GdsEditorSyntaxEntry));
   syntax->start = g_strdup(start);
   strncpy(syntax->key, name, 32);  
   syntax->font = font;
   syntax->color = fg;
   syntax->bg = bg;
   syntax->flags = flags;
   syntax->type = type;
   gds_editor_compile_regex(start,&syntax->regex_start);
   gds_editor_compile_regex(end,&syntax->regex_end);
   return(g_list_append(entries, syntax));
}

GList *gds_editor_pattern_entry_new(GList *entries, gchar *name, gchar *start, GdkColor *fg, GdkColor *bg, GdkFont *font, guint16 flags, gint type)
{
   GdsEditorPatternEntry *pattern;
   pattern = g_malloc0(sizeof(GdsEditorPatternEntry));
   strncpy(pattern->key, name, 32);
   pattern->font = font;
   pattern->color = fg;
   pattern->bg = bg;
   pattern->flags = flags;
   pattern->type = type;
   gds_editor_compile_regex(start, &pattern->regex);
   return(g_list_append(entries, pattern));
}

GList *gds_editor_embedded_entry_new(GList *entries, gchar *name, gchar *start, gchar *end, gchar *words, GdkColor *fg, GdkColor *bg, GdkFont *font, guint16 flags, gint type)
{
   GdsEditorEmbeddedEntry *embedded;
   embedded = g_malloc0(sizeof(GdsEditorEmbeddedEntry));
   strncpy(embedded->key, name, 32);
   embedded->font = font;
   embedded->color = fg;
   embedded->bg = bg;
   embedded->flags = flags;
   embedded->type = type;
   gds_editor_compile_regex(start, &embedded->start);
   gds_editor_compile_regex(end, &embedded->end);
   gds_editor_compile_regex(words, &embedded->regex);
   return(g_list_append(entries, embedded));
}

void gds_editor_table_free(GdsEditorHighlightTable *table)
{
   GdsEditorSyntaxEntry *syntax;
   GdsEditorPatternEntry *pattern;
   GdsEditorEmbeddedEntry *embedded;
   
   GList *cur;
   cur = g_list_first(table->entries);
   while(cur)
   {
      if(table->type == SYNTAX_TABLE)
      {
         syntax = (GdsEditorSyntaxEntry *)cur->data;
         g_free(syntax->regex_start.buf.fastmap);
         syntax->regex_start.buf.fastmap = NULL;
         regfree(&syntax->regex_start.buf);
         g_free(syntax->regex_end.buf.fastmap);
         syntax->regex_end.buf.fastmap = NULL;
         regfree(&syntax->regex_end.buf);
      }
      else if(table->type == PATTERN_TABLE)
      {
         pattern = (GdsEditorPatternEntry *)cur->data;
         g_free(pattern->regex.buf.fastmap);
         pattern->regex.buf.fastmap = NULL;
         regfree(&pattern->regex.buf);
      }
      if(table->type == EMBEDDED_TABLE)
      {
         embedded = (GdsEditorEmbeddedEntry *)cur->data;
         g_free(embedded->start.buf.fastmap);
         embedded->start.buf.fastmap = NULL;
         regfree(&embedded->start.buf);
         g_free(embedded->end.buf.fastmap);
         embedded->end.buf.fastmap = NULL;
         regfree(&embedded->end.buf);
         g_free(embedded->regex.buf.fastmap);
         embedded->regex.buf.fastmap = NULL;
         regfree(&embedded->regex.buf);
      }
      g_free(cur->data);
      cur=g_list_next(cur);
   }
   g_list_free(table->entries);
   g_free(table);
}

void gds_editor_insert_text(GdsEditor *editor, const gchar *new_text, gint new_text_length, gint *position)
{
   GtkExText *text;
   g_return_if_fail(editor != NULL);
   g_return_if_fail(GTK_IS_EXTEXT(editor));
   text = GTK_EXTEXT(editor);
   if(!text->editable) return;
   gtk_extext_insert_text(text, new_text, new_text_length, position);
}

void gds_editor_delete_text(GdsEditor *editor, gint start_pos, gint end_pos)
{
   GtkExText *text;
   g_return_if_fail(editor != NULL);
   g_return_if_fail(GTK_IS_EXTEXT(editor));
   text = GTK_EXTEXT(editor);
   if(!text->editable) return;
   gtk_extext_delete_text(text, start_pos, end_pos);
}

gchar *gds_editor_get_chars(GdsEditor *editor, gint start_pos, gint end_pos)
{
   GtkExText *text;
   gchar *chars;
   g_return_val_if_fail(editor != NULL, NULL);
   g_return_val_if_fail(GTK_IS_EXTEXT(editor), NULL);
   text = GTK_EXTEXT(editor);
   chars = gtk_extext_get_chars(text, start_pos, end_pos);
   return(chars);
}

void gds_editor_set_editable(GdsEditor *editor, gboolean editable)
{
   g_return_if_fail(editor != NULL);
   g_return_if_fail(GTK_IS_EXTEXT(editor));

   gtk_extext_set_editable(GTK_EXTEXT(editor), editable);
}

static void gds_editor_dnd_drop(GdsEditor *editor, GdkDragContext *context, gint x, gint y, GtkSelectionData *selection_data, guint info, guint time, gpointer data)
{
   GtkExTextLineData *linedata;
   gint offset = 0;

   if(info == TARGET_COLOR)
   {
      guint16 *vals;
      gchar string[] = "#000000";

      if (selection_data->length < 0)
         return;
    
      if ((selection_data->format != 16) || (selection_data->length != 8))
      {
         g_warning ("Received invalid color data\n");
         return;
      }
      
      vals = (guint16 *)selection_data->data;
      vals[0] /= 256;
      vals[1] /= 256;
      vals[2] /= 256;
      g_snprintf(string, sizeof(string), "#%02X%02X%02X", vals[0], vals[1], vals[2]);
      linedata = gtk_extext_get_line_by_offset(GTK_EXTEXT(editor), y, NULL);
      offset = gtk_extext_get_column_by_offset(GTK_EXTEXT(editor), linedata, x, NULL);
      if(offset < 0) offset = 0;
      offset += linedata->startpos;
      gtk_extext_insert_text(GTK_EXTEXT(editor), string, strlen(string), &offset);
   }
}
