/* Copyright (C) 2010 The giomm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <giomm.h>
#include <glibmm.h>
#include <iostream>

// The main loop.
Glib::RefPtr<Glib::MainLoop> loop;

// A main loop idle callback to quit when the main loop is idle.
bool main_loop_idle()
{
  loop->quit();
  return false;
}

// A callback to finish creating a DBusProxy that was asynchronously created
// for the user session's bus and then try to call the well known 'ListNames'
// method.
void dbus_proxy_available(Glib::RefPtr<Gio::AsyncResult>& result)
{
  Glib::RefPtr<Gio::DBus::Proxy> proxy = Gio::DBus::Proxy::create_finish(result);

  if(!proxy)
  {
    std::cerr << "The proxy to the user's session bus was not successfully "
      "created." << std::endl;
    loop->quit();
    return;
  }

  try
  {
    // The proxy's call method returns a tuple of the value(s) that the method
    // call produces so just get the tuple as a VariantContainerBase.
    const Glib::VariantContainerBase result = proxy->call_sync("ListNames");

    // Now extract the single item in the variant container which is the
    // array of strings (the names).
    Glib::Variant< std::vector<Glib::ustring> > names_variant;
    result.get(names_variant);

    // Get the vector of strings.
    std::vector<Glib::ustring> names = names_variant.get();

    std::cout << "The names on the message bus are:" << std::endl;

    for(unsigned i = 0; i < names.size(); i++)
      std::cout << names[i] << "." << std::endl;
  }
  catch(const Glib::Error& error)
  {
    std::cerr << "Got an error: '" << error.what() << "'." << std::endl;
  }

  // Connect an idle callback to the main loop to quit when the main loop is
  // idle now that the method call is finished.
  Glib::signal_idle().connect(sigc::ptr_fun(&main_loop_idle));
}

int main(int, char**)
{
  std::locale::global(std::locale(""));
  Gio::init();

  loop = Glib::MainLoop::create();

  // Get the user session bus connection.
  Glib::RefPtr<Gio::DBus::Connection> connection =
    Gio::DBus::Connection::get_sync(Gio::DBus::BUS_TYPE_SESSION);

  // Check for an unavailable connection.
  if(!connection)
  {
    std::cerr << "The user's session bus is not available." << std::endl;
    return 1;
  }

  // Create the proxy to the bus asynchronously.
  Gio::DBus::Proxy::create(connection, "org.freedesktop.DBus",
    "/org/freedesktop/DBus", "org.freedesktop.DBus",
    sigc::ptr_fun(&dbus_proxy_available));

  loop->run();

  return 0;
}
