/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <gtk/gtk.h>
#include "gladeconfig.h"

#ifdef USE_GNOME
#include <gnome.h>
#endif

#include "editor.h"
#include "gb.h"
#include "gbwidget.h"
#include "glade.h"
#include "glade_gnome.h"
#include "glade_gtk12lib.h"
#include "glade_plugin.h"
#include "glade_project.h"
#include "load.h"
#include "palette.h"
#include "property.h"
#include "save.h"
#include "source.h"
#include "tree.h"
#include "utils.h"

#ifdef USE_GNOME
#include "glade_gnomelib.h"
#endif

/*
 * The hash table associates a Gtk widget class name with a GbWidget struct,
 * which contains a table of functions used by the builder, e.g. creating a
 * new widget, saving the widget etc.
 */
static GHashTable *gb_widget_table = NULL;

/* Tooltips for the created widgets */
static GtkTooltips *gb_widget_tooltips;


/* These are used for outputting signal handler prototypes. */
#define GB_PARAM_INDENT		40
#define GB_PARAM_TYPE_WIDTH	16

const gchar *GladeMenuItemStockIndexKey = "GladeMenuItemStockIndexKey";
const gchar *GladeMenuItemIconKey = "GladeMenuItemStockIconKey";


static void gb_widget_init_widget_lib (GladePaletteSectionData *sections);

static void on_widget_destroy (GtkWidget * widget,
			       gpointer data);
static void gb_widget_free_accelerators (GtkWidget * widget);
static void gb_widget_free_signals (GtkWidget * widget);

#ifdef GLADE_STYLE_SUPPORT
static void show_color_properties (GdkColor colors[],
				   gchar * name);
#endif
static void show_accelerators (GtkWidget * widget,
			       GbWidgetGetArgData * data);
static void show_signals (GtkWidget * widget,
			  GbWidgetGetArgData * data);

static void set_standard_properties (GtkWidget * widget,
				     GbWidgetSetArgData * data);
static void set_position_properties (GtkWidget * widget,
				     GbWidgetSetArgData * data);
static void set_special_child_properties (GtkWidget * widget,
					  GbWidgetSetArgData * data);
static void set_lang_specific_properties (GtkWidget * widget,
					  GbWidgetSetArgData * data);
#if 0
static void apply_style (GtkWidget * widget,
			 GbWidgetSetArgData * data);
static gboolean apply_colors (GtkWidget * widget,
			      GbWidgetSetArgData * data,
			      GdkColor colors[],
			      GdkColor new_colors[],
			      gchar * name);
#endif
static void apply_accelerators (GtkWidget * widget,
				GbWidgetSetArgData * data);
static void apply_signals (GtkWidget * widget,
			   GbWidgetSetArgData * data);

static void add_standard_top_menu_items (GtkWidget * widget,
					 GbWidgetCreateMenuData * data);
static void add_standard_bottom_menu_items (GtkWidget * widget,
					    GbWidgetCreateMenuData * data);

static void gb_widget_load_properties (GbWidgetSetArgData * data);
static void load_special_child_properties (GbWidgetSetArgData * data);
static void load_accelerator (GbWidgetSetArgData * data);
static void load_signal (GbWidgetSetArgData * data);
static void load_style (GtkWidget * widget,
			GbWidgetSetArgData * data);
#ifdef GLADE_STYLE_SUPPORT
static gboolean load_colors (GbWidgetSetArgData * data,
			     GdkColor colors[],
			     gchar * name,
			     gchar * element,
			     gchar * cdata);
#endif

static void gb_add_properties (gint tag_index,
			       gint cdata_index,
			       gint line_number,
			       GbLoadPropControl *p);
static void gb_realloc_properties_if_needed (GbLoadPropControl *p);

static void get_standard_properties (GtkWidget * widget,
				     GbWidgetGetArgData * data);

static void get_position_properties (GtkWidget * widget,
				     GbWidgetGetArgData * data);
static void get_lang_specific_properties (GtkWidget * widget,
					  GbWidgetGetArgData * data);

#ifdef GLADE_STYLE_SUPPORT
static void save_style (GtkWidget * widget,
			GbWidgetGetArgData * data);
static void save_colors (GbWidgetGetArgData * data,
			 gboolean save_all,
			 GdkColor colors[],
			 GdkColor default_colors[],
			 gchar * name);
#endif
static void save_accelerators (GtkWidget * widget,
			       GbWidgetGetArgData * data);
static void save_signals (GtkWidget * widget,
			  GbWidgetGetArgData * data);


static void gb_widget_write_signals_source (GtkWidget * widget,
					    GbWidgetWriteSourceData * data);
static void gb_widget_write_signal_connection_source (GbWidgetWriteSourceData * data,
						      const gchar *signal_name,
						      const gchar *connect_object,
						      gboolean connect_after,
						      const gchar *handler_data,
						      const gchar *handler);
static gchar *get_type_name (GtkType type,
			     gboolean * is_pointer);
static gchar *get_gdk_event (gchar * signal_name);
static gchar **lookup_signal_arg_names (gchar * type,
					gchar * signal_name);
static void gb_widget_write_accelerators_source (GtkWidget * widget,
					    GbWidgetWriteSourceData * data);

static void gb_widget_add_alignment (GtkWidget * menuitem,
				     GtkWidget * widget);
static void gb_widget_remove_alignment (GtkWidget * menuitem,
					GtkWidget * widget);
static void gb_widget_add_event_box (GtkWidget * menuitem,
				     GtkWidget * widget);
static void gb_widget_remove_event_box (GtkWidget * menuitem,
					GtkWidget * widget);
static void gb_widget_redisplay_window (GtkWidget * menuitem,
					GtkWidget * widget);
static void gb_widget_add_scrolled_window (GtkWidget * menuitem,
					   GtkWidget * widget);
static void gb_widget_remove_scrolled_window (GtkWidget * menuitem,
					      GtkWidget * widget);

static void table_foreach (GtkTable * table,
			   GtkCallback callback,
			   gpointer callback_data);
static GtkNotebookPage *find_notebook_page (GtkWidget * widget,
					    GtkWidget * current_child,
					    gint * pos);



/*************************************************************************
 * Initialization functions
 *************************************************************************/

void
gb_widgets_init ()
{
  gb_widget_table = g_hash_table_new (g_str_hash, g_str_equal);

#ifdef GLADE_STYLE_SUPPORT
  gb_widget_reset_gb_styles ();
#endif

  /* Create tooltips */
  gb_widget_tooltips = gtk_tooltips_new ();

  gb_widget_init_widget_lib (get_gtk_widgets());
#ifdef USE_GNOME
  gb_widget_init_widget_lib (get_gnome_widgets());
#endif

  glade_plugin_load_plugins ();
}

static void
gb_widget_init_widget_lib (GladePaletteSectionData *sections)
{
  gint index, j;
  GladeWidgetInitData *gwid;
  GladePaletteSectionData *palsec;
  GbWidget *(*init_func) ();
  GbWidget *gbwidget;

  index = 0;
  while (1)
    {
      j = 0;
      palsec = &sections[index];
      index++;
      if (!palsec->section)
        break;
      while (1)
        {
          gwid = &palsec->widgets[j];
          j++;
          if (!gwid->name)
            break;  
          init_func = gwid->init_func;
          gbwidget = (*init_func) ();
	  gb_widget_register_gbwidget (gwid->name, gbwidget);
	  palette_add_gbwidget (gbwidget, palsec->section, gwid->name);
        }
    }
}


/* Adds a GbWidget to the hash of known GbWidgets. The class_name is used
   directly, so it must be a constant string literal or g_strdup'ed. */
void
gb_widget_register_gbwidget (const gchar *class_name,
			     GbWidget    *gbwidget)
{
  g_hash_table_insert (gb_widget_table, (gpointer) class_name, gbwidget);
}


/* This returns the GbWidget struct corresponding to the given class name. */
GbWidget *
gb_widget_lookup_class (const gchar *class_name)
{
  GbWidget *gbwidget;

  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
  return gbwidget;
}


/* This returns the GbWidget struct corresponding to the given widget. */
GbWidget *
gb_widget_lookup (GtkWidget *widget)
{
  GbWidget *gbwidget;
  gchar *class_name;

  if (GLADE_IS_CUSTOM_WIDGET (widget))
    class_name = "Custom";
  else
    class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));

  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
  return gbwidget;
}


gchar*
gb_widget_get_type_name (GtkWidget *widget)
{
  if (GLADE_IS_CUSTOM_WIDGET (widget))
    return "Custom";
  else
    return gtk_type_name (GTK_OBJECT_TYPE (widget));
}


void
gb_widget_init_struct (GbWidget * gbwidget)
{
  gbwidget->pixmap_struct = NULL;
  gbwidget->gdkpixmap = NULL;
  gbwidget->mask = NULL;
  gbwidget->tooltip = NULL;
  gbwidget->properties_page_number = GB_PROPERTIES_NOT_CREATED;
  gbwidget->child_properties_page_number = GB_PROPERTIES_NOT_CREATED;

  gbwidget->gb_widget_new = NULL;
  gbwidget->gb_widget_create_from_widget = NULL;

  gbwidget->gb_widget_create_properties = NULL;
  gbwidget->gb_widget_get_properties = NULL;
  gbwidget->gb_widget_set_properties = NULL;

  gbwidget->gb_widget_add_child = NULL;
  gbwidget->gb_widget_get_child = NULL;

  gbwidget->gb_widget_create_child_properties = NULL;
  gbwidget->gb_widget_get_child_properties = NULL;
  gbwidget->gb_widget_set_child_properties = NULL;

  gbwidget->gb_widget_write_add_child_source = NULL;

  gbwidget->gb_widget_create_popup_menu = NULL;

  gbwidget->gb_widget_write_source = NULL;

  gbwidget->gb_widget_destroy = NULL;
}



/*************************************************************************
 * Functions for creating & destroying GbWidgets
 *************************************************************************/

GtkWidget *
gb_widget_new (const gchar * class_name, GtkWidget * parent)
{
  return gb_widget_new_full (class_name, TRUE, parent, NULL, 0, 0, NULL,
			     GB_CREATING, NULL);
}


/* Creates a new widget.
 * class_name is the name of the widget class, e.g. 'GtkLabel'
 * create_default_name is TRUE if you want a default name to be created,
 *   e.g. 'label1'.
 * parent is the widget that the new widget will be added beneath, so that
 *   the callback function knows where to put the widget.
 * current_child is the widget that the new widget will replace, or NULL
 *   if the new widget is just being added. It is used when replacing
 *   placeholders.
 * x & y are the coordinates of the new widget if it is being added to a
 *   GtkFixed container.
 * callback is the function to call once the widget is created to actually
 *   add it to the parent. Some widgets require dialog boxes for creating
 *   them (e.g. the dialog box to set the number of rows/cols in a table).
 *   So we need to provide a function to be called after this is done
 *   (we could have possibly chosen to use modal dialogs instead.)
 * action is either GB_CREATING or GB_LOADING. When loading widgets we don't
 *   want dialog boxes to pop up when they are being created.
 * loading_data is set when action is GB_LOADING, and contains the data used
 *   while loading, so that the GbWidgets can get any properties they need to
 *   create the widget without popping up a dialog box.
 */
GtkWidget *
gb_widget_new_full (const gchar * class_name, gboolean create_default_name,
		    GtkWidget * parent, GtkWidget * current_child,
		    gint x, gint y, GbWidgetNewCallback callback,
		    GbWidgetAction action, GbWidgetSetArgData * loading_data)
{
  GbWidgetNewData *data;
  GtkWidget *new_widget;
  GbWidget *gbwidget;
  gint type;

  gbwidget = gb_widget_lookup_class (class_name);
  g_return_val_if_fail (gbwidget != NULL, NULL);

  /* Note that for custom widgets this won't be found, and so will be 0. */
  type = gtk_type_from_name (class_name);

  data = g_new (GbWidgetNewData, 1);
  /* Don't set data->name to NULL, since many widgets use it to set the label
     of the new widget. */
  data->project = current_project;
  data->name = create_default_name ? glade_project_new_widget_name (data->project, class_name) : g_strdup ("");
  data->callback = callback;
  data->parent = parent;
  if (parent)
    gtk_widget_ref (parent);
  data->current_child = current_child;
  if (current_child)
    gtk_widget_ref (current_child);
  data->x = x;
  data->y = y;
  data->widget_data = gb_widget_new_widget_data ();
  data->action = action;
  data->loading_data = loading_data;

  if (gbwidget->gb_widget_new)
    new_widget = (gbwidget->gb_widget_new) (data);
  else if (type != 0)
    new_widget = GTK_WIDGET (gtk_object_newv (type, 0, NULL));

  /* If the widget has been created immediately, then we can finish it off now,
     and free the GbWidgetNewData struct, otherwise we leave that to the
     dialog. */
  if (new_widget)
    {
      gb_widget_initialize (new_widget, data);
      if (data->callback)
	(*data->callback) (new_widget, data);
      gb_widget_free_new_data (data);
    }

  return new_widget;
}


/* This turns a normal widget into a GbWidget, adding a GbWidgetData struct
   and the necessary signals. The widget must have its parent set already.
   If name is non-NULL, the widget's name will be set to the name with a
   unique ID added on to it, e.g. "ok_button1".
   NOTE: when loading, you should not create the names of any widgets,
   since it may clash with a widget loaded later. Instead leave the name as
   NULL. glade_project_ensure_widgets_named () will be called after the
   project is loaded, and any widgets without names will have names
   generated for them. */
void
gb_widget_create_from (GtkWidget * widget,
		       const gchar * name)
{
  GbWidget *gbwidget;
  GbWidgetData *wdata;

  MSG1 ("In create_from, widget name: %s", gtk_widget_get_name (widget));
  if (name)
    {
      gtk_widget_set_name (widget,
			   glade_project_new_widget_name (current_project,
							  name));
    }
  wdata = gb_widget_new_widget_data ();
  gb_widget_real_initialize (widget, wdata);

  /* Call any function the GbWidget has for setting up the widget to be used
     within Glade. */
  gbwidget = gb_widget_lookup (widget);
  g_return_if_fail (gbwidget != NULL);
  if (gbwidget->gb_widget_create_from_widget)
    {
      GbWidgetCreateFromData data;

      data.project = current_project;
      (gbwidget->gb_widget_create_from_widget) (widget, &data);
    }
}


GbWidgetData *
gb_widget_new_widget_data ()
{
  GbWidgetData *widget_data = g_new (GbWidgetData, 1);
  widget_data->flags = GB_VISIBLE | GB_SENSITIVE | GB_STYLE_IS_UNNAMED
    | GB_STYLE_PROPAGATE | GB_SIZE_NOT_ALLOCATED;
  widget_data->x = 0;
  widget_data->y = 0;
  widget_data->width = 0;
  widget_data->height = 0;
  widget_data->events = 0;
  widget_data->tooltip = NULL;
  widget_data->signals = NULL;
  widget_data->accelerators = NULL;
#ifdef GLADE_STYLE_SUPPORT
  widget_data->gbstyle = gb_widget_default_gb_style;
  gb_widget_ref_gb_style (gb_widget_default_gb_style);
#endif

  /* C options. */
  widget_data->source_file = NULL;
  widget_data->public_field = 1;

  /* C++ options. */
  widget_data->cxx_separate_file = 0;
  widget_data->cxx_use_heap = 0;
  widget_data->cxx_separate_class = 0;
  widget_data->cxx_visibility = 0;

  return widget_data;
}


void
gb_widget_initialize (GtkWidget * widget, GbWidgetNewData * data)
{
  if (data->name && data->name[0] != '\0')
    gtk_widget_set_name (widget, data->name);
  gb_widget_real_initialize (widget, data->widget_data);

  /* Now we set the widget's real style */
  /* FIXME: check if style should be propagated down from an ancestor? */
#if 0
  if (widget->style != data->widget_data->gbstyle->style)
    {
      gtk_widget_set_style (widget, data->widget_data->gbstyle->style);
    }
#endif

  /* FIXME: GTK workarounds to make sure that some widgets have reasonable
     sizes initially. Quite a few widgets default to a width and height of 0,
     which means that if there is little space available they will disappear,
     so we may need to do more here. */
  if (GTK_IS_ARROW (widget))
    gtk_widget_set_usize (widget, 16, 16);

  /* Set this to NULL so we don't try to free it later. */
  data->widget_data = NULL;
}



void
gb_widget_real_initialize (GtkWidget * widget, GbWidgetData * wdata)
{
  gtk_object_set_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY, wdata);
  gtk_signal_connect (GTK_OBJECT (widget), "destroy",
		      GTK_SIGNAL_FUNC (on_widget_destroy), NULL);

  editor_add_mouse_signals (widget);
  editor_add_draw_signals (widget);
  editor_add_key_signals (widget);

#ifdef USE_GNOME
  /* FIXME: GnomeLibs 1.0.1 workaround - floating GnomeDockItem's don't work
     properly if we show them before adding to the GnomeDock. */
  if (!GTK_IS_WINDOW (widget) && !GTK_IS_MENU (widget)
      && !GNOME_IS_DOCK_ITEM (widget))
    gtk_widget_show (widget);
#else
  if (!GTK_IS_WINDOW (widget) && !GTK_IS_MENU (widget))
    gtk_widget_show (widget);
#endif
}


/* This returns TRUE if it is OK to complete the new() procedure, i.e. that
   the widget to replace or the parent widget still exist. It is used after
   the OK button is pressed in the dialog boxes for creating new tables/boxes.
   FIXME: I'm not too sure what we should do here. */
gboolean
gb_widget_can_finish_new (GbWidgetNewData * data)
{
  if (data->current_child)
    {
      if (GTK_OBJECT_DESTROYED (data->current_child)
	  || data->current_child->parent == NULL)
	return FALSE;
    }
  else if (data->parent)
    {
      if (GTK_OBJECT_DESTROYED (data->parent)
	  || (data->parent->parent == NULL && !GTK_IS_WINDOW (data->parent)))
	return FALSE;
    }
  return TRUE;
}


void
gb_widget_free_new_data (GbWidgetNewData * data)
{
  g_free (data->name);
  g_free (data->widget_data);
  if (data->parent)
    gtk_widget_unref (data->parent);
  if (data->current_child)
    gtk_widget_unref (data->current_child);
  g_free (data);
}


static void
on_widget_destroy (GtkWidget * widget, gpointer user_data)
{
  GbWidget *gbwidget;
  GbWidgetData *widget_data;
  GbWidgetDestroyData data;

  MSG1 ("IN on_widget_destroy widget:%s", gtk_widget_get_name (widget));

  /* Make sure we don't try to show its properties after it is destroyed. */
  if (property_get_widget () == widget)
    property_set_widget (NULL);

  /* If the entire project is being destroyed, we don't need to update the
     selection or the widget tree. */
  if (!GTK_OBJECT_DESTROYED (current_project))
    {
      editor_remove_widget_from_selection (widget);
      tree_remove_widget (widget);
    }

  gbwidget = gb_widget_lookup (widget);
  g_return_if_fail (gbwidget != NULL);

  /* Call the GbWidget destroy function, if it has one. */
  data.project = current_project;
  if (gbwidget->gb_widget_destroy)
    (gbwidget->gb_widget_destroy) (widget, &data);

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);

  /* Release the ID. */
  if (widget->name)
    glade_project_release_widget_name (current_project, widget->name);

  g_free (widget_data->tooltip);
  gb_widget_free_accelerators (widget);
  gb_widget_free_signals (widget);
  g_free (widget_data->source_file);
  g_free (widget_data);
  g_free (gb_widget_get_child_name (widget));
  MSG1 ("OUT on_widget_destroy widget:%s", gtk_widget_get_name (widget));
}


static void
gb_widget_free_accelerators (GtkWidget * widget)
{
  GbWidgetData *widget_data;
  GList *item;
  GbAccelerator *accelerator;

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  if (widget_data->accelerators)
    {
      item = widget_data->accelerators;
      while (item)
	{
	  accelerator = (GbAccelerator *) item->data;
	  g_free (accelerator->key);
	  g_free (accelerator->signal);
	  item = item->next;
	}
      g_list_free (widget_data->accelerators);
      widget_data->accelerators = NULL;
    }
}


static void
gb_widget_free_signals (GtkWidget * widget)
{
  GbWidgetData *widget_data;
  GList *item;
  GbSignal *signal;

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  if (widget_data->signals)
    {
      item = widget_data->signals;
      while (item)
	{
	  signal = (GbSignal *) item->data;
	  g_free (signal->name);
	  g_free (signal->handler);
	  g_free (signal->object);
	  g_free (signal->data);
	  item = item->next;
	}
      g_list_free (widget_data->signals);
      widget_data->signals = NULL;
    }
}


/*************************************************************************
 * Functions for getting/setting the child name of a widget.
 * The child name is used to identify special widgets which have to
 * be treated differently. e.g. Dialog buttons.
 *************************************************************************/

static const gchar *glade_child_name_key  = "glade-child-name";
static GQuark       glade_child_name_key_id = 0;


/* Returns the child name of the widget. */
gchar*
gb_widget_get_child_name (GtkWidget *widget)
{
  if (!glade_child_name_key_id)
    glade_child_name_key_id = g_quark_from_static_string (glade_child_name_key);
  return gtk_object_get_data_by_id (GTK_OBJECT (widget),
				    glade_child_name_key_id);
}


/* Sets the child name of the widget. The child_name string is duplicated. */
void
gb_widget_set_child_name (GtkWidget *widget, gchar *child_name)
{
  if (!glade_child_name_key_id)
    glade_child_name_key_id = g_quark_from_static_string (glade_child_name_key);
  /* Free any existing child name. */
  g_free (gtk_object_get_data_by_id (GTK_OBJECT (widget),
				     glade_child_name_key_id));
  gtk_object_set_data_by_id (GTK_OBJECT (widget), glade_child_name_key_id,
			     g_strdup (child_name));
}


/*************************************************************************
 * Functions for creating the page of properties specific to this widget
 *************************************************************************/

/* Returns the page number of the new page in the notebook which contains the
   widget's specific properties, or GB_PROPERTIES_NOT_NEEDED if it has none. */
gint
gb_widget_create_properties (GtkWidget * widget)
{
  GtkWidget *page;
  gint page_number;
  GbWidgetCreateArgData data;
  GbWidget *gbwidget;

  gbwidget = gb_widget_lookup (widget);
  g_return_val_if_fail (gbwidget != NULL, GB_PROPERTIES_NOT_NEEDED);

  if (gbwidget->gb_widget_create_properties || GTK_IS_CONTAINER (widget))
    {
      /* Create skeleton of properties page, so gbwidget just has to add
         properties */
      page = gtk_table_new (10, 3, FALSE);
      gtk_widget_show (page);
      page_number = property_add_gbwidget_page (page);
      property_set_table_position (page, 0);

      /* If widget is a container add a border width property */
      if (GTK_IS_CONTAINER (widget))
	{
	  gchar *class_name, buf[128];

	  class_name = gb_widget_get_type_name (widget);
	  sprintf (buf, "%s::border_width", class_name);
	  property_add_int_range (buf, _("Border Width:"),
			     _("The width of the border around the container"),
				  0, 1000, 1, 10, 1);
	}

      data.project = current_project;
      if (gbwidget->gb_widget_create_properties)
	(gbwidget->gb_widget_create_properties) (widget, &data);
      return page_number;
    }
  else
    {
      return GB_PROPERTIES_NOT_NEEDED;
    }
}


/*************************************************************************
 * Functions for creating the page of place properties specific to this
 * widget.
 *************************************************************************/

/* Returns the page number of the new page in the notebook which contains the
   widget's properties which applt to any children of the widget,
   or GB_PROPERTIES_NOT_NEEDED if no extra properties are needed for its
   children. */
gint
gb_widget_create_child_properties (GtkWidget * widget)
{
  static GHashTable *page_hash_table = NULL;

  GtkWidget *page;
  gint page_number;
  GbWidgetCreateChildArgData data;
  GbWidget *gbwidget;

  /* Create a hash table to contain functions already called to create child
     packing properties together with the page numbers they returned.
     This lets us use the same functions for multiple widgets, e.g. GtkHBox
     and GtkVBox, GtkHPaned and GtkVPaned. */
  if (page_hash_table == NULL)
    page_hash_table = g_hash_table_new (NULL, NULL);

  gbwidget = gb_widget_lookup (widget);
  g_return_val_if_fail (gbwidget != NULL, GB_PROPERTIES_NOT_NEEDED);

  if (gbwidget->gb_widget_create_child_properties)
    {
      /* First see if the function has already been called. Note that the
	 page numbers in the hash have 1 added to them so we can detect empty
	 values (they won't clash with page 0). */
      page_number = GPOINTER_TO_INT (g_hash_table_lookup (page_hash_table, gbwidget->gb_widget_create_child_properties));
      if (page_number)
	return page_number - 1;

      /* Create skeleton of properties page, so gbwidget just has to add
         properties */
      page = gtk_table_new (10, 3, FALSE);
      gtk_widget_show (page);
      page_number = property_add_child_packing_page (page);
      g_hash_table_insert (page_hash_table,
			   gbwidget->gb_widget_create_child_properties,
			   GINT_TO_POINTER (page_number + 1));
      property_set_table_position (page, 0);

      data.project = current_project;
      (gbwidget->gb_widget_create_child_properties) (widget, &data);
      return page_number;
    }
  else
    {
      return GB_PROPERTIES_NOT_NEEDED;
    }
}


/*************************************************************************
 * Functions for showing the widget's properties
 *************************************************************************/

void
gb_widget_show_properties (GtkWidget * widget)
{
  GbWidgetGetArgData data;
  GbWidget *gbwidget, *parent_gbwidget = NULL;
  GbWidgetData *widget_data;
  gint page, child_packing_page;

  /* If properties of widget are already shown, just return */
  if (property_get_widget () == widget)
    return;

  /* If widget is a placeholder reset the properties window and return */
  if (GB_IS_PLACEHOLDER (widget))
    {
      property_set_widget (NULL);
      return;
    }

  gbwidget = gb_widget_lookup (widget);
  g_return_if_fail (gbwidget != NULL);

  /* Turn off auto-apply so we can set properties without the 'changed'
     callbacks calling gb_widget_apply_properties (). */
  property_set_auto_apply (FALSE);

  /* Need this here to make sure properties notebook is sensitive */
  property_set_widget (widget);

  page = gbwidget->properties_page_number;
  /* If widget's properties page hasn't been created, create it now */
  if (page == GB_PROPERTIES_NOT_CREATED)
    {
      page = gb_widget_create_properties (widget);
      gbwidget->properties_page_number = page;
    }

  /* Show the widget's own properties page if it has one.
     Need to show the page before setting properties because of the
     Text widget - it must be realized before setting the text :-( */
  if (page == GB_PROPERTIES_NOT_NEEDED)
    property_hide_gbwidget_page ();
  else
    property_show_gbwidget_page (page);

  /* Now see if the parent has child packing properties that need to be
     created or shown. */
  if (widget->parent)
    {
      parent_gbwidget = gb_widget_lookup (widget->parent);

      /* parent_gbwidget may be NULL, e.g. for GnomeDockItems. */
      if (parent_gbwidget)
	{
	  child_packing_page = parent_gbwidget->child_properties_page_number;
	  /* If widget's properties page hasn't been created, create it now */
	  if (child_packing_page == GB_PROPERTIES_NOT_CREATED)
	    {
	      child_packing_page = gb_widget_create_child_properties (widget->parent);
	      parent_gbwidget->child_properties_page_number = child_packing_page;
	    }
      
	  if (child_packing_page == GB_PROPERTIES_NOT_NEEDED)
	    property_hide_child_packing_page ();
	  else
	    property_show_child_packing_page (child_packing_page);
	}
    }

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (widget_data != NULL);
  data.project = current_project;
  data.action = GB_SHOWING;
  data.widget_data = widget_data;
  get_standard_properties (widget, &data);

  if (gbwidget->gb_widget_get_properties)
    (gbwidget->gb_widget_get_properties) (widget, &data);

  if (parent_gbwidget && parent_gbwidget->gb_widget_get_child_properties)
    (parent_gbwidget->gb_widget_get_child_properties) (widget->parent, widget,
						       &data);

  /* Turn auto-apply back on again */
  property_set_auto_apply (TRUE);
}


/* This is called when the widget's size or position has changed, so that we
   should update the values shown in the properties editor. */
void
gb_widget_show_position_properties (GtkWidget * widget)
{
  GbWidgetGetArgData data;
  GbWidgetData *widget_data;

  /* Make sure this is the widget shown in the properties editor. */
  if (property_get_widget () != widget)
    return;

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (widget_data != NULL);
  data.project = current_project;
  data.action = GB_SHOWING;
  data.widget_data = widget_data;

  property_set_auto_apply (FALSE);
  get_position_properties (widget, &data);
  property_set_auto_apply (TRUE);
}



#ifdef GLADE_STYLE_SUPPORT
void
gb_widget_show_style (GtkWidget * widget)
{
  GbWidgetData *wdata;
  GbStyle *gbstyle;
  GtkStyle *style = widget->style;
  gchar buffer[128];
  gint i;

  wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (wdata != NULL);
  gbstyle = wdata->gbstyle;

  property_set_bool (GbStylePropagate, wdata->flags & GB_STYLE_PROPAGATE);

  property_set_dialog (GbStyleName, wdata->flags & GB_STYLE_IS_UNNAMED ?
		       "" : gbstyle->name, NULL);
  property_set_font (GbStyleFont, style->font, gbstyle->xlfd_fontname);

  /* Colors */
  show_color_properties (style->fg, "fg");
  show_color_properties (style->bg, "bg");
  show_color_properties (style->text, "text");
  show_color_properties (style->base, "base");

  /* Background pixmaps */
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      sprintf (buffer, "GtkStyle::%s[%s]", GbBgPixmapName, GbStateNames[i]);
      property_set_bgpixmap (buffer, style->bg_pixmap[i],
			     gbstyle->bg_pixmap_filenames[i]);
    }
}


static void
show_color_properties (GdkColor colors[], gchar * name)
{
  gint state;
  gchar buf[128];

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      sprintf (buf, "GtkStyle::%s[%s]", name, GbStateNames[state]);
      property_set_color (buf, &colors[state]);
    }
}
#endif


static void
show_accelerators (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *element = data->widget_data->accelerators;
  property_clear_accelerators ();
  while (element)
    {
      property_add_accelerator ((GbAccelerator *) element->data);
      element = element->next;
    }
}


static void
show_signals (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *element = data->widget_data->signals;
  property_clear_signals ();
  MSG1 ("Num signals: %i", g_list_length (element));
  while (element)
    {
      property_add_signal ((GbSignal *) element->data);
      element = element->next;
    }
}



/*************************************************************************
 * Functions for applying properties to a widget
 *************************************************************************/

void
gb_widget_apply_properties (GtkWidget * widget, GtkWidget * property)
{
  GbWidgetSetArgData data;
  GbWidget *gbwidget;
  GbWidgetData *widget_data;

  MSG1 ("Applying properties: %s", gtk_widget_get_name (widget));

  gbwidget = gb_widget_lookup (widget);
  g_return_if_fail (gbwidget != NULL);

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (widget_data != NULL);

  data.project = current_project;
  data.action = GB_APPLYING;
  data.widget_data = widget_data;
  data.property_to_apply = property;

  set_standard_properties (widget, &data);

  MSG ("Calling widget's own apply_properties");
  if (gbwidget->gb_widget_set_properties)
    (gbwidget->gb_widget_set_properties) (widget, &data);
  MSG ("Called widget's own apply_properties");
}


static void
set_standard_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gchar *name, *tooltip, *events_str, *ext_events;
  gboolean visible, sensitive, can_default, has_default, can_focus, has_focus;
  gint events, i;

  /* Properties on widget page */
  /* When pasting, we may want to discard the names from the XML, and generate
     a new name instead. */
  if (data->action == GB_LOADING && data->xml_buffer && data->discard_names)
    {
      gchar *class_name = gb_widget_get_type_name (widget);
      name = glade_project_new_widget_name (data->project, class_name);
      gtk_widget_set_name (widget, name);

      if (widget->parent == NULL)
	glade_project_component_changed (data->project, widget);
    }
  else
    {
      name = gb_widget_input_string (data, GbName);
      if (data->apply)
	{
	  /* If we are loading, the name will be set when the widget is added
	     to the tree, so we don't need to set it here. */
	  if (data->action != GB_LOADING)
	    tree_rename_widget (widget, name);
	  gtk_widget_set_name (widget, name);
	  /* If widget is a toplevel window/dialog set the component's name in
	     the project window */
	  if (widget->parent == NULL)
	    glade_project_component_changed (data->project, widget);

	  /* If we are loading, we have to check if the name has a trailing ID
	     and if so we reserve it so no other widget can use it. */
	  if (data->action == GB_LOADING)
	    glade_project_reserve_name (data->project, name);
	}
    }

  /* If widget is a container, show the border width */
  if (GTK_IS_CONTAINER (widget))
    {
      gchar buf[128];
      gint border_width;
      gchar *class_name;

      class_name = gb_widget_get_type_name (widget);
      sprintf (buf, "%s::border_width", class_name);
      border_width = gb_widget_input_int (data, buf);
      if (data->apply && GTK_CONTAINER (widget)->border_width != border_width)
	{
	  gtk_container_set_border_width (GTK_CONTAINER (widget),
					  border_width);
	  if (data->action == GB_APPLYING)
	    editor_refresh_widget (widget);
	}
    }

  /* Language-specific properties. */
  set_lang_specific_properties (widget, data);

  /* Special child properties page */
  set_special_child_properties (widget, data);

  /* Properties on standard page */
  set_position_properties (widget, data);

  visible = gb_widget_input_bool (data, GbVisible);
  if (data->apply)
    {
      if (visible)
	wdata->flags |= GB_VISIBLE;
      else
	wdata->flags &= ~GB_VISIBLE;
    }

  sensitive = gb_widget_input_bool (data, GbSensitive);
  if (data->apply)
    {
      if (sensitive)
	wdata->flags |= GB_SENSITIVE;
      else
	wdata->flags &= ~GB_SENSITIVE;
    }

  tooltip = gb_widget_input_string (data, GbTooltip);
  if (data->apply)
    {
      g_free (wdata->tooltip);
      if (tooltip && tooltip[0] == '\0')
	tooltip = NULL;
      wdata->tooltip = g_strdup (tooltip);

      /* SPECIAL CODE: toolbar children's tooltips use the toolbar's tooltips
	 object rather than our tooltip object. */
      if (widget->parent && GTK_IS_TOOLBAR (widget->parent))
	gtk_tooltips_set_tip (GTK_TOOLBAR (widget->parent)->tooltips, 
			      widget, tooltip, NULL);
      else
	gtk_tooltips_set_tip (gb_widget_tooltips, widget, tooltip, NULL);
    }

  can_default = gb_widget_input_bool (data, GbCanDefault);
  if (data->apply)
    {
      if (can_default)
	GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_DEFAULT);
      else
	GTK_WIDGET_UNSET_FLAGS (widget, GTK_CAN_DEFAULT);
      if (data->action == GB_APPLYING)
	editor_refresh_widget (widget);
    }

  has_default = gb_widget_input_bool (data, GbHasDefault);
  if (data->apply)
    {
      if (has_default)
	wdata->flags |= GB_GRAB_DEFAULT;
      else
	wdata->flags &= ~GB_GRAB_DEFAULT;
    }

  /* Different widgets have different default values for CAN_FOCUS, so when
     we load an XML file we must make sure that we always set it or unset it.
     Also, since we don't save the can_focus flag if it is false, we must make
     sure that we apply it anyway when loading. */
  can_focus = gb_widget_input_bool (data, GbCanFocus);
  if (!data->apply)
    can_focus = FALSE;
  if (data->apply || data->action == GB_LOADING)
    {
      if (can_focus)
	GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_FOCUS);
      else
	GTK_WIDGET_UNSET_FLAGS (widget, GTK_CAN_FOCUS);
    }
  has_focus = gb_widget_input_bool (data, GbHasFocus);
  if (data->apply)
    {
      if (has_focus)
	wdata->flags |= GB_GRAB_FOCUS;
      else
	wdata->flags &= ~GB_GRAB_FOCUS;
    }

  /* Events & ext events. */
  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      if (data->action == GB_APPLYING)
	{
	  events = property_events_string_to_value (gb_widget_input_string (data,
								 GbEvents));
	  if (data->apply)
	    wdata->events = events;
	}
      else
	{
	  events_str = gb_widget_input_string (data, GbEvents);
	  if (data->apply)
	    {
	      for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
		{
		  if (glade_util_strstr (events_str, GbEventMaskSymbols[i]))
		    wdata->events |= GbEventMaskValues[i];
		}
	    }
	}

      ext_events = gb_widget_input_choice (data, GbExtEvents);
      if (data->apply)
	{
	  for (i = 0; GbExtensionModeChoices[i]; i++)
	    {
	      if (!strcmp (ext_events, GbExtensionModeChoices[i])
		  || !strcmp (ext_events, GbExtensionModeSymbols[i]))
		{
		  gtk_widget_set_extension_events (widget, GbExtensionModeValues
						   [i]);
		  break;
		}
	    }
	}
    }

  if (data->action == GB_APPLYING)
    {
#if 0
      apply_style (widget, data);
#endif
      apply_accelerators (widget, data);
      apply_signals (widget, data);
    }
  else
    {
      load_style (widget, data);
      data->widget_data->signals = data->signals;
      data->widget_data->accelerators = data->accelerators;
    }
}


static void
set_special_child_properties (GtkWidget * widget,
			      GbWidgetSetArgData * data)
{
  GtkWidget *parent = widget->parent;
  GbWidget *gbparent;

  if (!parent)
    return;

  /* When pasting a widget to replace an existing widget, the child properties
     will already have been set, so we just return. */
  if (data->action == GB_LOADING && data->xml_buffer && data->replacing_widget)
    return;

  gbparent = gb_widget_lookup (parent);

  /* Tell the load functions to use the child properties array. */
  data->loading_type = GB_CHILD_PROPERTIES;

  if (gbparent && gbparent->gb_widget_set_child_properties)
    {
      (gbparent->gb_widget_set_child_properties) (parent, widget, data);
    }

  data->loading_type = GB_STANDARD_PROPERTIES;
}


static void
set_position_properties (GtkWidget * widget,
			 GbWidgetSetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gint x, y, w, h;
  gboolean applyX, applyY, applyWidth, applyHeight;
  gboolean set_usize = FALSE, set_upos = FALSE;

  x = gb_widget_input_int (data, GbX);
  applyX = data->apply;
  y = gb_widget_input_int (data, GbY);
  applyY = data->apply;
  w = gb_widget_input_int (data, GbWidth);
  applyWidth = data->apply;
  h = gb_widget_input_int (data, GbHeight);
  applyHeight = data->apply;

  /* When loading we need to remember which values have been set explicitly. */
  if (data->action == GB_LOADING)
    {
      if (applyX)
	wdata->flags |= GB_X_SET;
      if (applyY)
	wdata->flags |= GB_Y_SET;
      if (applyWidth)
	wdata->flags |= GB_WIDTH_SET;
      if (applyHeight)
	wdata->flags |= GB_HEIGHT_SET;
    }

#if 0
  g_print ("In set_position_properties X:%i Y:%i W:%i H:%i\n", x, y, w, h);
#endif
  if (GTK_IS_WINDOW (widget))
    {
      if (applyX)
	wdata->x = x;
      if (applyY)
	wdata->y = y;

      if (applyWidth && wdata->width != w)
	{
	  wdata->width = w;
	  set_usize = TRUE;
	}
      if (applyHeight && wdata->height != h)
	{
	  wdata->height = h;
	  set_usize = TRUE;
	}

      if (set_usize)
	{
	  gint w = wdata->flags & GB_WIDTH_SET ? wdata->width : -1;
	  gint h = wdata->flags & GB_HEIGHT_SET ? wdata->height : -1;
	  gb_widget_set_usize (widget, w, h);
	}
    }
  else if (widget->parent && (GTK_IS_FIXED (widget->parent)
			      || GTK_IS_LAYOUT (widget->parent)))
    {
      /* When pasting a widget to replace an existing widget, the size &
	 position will be set in the replace_child function. */
      if (data->action == GB_LOADING && data->xml_buffer
	  && data->replacing_widget)
	{
	  return;
	}


      if (applyWidth && wdata->width != w)
	{
	  wdata->width = w;
	  set_usize = TRUE;
	}
      if (applyHeight && wdata->height != h)
	{
	  wdata->height = h;
	  set_usize = TRUE;
	}
      if (set_usize)
	gb_widget_set_usize (widget, wdata->width, wdata->height);

      if (applyX && wdata->x != x)
	{
	  wdata->x = x;
	  set_upos = TRUE;
	}
      if (applyY && wdata->y != y)
	{
	  wdata->y = y;
	  set_upos = TRUE;
	}
      if (set_upos)
	{
	  if (GTK_IS_FIXED (widget->parent))
	    {
	      /* FIXME: Do we need both? Widget with windows aren't moved if we
		 only use gtk_fixed_move(). */
	      gtk_fixed_move (GTK_FIXED (widget->parent), widget,
			      wdata->x, wdata->y);
	      gtk_widget_set_uposition (widget, wdata->x, wdata->y);
	    }
	  else
	    {
	      gtk_layout_move (GTK_LAYOUT (widget->parent), widget,
			       wdata->x, wdata->y);
	    }
	}
    }
  else
    {
      /* X & Y aren't applicable to widgets in standard containers, so we
	 make sure they aren't set. */
      wdata->flags &= ~(GB_X_SET | GB_Y_SET);

      if (applyWidth && wdata->width != w)
	{
	  wdata->width = w;
	  set_usize = TRUE;
	}
      if (applyHeight && wdata->height != h)
	{
	  wdata->height = h;
	  set_usize = TRUE;
	}

      if (set_usize)
	{
	  gint w = wdata->flags & GB_WIDTH_SET ? wdata->width : -1;
	  gint h = wdata->flags & GB_HEIGHT_SET ? wdata->height : -1;
	  gb_widget_set_usize (widget, w, h);
	}
      MSG2 ("*** Width set:%i Height set:%i", wdata->flags & GB_WIDTH_SET,
	    wdata->flags & GB_HEIGHT_SET);
    }
}


#if 0
static void
apply_style (GtkWidget * widget,
	     GbWidgetSetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  GtkStyle *style = widget->style, *old_style = NULL;
  GbStyle *gbstyle = wdata->gbstyle, *new_gbstyle;
  GdkFont *font = NULL;
  gchar *style_name, *xlfd_fontname;
  GdkColor fg[GB_NUM_STYLE_STATES];
  GdkColor bg[GB_NUM_STYLE_STATES];
  GdkColor text[GB_NUM_STYLE_STATES];
  GdkColor base[GB_NUM_STYLE_STATES];
  GdkPixmap *bg_pixmap[GB_NUM_STYLE_STATES];
  gchar *bg_pixmap_filenames[GB_NUM_STYLE_STATES];
  gint recreate = FALSE, redraw = FALSE, i;
  gchar buf[128], *filename;
  gboolean named_style;

  style_name = gb_widget_input_dialog (data, GbStyleName);
  named_style = (style_name[0] == '\0') ? FALSE : TRUE;
  if (data->apply)
    {
      if (named_style)
	{
	  new_gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, style_name);
	  g_return_if_fail (new_gbstyle != NULL);
	  if (new_gbstyle != gbstyle)
	    {
	      gbstyle = new_gbstyle;
	      gb_widget_set_gb_style (widget, gbstyle);
	      wdata->flags &= ~GB_STYLE_IS_UNNAMED;
	      redraw = TRUE;
	    }
	}
      else
	{
	  wdata->flags |= GB_STYLE_IS_UNNAMED;
	}
    }

  font = gb_widget_input_font (data, GbStyleFont, &xlfd_fontname);
  if (data->apply)
    {
      if (font != style->font)
	recreate = TRUE;
    }

  recreate |= apply_colors (widget, data, style->fg, fg, "fg");
  recreate |= apply_colors (widget, data, style->bg, bg, "bg");
  recreate |= apply_colors (widget, data, style->text, text, "text");
  recreate |= apply_colors (widget, data, style->base, base, "base");

  /* Background pixmaps */
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      sprintf (buf, "GtkStyle::%s[%s]", GbBgPixmapName, GbStateNames[i]);
      bg_pixmap[i] = gb_widget_input_bgpixmap (data, buf, &filename);
      bg_pixmap_filenames[i] = filename;
      if (data->apply)
	{
	  if (bg_pixmap[i] != style->bg_pixmap[i])
	    recreate = TRUE;
	}
    }

  if (recreate)
    {
      old_style = style;

      /* If the widget is supposedly using an unnamed GbStyle, but currently is
         actually using a named GbStyle (for convenience), then we need to
         create a copy of the GbStyle and place our new style in it. */
      if ((wdata->flags & GB_STYLE_IS_UNNAMED) && gbstyle->name)
	{
	  gbstyle = gb_widget_copy_gb_style (gbstyle);
	  g_free (gbstyle->name);
	  gbstyle->name = NULL;
	}

      style = gtk_style_new ();
      for (i = 0; i < GB_NUM_STYLE_STATES; i++)
	{
	  style->fg[i] = fg[i];
	  style->bg[i] = bg[i];
	  style->text[i] = text[i];
	  style->base[i] = base[i];
	  style->bg_pixmap[i] = bg_pixmap[i];
	  if (bg_pixmap[i])
	    gdk_pixmap_ref (bg_pixmap[i]);

	  if (gbstyle->bg_pixmap_filenames[i] != bg_pixmap_filenames[i])
	    {
	      g_free (gbstyle->bg_pixmap_filenames[i]);
	      gbstyle->bg_pixmap_filenames[i] = g_strdup (bg_pixmap_filenames
							  [i]);
	    }
	}
      if (font)
	{
	  gdk_font_unref (style->font);
	  style->font = font;
	  gdk_font_ref (style->font);
	}
      if (strcmp (gbstyle->xlfd_fontname, xlfd_fontname))
	{
	  g_free (gbstyle->xlfd_fontname);
	  gbstyle->xlfd_fontname = g_strdup (xlfd_fontname);
	}

      gbstyle->style = style;
      gtk_style_ref (style);
      gb_widget_set_gb_style (widget, gbstyle);
    }


  /* If a named style has been changed/recreated we have to update all
     widget's that use it. */
  if (recreate || redraw)
    {
      if (named_style)
	{
	  gb_widget_update_gb_styles (gbstyle, gbstyle);
	}
      else
	{
	  editor_refresh_widget (widget);
	}
    }

  if (old_style)
    gtk_style_unref (old_style);
}


/* This makes sure a widget's gbstyle & its style are up to date, and
   if the propagate flag is set it also updates any children.
   But it won't change a descendant's style if it has been set explicitly.
   FIXME: only propagates one level at present, and always sets child's style,
   even if it has a different GbStyle! */
void
gb_widget_set_gb_style (GtkWidget * widget,
			GbStyle * gbstyle)
{
  GbWidgetData *wdata;

  if (!GB_IS_PLACEHOLDER (widget))
    {
      if (widget->style != gbstyle->style)
	gtk_widget_set_style (widget, gbstyle->style);
    }

  wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  if (wdata)
    {
      if (wdata->gbstyle != gbstyle)
	{
	  gb_widget_unref_gb_style (wdata->gbstyle);
	  wdata->gbstyle = gbstyle;
	  gb_widget_ref_gb_style (gbstyle);
	}
      /* If propagate style flags is set, propagate style to children */
      if (wdata->flags & GB_STYLE_PROPAGATE)
	gb_widget_children_foreach (widget, (GtkCallback) gb_widget_set_gb_style,
			   gbstyle);
    }
}


static gboolean
apply_colors (GtkWidget * widget, GbWidgetSetArgData * data, GdkColor colors[],
	      GdkColor new_colors[], gchar * name)
{
  gint state;
  gchar buf[128];
  GdkColor *color;
  gboolean need_redraw = FALSE;

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      sprintf (buf, "GtkStyle::%s[%s]", name, GbStateNames[state]);

      color = gb_widget_input_color (data, buf);
      if (data->apply)
	{
	  new_colors[state] = *color;
	  if (!gdk_color_equal (&new_colors[state], &colors[state]))
	    need_redraw = TRUE;
	}
      else
      {
	/* Copy the old values across. */
	new_colors[state] = colors[state];
      }
    }
  return need_redraw;
}
#endif


/* Currently this frees all the GbAccelerators and creates them from scratch */
static void
apply_accelerators (GtkWidget * widget, GbWidgetSetArgData * data)
{
  if (data->property_to_apply == NULL
      || property_is_accel_clist (data->property_to_apply))
    {
      gb_widget_free_accelerators (widget);
      data->widget_data->accelerators = property_get_accelerators ();
    }
}


/* Currently this frees all the GbSignals and creates them from scratch. */
static void
apply_signals (GtkWidget * widget, GbWidgetSetArgData * data)
{
  if (data->property_to_apply == NULL
      || property_is_signal_clist (data->property_to_apply))
    {
      gb_widget_free_signals (widget);
      data->widget_data->signals = property_get_signals ();
    }
}



/*************************************************************************
 * Functions for showing the popup context-sensitive menu of a widget
 *************************************************************************/

void
gb_widget_show_popup_menu (GtkWidget * widget,
			   GdkEventButton * event)
{
  static GtkWidget *menu = NULL;

  GbWidget *gbwidget;
  gchar *name;
  GtkWidget *menuitem, *ancestor, *submenu, *child;
  GbWidgetCreateMenuData data;

  gbwidget = gb_widget_lookup (widget);
  g_return_if_fail (gbwidget != NULL);

  name = gtk_widget_get_name (widget);
  if (GB_IS_PLACEHOLDER (widget))
    name = "Placeholder";

  /* FIXME: bug workaround - delete last popup menu */
  if (menu)
    gtk_widget_destroy (menu);

  menu = gtk_menu_new ();
  /* doesn't call menuitem's 'activate' handler if we do this
     gtk_signal_connect_object (GTK_OBJECT (menu), "deactivate",
     GTK_SIGNAL_FUNC(gtk_widget_destroy),
     GTK_OBJECT(menu));
   */
  menuitem = gtk_menu_item_new_with_label (name);
  gtk_widget_show (menuitem);
  gtk_widget_set_sensitive (menuitem, FALSE);
  gtk_menu_append (GTK_MENU (menu), menuitem);

  data.project = current_project;
  data.menu = menu;
  data.child = NULL;
  add_standard_top_menu_items (widget, &data);

  if (gbwidget->gb_widget_create_popup_menu)
    (gbwidget->gb_widget_create_popup_menu) (widget, &data);

  add_standard_bottom_menu_items (widget, &data);

  child = widget;
  ancestor = widget->parent;
  while (ancestor)
    {
      name = gtk_widget_get_name (ancestor);
      if (GB_IS_PLACEHOLDER (ancestor))
	name = "Placeholder";

      /* Skip widgets which aren't GbWidgets */
      if (GB_IS_GB_WIDGET (ancestor))
	{
	  /* Add a separator */
	  menuitem = gtk_menu_item_new ();
	  gtk_menu_append (GTK_MENU (menu), menuitem);
	  gtk_widget_show (menuitem);

	  menuitem = gtk_menu_item_new_with_label (name);
	  gtk_widget_show (menuitem);
	  gtk_menu_append (GTK_MENU (menu), menuitem);

	  /* Create submenu */
	  submenu = gtk_menu_new ();
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), submenu);

	  data.menu = submenu;
	  data.child = child;
	  add_standard_top_menu_items (ancestor, &data);

	  /* Call ancestors function to add any menu items */
	  gbwidget = gb_widget_lookup (ancestor);
	  if (gbwidget != NULL && gbwidget->gb_widget_create_popup_menu)
	    (gbwidget->gb_widget_create_popup_menu) (ancestor, &data);

	  add_standard_bottom_menu_items (ancestor, &data);
	}
      child = ancestor;
      ancestor = ancestor->parent;
    }

  MSG ("showing popup menu");
  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
		  event->button, event->time);
}


static void
add_standard_top_menu_items (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;

  menuitem = gtk_menu_item_new_with_label (_("Select"));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_on_select_activate), widget);
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
}


static void
add_standard_bottom_menu_items (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;

  /* For widgets which can handle scrolling, we add commands to add or remove
     a parent scrolled window. */
  if (GTK_WIDGET_CLASS (GTK_OBJECT (widget)->klass)->set_scroll_adjustments_signal)
    {
      if (widget->parent && GTK_IS_SCROLLED_WINDOW (widget->parent))
	{
	  menuitem = gtk_menu_item_new_with_label (_("Remove Scrolled Window"));
	  gtk_widget_show (menuitem);
	  gtk_menu_append (GTK_MENU (data->menu), menuitem);
	  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			      GTK_SIGNAL_FUNC (gb_widget_remove_scrolled_window),
			      widget);
	}
      else
	{
	  menuitem = gtk_menu_item_new_with_label (_("Add Scrolled Window"));
	  gtk_widget_show (menuitem);
	  gtk_menu_append (GTK_MENU (data->menu), menuitem);
	  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			      GTK_SIGNAL_FUNC (gb_widget_add_scrolled_window),
			      widget);
	}
    }

  if (!GTK_IS_WINDOW (widget)
      && !GTK_IS_MISC (widget)
      && !GTK_IS_ALIGNMENT (widget)
      && !GB_IS_PLACEHOLDER (widget))
    {
      if (GTK_IS_ALIGNMENT (widget->parent))
	{
	  menuitem = gtk_menu_item_new_with_label (_("Remove Alignment"));
	  gtk_widget_show (menuitem);
	  gtk_menu_append (GTK_MENU (data->menu), menuitem);
	  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_widget_remove_alignment), widget);
	}
      else
	{
	  menuitem = gtk_menu_item_new_with_label (_("Add Alignment"));
	  gtk_widget_show (menuitem);
	  gtk_menu_append (GTK_MENU (data->menu), menuitem);
	  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			 GTK_SIGNAL_FUNC (gb_widget_add_alignment), widget);
	}
    }

  if (GTK_WIDGET_NO_WINDOW (widget)
      && !GTK_IS_EVENT_BOX (widget)
      && !GB_IS_PLACEHOLDER (widget))
    {
      if (GTK_IS_EVENT_BOX (widget->parent))
	{
	  menuitem = gtk_menu_item_new_with_label (_("Remove Event Box"));
	  gtk_widget_show (menuitem);
	  gtk_menu_append (GTK_MENU (data->menu), menuitem);
	  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (gb_widget_remove_event_box), widget);
	}
      else
	{
	  menuitem = gtk_menu_item_new_with_label (_("Add Event Box"));
	  gtk_widget_show (menuitem);
	  gtk_menu_append (GTK_MENU (data->menu), menuitem);
	  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			 GTK_SIGNAL_FUNC (gb_widget_add_event_box), widget);
	}
    }

  if (GTK_IS_WINDOW (widget))
    {
      menuitem = gtk_menu_item_new_with_label (_("Redisplay"));
      gtk_widget_show (menuitem);
      gtk_menu_append (GTK_MENU (data->menu), menuitem);
      gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			  GTK_SIGNAL_FUNC (gb_widget_redisplay_window),
			  widget);
    }

  menuitem = gtk_menu_item_new_with_label (_("Cut"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_on_cut_activate), widget);
 
  menuitem = gtk_menu_item_new_with_label (_("Copy"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_on_copy_activate), widget);
 
  menuitem = gtk_menu_item_new_with_label (_("Paste"));
  gtk_widget_show (menuitem);
  if (!widget->parent)
    gtk_widget_set_sensitive (menuitem, FALSE);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_on_paste_activate), widget);
 
  menuitem = gtk_menu_item_new_with_label (_("Delete"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (editor_on_delete_activate), widget);
}



/*************************************************************************
 * Functions for loading project files
 *************************************************************************/

/* This reads in a widget, and calls itself recursively to read it's children.
   It assumes a '<widget>' has just been read, and reads up to and including
   '</widget>'. The newly-created widget is returned, mainly so that
   glade_clipboard_paste () can show any windows which are pasted. */
GtkWidget*
gb_widget_load (GtkWidget * widget, GbWidgetSetArgData * data, GtkWidget * parent)
{
  gchar *class_name, *child_name;
  GbWidget *gbwidget, *ancestor_gbwidget;
  GtkWidget *ancestor;
  GbWidgetData *wdata;

  data->action = GB_LOADING;
  data->loading_type = GB_STANDARD_PROPERTIES;
  gb_widget_load_properties (data);

  if (data->status != GLADE_STATUS_OK)
    {
      MSG ("Load error");
      return NULL;
    }

#ifdef GLADE_DEBUG
  {
    gint i;
    for (i = 0; i < data->properties.n; i++)
      {
	MSG3 ("Line number: %i Property: %s Value: %s",
	      data->properties.ptr[i].line_number,
	      data->buffer.ptr + data->properties.ptr[i].tag_index,
	      data->buffer.ptr + data->properties.ptr[i].cdata_index);

      }
  }
#endif
  class_name = load_get_value (data, "class");
  MSG1 ("Reading Class: %s", class_name);
  if (!class_name)
    {
      MSG ("Load error");
      data->status = GLADE_STATUS_CLASS_NAME_MISSING;
      return NULL;
    }

  child_name = load_get_value (data, "child_name");
  /* SPECIAL CODE: when pasting a widget to replace an existing widget, check
     if the child name should be transferred to the new widget. */
  if (data->xml_buffer && data->replacing_widget && child_name)
    {
      /* These child names should never be transferred. */
      if (!strcmp (child_name, "FileSel:ok_button")
	  || !strcmp (child_name, "FileSel:cancel_button")
	  || !strcmp (child_name, "ColorSel:ok_button")
	  || !strcmp (child_name, "ColorSel:cancel_button")
	  || !strcmp (child_name, "ColorSel:help_button")
	  || !strcmp (child_name, "FontSel:ok_button")
	  || !strcmp (child_name, "FontSel:cancel_button")
	  || !strcmp (child_name, "FontSel:apply_button")
	  || !strcmp (child_name, "InputDialog:save_button")
	  || !strcmp (child_name, "InputDialog:close_button")
	  || !strcmp (child_name, "Dialog:vbox")
	  || !strcmp (child_name, "Dialog:action_area")
	  || !strcmp (child_name, "GtkCombo:entry")
	  || !strcmp (child_name, "GnomeApp:dock")
	  || !strcmp (child_name, "GnomeApp:appbar")
	  || !strcmp (child_name, "GnomeDialog:vbox")
	  || !strcmp (child_name, "GnomeDialog:action_area")
	  || !strcmp (child_name, "GnomeEntry:entry")
	  )
	{
	  child_name = NULL;
	}
      else if (!strcmp (child_name, "Toolbar:button"))
	{
	  /* Toolbar buttons only retain their child name if they are pasted
	     into a toolbar. */
	  if (parent == NULL || !GTK_IS_TOOLBAR (parent))
	    child_name = NULL;
	}
    }

  if (!g_strcasecmp (class_name, "placeholder"))
    {
      MSG ("found placeholder");
      widget = editor_new_placeholder ();
      if (!gb_widget_add_child (parent, data, widget))
	{
	  data->status = GLADE_STATUS_ERROR;
	  return NULL;
	}
      if (child_name)
	gb_widget_set_child_name (widget, child_name);
      return NULL;
    }

  gbwidget = gb_widget_lookup_class (class_name);
  if (gbwidget == NULL)
    {
      MSG ("Load error");
      data->status = GLADE_STATUS_CLASS_UNKNOWN;
      load_add_error_message_with_tag (data, GLADE_LINE_PROPERTY,
				       glade_get_error_message (data->status),
				       "class", class_name);
      return NULL;
    }

  /* If this is a special child of a widget, step through the ancestors of
     the widget and try to find it. Note that some special child widgets
     have to be created normally, and then added by the parent container,
     e.g. notebook tabs and clist titles - see gb_widget_add_child(). */
  if (child_name)
    {
      MSG1 ("Child name: %s", child_name);
      ancestor = parent;
      while (ancestor)
	{
	  ancestor_gbwidget = gb_widget_lookup (ancestor);
	  if (ancestor_gbwidget && ancestor_gbwidget->gb_widget_get_child)
	    {
	      widget = (ancestor_gbwidget->gb_widget_get_child) (ancestor,
								 child_name);
	      if (widget)
		break;
	    }

	  ancestor = ancestor->parent;
	}

#if 0
      if (!widget)
	g_print ("Child widget %s not found - may be a problem\n", child_name);
#endif
    }

  /* If this is a standard widget, we need to create it and add it to its
     parent. If the widget has already been created by its parent, we can
     just set the properties. */
  if (widget == NULL)
    {
      MSG ("widget == NULL, has not been created by parent.");
      widget = gb_widget_new_full (class_name, FALSE, parent, NULL,
				   0, 0, NULL, GB_LOADING, data);
      wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
      data->widget_data = wdata;

      if (parent == NULL)
	{
	  glade_project_add_component (data->project, widget);
	}
      else
	{
	  if (!gb_widget_add_child (parent, data, widget))
	    {
	      data->status = GLADE_STATUS_ERROR;
	      return NULL;
	    }
	}

      /* This must come after add_child so we can set special child properties,
         and also we may need to realize the widget. It must also come after
	 glade_project_add_component(), in case setting any properties results
	 in a signal being emitted from the project. */
      set_standard_properties (widget, data);

      tree_add_widget (widget);
    }
  else
    {
      MSG ("widget != NULL, has been created by parent.");
      wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
      if (wdata == NULL)
	g_warning ("Null widget_data\n");
      data->widget_data = wdata;
      set_standard_properties (widget, data);
      tree_add_widget (widget);
    }

  if (child_name)
    gb_widget_set_child_name (widget, child_name);

  if (data->status != GLADE_STATUS_OK)
    {
      MSG ("Load error");
      return NULL;
    }

  MSG ("Calling widgets set_properties()");
  if (gbwidget->gb_widget_set_properties)
    (gbwidget->gb_widget_set_properties) (widget, data);
  MSG ("Called widgets set_properties()");

  if (data->status != GLADE_STATUS_OK)
    {
      MSG ("Load error");
      return NULL;
    }

  /* All the necessary properties should have been copied from the buffer
     by now, se we can clear it. Note: actually we use it below but that
     should be OK?? */
  load_buffer_release (data);

  /* When pasting, we only ever replace the top widget. All children are added
     as normal. */
  data->replacing_widget = FALSE;

  MSG ("Loading children");

  /* load children. */
  while (data->token_type == GB_TOKEN_START_TAG)
    {
      if (!strcmp (data->buffer.ptr + data->token, "widget"))
	{
	  /* We have to reset the widget_data since this may have been changed
	     while loading the last child (and its children) */
	  data->widget_data = wdata;
	  gb_widget_load (NULL, data, widget);
	  if (data->status != GLADE_STATUS_OK)
	    {
	      MSG ("Load error");
	      return NULL;
	    }
	}
      else
	{
	  break;
	}
      load_token_skip_whitespace (data);
      if (data->status != GLADE_STATUS_OK)
	{
	  MSG ("Load error");
	  return NULL;
	}
    }

  MSG ("Loaded children");

  if (data->token_type != GB_TOKEN_END_TAG
      || strcmp (data->buffer.ptr + data->token, "widget"))
    {
      data->status = GLADE_STATUS_END_TAG_EXPECTED;
      load_add_error_message (data, data->line_number, _("End tag expected"),
			      "widget");
      MSG ("Load error");
      return NULL;
    }

  /* If the widget is a table, we make sure all empty cells have placeholders
     in them. */
  if (GTK_IS_TABLE (widget))
    {
      gb_table_update_placeholders (widget, -1, -1);
    }

  return widget;
}


/* Adds a widget to a parent, when loading or pasting from the clipboard.
   Returns TRUE on success. */
gboolean
gb_widget_add_child (GtkWidget * parent,
		     GbWidgetSetArgData * data,
		     GtkWidget * child)
{
  GbWidget *gbwidget;

  MSG2 ("Adding %s to %s", gtk_widget_get_name (child),
	parent ? gtk_widget_get_name (parent) : "NULL");

  g_return_val_if_fail (parent != NULL, FALSE);

  /* If we are pasting, the top widget replaces the parent rather than being
     added to it. */
  if (data->xml_buffer && data->replacing_widget)
    {
      return gb_widget_replace_child (parent, data->replacing_widget, child);
    }

  gbwidget = gb_widget_lookup (parent);
  g_return_val_if_fail (gbwidget != NULL, FALSE);

  if (gbwidget->gb_widget_add_child)
    {
      gbwidget->gb_widget_add_child (parent, child, data);
    }
  else if (GTK_IS_BIN (parent))
    {
      if (GTK_BIN (parent)->child)
	gtk_container_remove (GTK_CONTAINER (parent), GTK_BIN (parent)->child);
      gtk_container_add (GTK_CONTAINER (parent), child);
    }
  else if (GTK_IS_CONTAINER (parent))
    {
      gtk_container_add (GTK_CONTAINER (parent), child);
    }

  return TRUE;
}


/* This will load all the properties/accelerators/signals of a widget.
   It assumes a <widget> has just been read and reads up to a <widget>,
   </widget>. */
static void
gb_widget_load_properties (GbWidgetSetArgData * data)
{
  gint property;

  /* Initialise the widget's properties. */
  data->properties.n = 0;
  data->child_properties.n = 0;

  data->accelerators = NULL;
  data->signals = NULL;
#ifdef GLADE_STYLE_SUPPORT
  data->gbstyle = NULL;
#endif

  for (;;)
    {
      load_token_skip_whitespace (data);
      if (data->status != GLADE_STATUS_OK)
	{
	  MSG ("Load error");
	  return;
	}

      if (!strcmp (data->buffer.ptr + data->token, "widget"))
	return;

      if (data->token_type != GB_TOKEN_START_TAG)
	{
	  data->status = GLADE_STATUS_START_TAG_EXPECTED;
	  MSG ("Load error");
	  return;
	}

      if (!strcmp (data->buffer.ptr + data->token, "style"))
	gb_widget_load_style (data);
      else if (!strcmp (data->buffer.ptr + data->token, "accelerator"))
	load_accelerator (data);
      else if (!strcmp (data->buffer.ptr + data->token, "signal"))
	load_signal (data);
      else if (!strcmp (data->buffer.ptr + data->token, "child"))
	load_special_child_properties (data);
      else
	{
	  property = data->token;
	  load_token (data);
	  if (data->status != GLADE_STATUS_OK)
	    {
	      MSG ("Load error");
	      return;
	    }

	  /* Remember the property could be empty, so we may get the end tag.*/
	  if (data->token_type == GB_TOKEN_START_TAG)
	    {
	      data->status = GLADE_STATUS_DATA_EXPECTED;
	      load_add_error_message (data, data->line_number,
				      _("Data expected"),
				      data->buffer.ptr + property);
	      MSG ("Load error");
	      return;
	    }

	  if (data->token_type == GB_TOKEN_DATA)
	    {
              gb_add_properties (property, data->token, data->line_number,
				 &data->properties);

	      load_token (data);
	      if (data->status != GLADE_STATUS_OK)
		{
		  MSG ("Load error");
		  return;
		}
	      if (data->token_type != GB_TOKEN_END_TAG
		  || strcmp (data->buffer.ptr + data->token,
			     data->buffer.ptr + property))
		{
		  data->status = GLADE_STATUS_END_TAG_EXPECTED;
		  load_add_error_message (data, data->line_number,
					  _("End tag expected"),
					  data->buffer.ptr + property);
		  MSG ("Load error");
		  return;
		}
	    }
	  else
	    /* An empty tag may have been read. Check the start and end tags
	       match. */
	    {
	      if (strcmp (data->buffer.ptr + data->token, data->buffer.ptr + property))
		{
		  data->status = GLADE_STATUS_END_TAG_EXPECTED;
		  load_add_error_message (data, data->line_number,
					  _("End tag expected"),
					  data->buffer.ptr + property);
		  MSG ("Load error");
		  return;
		}
              gb_add_properties (property, 0 ,data->line_number,
				 &data->properties); 
	    }
	}
    }
}


/* This will load all the child properties of a widget, e.g. its packing
   options. It assumes a <child> has just been read, and reads up to to
   </child>. */
static void
load_special_child_properties (GbWidgetSetArgData * data)
{
  gint element_index, cdata_index;

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);

      if (data->status != GLADE_STATUS_OK)
	{
	  if (data->status == GLADE_STATUS_START_TAG_EXPECTED
	      && data->token_type == GB_TOKEN_END_TAG
	      && (!strcmp (data->buffer.ptr + data->token, "child")))
	    {
	      data->status = GLADE_STATUS_OK;
	    }
	  return;
	}

      gb_add_properties (element_index, cdata_index, data->line_number,
			 &data->child_properties);
    }
}


/* This will load all the accelerators of a widget. It assumes a <accelerator>
   has just been read and reads up to to </accelerator>. */
static void
load_accelerator (GbWidgetSetArgData * data)
{
  GbAccelerator *accel;
  gint element_index, cdata_index;
  gchar *element, *cdata;

  accel = g_new (GbAccelerator, 1);
  accel->modifiers = 0;
  accel->key = NULL;
  accel->signal = NULL;

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);
      element = data->buffer.ptr + element_index;
      cdata = data->buffer.ptr + cdata_index;

      if (data->status != GLADE_STATUS_OK)
	{
	  if (data->status == GLADE_STATUS_START_TAG_EXPECTED
	      && data->token_type == GB_TOKEN_END_TAG
	      && (!strcmp (data->buffer.ptr + data->token, "accelerator")))
	    {
	      data->accelerators = g_list_append (data->accelerators, accel);
	      data->status = GLADE_STATUS_OK;
	    }
	  else
	    {
	      g_free (accel->key);
	      g_free (accel->signal);
	      g_free (accel);
	    }
	  return;
	}

      if (!strcmp (element, "modifiers"))
	{
	  /* We simple scan for the flags in the cdata, although they should
	     be separated by '|'. */
	  if (glade_util_strstr (cdata, "GDK_CONTROL_MASK"))
	    accel->modifiers |= GDK_CONTROL_MASK;
	  if (glade_util_strstr (cdata, "GDK_SHIFT_MASK"))
	    accel->modifiers |= GDK_SHIFT_MASK;
	  if (glade_util_strstr (cdata, "GDK_MOD1_MASK"))
	    accel->modifiers |= GDK_MOD1_MASK;
	}
      else if (!strcmp (element, "key"))
	{
	  if ((strlen (cdata) < 5) || strncmp (cdata, "GDK_", 4))
	    {
	      load_add_error_message_with_tag (data, GLADE_LINE_PROPERTY,
					       _("Invalid accelerator key"),
					       element, cdata);
	    }
	  else
	    {
	      g_free (accel->key);
	      /* The initial 'GDK_' is removed. */
	      accel->key = g_strdup (cdata + 4);
	    }
	}
      else if (!strcmp (element, "signal"))
	{
	  g_free (accel->signal);
	  accel->signal = g_strdup (cdata);
	}
      else
	load_add_error_message_with_tag (data, GLADE_LINE_PROPERTY,
					 _("Unknown tag"), element, cdata);
    }
}


/* This will load all the accelerators of a widget. It assumes a <signal>
   has just been read and reads up to to </signal>. */
static void
load_signal (GbWidgetSetArgData * data)
{
  GbSignal *signal;
  gint element_index, cdata_index;
  gchar *element, *cdata;

  signal = g_new (GbSignal, 1);
  signal->name = NULL;
  signal->handler = NULL;
  signal->object = NULL;
  signal->after = FALSE;
  signal->data = NULL;
  signal->last_modification_time = 0;

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);
      element = data->buffer.ptr + element_index;
      cdata = data->buffer.ptr + cdata_index;

      if (data->status != GLADE_STATUS_OK)
	{
	  if (data->status == GLADE_STATUS_START_TAG_EXPECTED
	      && data->token_type == GB_TOKEN_END_TAG
	      && (!strcmp (data->buffer.ptr + data->token, "signal")))
	    {
	      data->signals = g_list_append (data->signals, signal);
	      data->status = GLADE_STATUS_OK;
	      MSG ("Signal loaded OK");
	    }
	  else
	    {
	      g_free (signal->name);
	      g_free (signal->handler);
	      g_free (signal->object);
	      g_free (signal->data);
	      g_free (signal);
	      MSG1 ("ERROR loading signal: %i", data->status);
	    }
	  return;
	}

      MSG2 ("Signal element:%s cdata:%s", element, cdata);

      if (!strcmp (element, "name"))
	{
	  g_free (signal->name);
	  signal->name = g_strdup (cdata);
	}
      else if (!strcmp (element, "handler"))
	{
	  g_free (signal->handler);
	  signal->handler = g_strdup (cdata);
	}
      else if (!strcmp (element, "object"))
	{
	  g_free (signal->object);
	  signal->object = g_strdup (cdata);
	}
      else if (!strcmp (element, "after"))
	{
	  signal->after = load_parse_bool (data, cdata);
	  if (data->status == GLADE_STATUS_INVALID_VALUE)
	    {
	      signal->after = FALSE;
	      load_add_error_message_with_tag (data, GLADE_LINE_CURRENT,
					       _("Invalid boolean value"),
					       element, cdata);
	      data->status = GLADE_STATUS_OK;
	    }
	}
      else if (!strcmp (element, "data"))
	{
	  g_free (signal->data);
	  signal->data = g_strdup (cdata);
	}
      else if (!strcmp (element, "last_modification_time"))
	{
	  signal->last_modification_time = load_parse_date (data, cdata);
	  if (data->status == GLADE_STATUS_INVALID_VALUE)
	    {
	      signal->last_modification_time = 0;
	      load_add_error_message_with_tag (data, GLADE_LINE_CURRENT,
					       _("Invalid date"), element,
					       cdata);
	      data->status = GLADE_STATUS_OK;
	    }
	}
      else
	load_add_error_message (data, data->line_number, _("Unknown tag"),
				element);
    }
}


/* This will load a style. It assumes a <style> has just been read and reads
   up to to </style>. If the style is named it is added to the GbStyle hash.
   The created GbStyle is returned in data->gbstyle. */
void
gb_widget_load_style (GbWidgetSetArgData * data)
{
  gchar *element, *cdata;
  gint element_index, cdata_index;

  /* This is just a dummy function to skip any styles in the XML for
     backwards compatability. */

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);
      element = data->buffer.ptr + element_index;
      cdata = data->buffer.ptr + cdata_index;

      if (data->status != GLADE_STATUS_OK)
	{
	  if (data->status == GLADE_STATUS_START_TAG_EXPECTED
	      && data->token_type == GB_TOKEN_END_TAG
	      && (!strcmp (data->buffer.ptr + data->token, "style")))
	    {
	      return;
	    }
	}
    }
}

#if 0
void
gb_widget_load_style (GbWidgetSetArgData * data)
{
  GbStyle *gbstyle;
  GtkStyle *style;
  gchar *element, *cdata;
  gint element_index, cdata_index;
  gboolean loading_default_style = FALSE;

  gbstyle = gb_widget_new_gb_style ();
  /* We copy the default style, since only changed values will appear in the
     input file. */
  style = gbstyle->style = gtk_style_copy (gb_widget_default_gb_style->style);
  gtk_style_ref (style);

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);
      element = data->buffer.ptr + element_index;
      cdata = data->buffer.ptr + cdata_index;

      if (data->status != GLADE_STATUS_OK)
	{
	  if (data->status == GLADE_STATUS_START_TAG_EXPECTED
	      && data->token_type == GB_TOKEN_END_TAG
	      && (!strcmp (data->buffer.ptr + data->token, "style")))
	    {
	      /* Make sure the new GbStyle has a font name set. */
	      if (!gbstyle->xlfd_fontname)
		{
		  gbstyle->xlfd_fontname
		    = g_strdup (gb_widget_default_gb_style->xlfd_fontname);
		}

	      /* If we are loading the Default style, replace current one. */
	      if (loading_default_style)
		{
		  gb_widget_destroy_gb_style (gb_widget_default_gb_style, TRUE);
		  gb_widget_default_gb_style = gbstyle;
		}

	      /* If the style is named, add it to the hash. */
	      if (gbstyle->name)
		g_hash_table_insert (gb_style_hash, gbstyle->name, gbstyle);

	      data->status = GLADE_STATUS_OK;
	      return;
	    }
	  MSG ("Load error");
	  return;
	}

      if (!strcmp (element, "style_name"))
	{
	  g_free (gbstyle->name);
	  gbstyle->name = g_strdup (cdata);
	  if (!strcmp (gbstyle->name, GB_STYLE_DEFAULT))
	    loading_default_style = TRUE;
	  MSG1 ("Loading named style:%s", gbstyle->name);
	}
      else if (!strcmp (element, "style_font"))
	{
	  GdkFont *font;
	  g_free (gbstyle->xlfd_fontname);
	  gbstyle->xlfd_fontname = g_strdup (cdata);
	  font = gdk_font_load (cdata);
	  if (font)
	    {
	      gdk_font_unref (style->font);
	      style->font = font;
	      gdk_font_ref (font);
	    }
	  else
	    load_add_error_message_with_tag (data, GLADE_LINE_CURRENT,
					     _("Couldn't load font"),
					     element, cdata);
	}
      else
	{
	  gboolean found;

	  found = load_colors (data, style->fg, "fg", element, cdata);
	  if (!found)
	    found = load_colors (data, style->bg, "bg", element, cdata);
	  if (!found)
	    found = load_colors (data, style->text, "text", element, cdata);
	  if (!found)
	    found = load_colors (data, style->base, "base", element, cdata);

	  /* Check if element is a color or bgpixmap of one of the states. */
	  if (!found)
	    {
	      gint state;
	      gchar buf[128];
	      GdkPixmap *gdkpixmap;

	      for (state = 0; state < GB_NUM_STYLE_STATES; state++)
		{
		  /* Remember the tag will be converted to lowercase. */
		  sprintf (buf, "GtkStyle::%s-%s", GbBgPixmapName,
			   GbStateNames[state]);
		  g_strdown (buf);
		  if (!strcmp (element,
			       glade_util_find_start_of_tag_name (buf))
		      && cdata[0] != '\0')
		    {
		      g_free (gbstyle->bg_pixmap_filenames[state]);
		      gbstyle->bg_pixmap_filenames[state] = g_strdup (cdata);

		      gdkpixmap = gdk_pixmap_create_from_xpm (NULL,
							      NULL,
					       &style->bg[GTK_STATE_NORMAL],
							      cdata);
		      if (gdkpixmap)
			{
			  style->bg_pixmap[state] = gdkpixmap;
			}
		      else
			load_add_error_message_with_tag (data,
							 GLADE_LINE_CURRENT,
							 _("Couldn't load pixmap"),
							 element, cdata);

		      found = TRUE;
		    }
		}
	    }

	  if (!found)
	    load_add_error_message_with_tag (data, GLADE_LINE_CURRENT,
					     _("Unknown tag"), element, cdata);
	}
    }
}


static gboolean
load_colors (GbWidgetSetArgData * data, GdkColor colors[], gchar * name,
	     gchar * element, gchar * cdata)
{
  gint state;
  gchar buf[128];
  GdkColor *color;

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      /* Remember the tag will be converted to lowercase. */
      sprintf (buf, "GtkStyle::%s-%s", name, GbStateNames[state]);
      g_strdown (buf);
      if (!strcmp (element, glade_util_find_start_of_tag_name (buf)))
	{
	  color = load_parse_color (data, cdata);
	  if (color)
	    {
	      MSG3 ("RGB: %i,%i,%i", color->red, color->green, color->blue);
	      colors[state] = *color;
	      return TRUE;
	    }
	  else
	    {
	      load_add_error_message_with_tag (data, GLADE_LINE_PROPERTY,
					       _("Invalid color"), element,
					       cdata);
	      return TRUE;
	    }
	}
    }
  return FALSE;
}
#endif


/* This applies the loaded style to the widget. */
static void
load_style (GtkWidget * widget, GbWidgetSetArgData * data)
{
  gchar *gbstyle_name;

  /* Check if its using a named style. */
  gbstyle_name = load_get_value (data, GbStyleName);
  if (gbstyle_name)
    {

    }

}


/* Utility functions for loading. */
void
gb_init_load_properties (GbLoadPropControl *p)
{
  p->n = 0;
  p->space = GB_LOADED_PROPERTIES_INC;
  p->ptr = g_new (GbLoadedProperty, GB_LOADED_PROPERTIES_INC);
}


static void
gb_add_properties (gint tag_index, gint cdata_index,
		   gint line_number, GbLoadPropControl *p)
{
    gb_realloc_properties_if_needed (p);
    p->ptr[p->n].tag_index   = tag_index;
    p->ptr[p->n].cdata_index = cdata_index;
    p->ptr[p->n].line_number = line_number;
    p->n++;
}


static void
gb_realloc_properties_if_needed (GbLoadPropControl *p)
{
  if (p->n == p->space)
    {
      p->space += GB_LOADED_PROPERTIES_INC;
      p->ptr = g_realloc (p->ptr, sizeof(GbLoadedProperty) * p->space);
    }
}


void
gb_free_load_properties (GbLoadPropControl *p)
{
  p->n = 0;
  p->space = 0;
  if (p->ptr)
     g_free (p->ptr);
  p->ptr = NULL;
}


void
gb_init_buffer_struct (int pos, int space, GbBuffControl * p)
{
  p->space = space;
  p->ptr = g_new (gchar, p->space);
  p->ptr[0] = '\0';
  p->pos = pos;  
}


/*************************************************************************
 * Functions for saving project files
 *************************************************************************/

void
gb_widget_save (GtkWidget * widget,
		GbWidgetGetArgData * data)
{
  GbWidget *gbwidget, *parent_gbwidget;
  GbWidgetData *widget_data;
  gchar *class_name, *child_name;

  child_name = gb_widget_get_child_name (widget);

  /* SPECIAL CODE: Don't save a placeholder if its parent is a table, since we
     don't really want placeholders in the XML when the interface is finished,
     but it is quite possible that some table cells will be left blank. */
  if (GB_IS_PLACEHOLDER (widget))
    {
      if (!GTK_IS_TABLE (widget->parent))
	{
	  save_newline (data);
	  save_start_tag (data, "widget");
	  save_string (data, "class", "Placeholder");
	  if (child_name)
	    save_string (data, "child_name", child_name);
	  save_end_tag (data, "widget");
	}
      return;
    }

  class_name = gb_widget_get_type_name (widget);

  /* Write out widget data and any child widgets */
  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  /* If this isn't a gbwidget, skip it. */
  if (widget_data)
    {
      data->action = GB_SAVING;
      data->widget_data = widget_data;

      save_newline (data);
      save_start_tag (data, "widget");
      save_string (data, "class", class_name);
      if (child_name)
	save_string (data, "child_name", child_name);

      get_standard_properties (widget, data);

      /* Call gbwidgets save function for any extra properties */
      gbwidget = gb_widget_lookup_class (class_name);
      g_return_if_fail (gbwidget != NULL);

      if (gbwidget->gb_widget_get_properties)
	(gbwidget->gb_widget_get_properties) (widget, data);

      /* Call parent widget's function to save child packing properties. */
      if (widget->parent)
	{
	  parent_gbwidget = gb_widget_lookup (widget->parent);
	  /* parent_gbwidget may be NULL, e.g. for GnomeDockItems. */
	  if (parent_gbwidget
	      && parent_gbwidget->gb_widget_get_child_properties)
	    (parent_gbwidget->gb_widget_get_child_properties) (widget->parent,
							       widget, data);
	}
    }

  /* Recursively save children. */
  gb_widget_children_foreach (widget, (GtkCallback) gb_widget_save, data);

  if (widget_data)
    {
      save_end_tag (data, "widget");
    }
}


/* This outputs all the standard widget properties, for showing or saving.
   Note that when saving we try not to save default values. */
static void
get_standard_properties (GtkWidget * widget,
			 GbWidgetGetArgData * data)
{
  gboolean can_default, has_default, can_focus, has_focus, visible, sensitive;
  GbWidgetData *wdata = data->widget_data;
  gchar *name, *class_name;
  GbWidgetAction action = data->action;

  /* Save widget's name if it has been set */
  name = gtk_widget_get_name (widget);
  class_name = gb_widget_get_type_name (widget);
  if (action != GB_SAVING || strcmp (name, class_name))
    gb_widget_output_string (data, GbName, name);

  /* If widget is a container, save the border width */
  if (GTK_IS_CONTAINER (widget))
    {
      gint border_width;
      gchar buf[128];
      border_width = GTK_CONTAINER (widget)->border_width;
      if (action != GB_SAVING || border_width > 0)
	{
	  sprintf (buf, "%s::border_width", class_name);
	  gb_widget_output_int (data, buf, border_width);
	}
    }

  get_position_properties (widget, data);
  get_lang_specific_properties (widget, data);

  visible = wdata->flags & GB_VISIBLE;
  if (action != GB_SAVING || !visible)
    gb_widget_output_bool (data, GbVisible, visible);
  sensitive = wdata->flags & GB_SENSITIVE;
  if (action != GB_SAVING || !sensitive)
    gb_widget_output_bool (data, GbSensitive, sensitive);

  /* Only widgets with windows can have tooltips at present. */
  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      if (action != GB_SAVING || wdata->tooltip)
	gb_widget_output_translatable_string (data, GbTooltip, wdata->tooltip);
      if (action == GB_SHOWING)
	property_set_sensitive (GbTooltip, TRUE);
    }
  else if (action == GB_SHOWING)
    {
      /* N/A stands for 'Not Applicable'. It is used when a standard widget
	 property does not apply to the current widget. e.g. widgets without
	 windows can't use the Events property. This appears in the property
	 editor and so should be a short abbreviation. */
      gb_widget_output_string (data, GbTooltip, _("N/A"));
      property_set_sensitive (GbTooltip, FALSE);
    }

  can_default = GTK_WIDGET_CAN_DEFAULT (widget);
  if (action != GB_SAVING || can_default)
    gb_widget_output_bool (data, GbCanDefault, can_default);
  has_default = wdata ? wdata->flags & GB_GRAB_DEFAULT : 0;
  if (action != GB_SAVING || has_default)
    gb_widget_output_bool (data, GbHasDefault, has_default);

  can_focus = GTK_WIDGET_CAN_FOCUS (widget);
  if (action != GB_SAVING || can_focus)
    gb_widget_output_bool (data, GbCanFocus, can_focus);
  has_focus = wdata ? wdata->flags & GB_GRAB_FOCUS : 0;
  if (action != GB_SAVING || has_focus)
    gb_widget_output_bool (data, GbHasFocus, has_focus);

  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      GdkExtensionMode ext_mode = gtk_widget_get_extension_events (widget);
      gint i;

      if (action == GB_SAVING)
	{
	  if (wdata && wdata->events)
	    {
	      gchar buffer[1024];
	      gboolean written_first = FALSE;

	      buffer[0] = '\0';
	      for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
		{
		  if (wdata->events & GbEventMaskValues[i])
		    {
		      if (written_first)
			strcat (buffer, " | ");
		      strcat (buffer, GbEventMaskSymbols[i]);
		      written_first = TRUE;
		    }
		}

	      gb_widget_output_string (data, GbEvents, buffer);
	    }
	}
      else
	{
	  gb_widget_output_string (data, GbEvents,
			   property_events_value_to_string (wdata->events));
	}

      /* Don't save default extension mode ('None') */
      if (action != GB_SAVING || ext_mode != GDK_EXTENSION_EVENTS_NONE)
	{
	  for (i = 0; GbExtensionModeChoices[i]; i++)
	    {
	      if (GbExtensionModeValues[i] == ext_mode)
		gb_widget_output_choice (data, GbExtEvents, i,
					 GbExtensionModeSymbols[i]);
	    }
	}
      if (action == GB_SHOWING)
	{
	  property_set_sensitive (GbEvents, TRUE);
	  property_set_sensitive (GbExtEvents, TRUE);
	}
    }
  else if (action == GB_SHOWING)
    {
      gb_widget_output_dialog (data, GbEvents, _("N/A"), NULL);
      property_set_sensitive (GbEvents, FALSE);
      gb_widget_output_choice (data, GbExtEvents, 0, GbExtensionModeSymbols[0]);
      property_set_sensitive (GbExtEvents, FALSE);
    }

  if (action == GB_SHOWING)
    {
#ifdef GLADE_STYLE_SUPPORT
      gb_widget_show_style (widget);
#endif
      show_accelerators (widget, data);
      show_signals (widget, data);
    }
  else
    {
#ifdef GLADE_STYLE_SUPPORT
      save_style (widget, data);
#endif
      save_accelerators (widget, data);
      save_signals (widget, data);
    }
}


/* This is used when saving or displaying properties. */
static void
get_lang_specific_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{ guint i;

  GbWidgetData *wdata = data->widget_data;
  GbWidgetAction action = data->action;

  /* Note that when saving we only save the property if it is not the
     default value. Otherwise we get lots of unnecessary XML output. */

  /* C options. */
  if (action != GB_SAVING || wdata->source_file)
    gb_widget_output_filename (data, GbCSourceFile, wdata->source_file);
  if (action != GB_SAVING || wdata->public_field == 0)
    gb_widget_output_bool (data, GbCPublic, wdata->public_field);
  if (data->action == GB_SHOWING)
    {
      /* We only want the source file set for toplevel widgets. */
      if (GTK_IS_WINDOW (widget) || GTK_IS_MENU (widget))
	property_set_sensitive (GbCSourceFile, TRUE);
      else
	property_set_sensitive (GbCSourceFile, FALSE);
    }

  /* C++ options. */
  if (action != GB_SAVING || wdata->cxx_separate_file == 1)
    gb_widget_output_bool (data, GbCxxSeparateFile, wdata->cxx_separate_file);
  if (action != GB_SAVING || wdata->cxx_use_heap == 1)
    gb_widget_output_bool (data, GbCxxUseHeap, wdata->cxx_use_heap);
  if (action != GB_SAVING || wdata->cxx_separate_class == 1)
    gb_widget_output_bool (data, GbCxxSeparateClass, wdata->cxx_separate_class);
  if (action != GB_SAVING || wdata->cxx_visibility != 0)
    for (i = 0; GbCxxVisibilityChoices[i]; i++)
	{
	  if (GbCxxVisibilityValues[i] == wdata->cxx_visibility)
	    gb_widget_output_choice (data, GbCxxVisibility, i,
				     GbCxxVisibilitySymbols[i]);
	}
}


/* This is used when loading or applying properties. */
static void
set_lang_specific_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gchar *filename;
  gboolean public_field;
  gboolean cxx_separate_file, cxx_use_heap, cxx_separate_class;
  gchar *cxx_visibility;

  /* C options. */
  filename = gb_widget_input_filename (data, GbCSourceFile);
  if (data->apply)
    {
      g_free (wdata->source_file);
      wdata->source_file = g_strdup (filename);
    }
  public_field = gb_widget_input_bool (data, GbCPublic);
  if (data->apply)
    wdata->public_field = public_field ? 1 : 0;

  /* C++ options. */
  cxx_separate_file = gb_widget_input_bool (data, GbCxxSeparateFile);
  if (data->apply)
    wdata->cxx_separate_file = cxx_separate_file ? 1 : 0;
  cxx_use_heap = gb_widget_input_bool (data, GbCxxUseHeap);
  if (data->apply)
    wdata->cxx_use_heap = cxx_use_heap ? 1 : 0;
  cxx_separate_class = gb_widget_input_bool (data, GbCxxSeparateClass);
  if (data->apply)
    wdata->cxx_separate_class = cxx_separate_class ? 1 : 0;
  cxx_visibility = gb_widget_input_choice (data, GbCxxVisibility);
  if (data->apply)
	{ guint i;
	  for (i = 0; GbCxxVisibilityChoices[i]; i++)
	    {
	      if (!strcmp (cxx_visibility, GbCxxVisibilityChoices[i])
		  || !strcmp (cxx_visibility, GbCxxVisibilitySymbols[i]))
		{
		  wdata->cxx_visibility = GbCxxVisibilityValues[i];
		  break;
		}
	    }
	}
}


static void
get_position_properties (GtkWidget * widget,
			 GbWidgetGetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gboolean xy_applicable = TRUE, wh_applicable = TRUE;
  gboolean show_xy_buttons = TRUE, show_wh_buttons = TRUE;

#if 0
  g_print ("In get_pos: %s WS:%s HS:%s ",
	gtk_widget_get_name (widget),
	wdata->flags & GB_WIDTH_SET ? "1" : "0",
	wdata->flags & GB_HEIGHT_SET ?  "1" : "0");
#endif

  /* Don't bother if widget's area hasn't been allocated yet, unless it is a
     window. */
  if (data->action == GB_SHOWING && (wdata->flags & GB_SIZE_NOT_ALLOCATED)
      && !GTK_IS_WINDOW (widget))
      return;

  if (GTK_IS_MENU (widget))
    {
      if (data->action == GB_SHOWING)
	{
	  gb_widget_output_int (data, GbX, widget->allocation.x);
	  gb_widget_output_int (data, GbY, widget->allocation.y);
	  gb_widget_output_int (data, GbWidth, widget->requisition.width);
	  gb_widget_output_int (data, GbHeight, widget->requisition.height);
	}
      xy_applicable = FALSE;
      wh_applicable = FALSE;
    }
  else if (GTK_IS_WINDOW (widget))
    {
      /* Toplevel window widgets */
      if (data->action == GB_SHOWING)
	{
	  gb_widget_output_int (data, GbX, wdata->x);
	  gb_widget_output_int (data, GbY, wdata->y);
	  gb_widget_output_int (data, GbWidth, wdata->width);
	  gb_widget_output_int (data, GbHeight, wdata->height);
	}
      else
	{
	  if (wdata->flags & GB_X_SET)
	    gb_widget_output_int (data, GbX, wdata->x);
	  if (wdata->flags & GB_Y_SET)
	    gb_widget_output_int (data, GbY, wdata->y);
	  if (wdata->flags & GB_WIDTH_SET)
	    gb_widget_output_int (data, GbWidth, wdata->width);
	  if (wdata->flags & GB_HEIGHT_SET)
	    gb_widget_output_int (data, GbHeight, wdata->height);
	}
      if (GTK_WINDOW (widget)->position != GTK_WIN_POS_NONE)
	xy_applicable = FALSE;
    }
  else if (widget->parent && (GTK_IS_FIXED (widget->parent)
			      || GTK_IS_LAYOUT (widget->parent)))
    {
      /* Widgets in fixed or layout containers. Note that for widgets in a 
       layout the allocation is relative to the window origin and changes as
       the layout is scrolled, so don't output that as the x & y coords. */
      gb_widget_output_int (data, GbX, wdata->x);
      gb_widget_output_int (data, GbY, wdata->y);
      gb_widget_output_int (data, GbWidth, wdata->width);
      gb_widget_output_int (data, GbHeight, wdata->height);
      show_xy_buttons = FALSE;
      show_wh_buttons = FALSE;
    }
  else
    {
      /* Widgets in standard containers */
      if (data->action == GB_SHOWING)
	{
	  gb_widget_output_int (data, GbX, widget->allocation.x);
	  gb_widget_output_int (data, GbY, widget->allocation.y);

	  /* If the width or height has been set explicitly we show the size
	     set, else we show the current requisition. We always remember
	     what we have shown in wdata->width & height so we know if the
	     user changes it. */
	  if (!(wdata->flags & GB_WIDTH_SET))
	    wdata->width = widget->requisition.width;
	  gb_widget_output_int (data, GbWidth, wdata->width);

	  if (!(wdata->flags & GB_HEIGHT_SET))
	    wdata->height = widget->requisition.height;
	  gb_widget_output_int (data, GbHeight, wdata->height);

	}
      else
	{
	  /* Only save if user has set it explicitly. */
	  if (wdata->flags & GB_WIDTH_SET)
	    gb_widget_output_int (data, GbWidth, wdata->width);
	  if (wdata->flags & GB_HEIGHT_SET)
	    gb_widget_output_int (data, GbHeight, wdata->height);
	}
      xy_applicable = FALSE;
    }

  /* Show the buttons for setting the size & positions explicitly, if
     applicable, and set the values sensitive if they are currently set. */
  if (data->action == GB_SHOWING)
    {
      if (xy_applicable)
	{
	  gboolean xsensitive = (wdata->flags & GB_X_SET) ? TRUE : FALSE;
	  gboolean ysensitive = (wdata->flags & GB_Y_SET) ? TRUE : FALSE;

	  property_set_sensitive_full (GbX, TRUE, xsensitive, show_xy_buttons);
	  property_set_sensitive_full (GbY, TRUE, ysensitive, show_xy_buttons);
	}
      else
	{
	  property_set_sensitive_full (GbX, FALSE, FALSE, FALSE);
	  property_set_sensitive_full (GbY, FALSE, FALSE, FALSE);
	}

      if (wh_applicable)
	{
	  gboolean wsensitive = (wdata->flags & GB_WIDTH_SET) ? TRUE : FALSE;
	  gboolean hsensitive = (wdata->flags & GB_HEIGHT_SET) ? TRUE : FALSE;

	  property_set_sensitive_full (GbWidth, TRUE, wsensitive,
				       show_wh_buttons);
	  property_set_sensitive_full (GbHeight, TRUE, hsensitive,
				       show_wh_buttons);
	}
      else
	{
	  property_set_sensitive_full (GbWidth, FALSE, FALSE, FALSE);
	  property_set_sensitive_full (GbHeight, FALSE, FALSE, FALSE);
	}
    }

#if 0
  g_print ("\n");
#endif
}


#ifdef GLADE_STYLE_SUPPORT
static void
save_style (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  g_return_if_fail (wdata != NULL);

  /* Don't save the style if it's the default. */
  if (wdata->gbstyle != gb_widget_default_gb_style)
    {
      /* If the widget is using a named style, just save the name. */
      if (!(wdata->flags & GB_STYLE_IS_UNNAMED))
	{
	  save_string (data, GbStyleName, wdata->gbstyle->name);
	}
      else
	{
	  gb_widget_save_style (wdata->gbstyle, data, FALSE);
	}
    }
  /* The default propagate value is TRUE. */
  if (!(wdata->flags & GB_STYLE_PROPAGATE))
    save_bool (data, GbStylePropagate, wdata->flags & GB_STYLE_PROPAGATE);
}


void
gb_widget_save_style (GbStyle * gbstyle, GbWidgetGetArgData * data,
		      gboolean save_all)
{
  GtkStyle *style = gbstyle->style;
  GtkStyle *default_style = gb_widget_default_gb_style->style;
  gchar buffer[128];
  gint i;

  save_start_tag (data, "style");

  if (gbstyle->name)
    save_string (data, GbStyleName, gbstyle->name);
  if (save_all
      || strcmp (gbstyle->xlfd_fontname, gb_widget_default_gb_style->xlfd_fontname))
    {
      save_string (data, GbStyleFont, gbstyle->xlfd_fontname);
    }

  /* Colors */
  default_style = gb_widget_default_gb_style->style;
  save_colors (data, save_all, style->fg, default_style->fg, "fg");
  save_colors (data, save_all, style->bg, default_style->bg, "bg");
  save_colors (data, save_all, style->text, default_style->text, "text");
  save_colors (data, save_all, style->base, default_style->base, "base");

  /* Background pixmaps */
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      /* Can't use '[' or ']' in XML element names */
      sprintf (buffer, "GtkStyle::%s-%s", GbBgPixmapName, GbStateNames[i]);
      if (gbstyle->bg_pixmap_filenames[i])
	{
	  if (save_all
	      || gb_widget_default_gb_style->bg_pixmap_filenames[i] == NULL
	      || strcmp (gbstyle->bg_pixmap_filenames[i],
			 gb_widget_default_gb_style->bg_pixmap_filenames[i]))
	    {
	      save_string (data, buffer, gbstyle->bg_pixmap_filenames[i]);
	    }
	}
    }

  save_end_tag (data, "style");
}


static void
save_colors (GbWidgetGetArgData * data, gboolean save_all, GdkColor colors[],
	     GdkColor default_colors[], gchar * name)
{
  gint state;
  gchar buf[128];

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      if (save_all
	  || colors[state].red != default_colors[state].red
	  || colors[state].green != default_colors[state].green
	  || colors[state].blue != default_colors[state].blue)
	{
	  /* Can't use ':', '[' or ']' in XML element names */
	  sprintf (buf, "GtkStyle::%s-%s", name, GbStateNames[state]);
	  save_color (data, buf, &colors[state]);
	}
    }
}
#endif


static void
save_accelerators (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *item;
  GbAccelerator *accel;
  gchar key_buffer[128];

  if (data->widget_data == NULL)
    return;
  item = data->widget_data->accelerators;
  while (item)
    {
      accel = (GbAccelerator *) item->data;

      save_start_tag (data, "accelerator");
      save_string (data, "modifiers",
		   glade_util_create_modifiers_string (accel->modifiers));

      /* Make sure we don't overflow the buffer. */
      if (strlen (accel->key) < 100)
	{
	  strcpy (key_buffer, "GDK_");
	  strcat (key_buffer, accel->key);
	  save_string (data, "key", key_buffer);
	}

      save_string (data, "signal", accel->signal);
      save_end_tag (data, "accelerator");
      item = item->next;
    }
}


static void
save_signals (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *item;
  GbSignal *signal;

  if (data->widget_data == NULL)
    return;
  item = data->widget_data->signals;
  while (item)
    {
      signal = (GbSignal *) item->data;

      save_start_tag (data, "signal");
      if (signal->name)
	save_string (data, "name", signal->name);
      if (signal->handler)
	save_string (data, "handler", signal->handler);
      if (signal->object)
	save_string (data, "object", signal->object);
      if (signal->after)
	save_bool (data, "after", signal->after);
      if (signal->data)
	save_string (data, "data", signal->data);
      if (signal->last_modification_time != 0)
	save_date (data, "last_modification_time",
		   signal->last_modification_time);
      save_end_tag (data, "signal");

      item = item->next;
    }
}



/*************************************************************************
 * Functions for writing C source code
 *************************************************************************/

void
gb_widget_write_source (GtkWidget * widget,
			GbWidgetWriteSourceData * data)
{
  GtkWidget *parent;
  GbWidget *gbwidget;
  GbWidgetData *widget_data;
  gchar *class_name, *child_name;
  gint source_len;

  /* This is a temporary(?) kludge so that the code for GtkDialogs is OK.
     We stop the source code for the action_area from being written.
     GtkDialog & GnomeDialog need to output the code for their vbox &
     action_area children themselves, since they need to output special code
     to access them, e.g. "GTK_DIALOG (dialog1)->vbox". However, the vbox
     is the parent of the action_area, and so we have to stop the action_area
     being output using the standard code since that won't work.
     The problem is due to the dialogs having 2 special children, where one
     is a descendant of the other. I don't think this occurs anywhere else. */
  child_name = gb_widget_get_child_name (widget);
  if (child_name && data->create_widget)
    {
      if (!strcmp (child_name, "Dialog:action_area")
	  || !strcmp (child_name, "GnomeDialog:action_area"))
	return;
    }

  parent = data->parent;

  class_name = gb_widget_get_type_name (widget);

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  /* If this isn't a gbwidget, skip it. */
  if (widget_data)
    {
      gbwidget = gb_widget_lookup_class (class_name);
      g_return_if_fail (gbwidget != NULL);

      /* For custom widgets, we don't have a default widget to compare with,
	 so all properties should be set. */
      if (GLADE_IS_CUSTOM_WIDGET (widget))
	{
	  data->standard_widget = NULL;
	}
      else
	{
	  data->standard_widget = (GtkWidget *) g_hash_table_lookup (data->standard_widgets, class_name);
	  if (data->standard_widget == NULL)
	    {
#ifdef USE_GNOME
	      /* FIXME: GnomeLibs 1.0.1 workaround - gtk_object_newv doesn't
		 return a valid GnomeAppBar or GnomeDateEdit, so we create it
		 ourself. */
	      if (!strcmp (class_name, "GnomeAppBar"))
		data->standard_widget = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
	      else if (!strcmp (class_name, "GnomeDateEdit"))
		data->standard_widget = gnome_date_edit_new ((time_t) 0, TRUE,
							     TRUE);
	      else
		data->standard_widget = GTK_WIDGET (gtk_object_newv (gtk_type_from_name (class_name), 0, NULL));
#else
	      data->standard_widget = GTK_WIDGET (gtk_object_newv (gtk_type_from_name (class_name), 0, NULL));
#endif
	      g_hash_table_insert (data->standard_widgets, class_name,
				   data->standard_widget);
	    }
	}

      data->real_wname = source_create_valid_identifier (gtk_widget_get_name (widget));
      if (data->use_component_struct && widget_data->public_field)
	data->wname = g_strdup_printf ("%s->%s", data->component_name,
				       data->real_wname);
      else
	data->wname = data->real_wname;

      data->widget_data = widget_data;
      data->write_children = TRUE;
      if (gbwidget->gb_widget_write_source)
	(gbwidget->gb_widget_write_source) (widget, data);
      else
	source_add (data, "  /* Skipping %s: unimplemented. */\n", class_name);

      /* Make sure there is a blank line after each widget, for readability. */
      source_len = data->source_buffers[GLADE_SOURCE]->len;
      if (source_len > 2
	  && (data->source_buffers[GLADE_SOURCE]->str[source_len - 1] != '\n'
	      || data->source_buffers[GLADE_SOURCE]->str[source_len - 2] != '\n'))
	source_add (data, "\n");

      if (data->wname != data->real_wname)
	g_free (data->real_wname);
      g_free (data->wname);
      data->wname = NULL;
      data->real_wname = NULL;
      data->parent = widget;
    }
  else if (GB_IS_PLACEHOLDER (widget) && parent)
    {
      if (GTK_IS_NOTEBOOK (parent))
	{
	  /* SPECIAL CODE: If notebook pages are empty (i.e. placeholders),
	     we create dummy widgets instead, so it still runs OK. */
	  if (child_name == NULL)
	    {
	      gchar *wname, *parent_name;
	  
	      wname = "empty_notebook_page";
	      /* Make sure the dummy widget is declared. */
	      source_ensure_decl (data, "  GtkWidget *empty_notebook_page;\n");

	      parent_name = gtk_widget_get_name (parent);
	      parent_name = source_create_valid_identifier (parent_name);
	      source_add (data,
			  "  %s = gtk_vbox_new (FALSE, 0);\n"
			  "  gtk_widget_show (%s);\n"
			  "  gtk_container_add (GTK_CONTAINER (%s), %s);\n"
			  "\n",
			  wname, wname, parent_name, wname);
	      g_free (parent_name);
	    }
	  else
	    {
	      /* For empty notebook tabs, we increment the 'last_child' written
		 value. */
	      gint col = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (parent),
							       "last_child"));
	      gtk_object_set_data (GTK_OBJECT (parent), "last_child",
				   GINT_TO_POINTER (col + 1));
	    }
	}
      else if (GTK_IS_CLIST (parent))
	{
	  /* For empty clist/ctree titles, we increment the 'last_child'
	     written value. */
	  if (child_name && (!strcmp (child_name, "CList:title")
			     || !strcmp (child_name, "CTree:title")))
	    {
	      gint col = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (parent),
							       "last_child"));
	      gtk_object_set_data (GTK_OBJECT (parent), "last_child",
				   GINT_TO_POINTER (col + 1));
	    }
	}
    }


  /* Recursively write source for children.
     We need to reset the parent after the children have been written. */
  data->create_widget = TRUE;
  if (data->write_children)
    gb_widget_children_foreach (widget, (GtkCallback) gb_widget_write_source, data);

  /* We need to reset some of the members of the GbWidgetWriteSourceData struct
     so that they work OK for the next sibling. */
  data->parent = parent;

  /* SPECIAL CODE: Finish off the GnomeUIInfo struct if we are building a
     Gnome menu. */
#ifdef USE_GNOME
  if (data->project->gnome_support && GTK_IS_MENU_SHELL (widget))
    {
      glade_gnome_finish_menu_source (GTK_MENU_SHELL (widget), data);
    }
#endif
}


/* This is called by each GbWidget's write_source function to write all the
   common source code, including code to add the widget to its parent. */
void
gb_widget_write_standard_source (GtkWidget * widget,
				 GbWidgetWriteSourceData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gint i, width, height, can_focus, can_default;

  /* FIXME: unfinished. public fields were to be added to the component struct.
     Private fields would just be declared within the creation function. */
  if (wdata->public_field)
    source_add_decl (data, "  GtkWidget *%s;\n", data->real_wname);
  else
    source_add_decl (data, "  GtkWidget *%s;\n", data->real_wname);

  if (data->set_widget_names)
    source_add (data, "  gtk_widget_set_name (%s, \"%s\");\n", data->wname,
		data->real_wname);

  if (!data->use_component_struct)
    {
      /* For toplevel widgets we don't ref the widget, since if we do it never
	 gets destroyed. This may be a bug in GTK+ 1.2.3. */
      if (data->parent == NULL)
	{
	  source_add (data,
		      "  gtk_object_set_data (GTK_OBJECT (%s), %s, %s);\n",
		      data->component_name,
		      source_make_string (data->real_wname, FALSE),
		      data->wname);
	}
      else
	{
	  source_add (data,
		      "  gtk_widget_ref (%s);\n"
		      "  gtk_object_set_data_full (GTK_OBJECT (%s), %s, %s,\n"
		      "                            (GtkDestroyNotify) gtk_widget_unref);\n",
		      data->wname,
		      data->component_name,
		      source_make_string (data->real_wname, FALSE),
		      data->wname);
	}
    }

  /* SPECIAL CODE: menus are not shown here. */
  if (widget->parent && wdata->flags & GB_VISIBLE && !GTK_IS_MENU (widget))
    {
#ifdef USE_GNOME
      /* FIXME: GnomeDruidPageStandard bug workaround. It needs show_all(). */
      if (GNOME_IS_DRUID_PAGE_STANDARD (widget))
	source_add (data, "  gtk_widget_show_all (%s);\n", data->wname);
      else
	source_add (data, "  gtk_widget_show (%s);\n", data->wname);
#else
      source_add (data, "  gtk_widget_show (%s);\n", data->wname);
#endif
    }

  /* Output code to add widget to parent. */
  gb_widget_write_add_child_source (widget, data);

  if (wdata->flags & (GB_X_SET | GB_Y_SET))
    {
      /* SPECIAL CODE: Widgets in a GtkLayout are positioned with
	 gtk_layout_put(). */
      if (!widget->parent || !GTK_IS_LAYOUT (widget->parent))
	{
	  source_add (data, "  gtk_widget_set_uposition (%s, %i, %i);\n",
		      data->wname,
		      wdata->flags & GB_X_SET ? wdata->x : -2,
		      wdata->flags & GB_Y_SET ? wdata->y : -2);
	}
    }

  if (wdata->flags & (GB_WIDTH_SET | GB_HEIGHT_SET))
    {
      width = wdata->flags & GB_WIDTH_SET ? wdata->width : -2;
      height = wdata->flags & GB_HEIGHT_SET ? wdata->height : -2;

      /* GTK BUG WORKAROUND - a combo should manage the size of its entry.
	 I think this isn't needed any more (GTK+ 1.2.3). */
#if 0
      if (GTK_IS_COMBO (widget))
	source_add(data,
		   "  gtk_widget_set_usize (GTK_COMBO (%s)->entry, %i, %i);\n",
		   data->wname, width - 16 < 0 ? -2 : width - 16, height);
#endif

      source_add (data, "  gtk_widget_set_usize (%s, %i, %i);\n",
		  data->wname, width, height);
    }


  if (GTK_IS_CONTAINER (widget))
    {
      if (GTK_CONTAINER (widget)->border_width != 0)
	{
	  source_add (data,
		      "  gtk_container_set_border_width (GTK_CONTAINER (%s), %i);\n",
		      data->wname, GTK_CONTAINER (widget)->border_width);
	}
    }


  /* FIXME: Kludge to set separator menu items insensitive, so that they are
     skipped when using the cursor keys to move up/down the menu. */
  if (!(wdata->flags & GB_SENSITIVE)
      || (GTK_IS_MENU_ITEM (widget) && GTK_BIN (widget)->child == NULL))
    source_add (data, "  gtk_widget_set_sensitive (%s, FALSE);\n", data->wname);

  can_focus = GTK_WIDGET_CAN_FOCUS (widget);
  if (!data->standard_widget
      || can_focus != GTK_WIDGET_CAN_FOCUS (data->standard_widget))
    {
      if (can_focus)
	source_add (data, "  GTK_WIDGET_SET_FLAGS (%s, GTK_CAN_FOCUS);\n",
		    data->wname);
      /* SPECIAL CODE: toolbar widgets can't get focus by default. */
      else if (!gb_toolbar_is_toolbar_button (widget))
	source_add (data, "  GTK_WIDGET_UNSET_FLAGS (%s, GTK_CAN_FOCUS);\n",
		    data->wname);
    }
  can_default = GTK_WIDGET_CAN_DEFAULT (widget);
  if (!data->standard_widget
      || can_default != GTK_WIDGET_CAN_DEFAULT (data->standard_widget))
    {
      if (can_default)
	source_add (data, "  GTK_WIDGET_SET_FLAGS (%s, GTK_CAN_DEFAULT);\n",
		    data->wname);
      else
	source_add (data, "  GTK_WIDGET_UNSET_FLAGS (%s, GTK_CAN_DEFAULT);\n",
		    data->wname);
    }

  if (wdata->flags & GB_GRAB_FOCUS)
    {
      if (data->focus_widget)
	g_warning ("Multiple widgets with 'Has Focus' set: %s, %s",
		   data->focus_widget, data->real_wname);
      else
	data->focus_widget = g_strdup (data->wname);
    }

  if (wdata->flags & GB_GRAB_DEFAULT)
    {
      if (data->default_widget)
	g_warning ("Multiple widgets with 'Has Default' set: %s, %s",
		   data->default_widget, data->real_wname);
      else
	data->default_widget = g_strdup (data->wname);
    }

  /* SPECIAL CODE: toolbar children use the toolbar's tooltips. */
  if (wdata->tooltip && (!widget->parent || !GTK_IS_TOOLBAR (widget->parent)))
    {
      data->need_tooltips = TRUE;
      source_add (data, "  gtk_tooltips_set_tip (tooltips, %s, %s, NULL);\n",
		  data->wname,
		  source_make_string (wdata->tooltip, data->use_gettext));
    }

  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      GdkExtensionMode ext_mode = gtk_widget_get_extension_events (widget);
      gboolean written_first = FALSE;

      if (wdata->events)
	{
	  source_add (data, "  gtk_widget_set_events (%s, ", data->wname);
	  for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
	    {
	      if (wdata->events & GbEventMaskValues[i])
		{
		  if (!written_first)
		    source_add (data, "%s", GbEventMaskSymbols[i]);
		  else
		    source_add (data, " | %s", GbEventMaskSymbols[i]);
		  written_first = TRUE;
		}
	    }
	  source_add (data, ");\n");
	}

      if (ext_mode != GDK_EXTENSION_EVENTS_NONE)
	{
	  for (i = 0; GbExtensionModeChoices[i]; i++)
	    {
	      if (GbExtensionModeValues[i] == ext_mode)
		source_add (data, "  gtk_widget_set_extension_events (%s, %s);\n",
			    data->wname, GbExtensionModeSymbols[i]);
	    }
	}
    }

  gb_widget_write_signals_source (widget, data);
  gb_widget_write_accelerators_source (widget, data);
}


void
gb_widget_write_add_child_source (GtkWidget * widget,
				  GbWidgetWriteSourceData * data)
{
  GbWidget *gbparent;
  GtkWidget *parent;
  gchar *parent_name;

  /* If the widget is created automatically by its parent, we don't need
     to add source to add it. */
  if (!data->create_widget)
    return;

  /* SPECIAL CODE: to handle menus. */
  if (GTK_IS_MENU (widget))
    parent = gtk_menu_get_attach_widget (GTK_MENU (widget));
  else
    parent = data->parent;

  /* Return if no parent exists */
  if (!parent)
    return;

  parent_name = gtk_widget_get_name (parent);
  parent_name = source_create_valid_identifier (parent_name);

  MSG2 ("Adding %s to %s", data->wname, parent_name);

  /* If the GbWidget has its own function to output source code to add a child,
     we use that, else we just output the default "gtk_container_add ()". */
  gbparent = gb_widget_lookup (parent);
  if (gbparent && gbparent->gb_widget_write_add_child_source)
    {
      (gbparent->gb_widget_write_add_child_source) (parent, parent_name,
						    widget, data);
    }
  else
    {
      source_add (data, "  gtk_container_add (GTK_CONTAINER (%s), %s);\n",
		  parent_name, data->wname);
    }

  g_free (parent_name);
}


void
gb_widget_write_signals_source (GtkWidget * widget,
				GbWidgetWriteSourceData * data)
{
  GList *item;
  GbSignal *signal;
  gboolean skip_handler_code;
  gchar *handler;

  item = data->widget_data->signals;
  while (item)
    {
      signal = (GbSignal *) item->data;
      item = item->next;
      skip_handler_code = FALSE;

      /* If we're appending new/changed signals and this is an old one, then
	 skip it. Note that we also skip it if the last mod time is 0, which
	 will happen for XML files created before the last mod tag was added.*/
      MSG2 ("Creating Signals:%s Handler: %s",
	    data->creating_callback_files ? "TRUE" : "FALSE", signal->handler);
      MSG1 ("LastMod:%s", ctime (&signal->last_modification_time));
      MSG1 ("LastWrite:%s", ctime (&data->last_write_time));

      if (!data->creating_callback_files
	  && (signal->last_modification_time <= data->last_write_time))
	{
	  MSG1 ("Skipping signal: %s", signal->handler);
	  skip_handler_code = TRUE;
	}

      if (!signal->name)
	{
	  /* FIXME: store warnings */
	  g_warning ("Signal name not set");
	  continue;
	}

      if (!signal->handler)
	{
	  /* FIXME: store warnings */
	  g_warning ("Signal handler not set");
	  continue;
	}

      /* Make sure handler function name is valid. */
      handler = source_create_valid_identifier (signal->handler);

      /* Output code to connect signal. */
      gb_widget_write_signal_connection_source (data, signal->name,
						signal->object, signal->after,
						signal->data, handler);

      /* We don't need to output code for standard GTK functions. */
      if (!strcmp (handler, "gtk_widget_hide")
	  || !strcmp (handler, "gtk_widget_destroy"))
	skip_handler_code = TRUE;


      /* If we're appending new/changed signals and this is an old one, we
	 don't want to output the source code for it. */
      if (!skip_handler_code)
	{
	  gb_widget_write_signal_handler_source (widget, data, signal->name,
						 handler);
	}
      g_free (handler);
    }
}


/* This outputs a signal handler declaration and empty function. handler
   should be a valid identifier. */
void
gb_widget_write_signal_handler_source (GtkWidget *widget,
				       GbWidgetWriteSourceData * data,
				       const gchar *signal_name,
				       const gchar *handler)
{
  guint signal_id;
  GtkSignalQuery *query_info;
  gint param, i;
  gchar buffer[1024];
  gchar *ret_type, *pos, *type_name, *arg_name, *object_name, *object_arg;
  gchar *object_arg_start;
  gboolean is_pointer;
  gint param_num, widget_num, event_num, callback_num;
  gint handler_len, object_name_len, type_name_len, num_spaces;
  gint *arg_num;
  gchar **arg_names;
  gchar *signal_name_copy;

  /* Check if we have already output the handler. */
  if (g_hash_table_lookup (data->handlers_output, handler))
    return;

  /* Remember that we have output the handler. */
  g_hash_table_insert (data->handlers_output, g_strdup (handler), "Output");


  signal_id = gtk_signal_lookup (signal_name, GTK_OBJECT_TYPE (widget));
  g_return_if_fail (signal_id != 0);

  query_info = gtk_signal_query (signal_id);
  g_return_if_fail (query_info != NULL);

  /* Output the return type and function name. */
  ret_type = get_type_name (query_info->return_val, &is_pointer);
  pos = buffer;
  sprintf (pos, "%s%s\n%s", ret_type, is_pointer ? "*" : "", handler);
  pos += strlen (pos);

  handler_len = strlen (handler);
  if (handler_len >= GB_PARAM_INDENT - 1)
    {
      *pos++ = '\n';
      for (i = 0; i < GB_PARAM_INDENT; i++)
	*pos++ = ' ';
    }
  else
    {
      num_spaces = GB_PARAM_INDENT - handler_len - 1;
      for (i = 0; i < num_spaces; i++)
	*pos++ = ' ';
    }

  /* Output the signal object type and the argument name. We assume the
     type is a pointer - I think that is OK. We remove "Gtk" and convert
     to lower case for the argument name. */
  object_name = gtk_type_name (query_info->object_type);
  sprintf (pos, "(%s ", object_name);
  pos += strlen (pos);
  object_name_len = strlen (object_name);
  if (object_name_len + 1 < GB_PARAM_TYPE_WIDTH)
    {
      num_spaces = GB_PARAM_TYPE_WIDTH - object_name_len - 1;
      for (i = 0; i < num_spaces; i++)
	*pos++ = ' ';
    }
  object_arg = (!strncmp (object_name, "Gtk", 3)) ? object_name + 3 : object_name;
  object_arg_start = pos;
  sprintf (pos, "*%s", object_arg);
  pos += strlen (pos);
  g_strdown (object_arg_start);
  if (!strcmp (object_arg_start, "widget"))
    widget_num++;
  sprintf (pos, ",\n");
  pos += strlen (pos);

  /* Convert signal name to use underscores rather than dashes '-'. */
  signal_name_copy = g_strdup (query_info->signal_name);
  for (i = 0; signal_name_copy[i]; i++)
    {
      if (signal_name_copy[i] == '-')
	signal_name_copy[i] = '_';
    }

  /* Output the signal parameters. */
  arg_names = lookup_signal_arg_names (object_name, signal_name_copy);

  param_num = 1;
  widget_num = event_num = callback_num = 0;
  for (param = 0; param < query_info->nparams; param++)
    {
      for (i = 0; i < GB_PARAM_INDENT; i++)
	*pos++ = ' ';

      if (arg_names)
	{
	  sprintf (pos, "%s,\n", arg_names[param]);
	  pos += strlen (pos);
	}
      else
	{
	  type_name = get_type_name (query_info->params[param], &is_pointer);
	  /* Most arguments to the callback are called "arg1", "arg2", etc.
	     GtkWidgets are called "widget", "widget2", ...
	     GdkEvents are called "event", "event2", ...
	     GtkCallbacks are called "callback", "callback2", ... */
	  if (!strcmp (type_name, "GtkWidget"))
	    {
	      arg_name = "widget";
	      arg_num = &widget_num;
	    }
	  else if (!strcmp (type_name, "GdkEvent"))
	    {
	      type_name = get_gdk_event (signal_name_copy);
	      arg_name = "event";
	      arg_num = &event_num;
	      is_pointer = TRUE;
	    }
	  else if (!strcmp (type_name, "GtkCallback"))
	    {
	      arg_name = "callback";
	      arg_num = &callback_num;
	    }
	  else
	    {
	      arg_name = "arg";
	      arg_num = &param_num;
	    }
	  sprintf (pos, "%s ", type_name);
	  pos += strlen (pos);
	  type_name_len = strlen (type_name);
	  if (type_name_len + 1 < GB_PARAM_TYPE_WIDTH)
	    {
	      num_spaces = GB_PARAM_TYPE_WIDTH - type_name_len - 1;
	      for (i = 0; i < num_spaces; i++)
		*pos++ = ' ';
	    }
	  if (!arg_num || *arg_num == 0)
	    sprintf (pos, "%s%s,\n", is_pointer ? "*" : " ", arg_name);
	  else
	    sprintf (pos, "%s%s%i,\n", is_pointer ? "*" : " ", arg_name,
		     *arg_num);
	  pos += strlen (pos);
	  
	  if (arg_num)
	    *arg_num += 1;
	}
    }

  /* Add the final user_data parameter, common to all handlers. */
  for (i = 0; i < GB_PARAM_INDENT; i++)
    *pos++ = ' ';
  sprintf (pos, "gpointer         user_data)");
  pos += strlen (pos);

  /* Output the declaration of the handler, which uses the same buffer. */
  source_add_to_buffer (data, GLADE_CALLBACK_DECLARATIONS,
			"\n%s;\n", buffer);

  /* Output the empty handler function, returning FALSE if the return type is
     bool (i.e. for the GdkEvent handlers). */
  source_add_to_buffer (data, GLADE_CALLBACK_SOURCE,
			"\n%s\n{\n\n%s}\n\n",
			buffer,
			query_info->return_val == GTK_TYPE_BOOL
			? "  return FALSE;\n" : "");

  g_free (signal_name_copy);
  g_free (query_info);
}


static void
gb_widget_write_signal_connection_source (GbWidgetWriteSourceData * data,
					  const gchar *signal_name,
					  const gchar *connect_object,
					  gboolean connect_after,
					  const gchar *handler_data,
					  const gchar *handler)
{
  if (connect_object && connect_object[0])
    {
      if (connect_after)
	{
	  source_add_to_buffer (data, GLADE_SIGNAL_CONNECTIONS,
				"  gtk_signal_connect_object_after (GTK_OBJECT (%s), \"%s\",\n"
				"                                   GTK_SIGNAL_FUNC (%s),\n"
				"                                   GTK_OBJECT (%s));\n",
				data->wname, signal_name, handler,
				connect_object);
	}
      else
	{
	  source_add_to_buffer (data, GLADE_SIGNAL_CONNECTIONS,
				"  gtk_signal_connect_object (GTK_OBJECT (%s), \"%s\",\n"
				"                             GTK_SIGNAL_FUNC (%s),\n"
				"                             GTK_OBJECT (%s));\n",
				data->wname, signal_name, handler,
				connect_object);
	}
    }
  else
    {
      if (connect_after)
	{
	  source_add_to_buffer (data, GLADE_SIGNAL_CONNECTIONS,
				"  gtk_signal_connect_after (GTK_OBJECT (%s), \"%s\",\n"
				"                            GTK_SIGNAL_FUNC (%s),\n"
				"                            %s);\n",
				data->wname, signal_name, handler,
				handler_data ? handler_data : "NULL");
	}
      else
	{
	  source_add_to_buffer (data, GLADE_SIGNAL_CONNECTIONS,
				"  gtk_signal_connect (GTK_OBJECT (%s), \"%s\",\n"
				"                      GTK_SIGNAL_FUNC (%s),\n"
				"                      %s);\n",
				data->wname, signal_name, handler,
				handler_data ? handler_data : "NULL");
	}
    }
}


static gchar *
get_type_name (GtkType type, gboolean * is_pointer)
{
  static gchar *GbTypeNames[] =
  {
    "char", "gchar",
    "bool", "gboolean",
    "int", "gint",
    "uint", "guint",
    "long", "glong",
    "ulong", "gulong",
    "float", "gfloat",
    "double", "gdouble",
    "string", "gchar",
    "enum", "gint",
    "flags", "gint",
    "boxed", "gpointer",
    "foreign", "gpointer",
    "callback", "GtkCallback",	/* ?? */
    "args", "gpointer",

    "pointer", "gpointer",
    "signal", "gpointer",
    "c_callback", "GtkCallback",	/* ?? */

    NULL
  };

  GtkType parent_type;
  gchar *type_name, *parent_type_name;
  gint i;

  *is_pointer = FALSE;
  type_name = gtk_type_name (type);
  for (i = 0; GbTypeNames[i]; i += 2)
    {
      if (!strcmp (type_name, GbTypeNames[i]))
	{
	  if (!strcmp (type_name, "string"))
	    *is_pointer = TRUE;
	  return GbTypeNames[i + 1];
	}
    }

  for (;;)
    {
      parent_type = gtk_type_parent (type);
      if (parent_type == 0)
	break;
      type = parent_type;
    }
  parent_type_name = gtk_type_name (type);
  if (!strcmp (parent_type_name, "GtkObject")
      || !strcmp (parent_type_name, "boxed")
      || !strcmp (parent_type_name, "GtkBoxed"))
    *is_pointer = TRUE;

  return type_name;
}


static gchar *
get_gdk_event (gchar * signal_name)
{
  static gchar *GbGDKEvents[] =
  {
    "button_press_event", "GdkEventButton",
    "button_release_event", "GdkEventButton",
    "motion_notify_event", "GdkEventMotion",
    "delete_event", "GdkEvent",
    "destroy_event", "GdkEvent",
    "expose_event", "GdkEventExpose",
    "key_press_event", "GdkEventKey",
    "key_release_event", "GdkEventKey",
    "enter_notify_event", "GdkEventCrossing",
    "leave_notify_event", "GdkEventCrossing",
    "configure_event", "GdkEventConfigure",
    "focus_in_event", "GdkEventFocus",
    "focus_out_event", "GdkEventFocus",
    "map_event", "GdkEvent",
    "unmap_event", "GdkEvent",
    "property_notify_event", "GdkEventProperty",
    "selection_clear_event", "GdkEventSelection",
    "selection_request_event", "GdkEventSelection",
    "selection_notify_event", "GdkEventSelection",
    "proximity_in_event", "GdkEventProximity",
    "proximity_out_event", "GdkEventProximity",
    "drag_begin_event", "GdkEventDragBegin",
    "drag_request_event", "GdkEventDragRequest",
    "drag_end_event", "GdkEventDragRequest",
    "drop_enter_event", "GdkEventDropEnter",
    "drop_leave_event", "GdkEventDropLeave",
    "drop_data_available_event", "GdkEventDropDataAvailable",
    "other_event", "GdkEventOther",
    "client_event", "GdkEventClient",
    "no_expose_event", "GdkEventNoExpose",
    NULL
  };

  gint i;

  for (i = 0; GbGDKEvents[i]; i += 2)
    {
      if (!strcmp (signal_name, GbGDKEvents[i]))
	return GbGDKEvents[i + 1];
    }
  return "GdkEvent";
}


/* This returns argument names to use for some known GTK signals. */
/* Note: Could possibly return the types to use as well, since several of
   them could be better, e.g. "GtkOrientation" instead of "gint". */
static gchar **
lookup_signal_arg_names (gchar * type, gchar * signal_name)
{
  /* Each arg array starts with the object type name and the signal name,
     and then signal arguments follow. Note that the spacing is hardcoded. */
  static gchar *GbArgTable[][8] =
  {
    {"GtkCList", "select_row",
     "gint             row",
     "gint             column",
     "GdkEvent        *event"},
    {"GtkCList", "unselect_row",
     "gint             row",
     "gint             column",
     "GdkEvent        *event"},
    {"GtkCList", "click_column",
     "gint             column"},

    {"GtkCList", "resize_column",
     "gint             column",
     "gint             width"},

    {"GtkCList", "extend_selection",
     "GtkScrollType    scroll_type",
     "gfloat           position",
     "gboolean         auto_start_selection"},
    {"GtkCList", "scroll_vertical",
     "GtkScrollType    scroll_type",
     "gfloat           position"},
    {"GtkCList", "scroll_horizontal",
     "GtkScrollType    scroll_type",
     "gfloat           position"},
    {"GtkContainer", "focus",
     "GtkDirectionType direction"},
    {"GtkCTree", "tree_select_row",
     "GList           *node",
     "gint             column"},
    {"GtkCTree", "tree_unselect_row",
     "GList           *node",
     "gint             column"},

    {"GtkCTree", "tree_expand",
     "GList           *node"},
    {"GtkCTree", "tree_collapse",
     "GList           *node"},
    {"GtkCTree", "tree_move",
     "GList           *node",
     "GList           *new_parent",
     "GList           *new_sibling"},
    {"GtkCTree", "change_focus_row_expansion",
     "GtkCTreeExpansionType expansion"},

    {"GtkEditable", "insert_text",
     "gchar           *new_text",
     "gint             new_text_length",
     "gint            *position"},
    {"GtkEditable", "delete_text",
     "gint             start_pos",
     "gint             end_pos"},
    {"GtkEditable", "set_editable",
     "gboolean         is_editable"},
    {"GtkEditable", "move_cursor",
     "gint             x",
     "gint             y"},
    {"GtkEditable", "move_word",
     "gint             num_words"},
    {"GtkEditable", "move_page",
     "gint             x",
     "gint             y"},
    {"GtkEditable", "move_to_row",
     "gint             row"},
    {"GtkEditable", "move_to_column",
     "gint             column"},

    {"GtkEditable", "kill_char",
     "gint             direction"},
    {"GtkEditable", "kill_word",
     "gint             direction"},
    {"GtkEditable", "kill_line",
     "gint             direction"},


    {"GtkInputDialog", "enable_device",
     "gint             deviceid"},
    {"GtkInputDialog", "disable_device",
     "gint             deviceid"},

    {"GtkListItem", "extend_selection",
     "GtkScrollType    scroll_type",
     "gfloat           position",
     "gboolean         auto_start_selection"},
    {"GtkListItem", "scroll_vertical",
     "GtkScrollType    scroll_type",
     "gfloat           position"},
    {"GtkListItem", "scroll_horizontal",
     "GtkScrollType    scroll_type",
     "gfloat           position"},

    {"GtkMenuShell", "move_current",
     "GtkMenuDirectionType direction"},
    {"GtkMenuShell", "activate_current",
     "gboolean         force_hide"},


    {"GtkNotebook", "switch_page",
     "GtkNotebookPage *page",
     "gint             page_num"},
    {"GtkStatusbar", "text_pushed",
     "guint            context_id",
     "gchar           *text"},
    {"GtkStatusbar", "text_popped",
     "guint            context_id",
     "gchar           *text"},
    {"GtkTipsQuery", "widget_entered",
     "GtkWidget       *widget",
     "gchar           *tip_text",
     "gchar           *tip_private"},
    {"GtkTipsQuery", "widget_selected",
     "GtkWidget       *widget",
     "gchar           *tip_text",
     "gchar           *tip_private",
     "GdkEventButton  *event"},
    {"GtkToolbar", "orientation_changed",
     "GtkOrientation   orientation"},
    {"GtkToolbar", "style_changed",
     "GtkToolbarStyle  style"},
    {"GtkWidget", "draw",
     "GdkRectangle    *area"},
    {"GtkWidget", "size_request",
     "GtkRequisition  *requisition"},
    {"GtkWidget", "size_allocate",
     "GtkAllocation   *allocation"},
    {"GtkWidget", "state_changed",
     "GtkStateType     state"},
    {"GtkWidget", "style_set",
     "GtkStyle        *previous_style"},

    {"GtkWidget", "install_accelerator",
     "gchar           *signal_name",
     "gchar            key",
     "gint             modifiers"},

    {"GtkWidget", "add_accelerator",
     "guint            accel_signal_id",
     "GtkAccelGroup   *accel_group",
     "guint            accel_key",
     "GdkModifierType  accel_mods",
     "GtkAccelFlags    accel_flags"},

    {"GtkWidget", "parent_set",
     "GtkObject       *old_parent"},

    {"GtkWidget", "remove_accelerator",
     "GtkAccelGroup   *accel_group",
     "guint            accel_key",
     "GdkModifierType  accel_mods"},

    {"GtkWidget", "debug_msg",
     "gchar           *message"},
    {"GtkWindow", "move_resize",
     "gint            *x",
     "gint            *y",
     "gint             width",
     "gint             height"},
    {"GtkWindow", "set_focus",
     "GtkWidget       *widget"},

    {"GtkWidget", "selection_get",
     "GtkSelectionData *data",
     "guint            info",
     "guint            time"},
    {"GtkWidget", "selection_received",
     "GtkSelectionData *data",
     "guint            time"},

    {"GtkWidget", "drag_begin",
     "GdkDragContext  *drag_context"},
    {"GtkWidget", "drag_end",
     "GdkDragContext  *drag_context"},
    {"GtkWidget", "drag_data_delete",
     "GdkDragContext  *drag_context"},
    {"GtkWidget", "drag_leave",
     "GdkDragContext  *drag_context",
     "guint            time"},
    {"GtkWidget", "drag_motion",
     "GdkDragContext  *drag_context",
     "gint             x",
     "gint             y",
     "guint            time"},
    {"GtkWidget", "drag_drop",
     "GdkDragContext  *drag_context",
     "gint             x",
     "gint             y",
     "guint            time"},
    {"GtkWidget", "drag_data_get",
     "GdkDragContext  *drag_context",
     "GtkSelectionData *data",
     "guint            info",
     "guint            time"},
    {"GtkWidget", "drag_data_received",
     "GdkDragContext  *drag_context",
     "gint             x",
     "gint             y",
     "GtkSelectionData *data",
     "guint            info",
     "guint            time"},

    {NULL}
  };

  gint i;

  for (i = 0; GbArgTable[i][0]; i++)
    {
      if (!strcmp (type, GbArgTable[i][0])
	  && !strcmp (signal_name, GbArgTable[i][1]))
	return &GbArgTable[i][2];
    }
  return NULL;
}


void
gb_widget_write_accelerators_source (GtkWidget * widget,
				     GbWidgetWriteSourceData * data)
{
  GList *item = data->widget_data->accelerators;
  GbAccelerator *accel;

  while (item)
    {
      accel = (GbAccelerator *) item->data;
      item = item->next;

      /* The code to create the accel_group is output in source.c */
      data->need_accel_group = TRUE;
      source_add (data,
		  "  gtk_widget_add_accelerator (%s, \"%s\", accel_group,\n"
		  "                              GDK_%s, %s,\n"
		  "                              GTK_ACCEL_VISIBLE);\n",
		  data->wname, accel->signal, accel->key,
		  glade_util_create_modifiers_string (accel->modifiers));
    }
}



/*************************************************************************
 * Functions for replacing widgets in the user interface
 *************************************************************************/

/*
 * Replace a child with a new child. Used to replace placeholders with
 * a widget when adding widgets, and also to replace widgets with
 * placeholders when deleting. Returns TRUE on success.
 * NOTE: gbwidgets do not currently have their own function for this,
 * but we'll probably add it at some point.
 */
gboolean
gb_widget_replace_child (GtkWidget * widget,
			 GtkWidget * current_child,
			 GtkWidget * new_child)
{
  gchar *child_name, *new_child_name;

  /* We move the child name to the new widget, except for toolbar buttons,
     in which case we leave the new child's child_name as it is, and we never
     copy the toolbar button child name to the new child. */
  child_name = gb_widget_get_child_name (current_child);
  new_child_name = gb_widget_get_child_name (new_child);
  if (new_child_name == NULL || strcmp (new_child_name, "Toolbar:button"))
    {
      if (child_name && strcmp (child_name, "Toolbar:button"))
	gb_widget_set_child_name (new_child, child_name);
      else
	gb_widget_set_child_name (new_child, NULL);
    }

#ifdef USE_GNOME
  if (GNOME_IS_DOCK_ITEM (widget))
    {
      gboolean is_floating;
      gint x, y;

      /* GnomeDockItem widgets which are floating automatically disappear when
	 the child is removed, so we remember the position and try to show it
	 again in the same place after adding the new widget. */
      is_floating = GNOME_DOCK_ITEM (widget)->is_floating ? TRUE : FALSE;
      if (is_floating)
	gnome_dock_item_get_floating_position (GNOME_DOCK_ITEM (widget),
					       &x, &y);
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_widget_hide (new_child);
      gtk_container_add (GTK_CONTAINER (widget), new_child);
      gtk_widget_show (new_child);

      if (is_floating)
	gnome_dock_item_detach (GNOME_DOCK_ITEM (widget), x, y);
    }
  else if (GTK_IS_BIN (widget))
#else
  if (GTK_IS_BIN (widget))
#endif
    {
      /* For a bin, we just delete the existing child and add the new one. */
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_container_add (GTK_CONTAINER (widget), new_child);

    }
  else if (GTK_IS_BOX (widget))
    {
      /* For a box, we find out the position of the current child, delete it,
         add the new one, and move it into position with reorder_child().
	 If the existing child is a placeholder and the new one is a menubar
	 or toolbar, we set the packing so it doesn't expand, as that is
	 probably what the user wants. */
      gboolean expand, fill;
      guint padding;
      GtkPackType pack_type;
      gint pos = glade_util_get_box_pos (GTK_BOX (widget), current_child);
      g_return_val_if_fail (pos != -1, FALSE);
      gtk_box_query_child_packing (GTK_BOX (widget), current_child,
				   &expand, &fill, &padding, &pack_type);

      /* If we are replacing a placeholder (i.e. we are adding a new widget),
	 we try to set the expand & fill options to reasonable defaults. */
      if (GB_IS_PLACEHOLDER (current_child))
	{
	  if (GTK_IS_LABEL (new_child)
	      || GTK_IS_BUTTON (new_child)
	      || GTK_IS_OPTION_MENU (new_child)
	      || GTK_IS_PROGRESS (new_child)
	      || GTK_IS_PROGRESS_BAR (new_child)
	      || GTK_IS_MENU_BAR (new_child)
	      || GTK_IS_TOOLBAR (new_child)
	      || GTK_IS_STATUSBAR (new_child))
	    {
	      expand = FALSE;
	      fill = FALSE;
	    }

	  /* In a vbox, entry & combo widgets should not expand/fill either. */
	  if (GTK_IS_VBOX (widget))
	    {
	      if (GTK_IS_ENTRY (new_child)
		  || GTK_IS_COMBO (new_child)
		  || GTK_IS_SPIN_BUTTON (new_child)
#ifdef USE_GNOME
		   || GNOME_IS_DATE_EDIT (new_child)
		   || GNOME_IS_FILE_ENTRY (new_child)
		   || GNOME_IS_NUMBER_ENTRY (new_child)
		   || GNOME_IS_PIXMAP_ENTRY (new_child)
#endif
		  )
		{
		  expand = FALSE;
		  fill = FALSE;
		}
	    }
	}

      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_container_add (GTK_CONTAINER (widget), new_child);

      gtk_box_set_child_packing (GTK_BOX (widget), new_child, expand, fill,
				 padding, pack_type);
      gtk_box_reorder_child (GTK_BOX (widget), new_child, pos);

    }
  else if (GTK_IS_TOOLBAR (widget))
    {
      /* For a toolbar, we find out the position of the current child, then
	 remove it. We call a special function to insert the new child in the
	 same position. */
      GtkToolbarChild *tbchild;
      gint pos;

      tbchild = glade_util_find_toolbar_child (widget, current_child, &pos,
					       NULL);
      g_return_val_if_fail (tbchild != NULL, FALSE);
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gb_toolbar_insert_toolbar_child (GTK_TOOLBAR (widget), new_child, pos);
    }
  else if (GTK_IS_LIST (widget))
    {
      /* For a list, we find out the position of the current child, delete it,
         and add the new one at the same position. */
      gint pos = gtk_list_child_position (GTK_LIST (widget), current_child);
      GList glist =
      {NULL, NULL, NULL};
      glist.data = current_child;
      gtk_list_remove_items (GTK_LIST (widget), &glist);
      glist.data = new_child;
      gtk_list_insert_items (GTK_LIST (widget), &glist, pos);

    }
  else if (GTK_IS_NOTEBOOK (widget))
    {
      /* For a notebook, we find out the position of the current child, delete
         it, and add the new one at the same position. If the current_child is
         a notebook tab, just replace it. */
      GtkWidget *tab_label, *notebook_page;
      gint pos;
      GtkNotebookPage *page = find_notebook_page (widget, current_child, &pos);
      g_return_val_if_fail (page != NULL, FALSE);
      if (page->child == current_child)
	{
	  tab_label = page->tab_label;
	  gtk_widget_ref (tab_label);
	  gtk_notebook_remove_page (GTK_NOTEBOOK (widget), pos);
	  gtk_notebook_insert_page (GTK_NOTEBOOK (widget), new_child, tab_label,
				    pos);
	  gtk_notebook_set_page (GTK_NOTEBOOK (widget), pos);
	  gtk_widget_unref (tab_label);
	}
      else
	{
	  notebook_page = page->child;
	  gtk_widget_ref (notebook_page);
	  gtk_notebook_remove_page (GTK_NOTEBOOK (widget), pos);
	  gtk_notebook_insert_page (GTK_NOTEBOOK (widget), notebook_page, new_child,
				    pos);
	  gtk_notebook_set_page (GTK_NOTEBOOK (widget), pos);
	  gtk_widget_unref (notebook_page);
	}

    }
  else if (GTK_IS_PANED (widget))
    {
      /* For paned, we find out the position of the current child, delete it,
         and add the new one at the same position. */
      gint pos = (GTK_PANED (widget)->child1 == current_child) ? 1 : 2;
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      if (pos == 1)
	gtk_paned_add1 (GTK_PANED (widget), new_child);
      else
	gtk_paned_add2 (GTK_PANED (widget), new_child);

    }
  else if (GTK_IS_TABLE (widget))
    {
      /* For a table, we find out the position & size of the current child,
         delete it, and add the new one in the same place. */
      gint left, right, top, bottom;
      GtkTableChild *tchild;
      GtkAttachOptions xoptions, yoptions;

      tchild = glade_util_find_table_child (GTK_TABLE (widget), current_child);
      g_return_val_if_fail (tchild != NULL, FALSE);

      left = tchild->left_attach;
      right = tchild->right_attach;
      top = tchild->top_attach;
      bottom = tchild->bottom_attach;

      xoptions = 0;
      if (tchild->xexpand)
	xoptions |= GTK_EXPAND;
      if (tchild->xshrink)
	xoptions |= GTK_SHRINK;
      if (tchild->xfill)
	xoptions |= GTK_FILL;

      yoptions = 0;
      if (tchild->yexpand)
	yoptions |= GTK_EXPAND;
      if (tchild->yshrink)
	yoptions |= GTK_SHRINK;
      if (tchild->yfill)
	yoptions |= GTK_FILL;

      /* If we are replacing a placeholder (i.e. we are adding a new widget),
	 we try to set the expand & fill options to reasonable defaults. */
      if (GB_IS_PLACEHOLDER (current_child))
	{
	  if (GTK_IS_LABEL (new_child)
	      || GTK_IS_BUTTON (new_child)
	      || GTK_IS_OPTION_MENU (new_child)
	      || GTK_IS_PROGRESS (new_child)
	      || GTK_IS_PROGRESS_BAR (new_child)
	      || GTK_IS_MENU_BAR (new_child)
	      || GTK_IS_TOOLBAR (new_child)
	      || GTK_IS_STATUSBAR (new_child))
	    {
	      xoptions = 0;
	      yoptions = 0;
	    }
	  else if (GTK_IS_ENTRY (new_child)
		   || GTK_IS_COMBO (new_child)
		   || GTK_IS_SPIN_BUTTON (new_child)
#ifdef USE_GNOME
		   || GNOME_IS_DATE_EDIT (new_child)
		   || GNOME_IS_FILE_ENTRY (new_child)
		   || GNOME_IS_NUMBER_ENTRY (new_child)
		   || GNOME_IS_PIXMAP_ENTRY (new_child)
#endif
		   )
	    {
	      xoptions = GTK_EXPAND | GTK_FILL;
	      yoptions = 0;
	    }
	}

      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_table_attach (GTK_TABLE (widget), new_child,
			left, right, top, bottom, xoptions, yoptions, 0, 0);

      /* Note that if we have just added a placeholder, but there is already
	 a widget at the same position in the table, the placeholder will be
	 destroyed immediately. Thus don't rely on the new widget still being
	 alive after calling gb_widget_replace_child(). */
      gb_table_update_placeholders (widget, -1, -1);
    }
  else if (GTK_IS_TREE (widget))
    {
      /* For a tree, we find out the position of the current child, delete it,
         and add the new one at the same position. */
      gint pos = gtk_tree_child_position (GTK_TREE (widget), current_child);
      GList glist =
      {NULL, NULL, NULL};
      glist.data = current_child;
      gtk_tree_remove_items (GTK_TREE (widget), &glist);
      gtk_tree_insert (GTK_TREE (widget), new_child, pos);

    }
  else if (GTK_IS_CLIST (widget))
    {
      /* For a clist, we check if the widget is a column title and if it is
	 we replace it. */
	gint pos;
	GtkCList *clist;

	clist = GTK_CLIST(widget);

	for (pos = 0; pos < clist->columns; pos++)
	{
	    if (clist->column[pos].button == current_child)
	    {
		gtk_clist_set_column_widget(clist, pos, new_child);
 	    }
	}
    }
  else if (GTK_IS_FIXED (widget) || GTK_IS_LAYOUT (widget))
    {
      GbWidgetData *wdata;
      gint x, y, w, h;

      /* For a fixed, we find the position and size of the existing child,
	 remove it, and add the new one in the same place. */
      wdata = gtk_object_get_data (GTK_OBJECT (current_child),
				   GB_WIDGET_DATA_KEY);
      g_return_val_if_fail (wdata != NULL, FALSE);

      x = wdata->x;
      y = wdata->y;
      w = wdata->width;
      h = wdata->height;

      wdata = gtk_object_get_data (GTK_OBJECT (new_child), GB_WIDGET_DATA_KEY);
      g_return_val_if_fail (wdata != NULL, FALSE);
      wdata->flags |= GB_X_SET | GB_Y_SET | GB_WIDTH_SET | GB_HEIGHT_SET;
      wdata->x = x;
      wdata->y = y;
      wdata->width = w;
      wdata->height = h;

      /* Reset the widget's uposition, just in case it gets added to a standard
	 container. */
      gtk_widget_set_uposition (current_child, -1, -1);

      /* FIXME: GTK+ 1.2.3 bug workaround. We need to ref the widget to stop
	 gtk_layout_remove() from issuing a warning. */
      gtk_widget_ref (current_child);
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_widget_unref (current_child);

      if (GTK_IS_FIXED (widget))
	{
	  gtk_fixed_put (GTK_FIXED (widget), new_child, x, y);
	  gtk_widget_set_uposition (new_child, x, y);
	}
      else
	{
	  gtk_layout_put (GTK_LAYOUT (widget), new_child, x, y);
	}
      gb_widget_set_usize (new_child, w, h);
    }

#ifdef USE_GNOME
  else if (GNOME_IS_DOCK (widget))
    {
      /* For a GnomeDock, we call gnome_dock_set_client_area (). It removes
	 the existing child automatically. */
      gnome_dock_set_client_area (GNOME_DOCK (widget), new_child);
    }

#endif

  else if (GTK_IS_CONTAINER (widget))
    {
      /* General code for container - has to remove all children and add back
         NOTE: this may not work for specialised containers.
         NOTE: need to ref widgets? */
      g_warning (_("replacing child of container - not implemented yet\n"));
      return FALSE;
    }

  return TRUE;
}


/*************************************************************************
 * Functions for showing/hiding tooltips of the widgets
 *************************************************************************/

gboolean
gb_widget_get_show_tooltips ()
{
  return GTK_TOOLTIPS (gb_widget_tooltips)->enabled;
}


void
gb_widget_set_show_tooltips (gboolean show)
{
  if (show)
    gtk_tooltips_enable (gb_widget_tooltips);
  else
    gtk_tooltips_disable (gb_widget_tooltips);
}


void
gb_widget_reset_tooltips ()
{
  gtk_object_destroy (GTK_OBJECT (gb_widget_tooltips));
  gb_widget_tooltips = gtk_tooltips_new ();
}



/*************************************************************************
 * Misc. Functions
 *************************************************************************/


/* This is a GTK bug workaround for combo widgets. They should manage the
   size of their GtkEntry, but they don't at present, so we have to set its
   size explicitly (and also in the source code output).
   I think this has been fixed. It seems to be OK in GTK+ 1.2.3. */
void
gb_widget_set_usize (GtkWidget *widget,
		     gint w,
		     gint h)
{
#if 0
  if (GTK_IS_COMBO (widget))
    gtk_widget_set_usize (GTK_COMBO (widget)->entry,
			  w - 16 < 0 ? -1 : w - 16, h);
#endif
  gtk_widget_set_usize (widget, w, h);
}


static GtkNotebookPage *
find_notebook_page (GtkWidget * widget, GtkWidget * current_child, gint * pos)
{
  GList *children;
  GtkNotebookPage *page;
  gint page_num = 0;

  children = GTK_NOTEBOOK (widget)->children;
  while (children)
    {
      page = children->data;
      if (page->child == current_child || page->tab_label == current_child)
	{
	  *pos = page_num;
	  return page;
	}
      children = children->next;
      page_num++;
    }
  return NULL;
}


/* This calls the given callback for each child of a widget. It gets round
   some of the quirks of the different versions of GTK, and descends menus
   as well. */
void
gb_widget_children_foreach (GtkWidget *widget,
			    GtkCallback callback,
			    gpointer data)
{
  /* SPECIAL CODE: for table, so we output in the reverse order. */
  if (GTK_IS_TABLE (widget))
    table_foreach (GTK_TABLE (widget), callback, data);
  else if (GTK_IS_CONTAINER (widget))
    gtk_container_forall (GTK_CONTAINER (widget), callback, data);
}


/* This function is used to iterate through the table children in reverse.
   It is needed so we output the XML file in the same order each time. */
static void
table_foreach (GtkTable * table,
	       GtkCallback callback,
	       gpointer callback_data)
{
  GList *children;
  GtkTableChild *child;

  children = g_list_last (table->children);
  while (children)
    {
      child = children->data;
      children = children->prev;

      (*callback) (child->widget, callback_data);
    }
}


/*************************************************************************
 * Common popup menu callbacks
 *************************************************************************/

static void
gb_widget_add_alignment (GtkWidget * menuitem,
			 GtkWidget * widget)
{
  GtkWidget *alignment, *parent;

  parent = widget->parent;
  alignment = gb_widget_new ("GtkAlignment", parent);

  gtk_widget_ref (widget);
  if (!gb_widget_replace_child (parent, widget, alignment))
    {
      glade_util_show_message_box (_("Couldn't insert GtkAlignment widget."));
      gtk_widget_destroy (alignment);
      gtk_widget_unref (widget);
      return;
    }
  if (GTK_BIN (alignment)->child)
    gtk_container_remove (GTK_CONTAINER (alignment),
			  GTK_BIN (alignment)->child);
  gtk_container_add (GTK_CONTAINER (alignment), widget);
  gtk_widget_unref (widget);
  tree_insert_widget_parent (alignment, widget);
}


static void
gb_widget_remove_alignment (GtkWidget * menuitem,
			    GtkWidget * widget)
{
  GtkWidget *alignment, *parent;

  alignment = widget->parent;
  g_return_if_fail (GTK_IS_ALIGNMENT (alignment));
  parent = alignment->parent;

  gtk_widget_ref (widget);
  gtk_widget_ref (alignment);
  gtk_container_remove (GTK_CONTAINER (alignment), widget);
  if (gb_widget_replace_child (parent, alignment, widget))
    {
      tree_remove_widget_parent (parent, widget);
    }
  else
    {
      glade_util_show_message_box (_("Couldn't remove GtkAlignment widget."));
      /* Try to put it back as it was. */
      gtk_container_add (GTK_CONTAINER (alignment), widget);
    }
  gtk_widget_unref (alignment);
  gtk_widget_unref (widget);
}


static void
gb_widget_add_event_box (GtkWidget * menuitem,
			 GtkWidget * widget)
{
  GtkWidget *event_box, *parent;

  parent = widget->parent;
  event_box = gb_widget_new ("GtkEventBox", parent);

  gtk_widget_ref (widget);
  if (!gb_widget_replace_child (parent, widget, event_box))
    {
      glade_util_show_message_box (_("Couldn't insert GtkEventBox widget."));
      gtk_widget_destroy (event_box);
      gtk_widget_unref (widget);
      return;
    }
  if (GTK_BIN (event_box)->child)
    gtk_container_remove (GTK_CONTAINER (event_box),
			  GTK_BIN (event_box)->child);
  gtk_container_add (GTK_CONTAINER (event_box), widget);
  gtk_widget_unref (widget);
  tree_insert_widget_parent (event_box, widget);
}

static void
gb_widget_remove_event_box (GtkWidget * menuitem,
			    GtkWidget * widget)
{
  GtkWidget *event_box, *parent;

  event_box = widget->parent;
  g_return_if_fail (GTK_IS_EVENT_BOX (event_box));
  parent = event_box->parent;

  gtk_widget_ref (widget);
  gtk_widget_ref (event_box);
  gtk_container_remove (GTK_CONTAINER (event_box), widget);
  if (gb_widget_replace_child (parent, event_box, widget))
    {
      tree_remove_widget_parent (parent, widget);
    }
  else
    {
      glade_util_show_message_box (_("Couldn't remove GtkEventBox widget."));
      /* Try to put it back as it was. */
      gtk_container_add (GTK_CONTAINER (event_box), widget);
    }
  gtk_widget_unref (event_box);
  gtk_widget_unref (widget);
}


static void
gb_widget_redisplay_window (GtkWidget * menuitem,
			    GtkWidget * widget)
{
  g_return_if_fail (GTK_IS_WINDOW (widget));

  glade_util_close_window (widget);
  gtk_widget_show (widget);
}


static void
gb_widget_add_scrolled_window (GtkWidget * menuitem,
			       GtkWidget * widget)
{
  GtkWidget *scrolledwin, *parent;

  parent = widget->parent;
  scrolledwin = gb_widget_new ("GtkScrolledWindow", parent);

  gtk_widget_ref (widget);
  if (!gb_widget_replace_child (parent, widget, scrolledwin))
    {
      glade_util_show_message_box (_("Couldn't insert GtkScrolledWindow widget."));
      gtk_widget_destroy (scrolledwin);
      gtk_widget_unref (widget);
      return;
    }
  if (GTK_BIN (scrolledwin)->child)
    gtk_container_remove (GTK_CONTAINER (scrolledwin),
			  GTK_BIN (scrolledwin)->child);
  gtk_container_add (GTK_CONTAINER (scrolledwin), widget);
  gtk_widget_unref (widget);
  tree_insert_widget_parent (scrolledwin, widget);
}


static void
gb_widget_remove_scrolled_window (GtkWidget * menuitem,
				  GtkWidget * widget)
{
  GtkWidget *scrolledwin, *parent;

  scrolledwin = widget->parent;
  g_return_if_fail (GTK_IS_SCROLLED_WINDOW (scrolledwin));
  parent = scrolledwin->parent;

  gtk_widget_ref (widget);
  gtk_widget_ref (scrolledwin);
  gtk_container_remove (GTK_CONTAINER (scrolledwin), widget);
  if (gb_widget_replace_child (parent, scrolledwin, widget))
    {
      tree_remove_widget_parent (parent, widget);
    }
  else
    {
      glade_util_show_message_box (_("Couldn't remove GtkScrolledWindow widget."));
      /* Try to put it back as it was. */
      gtk_container_add (GTK_CONTAINER (scrolledwin), widget);
    }
  gtk_widget_unref (scrolledwin);
  gtk_widget_unref (widget);
}


/*************************************************************************
 * Common functions used by gbwidgets.
 *************************************************************************/

/* This gets the child label for buttons or menuitems or subclasses.
   This is for showing or saving. */
void
gb_widget_output_child_label (GtkWidget * widget, GbWidgetGetArgData * data,
			      const gchar * Label)
{
  GtkWidget *child;
  gchar *label_text;

  child = GTK_BIN (widget)->child;

  /* Note that we don't want to save the child label if it is a GbWidget,
     since it will be saved as a separate widget. */
  if (child && GTK_IS_LABEL (child) && !GB_IS_GB_WIDGET (child))
    {
      label_text = glade_util_get_label_text (child);
      gb_widget_output_translatable_text (data, Label, label_text);
      g_free (label_text);

      if (data->action == GB_SHOWING)
	property_set_sensitive (Label, TRUE);
    }
  else
    {
      if (data->action == GB_SHOWING)
	{
	  gb_widget_output_text (data, Label, "");
	  property_set_sensitive (Label, FALSE);
	}
    }
}


/* This sets the child label for buttons/items/menuitems or subclasses.
   This is for applying or loading. */
void
gb_widget_input_child_label (GtkWidget * widget, GbWidgetSetArgData * data,
			     const gchar * Label)
{
  GtkWidget *child, *label;
  gchar *label_text;

  child = GTK_BIN (widget)->child;

  label_text = gb_widget_input_text (data, Label);
  if (data->apply)
    {
      if (child && GTK_IS_LABEL (child))
	{
	  /* We use parse_uline so letters can be underlined. */
	  gtk_label_parse_uline (GTK_LABEL (child), label_text);
	}
      else
	{
	  if (child != NULL)
	    gtk_container_remove (GTK_CONTAINER (widget), child);
	  if (GTK_IS_MENU_ITEM (widget))
	    {
	      label = gtk_accel_label_new ("");
	      gtk_accel_label_set_accel_widget (GTK_ACCEL_LABEL (label),
						widget);
	    }
	  else
	    {
	      label = gtk_label_new ("");
	    }
	  gtk_container_add (GTK_CONTAINER (widget), label);
	  gtk_label_parse_uline (GTK_LABEL (label), label_text);
	  gtk_widget_show (label);

	  /* Simple child labels are given different alignments according to
	     the parent. GtkButton and GtkToggleButton are centred. All the
	     others are aligned left. See the GTK _new_with_label() fns. */
	  if (data->action == GB_LOADING)
	    {
	      if (GTK_IS_CHECK_BUTTON (widget)
		  || GTK_IS_ITEM (widget))
		gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	    }
	}
    }
  /* This isn't very nice. When a text property is got from the property
     editor (i.e. when action is GB_APPLYING) it needs to be freed after. */
  if (data->action == GB_APPLYING)
    g_free (label_text);
}


void
gb_widget_create_child_label_popup_menu (GtkWidget	       *widget,
					 GbWidgetCreateMenuData *data)
{
  GtkWidget *menuitem;

  if (GTK_IS_LABEL (GTK_BIN (widget)->child))
    {
      menuitem = gtk_menu_item_new_with_label (_("Remove Label"));
      gtk_widget_show (menuitem);
      gtk_menu_append (GTK_MENU (data->menu), menuitem);
      gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			  GTK_SIGNAL_FUNC (gb_widget_remove_label), widget);
    }
}


void
gb_widget_remove_label (GtkWidget * menuitem,
			GtkWidget * widget)
{
  GtkWidget *child;

  g_return_if_fail (GTK_IS_BIN (widget));

  child = GTK_BIN (widget)->child;
  if (child && GTK_IS_LABEL (child))
    editor_delete_widget (child);
}


/*************************************************************************
 * Adjustment convenience functions - handles all 6 values.
 *************************************************************************/
void
gb_widget_output_adjustment (GbWidgetGetArgData * data,
			     const gchar * Values[],
			     GtkAdjustment * adjustment)
{
  if (Values[0])
    gb_widget_output_float (data, Values[0], adjustment->value);
  if (Values[1])
    gb_widget_output_float (data, Values[1], adjustment->lower);
  if (Values[2])
    gb_widget_output_float (data, Values[2], adjustment->upper);
  if (Values[3])
    gb_widget_output_float (data, Values[3], adjustment->step_increment);
  if (Values[4])
    gb_widget_output_float (data, Values[4], adjustment->page_increment);
  if (Values[5])
    gb_widget_output_float (data, Values[5], adjustment->page_size);
}


gboolean
gb_widget_input_adjustment (GbWidgetSetArgData * data,
			    const gchar * Values[],
			    GtkAdjustment * adjustment)
{
  gfloat value, lower, upper, step_inc, page_inc, page_size, tmp;

  value = adjustment->value;
  lower = adjustment->lower;
  upper = adjustment->upper;
  step_inc = adjustment->step_increment;
  page_inc = adjustment->page_increment;
  page_size = adjustment->page_size;

  if (Values[0])
    {
      tmp = gb_widget_input_float (data, Values[0]);
      if (data->apply)
	value = tmp;
    }

  if (Values[1])
    {
      tmp = gb_widget_input_float (data, Values[1]);
      if (data->apply)
	lower = tmp;
    }

  if (Values[2])
    {
      tmp = gb_widget_input_float (data, Values[2]);
      if (data->apply)
	upper = tmp;
    }

  if (Values[3])
    {
      tmp = gb_widget_input_float (data, Values[3]);
      if (data->apply)
	step_inc = tmp;
    }

  if (Values[4])
    {
      tmp = gb_widget_input_float (data, Values[4]);
      if (data->apply)
	page_inc = tmp;
    }

  if (Values[5])
    {
      tmp = gb_widget_input_float (data, Values[5]);
      if (data->apply)
	page_size = tmp;
    }

  /* Only return TRUE if one or more of the properties have changed. */
  if (adjustment->value != value
      || adjustment->lower != lower
      || adjustment->upper != upper
      || adjustment->step_increment != step_inc
      || adjustment->page_increment != page_inc
      || adjustment->page_size != page_size)
    {
      adjustment->value = value;
      adjustment->lower = lower;
      adjustment->upper = upper;
      adjustment->step_increment = step_inc;
      adjustment->page_increment = page_inc;
      adjustment->page_size = page_size;
      return TRUE;
    }
  return FALSE;
}


/*************************************************************************
 * Functions to input/output properties.
 *************************************************************************/

/* Inputting Properties - Loading or Applying. */

gchar*
gb_widget_input_string (GbWidgetSetArgData *data,
			const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_string (data, property);
  else
    return property_get_string (property, data->property_to_apply,
				&data->apply);
}


gchar*
gb_widget_input_text (GbWidgetSetArgData *data,
		      const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_text (data, property);
  else
    return property_get_text (property, data->property_to_apply,
			      &data->apply);
}


gint
gb_widget_input_int (GbWidgetSetArgData *data,
		     const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_int (data, property);
  else
    return property_get_int (property, data->property_to_apply,
			     &data->apply);
}


gint
gb_widget_input_optional_int (GbWidgetSetArgData *data,
			      const gchar *property,
			      gboolean *is_set)
{
  gint value;

  if (data->action == GB_LOADING)
    {
      value = load_int (data, property);
      if (is_set)
	*is_set = data->apply;
      return value;
    }
  else
    return property_get_optional_int (property, data->property_to_apply,
				      &data->apply, is_set);
}


gfloat
gb_widget_input_float (GbWidgetSetArgData *data,
		       const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_float (data, property);
  else
    return property_get_float (property, data->property_to_apply,
			       &data->apply);
}


gboolean
gb_widget_input_bool (GbWidgetSetArgData *data,
		      const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_bool (data, property);
  else
    return property_get_bool (property, data->property_to_apply,
			      &data->apply);
}


gchar*
gb_widget_input_choice (GbWidgetSetArgData *data,
			const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_choice (data, property);
  else
    return property_get_choice (property, data->property_to_apply,
				&data->apply);
}


gchar*
gb_widget_input_combo (GbWidgetSetArgData *data,
		       const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_combo (data, property);
  else
    return property_get_combo (property, data->property_to_apply,
			       &data->apply);
}


GdkColor*
gb_widget_input_color (GbWidgetSetArgData *data,
		       const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_color (data, property);
  else
    return property_get_color (property, data->property_to_apply,
			       &data->apply);
}


GdkPixmap*
gb_widget_input_bgpixmap (GbWidgetSetArgData *data,
			  const gchar *property,
			  gchar **filename)
{
  if (data->action == GB_LOADING)
    return load_bgpixmap (data, property, filename);
  else
    return property_get_bgpixmap (property, data->property_to_apply,
				  &data->apply, filename);
}


gpointer
gb_widget_input_dialog (GbWidgetSetArgData *data,
			const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_dialog (data, property);
  else
    return property_get_dialog (property, data->property_to_apply,
				&data->apply);
}


gchar*
gb_widget_input_filename (GbWidgetSetArgData *data,
			  const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_filename (data, property);
  else
    return property_get_filename (property, data->property_to_apply,
				  &data->apply);
}


gchar*
gb_widget_input_pixmap_filename (GbWidgetSetArgData *data,
				 const gchar *property)
{
  if (data->action == GB_LOADING)
    return load_pixmap_filename (data, property);
  else
    return property_get_filename (property, data->property_to_apply,
				  &data->apply);
}


GdkFont*
gb_widget_input_font (GbWidgetSetArgData *data,
		      const gchar *property,
		      gchar **xlfd_fontname)
{
  if (data->action == GB_LOADING)
    return load_font (data, property, xlfd_fontname);
  else
    return property_get_font (property, data->property_to_apply,
			      &data->apply, xlfd_fontname);
}


/* Outputting Properties - Saving or Showing. */
void
gb_widget_output_string (GbWidgetGetArgData *data,
			 const gchar *property,
			 const gchar *value)
{
  if (data->action == GB_SAVING)
    save_string (data, property, value);
  else
    property_set_string (property, value);
}


void
gb_widget_output_translatable_string (GbWidgetGetArgData *data,
				      const gchar *property,
				      const gchar *value)
{
  if (data->action == GB_SAVING)
    save_translatable_string (data, property, value);
  else
    property_set_string (property, value);
}


void
gb_widget_output_text (GbWidgetGetArgData *data,
		       const gchar *property,
		       const gchar *value)
{
  if (data->action == GB_SAVING)
    save_text (data, property, value);
  else
    property_set_text (property, value);
}


void
gb_widget_output_translatable_text (GbWidgetGetArgData *data,
				    const gchar *property,
				    const gchar *value)
{
  if (data->action == GB_SAVING)
    save_translatable_text (data, property, value);
  else
    property_set_text (property, value);
}


void
gb_widget_output_int (GbWidgetGetArgData *data,
		      const gchar *property,
		      gint value)
{
#if 0
  if (property == GbX)
    g_print ("X: %i ", value);
  else if (property == GbY)
    g_print ("Y: %i ", value);
  else if (property == GbWidth)
    g_print ("W: %i ", value);
  else if (property == GbHeight)
    g_print ("H: %i ", value);
#endif

  if (data->action == GB_SAVING)
    save_int (data, property, value);
  else
    property_set_int (property, value);
}


void
gb_widget_output_optional_int (GbWidgetGetArgData *data,
			       const gchar *property,
			       gint value,
			       gboolean is_set)
{
  if (data->action == GB_SAVING)
    {
      if (is_set)
	save_int (data, property, value);
    }
  else
    property_set_optional_int (property, value, is_set);
}


void
gb_widget_output_float (GbWidgetGetArgData *data,
			const gchar *property,
			gfloat value)
{
  if (data->action == GB_SAVING)
    save_float (data, property, value);
  else
    property_set_float (property, value);
}


void
gb_widget_output_bool (GbWidgetGetArgData *data,
		       const gchar *property,
		       gint value)
{
  if (data->action == GB_SAVING)
    save_bool (data, property, value);
  else
    property_set_bool (property, value);
}


void
gb_widget_output_choice (GbWidgetGetArgData *data,
			 const gchar *property,
			 gint value,
			 const gchar *symbol)
{
  if (data->action == GB_SAVING)
    save_choice (data, property, symbol);
  else
    property_set_choice (property, value);
}


void
gb_widget_output_combo (GbWidgetGetArgData *data,
			const gchar *property,
			const gchar *value)
{
  if (data->action == GB_SAVING)
    save_combo (data, property, value);
  else
    property_set_combo (property, value);
}


void
gb_widget_output_color (GbWidgetGetArgData *data,
			const gchar *property,
			GdkColor *value)
{
  if (data->action == GB_SAVING)
    save_color (data, property, value);
  else
    property_set_color (property, value);
}


void
gb_widget_output_bgpixmap (GbWidgetGetArgData *data,
			   const gchar *property,
			   GdkPixmap *value,
			   const gchar *filename)
{
  if (data->action == GB_SAVING)
    save_bgpixmap (data, property, filename);
  else
    property_set_bgpixmap (property, value, filename);
}


void
gb_widget_output_dialog (GbWidgetGetArgData *data,
			 const gchar *property,
			 const gchar *string,
			 gconstpointer value)
{
  if (data->action == GB_SAVING)
    save_dialog (data, property, value);
  else
    property_set_dialog (property, string, value);
}


void
gb_widget_output_filename (GbWidgetGetArgData *data,
			   const gchar *property,
			   const gchar *value)
{
  if (data->action == GB_SAVING)
    save_filename (data, property, value);
  else
    property_set_filename (property, value);
}


void
gb_widget_output_pixmap_filename (GbWidgetGetArgData *data,
				  const gchar *property,
				  const gchar *value)
{
  if (data->action == GB_SAVING)
    save_pixmap_filename (data, property, value);
  else
    property_set_filename (property, value);
}


void
gb_widget_output_font (GbWidgetGetArgData *data,
		       const gchar *property,
		       GdkFont *value,
		       const gchar *xlfd_fontname)
{
  if (data->action == GB_SAVING)
    save_font (data, property, xlfd_fontname);
  else
    property_set_font (property, value, xlfd_fontname);
}
