/* $Header: /cvs/gnome/gIDE/src/gI_globpatterns.c,v 1.5 2000/04/22 17:23:01 jpr Exp $ */
/*  gIDE
 *  Copyright (C) 1998-2000 Steffen Kern
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#ifdef HAVE_GTKTEXT_PATCH

#include <stdio.h>
#include <ctype.h>
#include <fnmatch.h>

#include <gtk/gtk.h>

#include "structs.h"
#include "gI_globpatterns.h"
#include "gI_hilite.h"
#include "gI_common.h"

typedef struct _GlobAssoc GlobAssoc;
struct _GlobAssoc {
  gchar *glob;
  gchar *pat;
};

/* The global glob<->pattern association list. */
static GList *globassoc = NULL;

/* --<local functions>------------------------------------------------ */
static gint
assoc_strict_cmp (GlobAssoc *ga1, GlobAssoc *ga2)
{
  g_assert (ga1);
  g_assert (ga2);

  return strcmp (ga1->glob, ga2->glob);
}

static gint
assoc_strict_cmp_2 (GlobAssoc *ga1, GlobAssoc *ga2)
{
  g_assert (ga1);
  g_assert (ga2);

  return strcmp (ga1->pat, ga2->pat);
}


static gint
assoc_glob_cmp (GlobAssoc *ga, const gchar *str)
{
  g_assert (ga);
  g_assert (str);

  return fnmatch (ga->glob, str, 0); /* FIXME: check the flags... */
}

/* --<exported functions>--------------------------------------------- */
/* gI_glob_assoc -- associates the glob pattern in glob with the
 * pattern in pat.  The glob strings are copied, so free them after the
 * call. */
void
gI_glob_assoc (gchar *glob, gchar *pat)
{
  GlobAssoc *assoc;

  g_return_if_fail (glob != NULL);
  g_return_if_fail (pat != NULL);

  assoc = g_new (GlobAssoc, 1);
  assoc->glob = g_strdup (glob);
  assoc->pat = g_strdup (pat);

  globassoc = g_list_prepend (globassoc, (gpointer)assoc);
}

/* gI_glob_unassoc -- removes glob from glob<->pattern structure, and
 * removes all associations to it.  It does *not* free the patterns.
 * The patterns is kept in a structure in gI_hilite, and is not really
 * part of the glob<->pattern association. */
void
gI_glob_unassoc (gchar *glob)
{
  GList *tmp;

  g_return_if_fail (glob != NULL);

  while ( (tmp = g_list_find_custom (globassoc, glob,
				     (GCompareFunc)assoc_strict_cmp)) ) {
    g_free (((GlobAssoc*)tmp->data)->glob);
    g_free (((GlobAssoc*)tmp->data)->pat);
    globassoc = g_list_remove_link (globassoc, tmp);
    g_list_free_1 (tmp);
  }
}

/* gI_glob_remove -- removes all instances of pat (found by pointer
 * comparison) from the glob<->pattern structur.  This function *must*
 * be called prior to freeing the pattern! */
void
gI_glob_remove (gchar *pat)
{
  GList *tmp;

  g_return_if_fail (pat != NULL);

  while ( (tmp = g_list_find_custom (globassoc, pat,
				     (GCompareFunc)assoc_strict_cmp_2)) ) {
    g_free (((GlobAssoc*)tmp->data)->glob);
    g_free (((GlobAssoc*)tmp->data)->pat);
    globassoc = g_list_remove_link (globassoc, tmp);
    g_list_free_1 (tmp);
  }
}

/* gI_glob_lookup -- finds pattern with glob-key matching string.  If
 * no such pattern is found (string matches non of the globs) NULL is
 * returned.  It will return the *first* pattern found with a glob
 * matching string, so in this way the assoc/unassoc works in a stack
 * like way.  It's not entirely a stack though, since the same pattern
 * can be associted to more than one pattern, unassoc'ing it will
 * remove them all!  It's only a stack in the sense that a glob that
 * matches first (was assoc'ed last) will match first. */
gI_HilitePattern *
gI_glob_lookup (gchar *string)
{
  GList *tmp;

  g_return_val_if_fail (string != NULL, NULL);

  tmp = g_list_find_custom (globassoc, string,
			    (GCompareFunc)assoc_glob_cmp);

  if (tmp) {
    return gI_hilite_get_pattern_by_name (((GlobAssoc*)tmp->data)->pat);
  } else {
    return NULL;
  }
}


#if !HAVE_GUILE
static gchar*
check_string (FILE *file) 
{
  gchar buf[MAXLEN];
  gint c, i;

  while (isspace ( (c = fgetc (file)) ));
  if (c != '"') {
    return NULL;
  }
  
  i = 0;
  while (((c = fgetc(file)) != EOF)) {
    if (c == '\\') {
      switch ( (c = fgetc(file)) ) {
      case 'n': 
	buf[i] = '\n';
	break;
      case '"':
	buf[i] = '"';
	break;
      case EOF:
	return NULL;
	break;
      default:
	buf[i] = c;
      }
    } else if (c == '"') {
      break;
    } else {
      buf[i] = c;
    }
    i++;
    if (i > MAXLEN - 2)
      break;
  }
  buf[i] = 0;
  return g_strdup(buf);
}

static gboolean
checkc (FILE *file, gint c)
{
  gint tmp;
  while (isspace ( ( tmp = fgetc (file)) ));
  if (tmp == c) {
    return TRUE;
  } else {
    return FALSE;
  }
}

void
gI_glob_load_globassoc (const gchar *filename)
{
  FILE *file = fopen (filename, "r");
  gchar *glob, *patname;

  if (!file) {
    g_warning ("Could not open globassoc config file: %s", filename);
    return;
  }

  while ( checkc (file, '(') ) {
    if (! (glob = check_string (file)) ) {
      gI_error_dialog( _("Syntax error in globassoc resource file") );
      break;
    }
    if (! (patname = check_string (file)) ) {
      gI_error_dialog( _("Syntax error in globassoc resource file") );
      g_free (glob);
      break;
    }
    
    gI_glob_assoc (glob, patname);

    g_free (glob);
    g_free (patname);

    if (! (checkc (file, ')')) ) {
      gI_error_dialog( _("Syntax error in globassoc resource file") );
    }
  }

  fclose (file);
}
#endif

#if 0
void
gI_hilite_save_globassoc (const gchar *filename)
{
  g_warning ("not implemented yet!");
  return;
}
#endif

#endif /*HAVE_GTKTEXT_PATCH*/
