/* $Header: /cvs/gnome/gIDE/src/gI_debug_watch.c,v 1.1 2000/04/09 16:37:35 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gide.h"
#include "gI_debug_watch.h"
#include "gI_debug_p.h"
#include "gI_common.h"

/* local structs */
typedef struct _item_data
{
	gchar*								expr;
	gchar*								item;
	glong								ptr;
} item_data;

/* prototypes for local functions */
static void watch_list_select(GtkWidget* widget, gint row, gint column,
	GdkEventKey* bevent, gI_watch_window* ww);
static gint watch_event_handler(GtkWidget* widget, GdkEventButton* event,
	gI_watch_window* ww);
static void watch_set_expression(GtkWidget* widget, entry_data* ed);
static void watch_install_popup(gI_watch_window* ww);
static void watch_eval_expr(GtkWidget* widget, gpointer data);
static void watch_set_expr(GtkWidget* widget, gpointer data);
static void watch_show_evalexpr(gchar* expr, gchar* data);
static void watch_eval_button_handler(GtkWidget* widget, GdkEventButton* event,
	item_data* id);

/* Local functions */
/*
 * Set a new watch expression
 */
static void
watch_set_expression(
	GtkWidget*							widget,
	entry_data*							ed
)
{
	gchar								cmdstr[STRLEN];
	gchar*								ptr;
	gchar*								value;

	ptr = gtk_entry_get_text(GTK_ENTRY(ed->entry));
	value = (gchar*)ed->data;

	g_snprintf(cmdstr, sizeof(cmdstr), "set %s = %s", value, ptr);

	/* destroy entry dialog and free memory */
	gtk_widget_destroy(ed->dialog);
	g_free(ed);

	/* set expression */
	debug_send_dbg_cmd(main_window->debug_window, cmdstr, TRUE);

	/* update expressions window */
	debug_send_dbg_cmd(main_window->debug_window, "info locals", TRUE);
	main_window->debug_window->wait_for = N_WAIT_EXPRESSIONS;
}


/*
 * Row selection callback for watch window
 */
static void
watch_list_select(
	GtkWidget*							widget,
	gint								row,
	gint								column,
	GdkEventKey*						bevent,
	gI_watch_window*					ww
)
{
	gchar*								expr;

	if(!bevent)
	{
		return;
	}

	if(bevent->type == GDK_2BUTTON_PRESS)
	{
		gtk_clist_get_text(GTK_CLIST(ww->list), row, 0, &expr);
		entry_dialog_data(expr, _("Set expression"), watch_set_expression,
			(gpointer)expr);
	}
}


/*
 * Event handler for watch window buttons
 */
static gint
watch_event_handler(
	GtkWidget*							widget,
	GdkEventButton*						event,
	gI_watch_window*					ww
)
{
	if(!event)
	{
		return FALSE;
	}

	if(event->button == 3)
	{
		if(!ww)
		{
			return FALSE;
		}

		gtk_item_factory_popup(GTK_ITEM_FACTORY(ww->popup_menu),
			(guint)event->x_root, (guint)event->y_root, event->button,
			event->time);
		return TRUE;
	}

	return FALSE;
}


/*
 * Popup menu callback to evaluate an expression
 */
static void
watch_eval_expr(
	GtkWidget*							widget,
	gpointer							data
)
{
	gchar*								expr;
	gchar*								type;
	GList*								selection;
	gI_watch_window*					ww;
	gchar								cmdstr[STRLEN];

	if(!main_window->debug_window)
	{
		return;
	}

	ww = main_window->debug_window->watch_window;
	if(!ww)
	{
		return;
	}

	selection = GTK_CLIST(ww->list)->selection;
	if(!selection)
	{
		return;
	}

	gtk_clist_get_text(GTK_CLIST(ww->list), (gint)selection->data, 0, &expr);
	gtk_clist_get_text(GTK_CLIST(ww->list), (gint)selection->data, 1, &type);

	if(!strchr(type, '*'))
	{
		g_snprintf(cmdstr, sizeof(cmdstr), "print (%s)", expr);
	}
	else
	{
		g_snprintf(cmdstr, sizeof(cmdstr), "print *(%s)", expr);
	}

	debug_send_dbg_cmd(main_window->debug_window, cmdstr, TRUE);
	main_window->debug_window->watch_window->data = g_strdup(expr);
	main_window->debug_window->wait_for = N_WAIT_EVALEXPR;
}


/*
 * Popup menu callback to set an expression
 */
static void
watch_set_expr(
	GtkWidget*							widget,
	gpointer							data
)
{
	gchar*								expr;
	GList*								selection;
	gI_watch_window*					ww;

	if(!main_window->debug_window)
	{
		return;
	}

	ww = main_window->debug_window->watch_window;
	if(!ww)
	{
		return;
	}

	selection = GTK_CLIST(ww->list)->selection;
	if(!selection)
	{
		return;
	}

	gtk_clist_get_text(GTK_CLIST(ww->list), (gint)selection->data, 0, &expr);
	entry_dialog_data(expr, _("Set Expression"), watch_set_expression,
		(gpointer)expr);
}


/*
 * Install the popup menu for the watch window
 */
static void
watch_install_popup(
	gI_watch_window*					ww
)
{
	GtkAccelGroup*						group;
	GtkItemFactoryEntry					ww_popup_items[] =
	{
		{ "/Evaluate", NULL, watch_eval_expr, 0 },
		{ "/Set", NULL, watch_set_expr, 0 }
	};

	group = gtk_accel_group_new();

	ww->popup_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Watch>", group);
	gtk_item_factory_create_items(ww->popup_menu,
		sizeof(ww_popup_items) / sizeof(ww_popup_items[0]), ww_popup_items,
		NULL);

	gtk_accel_group_attach(group, GTK_OBJECT(ww->window));
}


static void
watch_eval_button_handler(
	GtkWidget*							widget,
	GdkEventButton*						event,
	item_data*							id
)
{
	gchar								cmdstr[STRLEN];
	gchar*								ptr;
	gchar*								newitem;

	if(!event)
	{
		return;
	}

	if(event->type != GDK_2BUTTON_PRESS)
	{
		return;
	}

	ptr = id->item;
	while(*ptr != ' ' && *ptr != '\0')
	{
		ptr++;
	}
	*ptr = '\0';

	newitem = g_strconcat(id->expr, "->", id->item, NULL);

	if(id->ptr)
	{
		g_snprintf(cmdstr, sizeof(cmdstr), "print *(%s)", newitem);
	}
	else
	{
		g_snprintf(cmdstr, sizeof(cmdstr), "print (%s)", newitem);
	}

	debug_send_dbg_cmd(main_window->debug_window, cmdstr, TRUE);
	main_window->debug_window->watch_window->data = g_strdup(newitem);
	main_window->debug_window->wait_for = N_WAIT_EVALEXPR;

	g_free(newitem);
}


/*
 * Show the evaluation of an expression
 */
static void
watch_show_evalexpr(
	gchar*								expr,
	gchar*								data
)
{
	GtkWidget*							window;
	GtkWidget*							root;
	GtkWidget*							root_item;
	GtkWidget*							item;
	gchar*								start = NULL;
	gchar*								end = NULL;
	gchar*								ptr;
	gchar								subexpr[STRLEN];
	glong								endofexpr = 0;
	gchar*								retptr;
	gchar*								exprptr;
	gchar*								title;
	item_data*							id;
	glong								count = 0;

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	title = g_strconcat("Expression: ", expr, NULL);
	gtk_window_set_title(GTK_WINDOW(window), title);
	g_free(title);
	gtk_widget_set_usize(window, 350, 250);
	gtk_signal_connect_object(GTK_OBJECT(window), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), NULL);

	root = gtk_tree_new();
	gtk_container_add(GTK_CONTAINER(window), root);
	gtk_widget_show(root);

	root_item = gtk_tree_item_new_with_label(expr);
	gtk_tree_append(GTK_TREE(root), root_item);

	root = gtk_tree_new();
	gtk_tree_item_set_subtree(GTK_TREE_ITEM(root_item), root);
	gtk_widget_show(root);

	/* remove all '\n' from data */
	while((retptr = strchr(data, '\n')))
	{
		*retptr = ' ';
	}

	/* parse data and create tree */
	ptr = data;
	start = ptr;

	while(*ptr != '\0')
	{
		switch(*ptr)
		{
		case '"':
			ptr++;
			while(*ptr != '"')
			{
				ptr++;
			}
			ptr++;
			break;

		case ',':
			if(count > 1)
			{
				ptr++;
			}
			else
			{
				/* set retptr & abort here */
				end = ptr - 1;
			}
			break;

		case '{':
			count++;
			ptr++;
			break;

		case '}':
			count--;
			if(count == 1)
			{
				/* toplevel */
				end = ptr;
			}
			if(count == 0)
			{
				/* end */
				endofexpr = 1;
				end = ptr - 1;
			}
			ptr++;
			break;

		default:
			ptr++;
			break;
		}

		if(!start || !end)
		{
			continue;
		}

		if(*start == '{')
		{
			start++;
		}

		strncpy(subexpr, start, end - start + 1);
		subexpr[end - start + 1] = '\0';

		/* strip whitespace */
		exprptr = &subexpr[0];
		while(*exprptr == ' ')
		{
			exprptr++;
		}

		/* warning: memory is not free'd *LEAK* */
		id = (item_data*)g_malloc(sizeof(item_data));
		id->expr = g_strdup(expr);
		id->item = g_strdup(exprptr);
		if(strstr(subexpr, "= 0x") && !strstr(subexpr, "= {"))
		{
			id->ptr = 1;
		}
		else
		{
			id->ptr = 0;
		}

		/* new tree item */
		item = gtk_tree_item_new_with_label(exprptr);
		gtk_tree_append(GTK_TREE(root), item);
		gtk_signal_connect(GTK_OBJECT(item), "button_press_event",
			GTK_SIGNAL_FUNC(watch_eval_button_handler), (gpointer)id);
		gtk_widget_show(item);

		if(endofexpr)
		{
			break;
		}

		ptr += 2;
		start = ptr;
		end = NULL;
	}

	gtk_tree_item_expand(GTK_TREE_ITEM(root_item));
	gtk_widget_show(root_item);
	gtk_widget_show(window);
}


/* Public functions */
/*
 * Create a new watch window
 */
gI_watch_window*
gI_watch_window_new(
	void
)
{
	gI_watch_window*					ww;
	gchar* list_titles[] = { "Expression", "Type", "Value" };

	ww = (gI_watch_window*)g_malloc0(sizeof(gI_watch_window));
	ww->data = NULL;

	ww->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(ww->window), _("Expressions"));
	gtk_widget_set_usize(ww->window, 600, 250);
	gtk_signal_connect(GTK_OBJECT(ww->window), "destroy",
		GTK_SIGNAL_FUNC(gI_watch_window_destroy), (gpointer)ww);

	watch_install_popup(ww);

	ww->list = gtk_clist_new_with_titles(3, list_titles);
	gtk_clist_column_titles_passive(GTK_CLIST(ww->list));
	gtk_clist_set_column_width(GTK_CLIST(ww->list), 0, 100);
	gtk_clist_set_column_width(GTK_CLIST(ww->list), 1, 75);
	gtk_container_add(GTK_CONTAINER(ww->window), ww->list);
	gtk_signal_connect(GTK_OBJECT(ww->list), "select_row",
		GTK_SIGNAL_FUNC(watch_list_select), (gpointer)ww);
	gtk_signal_connect(GTK_OBJECT(ww->list), "button_press_event",
		GTK_SIGNAL_FUNC(watch_event_handler), (gpointer)ww);
	gtk_widget_show(ww->list);

	gtk_widget_show(ww->window);

	return ww;
}


/*
 * Destroy the watch window
 */
void
gI_watch_window_destroy(
	GtkWidget*							widget,
	gI_watch_window*					ww
)
{
	g_free(ww);
	ww = NULL;

	if(main_window->debug_window)
	{
		main_window->debug_window->watch_window = NULL;
	}
}


/*
 * Process the debugger output related to the watches
 */
void
gI_watch_process_output(
	gI_debug_window*					dw,
	gchar*								msg
)
{
	gchar*								retptr;
	gchar*								error;
	gchar*								ptr;

	if(!strncmp(msg, "Cannot access memory at", 23) ||
		!strncmp(msg, "Attempt to dereference a generic pointer", 40) ||
		!strncmp(msg, "No symbol \"", 11))
	{
		retptr = strchr(msg, '\n');
		if(!retptr)
		{
			/* this should not happen! */
		}
		*retptr = '\0';

		error = g_strconcat("\n    ", msg, "    \n", NULL);
		gI_error_dialog(error);
		g_free(error);

		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	ptr = strchr(msg, '=');
	if(!ptr)
	{
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}
	ptr += 2;

	retptr = strstr(msg, GDB_PROMPT);
	if(!retptr)
	{
		/* we have a problem! */
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}
	*retptr = '\0';

	if(dw->watch_window->data != NULL)
	{
		watch_show_evalexpr(dw->watch_window->data, ptr);

		g_free(dw->watch_window->data);
	}
	else
	{
		g_warning("problems with data in watch-window!\n");
	}

	dw->wait_for = N_WAIT_NOTHING;
}

