/*
 * dialog-search-replace.c: Dialog used to search/replace in gIDE
 * spreadsheet
 *
 * Authors:
 *   JP Rosevear (jpr@arcavia.com)
 */
#include <config.h>
#include "gide.h"
#include <glade/glade.h>
#include "dialogs.h"
#include "gI_plugin.h"
#include "gI_common.h"
#include "gI_search.h"

#define GLADE_FILE "search-replace.glade"

typedef struct 
{
	GideWindow *window;
	GideDocument *document;
	GtkWidget *dialog;
	GtkWidget *search_entry;
	GtkWidget *replace_entry;
	GtkWidget *back;
	GtkWidget *re;
	GtkWidget *cs;
} SearchReplace;

enum { BUTTON_SEARCH = 0, BUTTON_REPLACE, BUTTON_REPLACE_ALL, BUTTON_CLOSE };

static void
search_state_extract( SearchReplace *sr, SearchReplaceState *state,
		      gboolean replace )
{
	/* The search variables */
	if (state->search)
		g_free( state->search );
	state->search = gtk_editable_get_chars(GTK_EDITABLE(sr->search_entry),
					       0, -1);

	state->back =
		gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sr->back));
	state->re = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sr->re));
	state->cs = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sr->cs));

	/* The replace variables */
	if (replace) {
		if ( state->replace )
			g_free( state->replace );	
		state->replace =
			gtk_editable_get_chars(GTK_EDITABLE(sr->replace_entry),
					       0, -1);
	} else {
		if ( state->replace )
			g_free( state->replace );	
		state->replace = NULL;
	}
}

static void
search_not_found( SearchReplace *sr, SearchReplaceState *state )
{
	gchar *error;
	
	error = g_strdup_printf( _("[%s] not found!\n"), state->search );
	gI_error_dialog( error );
	g_free( error );
}

static void
search_cb (SearchReplace *sr, SearchReplaceState *state)
{
	search_state_extract( sr, state, FALSE );

	state->start = gI_text_get_point( sr->document );
	
	state->found = gI_document_search( sr->document, state );

	if ( state->found == -1 )
		search_not_found( sr, state );
}

static void
replace_cb (SearchReplace *sr, SearchReplaceState *state)
{
	search_state_extract( sr, state, TRUE );

	state->start = gI_text_get_point( sr->document );

	state->found = gI_document_search( sr->document, state );
	
	if (state->found >= 0) {
		gI_text_freeze( sr->document );
		gI_text_set_point( sr->document, state->found );

		gI_document_delete_selection( sr->document );
		gI_document_insert_text_at_point(sr->document, state->replace);

		gI_text_set_point( sr->document,
				   state->found + strlen( state->replace ) );
		gI_text_thaw( sr->document );
	} else {
		search_not_found( sr, state );
	}
}

static void
replace_all_cb (SearchReplace *sr, SearchReplaceState *state)
{
	gint replace = 0;
	
	search_state_extract( sr, state, TRUE );

	state->start = gI_text_get_point( sr->document );

	state->found = 0;
	while (state->found != -1) {
		state->found = gI_document_search( sr->document, state );

		if (state->found >= 0) {
			gI_text_set_point( sr->document, state->found );
			
			gI_document_delete_selection( sr->document );
			gI_document_insert_text_at_point( sr->document,
							  state->replace );

			gI_text_set_point( sr->document,
					   state->found +
					   strlen( state->replace ));

			replace++;		
		}
	}
	
	if ( replace > 0 ) {
		gchar *error;
		error = g_strdup_printf( _("%d occurrences replaced."),
					 replace);
		gI_error_dialog( error );
		g_free( error) ;
	} else {
		search_not_found( sr, state );
	}
}

static gint
sr_key_event (GtkWidget *sr, GdkEventKey *event)
{
	if (event->keyval == GDK_Escape) {
		gtk_widget_destroy (sr);
		return 1;
	}
	return 0;
}

static void
dialog_search_replace_impl (GideWindow *window, GladeXML *gui,
			    SearchReplaceState *state)
{
	SearchReplace *sr = g_new0 (SearchReplace, 1);
	gint bval = BUTTON_SEARCH;

	g_return_if_fail (sr);
	g_return_if_fail (window);
	
	sr->window = window;
	sr->document = gI_window_get_current_doc (window);
	sr->dialog = glade_xml_get_widget (gui, "dialog");
	sr->search_entry = glade_xml_get_widget (gui, "search_entry");
	sr->replace_entry = glade_xml_get_widget (gui, "replace_entry");
	sr->back = glade_xml_get_widget (gui, "back");
	sr->re = glade_xml_get_widget (gui, "re");
	sr->cs = glade_xml_get_widget (gui, "cs");

	/* Set defaults */
	if ( state->search )
		gtk_entry_set_text( GTK_ENTRY(sr->search_entry),
		state->search );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sr->back),
				      state->back );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sr->re), state->re );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sr->cs), state->cs );
	
	gnome_dialog_set_default (GNOME_DIALOG (sr->dialog), BUTTON_CLOSE);
	gtk_window_set_policy (GTK_WINDOW (sr->dialog), FALSE, TRUE, FALSE);

	gtk_widget_show_all (GNOME_DIALOG (sr->dialog)->vbox);

	bval = gnome_dialog_run (GNOME_DIALOG (sr->dialog));
	switch (bval) {
		
	case BUTTON_SEARCH:
		search_cb (sr, state);
		break;
		
	case BUTTON_REPLACE:
		replace_cb (sr, state);
		break;
		
	case BUTTON_REPLACE_ALL:
		replace_all_cb (sr, state);
		break;
		
	case BUTTON_CLOSE:
		break;
		
	case -1: /* close window */
		return;
		
	default: /* should never happen */
		break;
	}

	/* If the user canceled we have already returned */
	gnome_dialog_close (GNOME_DIALOG (sr->dialog));
}

/*
 * Wrapper around dialog_search_replace_impl
 * To libglade'ify it
 */
void
dialog_search_replace( GideWindow *window, SearchReplaceState *state )
{
	GladeXML *gui;

	g_return_if_fail (window != NULL);

	gui = glade_xml_new (GIDE_GLADEDIR "/" GLADE_FILE , NULL);
	if (!gui) {
		printf ("Could not find " GLADE_FILE "\n");
		return;
	}

	dialog_search_replace_impl( window, gui, state );
	gtk_object_unref (GTK_OBJECT (gui));
}












