/* vim: ts=4 sw=4 colorcolumn=80                                                
 * -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- *
 */
/* hex-document-mmap.c - `mmap` implementation of the HexBuffer iface.
 *
 * Based on code from aoeui, Copyright © 2007, 2008 Peter Klausler,
 * licensed by the author/copyright-holder under GPLv2 only.
 *
 * Source as adapted herein licensed for GHex to GPLv2+ with written
 * permission from Peter Klausler dated December 13, 2021 (see
 * associated git log).
 *
 * Copyright © 2021 Logan Rathbone
 */

#include "hex-buffer-mmap.h"

#define HEX_BUFFER_MMAP_ERROR hex_buffer_mmap_error_quark ()
GQuark
hex_buffer_mmap_error_quark (void)
{
  return g_quark_from_static_string ("hex-buffer-mmap-error-quark");
}

static char *invalid_path_msg = N_("The file appears to have an invalid path.");

struct _HexBufferMmap
{
	GObject parent_instance;

	GFile *file;
	GError *error;		/* no custom codes; use codes from errno */
	int last_errno;		/* cache in case we need to re-report errno error. */

	char *data;			/* buffer for modification and info */
	size_t payload;
	size_t mapped;
	size_t gap;
	char *tmpfile_path;	/* path to buffer tmpfile in mkstemp format */
	int fd;				/* file descriptor of tmpfile. */

	char *clean;		/* unmodified content, mmap'ed */
	size_t clean_bytes;
	int clean_fd;

	size_t pagesize;	/* is only fetched once and cached. */
};

static void hex_buffer_mmap_iface_init (HexBufferInterface *iface);

G_DEFINE_TYPE_WITH_CODE (HexBufferMmap, hex_buffer_mmap, G_TYPE_OBJECT,
		G_IMPLEMENT_INTERFACE (HEX_TYPE_BUFFER, hex_buffer_mmap_iface_init))


/* PRIVATE FUNCTIONS */

/* Helper wrapper for g_set_error and to cache errno */
static void
set_error (HexBufferMmap *self, const char *blurb)
{
	char *message = NULL;

	if (errno) {
	/* Translators:  the first '%s' is the blurb indicating some kind of an
	 * error has occurred (eg, 'An error has occurred', and the the 2nd '%s'
	 * is the standard error message that will be reported from the system
	 * (eg, 'No such file or directory').
	 */
		message = g_strdup_printf (_("%s: %s"), blurb, g_strerror (errno));
		g_debug ("%s: %s", __func__, message);
	}

	g_set_error (&self->error,
			HEX_BUFFER_MMAP_ERROR,
			errno,
			message ? message : blurb,
			g_strerror (errno));

	if (errno)
		self->last_errno = errno;

	g_free (message);
}

static inline
size_t buffer_gap_bytes (HexBufferMmap *self)
{
	return self->mapped - self->payload;
}

/* CONSTRUCTORS AND DESTRUCTORS */

static void
hex_buffer_mmap_init (HexBufferMmap *self)
{
	self->pagesize = getpagesize ();
	self->fd = -1;
}

static void
hex_buffer_mmap_dispose (GObject *gobject)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (gobject);

	/* chain up */
	G_OBJECT_CLASS(hex_buffer_mmap_parent_class)->dispose (gobject);
}

static void
hex_buffer_mmap_finalize (GObject *gobject)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (gobject);

	munmap (self->data, self->mapped);
	munmap (self->clean, self->clean_bytes);

	if (self->fd >= 0)
	{
		close (self->fd);
		unlink (self->tmpfile_path);
	}

	g_free (self->tmpfile_path);

	/* chain up */
	G_OBJECT_CLASS(hex_buffer_mmap_parent_class)->finalize (gobject);
}

static void
hex_buffer_mmap_class_init (HexBufferMmapClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
	
	gobject_class->finalize = hex_buffer_mmap_finalize;
	gobject_class->dispose = hex_buffer_mmap_dispose;
}

static void
hex_buffer_mmap_place_gap (HexBufferMmap *self, size_t offset)
{
	g_return_if_fail (HEX_IS_BUFFER_MMAP (self));

	size_t gapsize = buffer_gap_bytes (self);

	if (offset > self->payload)
		offset = self->payload;

	if (offset <= self->gap)
		memmove (self->data + offset + gapsize,
				self->data + offset,
				self->gap - offset);
	else
		memmove (self->data + self->gap,
			self->data + self->gap + gapsize,
			offset - self->gap);

	self->gap = offset;

	if (self->fd >= 0 && gapsize)
		memset (self->data + self->gap, ' ', gapsize);
}

static void
hex_buffer_mmap_resize (HexBufferMmap *self, size_t payload_bytes)
{
	void *p;
	char *old = self->data;
	int fd;
	int mapflags = 0;
	size_t map_bytes = payload_bytes;

	g_return_if_fail (HEX_IS_BUFFER_MMAP (self));

	/* Whole pages, with extras as size increases */
	map_bytes += self->pagesize - 1;
	map_bytes /= self->pagesize;
	map_bytes *= 11;
	map_bytes /= 10;
	map_bytes *= self->pagesize;

	if (map_bytes < self->mapped)
		munmap (old + map_bytes, self->mapped - map_bytes);

	if (self->fd >= 0  &&  map_bytes != self->mapped)
	{
		errno = 0;
		if (ftruncate (self->fd, map_bytes))
		{
			char *errmsg = g_strdup_printf (
					_("Could not adjust %s from %lu to %lu bytes"),
					self->tmpfile_path, (long)self->mapped, (long)map_bytes);

			set_error (self, errmsg);
			g_free (errmsg);
			return;
		}
	}

	if (map_bytes <= self->mapped)
	{
		self->mapped = map_bytes;
		return;
	}

	if (old)
	{
		/* attempt extension */
		errno = 0;
		p = mremap (old, self->mapped, map_bytes, MREMAP_MAYMOVE);
		if (p != MAP_FAILED)
			goto done;
	}

	/* new/replacement allocation */

	if ((fd = self->fd) >= 0)
	{
		mapflags |= MAP_SHARED;
		if (old) {
			munmap(old, self->mapped);
			old = NULL;
		}
	}
	else
	{
#ifdef MAP_ANONYMOUS
		mapflags |= MAP_ANONYMOUS;
#else
		mapflags |= MAP_ANON;
#endif
		mapflags |= MAP_PRIVATE;
	}

	errno = 0;
	p = mmap (0, map_bytes, PROT_READ|PROT_WRITE, mapflags, fd, 0);
	if (p == MAP_FAILED)
	{
		char *errmsg = g_strdup_printf (
			_("Fatal error: Memory mapping of file (%lu bytes, fd %d) failed"),
				(long)map_bytes, fd);

		set_error (self, errmsg);
		g_free (errmsg);
		return;
	}

	if (old)
	{
		memcpy(p, old, self->payload);
		munmap(old, self->mapped);
	}

done:
	self->data = p;
	self->mapped = map_bytes;
}

#define ADJUST_OFFSET_AND_BYTES				\
	if (offset >= self->payload)			\
		offset = self->payload;				\
	if (offset + bytes > self->payload)		\
		bytes = self->payload - offset;		\

size_t
hex_buffer_mmap_raw (HexBufferMmap *self,
		char **out, size_t offset, size_t bytes)
{
	g_assert (HEX_IS_BUFFER_MMAP (self));
	
	ADJUST_OFFSET_AND_BYTES

	if (!bytes) {
		*out = NULL;
		return 0;
	}

	if (offset < self->gap  &&  offset + bytes > self->gap)
		hex_buffer_mmap_place_gap (self, offset + bytes);

	*out = self->data + offset;
	if (offset >= self->gap)
		*out += buffer_gap_bytes (self);

	return bytes;
}

size_t
hex_buffer_mmap_copy_data (HexBufferMmap *self,
		void *out, size_t offset, size_t bytes)
{
	size_t left;

	g_assert (HEX_IS_BUFFER_MMAP (self));

	ADJUST_OFFSET_AND_BYTES

	left = bytes;
	if (offset < self->gap)
	{
		unsigned int before = self->gap - offset;

		if (before > bytes)
			before = bytes;

		memcpy (out, self->data + offset, before);

		out = (char *)out + before;
		offset += before;
		left -= before;

		if (!left)
			return bytes;
	}
	offset += buffer_gap_bytes (self);

	memcpy (out, self->data + offset, left);

	return bytes;
}

size_t
hex_buffer_mmap_delete (HexBufferMmap *self,
		     size_t offset, size_t bytes)
{
	g_assert (HEX_IS_BUFFER_MMAP (self));

	ADJUST_OFFSET_AND_BYTES

	hex_buffer_mmap_place_gap (self, offset);
	self->payload -= bytes;

	return bytes;
}
#undef ADJUST_OFFSET_AND_BYTES

static size_t
hex_buffer_mmap_insert (HexBufferMmap *self,
		const void *in, size_t offset, size_t bytes)
{
	g_assert (HEX_IS_BUFFER_MMAP (self));

	if (offset > self->payload)
		offset = self->payload;

	if (bytes > buffer_gap_bytes (self)) {
		hex_buffer_mmap_place_gap (self, self->payload);
		hex_buffer_mmap_resize (self, self->payload + bytes);
	}

	hex_buffer_mmap_place_gap (self, offset);

	if (in)
		memcpy (self->data + offset, in, bytes);
	else
		memset (self->data + offset, 0, bytes);

	self->gap += bytes;
	self->payload += bytes;

	return bytes;
}

size_t
hex_buffer_mmap_move (HexBufferMmap *to,
		size_t to_offset,
		HexBufferMmap *from,
		size_t from_offset,
		size_t bytes)
{
	char *raw = NULL;

	bytes = hex_buffer_mmap_raw (from, &raw, from_offset, bytes);
	hex_buffer_mmap_insert (to, raw, to_offset, bytes);

	return hex_buffer_mmap_delete (from, from_offset, bytes);
}

void 
hex_buffer_mmap_snap (HexBufferMmap *self)
{
	g_return_if_fail (HEX_IS_BUFFER_MMAP (self));

	if (self->fd >= 0)
	{
		hex_buffer_mmap_place_gap (self, self->payload);
		if (ftruncate (self->fd, self->payload)) {
			/* don't care */
		}
	}
}

char * hex_buffer_mmap_get_data (HexBuffer *buf,
		size_t offset,
		size_t len)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (buf);
	char *data;

	data = g_malloc (len);
	hex_buffer_mmap_copy_data (self, data, offset, len);

	return data;
}

char hex_buffer_mmap_get_byte (HexBuffer *buf,
		size_t offset)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (buf);
	char *cp;
	char c;

	cp = hex_buffer_mmap_get_data (buf, offset, 1);
	c = *cp;
	
	return c;
}

static size_t
hex_buffer_mmap_get_payload_size (HexBuffer *buf)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (buf);

	return self->payload;
}

static gboolean
hex_buffer_mmap_set_file (HexBuffer *buf, GFile *file)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (buf);
	const char *file_path;

	g_return_val_if_fail (G_IS_FILE (file), FALSE);

	file_path = g_file_peek_path (file);
	if (! file_path)
	{
		set_error (self, _(invalid_path_msg));
		return FALSE;
	}
	self->file = file;

	return TRUE;
}

/* helper */
static int
create_fd_from_path (HexBufferMmap *self, const char *path)
{
	int fd = -1;
	struct stat statbuf;

	errno = 0;

	if (stat (path, &statbuf))
	{
		if (errno != ENOENT) {
			set_error (self,
				_("Unable to retrieve file or directory information"));
			return -1;
		}

		errno = 0;
		fd = open(path, O_CREAT|O_TRUNC|O_RDWR,
				S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);

		if (fd < 0) {
			set_error (self, _("Unable to create file"));
			return -1;
		}
	} 
	else
	{
		/* FIXME - this is probably overkill - hex editor users may wish to
		 * open a 'non-regular' file.
		 */
		if (!S_ISREG(statbuf.st_mode)) {
			set_error (self, _("Not a regular file"));
			return -1;
		}

		fd = open (path, O_RDWR);

		if (fd < 0) {
			errno = 0;
			fd = open (path, O_RDONLY);
			if (fd < 0) {
				set_error (self, _("Unable to open file for reading"));
				return -1;
			}
		}
	}
	return fd;
}

static gboolean
create_buffer (HexBufferMmap *self)
{
	self->tmpfile_path = g_strdup ("hexmmapbufXXXXXX");
	errno = 0;
	self->fd = mkstemp (self->tmpfile_path);

	if (self->fd < 0) {
		set_error (self, _("Failed to open temporary file."));
		return FALSE;
	}

	return TRUE;
}

static gboolean
hex_buffer_mmap_read (HexBuffer *buf)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (buf);
	void *p;
	size_t bytes = 0;
	size_t pages;
	const char *file_path;
	int tmp_clean_fd;

	g_return_val_if_fail (G_IS_FILE (self->file), FALSE);

	file_path = g_file_peek_path (self->file);
	if (! file_path)
	{
		set_error (self, _(invalid_path_msg));
		return FALSE;
	}

	bytes = hex_buffer_util_get_file_size (self->file);
	pages = (bytes + self->pagesize - 1) / self->pagesize;

	/* Set up a clean buffer (read-only memory mapped version of O.G. file)
	 */
	if (self->clean)
		munmap (self->clean, self->clean_bytes);

	self->clean_bytes = bytes;
	self->clean = NULL;

	if (!pages)
		return FALSE;

	tmp_clean_fd = create_fd_from_path (self, file_path);
	if (tmp_clean_fd < 0)
		return FALSE;

	self->clean_fd = tmp_clean_fd;

	p = mmap (0, pages * self->pagesize, PROT_READ, MAP_SHARED,
			self->clean_fd, 0);

	if (p == MAP_FAILED)
		return FALSE;

	self->clean = p;

	/* Create dirty buffer for writing etc. */
	create_buffer (self);

	/* FIXME/TODO - sanity check against # of bytes read? */
	hex_buffer_mmap_insert (self, self->clean, 0, self->clean_bytes);

	return TRUE;
}

static gboolean hex_buffer_mmap_set_data (HexBuffer *buf,
		size_t offset,
		size_t len,
		size_t rep_len,
		char *data)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (buf);

	if (offset > self->payload)
	{
		g_debug ("%s: offset greater than payload size; returning.", __func__);
		return FALSE;
	}

	hex_buffer_mmap_insert (self, data, offset, len);
	hex_buffer_mmap_delete (self, offset + len, rep_len);

	return TRUE;
}

static gboolean hex_buffer_mmap_write_to_file (HexBuffer *buf,
		GFile *file)
{
	HexBufferMmap *self = HEX_BUFFER_MMAP (buf);
	char *raw;
	gboolean retval;

	g_return_val_if_fail (G_IS_FILE (self->file), FALSE);

	hex_buffer_mmap_raw (self, &raw, 0, self->payload);

	retval = g_file_replace_contents (self->file,
		/* const char* contents, */			raw,	
		/* gsize length, */					self->payload,
		/* const char* etag, */				NULL,
		/* gboolean make_backup, */			FALSE,	/* FIXME - make optional? */
		/* GFileCreateFlags flags, */		G_FILE_CREATE_NONE,
		/* char** new_etag, */				NULL,
		/* GCancellable* cancellable, */	NULL,	/* FIXME */
		/* GError** error */				&self->error);

	return retval;
}

/* PUBLIC FUNCTIONS */

HexBufferMmap *
hex_buffer_mmap_new (GFile *file)
{
	HexBufferMmap *self = g_object_new (HEX_TYPE_BUFFER_MMAP, NULL);

	if (file)
	{
		/* If a path is provided but it can't be set, nullify the object */
		if (! hex_buffer_mmap_set_file (HEX_BUFFER(self), file))
			g_clear_object (&self);
	}

	return self;
}

/* INTERFACE IMPLEMENTATION FUNCTIONS */

static void
hex_buffer_mmap_iface_init (HexBufferInterface *iface)
{
	iface->get_data = hex_buffer_mmap_get_data;
	iface->get_byte = hex_buffer_mmap_get_byte;
	iface->set_data = hex_buffer_mmap_set_data;
	iface->set_file = hex_buffer_mmap_set_file;
	iface->read = hex_buffer_mmap_read;
	iface->write_to_file = hex_buffer_mmap_write_to_file;
	iface->get_payload_size = hex_buffer_mmap_get_payload_size;
}
