/* main.c
 *
 * Copyright (C) 1999 Red Hat, Inc.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include "gfloppy.h"
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>

extern int errno;

GFloppy floppy;

static gint
option_menu_get_history (GtkOptionMenu *option_menu)
{
	GtkWidget *active_widget;

	g_return_val_if_fail (GTK_IS_OPTION_MENU (option_menu), -1);

	active_widget = gtk_menu_get_active (GTK_MENU (option_menu->menu));

	if (active_widget)
		return g_list_index (GTK_MENU_SHELL (option_menu->menu)->children,
				     active_widget);
	else
		return -1;
}

static void
signal_handler (int parm)
{
	gint pid;
	gint status = 0;
	gchar stat_char[3];
	
	pid = wait (&status);
	snprintf (stat_char, 2, "%d", status);
	
	if (pid <= 0)
		g_print ("Warning:  %s in sigchld.\n", strerror (errno));

	g_print ("exited with status %d\n", status);
	if (pid == floppy.pid)
		write (floppy.filedes[1], stat_char, strlen (stat_char));
}

static void
read_quit (gpointer  	     data,
	   gint		     source,
	   GdkInputCondition condition)
{
	gtk_main_quit ();
}

static void
start_format ()
{
	pipe (floppy.filedes);
	gtk_input_add_full (floppy.filedes[0], GDK_INPUT_READ, read_quit, NULL, NULL, NULL);
	signal (SIGCHLD, signal_handler);

	floppy.pid = fork ();
	if (floppy.pid < 0) {
		g_error ("unable to fork ().\nPlease free up some resources and try again.\n");
		_exit (1);
	}
	if (floppy.pid == 0) {
		/* child */
		signal (SIGCHLD, SIG_DFL);
		format_floppy (&floppy);
	}
	/* parent */
	gtk_main ();
}

static void
init_commands ()
{
	floppy.fdformat_cmd = NULL;
	if (g_file_test ("/usr/bin/fdformat", G_FILE_TEST_ISFILE))
		floppy.fdformat_cmd = g_strdup ("/usr/bin/fdformat");
	else if (g_file_test ("/bin/fdformat", G_FILE_TEST_ISFILE))
		floppy.fdformat_cmd = g_strdup ("/bin/fdformat");

	floppy.mke2fs_cmd = NULL;
	if (g_file_test ("/sbin/mke2fs", G_FILE_TEST_ISFILE))
		floppy.mke2fs_cmd = g_strdup ("/sbin/mke2fs");
	else if (g_file_test ("/usr/sbin/mke2fs", G_FILE_TEST_ISFILE))
		floppy.mke2fs_cmd = g_strdup ("/sbin/mke2fs");

	floppy.mformat_cmd = NULL;
	if (g_file_test ("/usr/bin/mformat", G_FILE_TEST_ISFILE))
		floppy.mformat_cmd = g_strdup ("/usr/bin/mformat");
	else if (g_file_test ("/bin/mformat", G_FILE_TEST_ISFILE))
		floppy.mformat_cmd = g_strdup ("/bin/mformat");

	floppy.badblocks_cmd = NULL;
	if (g_file_test ("/usr/bin/mbadblocks", G_FILE_TEST_ISFILE))
		floppy.badblocks_cmd = g_strdup ("/usr/bin/mbadblocks");
	else if (g_file_test ("/bin/mbadblocks", G_FILE_TEST_ISFILE))
		floppy.badblocks_cmd = g_strdup ("/bin/mbadblocks");

	if (floppy.mke2fs_cmd == NULL) {
		g_print ("Warning:  Unable to locate mke2fs.  Please confirm it is installed and try again\n");
		exit (1);
	}

}

static void
init_devices ()
{
	struct stat s;

	/* sanity check */
	if (floppy.device == NULL || *(floppy.device) == '\000') {
		g_print ("Usage:  gfloppy --device=DEVICE\n");
		exit (1);
	}
	stat (floppy.device, &s);
	if (!S_ISBLK(s.st_mode)) {
		g_print ("Waning:  %s is not a proper block device\n", floppy.device);
		exit (1);
	}
	if (strcmp (floppy.device, "/dev/fd0") == 0 && strcmp (floppy.device, "/dev/fd1") == 0) {
		g_print ("Waning:  The device,  %s, is not recognized\n", floppy.device);
		exit (1);
	}

	if (strncmp (floppy.device, "/dev/fd1", strlen ("/dev/fd1")) == 0)
		floppy.mdevice = g_strdup ("b:");
	else
		floppy.mdevice = g_strdup ("a:");
}

static void
set_floppy_extended_device ()
{
	switch (floppy.type) {
	case 0:
		floppy.extended_device = g_strdup_printf ("%sH1440", floppy.device);
		break;
	case 1:
		floppy.extended_device = g_strdup_printf ("%sh1200", floppy.device);
		break;
	case 2:
		floppy.extended_device = g_strdup_printf ("%sD720", floppy.device);
		break;
	case 3:
		floppy.extended_device = g_strdup_printf ("%sd360", floppy.device);
		break;
	default:
		g_assert_not_reached ();
	}
}

int
main (int argc, char *argv[])
{
	GtkWidget *dialog1;
	GtkWidget *ext2_entry;
	GtkWidget *type_option;
	GtkWidget *icon_frame;
	GtkWidget *icon;
	GtkWidget *device_label;
	GtkWidget *quick_format_button;
	gchar *device_string;
	GladeXML *xml;

	struct poptOption gfloppy_opts[] = {
		{"device", '\0', POPT_ARG_STRING, NULL, 0, NULL, NULL},
		{NULL, '\0', 0, NULL, 0, NULL, NULL}
	};

	gfloppy_opts[0].arg = &(floppy.device);
	gfloppy_opts[0].descrip = N_("The device to format");
	gfloppy_opts[0].argDescrip = N_("DEVICE");

	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);

	gnome_init_with_popt_table ("gfloppy", VERSION, argc, argv,
				    gfloppy_opts, 0, NULL);
	init_commands ();
	init_devices ();


	/* Now we can set up glade */
	glade_gnome_init();
        xml = glade_xml_new (GLADEDIR "/gfloppy.glade", NULL);
	dialog1 = glade_xml_get_widget (xml, "dialog1");
	quick_format_button = glade_xml_get_widget (xml, "quick_format_button");
	icon_frame = glade_xml_get_widget (xml, "icon_frame");
	type_option = glade_xml_get_widget (xml, "type_option");

	icon = gnome_pixmap_new_from_file ("/usr/share/pixmaps/mc/i-floppy.png");
	gtk_container_add (GTK_CONTAINER (icon_frame), icon);
	gtk_widget_show (icon);
	/* We do this to get around a bug in libglade.  Ideally we won't need
	 * to do this in the future. */
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog1), _("Format"), GNOME_STOCK_PIXMAP_SAVE);
	gnome_dialog_append_button (GNOME_DIALOG (dialog1), GNOME_STOCK_BUTTON_CLOSE);

	/* initialize our widgets */
	device_string = g_strdup_printf ("Formatting %s", floppy.device);
	device_label = glade_xml_get_widget (xml, "device_label");
	gtk_label_set_text (GTK_LABEL (device_label), device_string);
	g_free (device_string);

	if (floppy.fdformat_cmd == NULL) {
		/* We don't have fdformat.  We can't do a low-level format anymore. */
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (quick_format_button), TRUE);
		gtk_widget_set_sensitive (quick_format_button, FALSE);
	}

	if (floppy.mformat_cmd == NULL) {
		/* We don't have mtools.  Allow ext2 only. */
		ext2_entry = glade_xml_get_widget (xml, "ext2_entry");

		gtk_widget_hide (type_option);
		gtk_widget_show (ext2_entry);
	}

	if (gnome_dialog_run (GNOME_DIALOG (dialog1)) == 0) {
		GtkWidget *density_option;

		density_option = glade_xml_get_widget (xml, "density_option");
		g_assert (density_option != NULL);
		g_assert (quick_format_button != NULL);

		gtk_widget_set_sensitive (dialog1, FALSE);
		if (floppy.mformat_cmd) {
			/* Check to see which one is selected. */
			floppy.type = option_menu_get_history (GTK_OPTION_MENU (type_option));
		} else {
			floppy.type = GFLOPPY_E2FS;
		}
		floppy.size = option_menu_get_history (GTK_OPTION_MENU (density_option));
		set_floppy_extended_device ();
		floppy.quick_format = GTK_TOGGLE_BUTTON (quick_format_button)->active;

		start_format ();
	}
	return 0;
}

