/* GDM - The Gnome Display Manager
 * Copyright (C) 1998, 1999, 2000 Martin K. Petersen <mkp@mkp.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <gnome.h>
#include <syslog.h>

#include "gdm.h"
#include "misc.h"

static const gchar RCSid[]="$Id: misc.c,v 1.13 2001/07/06 08:53:56 jirka Exp $";


/* Configuration option variables */
extern gchar *GdmPidFile;
extern gboolean GdmDebug;

extern char **environ;


/**
 * gdm_fail:
 * @format: printf style format string
 * @...: Optional arguments
 *
 * Logs fatal error condition and aborts master daemon.  Also sleeps
 * for 30 seconds to avoid looping if gdm is started by init.  
 */

void 
gdm_fail (const gchar *format, ...)
{
    va_list args;
    gchar *s;

    va_start (args, format);
    s = g_strdup_vprintf (format, args);
    va_end (args);

    /* Log to both syslog and stderr */
    syslog (LOG_CRIT, s);
    fprintf (stderr, "%s\n", s);
    fflush (stderr);

    g_free (s);

    unlink (GdmPidFile);
    closelog ();

    /* Slow down respawning if we're started from init */
    if (getppid() == 1)
	sleep (30);

    exit (EXIT_FAILURE);
}


/**
 * gdm_info:
 * @format: printf style format string
 * @...: Optional arguments
 *
 * Log non-fatal information to syslog
 */

void 
gdm_info (const gchar *format, ...)
{
    va_list args;
    gchar *s;

    va_start (args, format);
    s = g_strdup_vprintf (format, args);
    va_end (args);
    
    syslog (LOG_INFO, s);
    
    g_free (s);
}


/**
 * gdm_error:
 * @format: printf style format string
 * @...: Optional arguments
 *
 * Log non-fatal error condition to syslog
 */

void 
gdm_error (const gchar *format, ...)
{
    va_list args;
    gchar *s;

    va_start (args, format);
    s = g_strdup_vprintf (format, args);
    va_end (args);
    
    syslog (LOG_ERR, s);
    
    g_free (s);
}


/**
 * gdm_debug:
 * @format: printf style format string
 * @...: Optional arguments
 *
 * Log debug information to syslog if debugging is enabled.
 */

void 
gdm_debug (const gchar *format, ...)
{
    va_list args;
    gchar *s;

    if (/*0 && */! GdmDebug) 
	return;

    va_start (args, format);
    s = g_strdup_vprintf (format, args);
    va_end (args);

    /* UGLY DEBUGGING HACK! */
    
    /*{ FILE *fp = fopen ("/tmp/foo.gdm", "a"); fprintf (fp, "%s\n", s); fflush (fp); fclose (fp); };*/
    
    syslog (LOG_ERR, s);	/* FIXME: LOG_DEBUG */
    
    g_free (s);
}


/**
 * gdm_setenv:
 * @var: Variable to set/unset
 * @value: Value to assign to the variable, NULL for unsetting
 *
 * Wrapper around putenv() because POSIX/SUS is incredibly stupid.
 * 
 *   char *foo; char *bar;
 *   putenv (foo); putenv (bar); free (foo); 
 *
 * is legal, while
 *
 *   char *foo;
 *   putenv (foo); free (foo); 
 *
 * isn't.
 *
 * Afterall, providing the programmer with a nice, consistent
 * interface is what the standard C Library is all about. - Duh!
 *
 * Note from George:
 * You cannot free the last env as it could have been something else
 * and could still be in the env!  We just have to leak, there is no
 * recourse.
 * -George
 */

#ifndef HAVE_SETENV
gint 
gdm_setenv (const gchar *var, const gchar *value)
{
#if 0
    static gchar *lastenv = NULL; /* Holds last successful assignment pointer */
#endif
    gchar *envstr;		  /* Temporary environment string */
    gint result;		  /* Return value from the putenv() call */

    /* `var' is a prerequisite */
    if (!var)
	return -1;

    /* `value' is a prerequisite */
    if (!value)
	return -1;

    envstr = g_strconcat (var, "=", value, NULL);

    /* If string space allocation failed then abort */
    if (!envstr)
	return -1;

    /* Stuff the resulting string into the environment */
    result = putenv (envstr);

#if 0
    /* If putenv() succeeded and lastenv is set, free the old pointer */
    if (result == 0 && lastenv)
	g_free (lastenv);

    /* Save the current string pointer for the next gdm_setenv call */
    lastenv = envstr;
#endif
    
    return result;
}
#endif

#ifndef HAVE_UNSETENV
gint 
gdm_unsetenv (const gchar *var)
{
#if 0
    static gchar *lastenv = NULL; /* Holds last successful assignment pointer */
#endif
    gchar *envstr;		  /* Temporary environment string */
    gint result;		  /* Return value from the putenv() call */

    /* `var' is a prerequisite */
    if (!var)
	return -1;

    envstr = g_strdup (var);

    /* If string space allocation failed then abort */
    if (!envstr)
	return -1;

    /* Stuff the resulting string into the environment */
    result = putenv (envstr);

#if 0
    /* If putenv() succeeded and lastenv is set, free the old pointer */
    if (result == 0 && lastenv)
	g_free (lastenv);

    /* Save the current string pointer for the next gdm_setenv call */
    lastenv = envstr;
#endif
    
    return result;
}
#endif

void
gdm_clearenv (void)
{
#ifdef HAVE_CLEARENV
	clearenv ();
#else
	environ[0] = NULL;
#endif
}

/* clear environment, but keep the i18n ones,
 * note that this leak memory so only use before exec */
void
gdm_clearenv_no_lang (void)
{
	int i;
	GList *li, *envs = NULL;

	for (i = 0; environ[i] != NULL; i++) {
		char *env = environ[i];
		if (strncmp (env, "LC_", 3) == 0 ||
		    strncmp (env, "LANG", 4) == 0 ||
		    strncmp (env, "LINGUAS", 7) == 0)
			envs = g_list_prepend (envs, g_strdup (env));
	}

	gdm_clearenv ();

	for (li = envs; li != NULL; li = li->next) {
		putenv (li->data);
	}

	g_list_free (envs);
}

/* Do strcasecmp but ignore locale */
int
gdm_strcasecmp_no_locale (const char *s1, const char *s2)
{
	int i;

	/* Error, but don't make them equal then */
	g_return_val_if_fail (s1 != NULL, G_MAXINT);
	g_return_val_if_fail (s2 != NULL, G_MININT);

	for (i = 0; s1[i] != '\0' && s2[i] != '\0'; i++) {
		char a = s1[i];
		char b = s2[i];

		if (a >= 'A' && a <= 'Z')
			a -= 'A' - 'a';
		if (b >= 'A' && b <= 'Z')
			b -= 'A' - 'a';

		if (a < b)
			return -1;
		else if (a > b)
			return 1;
	}

	/* find out which string is smaller */
	if (s2[i] != '\0')
		return -1; /* s1 is smaller */
	else if (s1[i] != '\0')
		return 1; /* s2 is smaller */
	else
		return 0; /* equal */
}

char **
gdm_split (const char *s)
{
	int argc;
	char **temp_argv;
	char **ret;
	int i;

	if (s == NULL)
		return NULL;

	if (poptParseArgvString (s, &argc, &temp_argv) != 0) {
		return g_strsplit (s, " ", -1);
	}

	ret = g_new (char *, argc+1);
	for (i = 0; i < argc; i++) {
		ret[i] = g_strdup (temp_argv[i]);
	}
	ret[i] = NULL;

	free (temp_argv);

	return ret;
}

char *
gdm_first_word (const char *s)
{
	int argc;
	char **temp_argv;
	char *ret;

	if (s == NULL)
		return NULL;

	if (poptParseArgvString (s, &argc, &temp_argv) != 0) {
		char *p;
		ret = g_strdup (s);
		p = strchr (ret, ' ');
		if (p != NULL)
			*p = '\0';
		return ret;
	}

	ret = g_strdup (temp_argv[0]);

	free (temp_argv);

	return ret;
}

/* EOF */
