/* GDM - The Gnome Display Manager
 * Copyright (C) 1998, 1999 Martin Kasper Petersen <mkp@SunSITE.auc.dk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* This file contains functions for controlling local X servers */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <strings.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>
#include <X11/Xlib.h>

#include "gdm.h"

extern gchar *gdmDisplayInit;
extern gchar *gdmAuthDir;
extern gchar *gdmLogDir;

extern gchar **gdm_arg_munch(const gchar *p);
extern void gdm_auth_secure_display(gdmDisplayType *);
extern void gdm_debug(const gchar *, ...);
extern void gdm_abort(const gchar *, ...);
extern void gdm_display_manage(gdmDisplayType *);
extern void gdm_putenv(gchar *);

void gdm_server_start(gdmDisplayType *d);
void gdm_server_stop(gdmDisplayType *d);
void gdm_server_restart(gdmDisplayType *d);
void gdm_server_usr1_handler(gint);

gdmDisplayType *d;
Display *dsp=NULL;


void 
gdm_server_start(gdmDisplayType *disp)
{    
    struct sigaction usr1;
    sigset_t usr1mask;
    int logfd;
    gchar *srvcmd=NULL;
    gchar **argv=NULL;
    
    d=disp;

    gdm_debug(_("gdm_server_start: %s"), d->name);

    /* Catch USR1 from X server */
    usr1.sa_handler = gdm_server_usr1_handler;
    usr1.sa_flags = SA_RESTART|SA_ONESHOT;
    sigemptyset(&usr1.sa_mask);

    if(sigaction(SIGUSR1, &usr1, NULL) < 0) {
        syslog(LOG_ERR, _("gdm_server_start: Error setting up USR1 signal handler"));
	exit(SERVER_ABORT);
    };

    sigemptyset(&usr1mask);
    sigaddset(&usr1mask, SIGUSR1);
    sigprocmask(SIG_UNBLOCK, &usr1mask, NULL);

    /* Log all output from spawned programs to a file */
    logfd=open(g_strconcat(gdmLogDir, "/", d->name, ".log", NULL),
	       O_CREAT|O_TRUNC|O_APPEND|O_WRONLY, 0666);

    if(logfd != -1) {
	dup2(logfd, 1);
	dup2(logfd, 2);
    }
    else
	syslog(LOG_ERR, _("gdm_server_start: Could not open logfile for display %s!"), d->name);

    /* Just in case we have an old server hanging around */
    if(d->servpid) {
	gdm_debug(_("gdm_server_start: Old server found (%d). Killing."), d->servpid);
	gdm_server_stop(d);
    };

    /* Secure display with cookie */
    gdm_auth_secure_display(d);
    gdm_putenv(g_strconcat("XAUTHORITY=", d->auth, NULL));
    gdm_putenv(g_strconcat("DISPLAY=", d->name, NULL));

    /* Fork into two processes. Parent remains the gdm slave
     * process. Child becomes the X server.
     */

    switch(d->servpid=fork()) {
	
    case 0:
	/* The X server expects USR1 to be SIG_IGN */
	usr1.sa_handler = SIG_IGN;
	usr1.sa_flags = SA_RESTART;
	sigemptyset(&usr1.sa_mask);

	if(sigaction(SIGUSR1, &usr1, NULL) < 0) {
	    syslog(LOG_ERR, _("gdm_server_start: Error setting USR1 to SIG_IGN"));
	    exit(SERVER_ABORT);
	};
	
	srvcmd=g_strconcat(d->command, " -auth ", gdmAuthDir, \
			      "/", d->name, ".xauth ", 
			      d->name, NULL);

	gdm_debug(_("gdm_server_start: '%s'"), srvcmd);
	
	argv=gdm_arg_munch(srvcmd);
	g_free(srvcmd);

	execv(argv[0], argv);

	syslog(LOG_ERR, _("gdm_server_start: Xserver not found: %s"), d->command);

	exit(SERVER_ABORT);
	break;
	
    case -1:
	syslog(LOG_ERR, _("gdm_server_start: Can't fork Xserver process!"));
	d->servpid=0;
	break;
	
    default:
	break;
    };

    d->servstat=SERVER_STARTED;

    /* Wait for X server to send ready signal */
    pause();
};


void
gdm_server_stop(gdmDisplayType *d)
{
    gdm_debug(_("gdm_server_stop: Server for %s going down!"), d->name);

    kill(d->servpid, SIGTERM);
    waitpid(d->servpid, 0, 0);
    d->servpid=0;
    d->servstat=SERVER_DEAD;

    if(unlink(d->auth) == -1)
	syslog(LOG_ERR, _("gdm_server_stop: Could not unlink auth file: %s!"), strerror(errno));
}


void
gdm_server_restart(gdmDisplayType *d)
{
    sigset_t usr1mask;
    struct sigaction usr1;

    gdm_debug(_("gdm_server_restart: Server for %s restarting!"), d->name);

    /* Create new cookie */
    gdm_auth_secure_display(d);
    gdm_putenv(g_strconcat("XAUTHORITY=", d->auth, NULL));
    gdm_putenv(g_strconcat("DISPLAY=", d->name, NULL));

    /* Catch USR1 from X server */
    usr1.sa_handler = gdm_server_usr1_handler;
    usr1.sa_flags = SA_RESTART|SA_ONESHOT;
    sigemptyset(&usr1.sa_mask);

    if(sigaction(SIGUSR1, &usr1, NULL) < 0) {
        syslog(LOG_ERR, _("gdm_server_start: Error setting up USR1 signal handler"));
	exit(SERVER_ABORT);
    };

    sigemptyset(&usr1mask);
    sigaddset(&usr1mask, SIGUSR1);
    sigprocmask(SIG_UNBLOCK, &usr1mask, NULL);

    /* Prepare to rise again */
    d->servstat=SERVER_PHOENIX;

    /* Reset X and force auth file reread. XCloseDisplay works on most server, but we play it safe */
    kill(d->servpid, SIGHUP);

    d->servstat=SERVER_STARTED;

    /* Wait for X server to send ready signal */
    pause();
}


void
gdm_server_usr1_handler(gint sig)
{
    sigset_t usr1mask;
    gint openretries=0;
    
    gdm_debug(_("gdm_server_usr1_handler: Starting display %s!"), d->name);

    d->servstat=SERVER_RUNNING;

    /* Block USR1 */
    sigemptyset(&usr1mask);
    sigaddset(&usr1mask, SIGUSR1);
    sigprocmask(SIG_BLOCK, &usr1mask, NULL);

    /* We keep our own (windowless) connection (dsp) open to avoid the
     * X server resetting due to lack of active connections. */

    gdm_debug(_("gdm_slave_start: Opening display %s"), d->name);
    dsp=NULL;

    while(openretries < 10 && dsp==NULL) {
	dsp=XOpenDisplay(d->name);

	if(!dsp) {
	    gdm_debug(_("gdm_server_usr1_handler: Sleeping %d on a retry"), openretries*2);
	    sleep(openretries*2);
	    openretries++;
	};
    };
    
    if(!dsp) 
	gdm_abort(_("gdm_server_usr1_handler: Could not open display %s"), d->name);

    /* Start greeter on display */
    gdm_display_manage(d);
}

		
/* EOF */
