/*
 * gbrun-form.c
 *
 * Gnome Basic Interpreter Form functions.
 *
 * Authors:
 *	Frank Chiulli  <fc-linux@home.com>
 *      Michael Meeks  <michael@helixcode.com>
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include <gbrun/gbrun-stack.h>

#include "gbrun-form.h"
#include "gbrun-form-item.h"

#define ITEM_NAME "gb-vb.form"

/*
 *  temporary.
 */
static gint
delete_event_cb (GtkWidget *window, GdkEventAny *e, gpointer data)
{
	gtk_main_quit ();
	return TRUE;
}

/*
 * _GBEFormFont defines a Microsoft font.
 */
/*
struct _GBEFormFont {
	gchar		*name;
	gfloat		 size;
	guint8		 charset;
	guint16		 weight;
	gboolean	 underline;
	gboolean	 italic;
	gboolean	 strike_thru;
};
*/

/*
 * _GBEForm defines a VB Form
 *
 * appearance
 * auto_redraw
 * back_color		Specifies the forms's background color.
 * border_style		Determines whether a single-line border appears
 *			around the form.
 * caption		The text that appears on the form.
 * clip_controls
 * control_box
 * draw_mode
 * draw_style
 * draw_width
 * enabled
 * fill_color
 * fill_style
 * font		 	Caption's font name, style and size.
 * font_transparent
 * fore_color
 * has_dc
 * height		The height of button in twips.
 * help_context_id
 * icon
 * key_preview
 * left			The distance between the internal left edge of an 
 *			object and the left edge of its container.
 * link_mode
 * link_topic
 * max_button
 * mdi_child
 * min_button
 * mouse_icon
 * mouse_ptr		The shape of the mouse cursor when the user moves the 
 *			mouse over the command button.
 * moveable
 * negotiate_menus
 * ole_drop_mode
 * palette
 * palette_mode
 * picture		The name of icon graphic image that appears on the 
 *			command button as long as the Style property is set to
 *			1-Graphical.
 * right_to_left
 * scale_height		The number of units for the vertical measurement of an
 *			object's interior
 * scale_left		The horizontal coordinates for the left edge of an 
 *			object.
 * scale_mode
 * scale_top
 * scale_width
 * show_in_taskbar
 * start_up_position
 * tag
 * top			Holds the number of twips from the command button's
 *			top edge to the Form window's top edge.
 * visible		Determines whether the command button appears or is
 *			hidden from the user.
 * whats_this_button
 * whats_this_help
 * width		The width of an object
 * window_state
 *
 * cmd_buttons
 * labels
 * text_boxes
 *
 * window
 * fixed
 *
 * Definitions:
 * twip		1,440th of an inch (the smallest screen measurement)
 */
/*	guint8		 appearance;
	gboolean	 auto_redraw;
	guint32		 back_color;
	guint		 border_style;
	gchar		*caption;
	gboolean	 clip_controls;
	gboolean	 control_box;
	guint8		 draw_mode;
	guint8		 draw_style;
	guint8		 draw_width;
	gboolean	 enabled;
	guint32		 fill_color;
	guint8		 fill_style;
	GBEFormFont	 font;
	gboolean	 font_transparent;
	guint32		 fore_color;
	gboolean	 has_dc;
	guint16		 height;
	guint8		 help_context_id;
	gchar		*icon;
	gboolean	 key_preview;
	guint16		 left;
	guint8		 link_mode;
	gchar		*link_topic;
	gboolean	 max_button;
	gboolean	 mdi_child;
	gboolean	 min_button;
	gchar		*mouse_icon;
	guint8		 mouse_ptr;
	gboolean	 moveable;
	gboolean	 negotiate_menus;
	guint8		 ole_drop_mode;
	guint16		 palette;
	guint8		 palette_mode;
	gchar		*picture;
	gboolean	 right_to_left;
	guint16		 scale_height;
	guint16		 scale_left;
	guint8		 scale_mode;
	guint16		 scale_top;
	guint16		 scale_width;
	gboolean	 show_in_taskbar;
	guint8		 start_up_position;
	gchar		*tag;
	guint16		 top;
	gboolean	 visible;
	gboolean	 whats_this_button;
	gboolean	 whats_this_help;
	guint16		 width;
	guint8		 window_State; */

/*
 * Form properties
 * p_caption		The text that appears on the form.
 * p_height		The height of a form.
 * p_left		The distance between the internal left edge of an
 *			object and the left edge of its container.
 * p_scale_height	The number of units for the vertical measurement of an
 *			object's interior.
 * p_scale_width	The number of units for the horizontal measurement of
 *			an object's interior.
 * p_top		The distance between the internal top edge of an object
 *			and the top edge of its container.
 * p_width		The width of a form.
 *
 */

enum {
	ARG_FIRST = 0,
	CAPTION,
	HEIGHT,
	LEFT,
	SCALE_HEIGHT,
	SCALE_WIDTH,
	TOP,
	WIDTH
};

/**
 * form_setarg
 *   @ec
 *   @object
 *   @property
 *   @val
 *
 *   Internal function.
 *   This function processes the form properties and calls the appropriate
 * GTK function to set the appropriate properties of a GTK object.
 **/
static gboolean
form_setarg (GBRunEvalContext *ec,
	     GBRunObject      *object,
	     int               property,
	     GBValue          *val)
{
	GBRunForm *form = GBRUN_FORM (object);
	GtkWidget *w = GTK_WIDGET (form->window);

	g_return_val_if_fail (w != NULL, FALSE);
	g_return_val_if_fail (form != NULL, FALSE);

	switch (property) {
	case CAPTION:
		gtk_window_set_title (form->window, val->v.s->str);
		return TRUE;

	case HEIGHT:
		w->requisition.height = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_window_set_default_size (form->window, w->requisition.width,
				             w->requisition.height);
		return TRUE;

	case LEFT:
		w->allocation.x = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_widget_set_uposition (w, w->allocation.x, w->allocation.y);
		return TRUE;

	case TOP:
		w->allocation.y = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_widget_set_uposition (w, w->allocation.x, w->allocation.y);
		return TRUE;
	
	case WIDTH:
		w->requisition.width = GBRUN_FORM_TWIPS_TO_X (val->v.i);
 		gtk_window_set_default_size (form->window, w->requisition.width, 
		                             w->requisition.height);
		return TRUE;

	default:
		g_warning ("form: Unhandled property '%d'", property);
		return FALSE;
	}
}

/**
 * form_getarg
 *   @ec
 *   @object
 *   @property
 *
 *   Internal function.
 **/
static GBValue *
form_getarg (GBRunEvalContext *ec,
	     GBRunObject      *object,
	     int               property)
{
	GBRunForm *form = GBRUN_FORM (object);

	g_return_val_if_fail (form != NULL, NULL);
	g_return_val_if_fail (form->window != NULL, NULL);
	
	switch (property) {
	case CAPTION:
		return gb_value_new_string_chars (form->window->title);

	case WIDTH: {
		guint i = GTK_WIDGET (form->window)->allocation.width;
		return gb_value_new_int (GBRUN_FORM_X_TO_TWIPS (i));
	}

	case HEIGHT: {
		guint i = GTK_WIDGET (form->window)->allocation.height;
		return gb_value_new_int (GBRUN_FORM_Y_TO_TWIPS (i));
	}
	default:
		g_warning ("form: Unhandled property '%d'", property);
		break;
	}

	return NULL;
}

static GBValue *
form_show (GBRunEvalContext *ec,
	   GBRunObject      *form,
	   GBValue         **args)
{
	/* FIXME: loads if not already loaded !? */
	if (args [0] || args [1])
		g_warning ("Modality & owner unimplemented");

	gtk_widget_show (GTK_WIDGET (GBRUN_FORM (form)->window));

	return gb_value_new_empty ();
}

static GBValue *
form_hide (GBRunEvalContext *ec,
	   GBRunObject      *form,
	   GBValue         **args)
{
	gtk_widget_hide (GTK_WIDGET (GBRUN_FORM (form)->window));

	return gb_value_new_empty ();
}

/**
 * gbrun_form_register
 **/
void
gbrun_form_register (void)
{
	gbrun_form_get_type ();
}


/**
 * gbrun_form_shutdown
 **/
void
gbrun_form_shutdown (void)
{	
}


/**
 * gbrun_form_pass_properties:
 *   @ec: 
 *   @obj: 
 *   @item: 
 * 
 *   This function takes the properties described by the form description,
 * promotes their values to the correct types as expected by gbrun_object_set_arg
 * etc.  It then calls gbrun_object_set_arg. Ideally we must move the promotion 
 * into gbrun_object_set_arg.
 **/
static void
gbrun_form_pass_properties (GBRunEvalContext *ec,
			    GBRunObject      *obj,
			    GBFormItem       *item)
{
	GSList *l;

	g_return_if_fail (item != NULL);
	g_return_if_fail (GBRUN_IS_OBJECT (obj));
	g_return_if_fail (GBRUN_IS_EVAL_CONTEXT (ec));
	
	for (l = item->properties; l; l = l->next) {
		GBFormProperty *prop = l->data;
		GBObjRef ref;
		
		ref.method = FALSE;
		ref.name   = prop->name;
		ref.parms  = NULL;

		if (gbrun_object_has_property (
			GBRUN_OBJECT_GET_CLASS (obj),
			prop->name, GBRUN_PROPERTY_WRITEABLE))
			gb_object_assign (GB_EVAL_CONTEXT (ec), GB_OBJECT (obj),
					  &ref, prop->value, FALSE);
		else
			g_warning ("Missing property '%s' on '%s' named '%s'", 
			           prop->name, gbrun_object_name (GBRUN_OBJECT (obj)),
				   item->name);
	}
}


/**
 * gbrun_form_show:
 *   @obj: 
 **/
void
gbrun_form_show (GBRunForm *form)
{
	g_return_if_fail (GBRUN_IS_FORM (form));
	
	gtk_widget_show_all (GTK_WIDGET (form->window));
}

static void
gbrun_form_add (GBRunForm         *form,
		GBRunFormItem     *item,
		const char        *name,
		const GBParseData *module)
{
	g_return_if_fail (item != NULL);
	g_return_if_fail (name != NULL);
	g_return_if_fail (form != NULL);

	item->form = form;
	item->name = g_strdup (name);

	gtk_fixed_put (form->fixed, item->widget, 0, 0);
}

gboolean
gbrun_form_invoke (GBRunEvalContext *ec,
		   GBRunForm        *form,
		   const char       *method)
{
	GBValue  *ignore;
	gboolean  ret;
	
	g_return_val_if_fail (form != NULL, FALSE);

	if (gbrun_object_has_method (GBRUN_OBJECT_GET_CLASS (form), method)) {
		GBObjRef ref;

		ref.method = TRUE;
		ref.name   = method;
		ref.parms  = NULL;

		if ((ignore = gbrun_objref_deref (ec, GB_OBJECT (form),
						  &ref, TRUE))) {
			gb_value_destroy (ignore);
			ret = TRUE;
		} else {
			if (gb_eval_exception (GB_EVAL_CONTEXT (ec))) {
				g_warning ("Error invoking '%s' : '%s", method,
					   gb_eval_context_get_text (GB_EVAL_CONTEXT (ec)));
				gb_eval_context_reset (GB_EVAL_CONTEXT (ec));
				ret = FALSE;
			}
		}
	}

	return ret;
}

/**
 * gbrun_form_new:
 *   @ec: 
 *   @item: 
 * 
 **/
void
gbrun_form_init (GBRunEvalContext  *ec,
		 GBRunForm         *form,
		 const GBParseData *pd)
{
	GSList        *l;
	GBFormItem    *item;

	g_return_if_fail (ec != NULL);
	g_return_if_fail (pd != NULL);
	g_return_if_fail (form != NULL);

	item = pd->form;
	g_return_if_fail (item != NULL);

	gbrun_form_pass_properties (ec, GBRUN_OBJECT (form), item);

	for (l = item->children; l; l = l->next) {
		GBFormItem       *i = l->data;
		GtkType           type;
		GBRunFormItem    *item;

		type = gb_gtk_type_from_name (i->type);
		if (!type) {
			g_warning ("Unknown sub-form type '%s'", i->type);
			continue;
		}

		item = gbrun_form_item_new (ec, type);

		if (GBRUN_IS_FORM_ITEM (item))
			gbrun_form_add (form, item, i->name, pd);
		else
			g_warning ("Non form item '%s' put into form", i->type);
		
		gbrun_form_pass_properties (ec, GBRUN_OBJECT (item), i);
		
		/* Add it to the stack */
		gbrun_object_ref (item);
		gbrun_stack_add  (ec, i->name,
				  gb_value_new_object (GB_OBJECT (item)),
				  GBRUN_STACK_MODULE);
	}
	gbrun_form_show (form);
}

void
gbrun_form_widget_set_color (GtkWidget         *widget,
			     GBRunFormColorType type,
			     GBLong             color)
{
	GdkColor  col;
	GtkStyle *style;
	GdkColor *array = NULL;

	col.red   = ((color >>  0) & 0xff) * 255;
	col.green = ((color >>  8) & 0xff) * 255;
	col.blue  = ((color >> 16) & 0xff) * 255;

	style = gtk_style_copy (widget->style);

/*	g_warning ("Setting %d color on widget to 0x%x (= %d, %d, %d)", type,
	color, col.red, col.green, col.blue);*/
	/*
	 * FIXME: should recurse down containment hierarchy perhaps ?
	 * at least until we hit a widget with the FormItemKey
	 */

	/* FIXME: these need testing / rationalizing */
	switch (type) {

	case GBRUN_FORM_COLOR_BACK:
		array = style->base;
		break;

	case GBRUN_FORM_COLOR_BORDER:
		array = style->base;
		break;

	case GBRUN_FORM_COLOR_FILL:
		array = style->bg;
		break;

	case GBRUN_FORM_COLOR_FORE:
		array = style->fg;
		break;

	case GBRUN_FORM_COLOR_MASK:
	default:
		g_warning ("Unknown color type");
		break;
	};

	if (array) {
		int i;
		for (i = GTK_STATE_NORMAL; i <= GTK_STATE_INSENSITIVE; i++)
			array [i] = col;
	}

	gtk_widget_set_style  (widget, style);
	gtk_widget_queue_draw (widget);
}

GBLong
gbrun_form_widget_get_color (GtkWidget         *widget,
			     GBRunFormColorType type,
			     GBLong             color)
{
	g_warning ("Unimplemented");

	return 0;
}

char *
gbrun_form_un_shortcutify (const char *txt, char *shortcut)
{
	char *ans;
	int   i;

	g_return_val_if_fail (txt != NULL, NULL);

	ans = g_strdup (txt);

	/* FIXME: some sort of escaping must happen */
	for (i = 0; ans [i]; i++)
		if (ans [i] == '&') {
			if (shortcut)
				*shortcut = ans [i + 1];
			ans [i] = ' ';
		}

	return ans;
}


static void
gbrun_form_destroy (GtkObject *object)
{
	GBRunForm *form = GBRUN_FORM (object);

	if (form->window)
		gtk_widget_destroy (GTK_WIDGET (form->window));
	form->window = NULL;
}

static void
gbrun_form_instance_init (GBRunForm *form)
{
	form->window = GTK_WINDOW (gtk_window_new (GTK_WINDOW_TOPLEVEL));
	gtk_signal_connect (GTK_OBJECT (form->window),
  		            "delete_event",
		            GTK_SIGNAL_FUNC (delete_event_cb),
		            form);

	form->fixed = GTK_FIXED (gtk_fixed_new ()); 

	gtk_container_add (GTK_CONTAINER (form->window),
			   GTK_WIDGET (form->fixed));
}

static void
gbrun_form_class_init (GBRunObjectClass *klass)
{
	GtkObjectClass *gtk_class = (GtkObjectClass *) klass;

	klass->set_arg = form_setarg;
	klass->get_arg = form_getarg;

	gbrun_object_add_property (klass, "caption",	  gb_type_string, CAPTION);
	gbrun_object_add_property (klass, "clientheight", gb_type_int, HEIGHT);
	gbrun_object_add_property (klass, "clientleft",   gb_type_int, LEFT);
	gbrun_object_add_property (klass, "scaleheight",  gb_type_int, SCALE_HEIGHT);
	gbrun_object_add_property (klass, "scalewidth",   gb_type_int, SCALE_WIDTH);
	gbrun_object_add_property (klass, "clienttop",    gb_type_int, TOP);
	gbrun_object_add_property (klass, "clientwidth",  gb_type_int, WIDTH);

	/*
	 * We nee the Objects collection here;
	 */
/*
	This line:

		Form1!Ctrl1.Text = "Hello"

	translates into this code:

		Form1.Controls.Item("Ctrl1").Text = "Hello"
*/

	gbrun_object_add_method_arg (
		klass, "sub;show;modal,integer,byval,0;ownerform,variant,byref,0;g", form_show);

	gbrun_object_add_method_arg (
		klass, "sub;hide;g", form_hide);

	gtk_class->destroy = gbrun_form_destroy;
}

GtkType
gbrun_form_get_type (void)
{
	static GtkType object_type = 0;

	if (!object_type) {
		static const GtkTypeInfo object_info = {
			ITEM_NAME,
			sizeof (GBRunForm),
			sizeof (GBRunFormClass),
			(GtkClassInitFunc)  gbrun_form_class_init,
			(GtkObjectInitFunc) gbrun_form_instance_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		object_type = gtk_type_unique (GBRUN_TYPE_OBJECT, &object_info);
		gtk_type_class (object_type);
	}

	return object_type;	
}
