/*
 * GNOME Basic Enterpreter library main.
 *
 * Author:
 *    Michael Meeks <mmeeks@gnu.org>
 *
 * Copyright 2000, Helix Code, Inc
 */

#include <gbrun/gbrun.h>
#include <gb/libgb.h>
#include <gb/gb-main.h>
#include <gb/gb-form.h>
#include <gb/gb-mmap-lex.h>
#include <gbrun/libgbrun.h>
#include <gbrun/gbrun-type.h>
#include <gbrun/gbrun-stack.h>
#include <gbrun/gbrun-statement.h>
#include <gbrun/objects/libgbobj.h>
#include <gbrun/objects/gba/libgba.h>
#include <gbrun/objects/gbrun-objects.h>

#undef PROJECT_DEBUG

#define HAVE_FORM_CODE

struct _GBRunProjectPriv {
	GBRunStreamProvider *provider;
	gpointer             user_data;
	GBProject           *gb_proj;
	GSList              *objects;

	GHashTable          *classes;
};

static void
project_destroy (GtkObject *obj)
{
	GBRunProject     *proj = GBRUN_PROJECT (obj);
	GBRunProjectPriv *priv = proj->priv;

	gb_project_destroy (priv->gb_proj);
	priv->gb_proj = NULL;

	while (priv->objects) {
		GBObject *obj = priv->objects->data;

		gb_object_unref (obj);
		priv->objects = g_slist_remove (priv->objects, obj);
	}

	g_free (priv);
}

static void
gbrun_project_copy (GBEvalContext  *ec,
		    GBObject       *src,
		    GBObject       *dest)
{
	g_warning ("FIXME: copying a project, wierd");
}

static gboolean
gbrun_project_assign (GBEvalContext  *ec,
		      GBObject       *object,
		      const GBObjRef *ref,
		      GBValue        *value,
		      gboolean        try_assign)
{
	if (!try_assign)
		g_error ("Can't assign to project");

	return FALSE;
}

static GBValue *
gbrun_project_deref (GBEvalContext  *gb_ec,
		     GBObject       *obj,
		     const GBObjRef *ref,
		     gboolean        try_deref)
{
	GSList *l;
	GBRunProject *proj = GBRUN_PROJECT (obj);

#ifdef PROJECT_DEBUG
	g_warning ("In proj deref '%d'", g_slist_length (proj->priv->objects));
#endif

	for (l = proj->priv->objects; l; l = l->next) {
		char *str = gtk_object_get_data (l->data, "Hack");

#ifdef PROJECT_DEBUG
		g_warning ("Comparing '%s' with registered object '%s'",
			   ref->name, str);
#endif

		if (!g_strcasecmp (ref->name, str))
			return gb_value_new_object (l->data);
	}

		/* FIXME: consider public / private here */
	for (l = proj->priv->objects; l; l = l->next) {
		GBValue *ret;
		
		if ((ret = gb_object_deref (gb_ec, GB_OBJECT (l->data),
					    ref, TRUE)) ||
		    gb_eval_exception (gb_ec))
			return ret;
	}

	return NULL;
}

static void
gbrun_project_class_init (GBObjectClass *klass)
{
	GtkObjectClass    *gtk_class  = (GtkObjectClass *) klass;
/*	GBRunProjectClass *proj_class = (GBRunProjectClass *) klass;*/
	
	klass->copy   = gbrun_project_copy;
	klass->assign = gbrun_project_assign;
	klass->deref  = gbrun_project_deref;

	gtk_class->destroy = project_destroy;
}

static void
gbrun_project_init (GBRunProject *proj)
{
	proj->priv = g_new0 (GBRunProjectPriv, 1);
}

GtkType
gbrun_project_get_type (void)
{
	static GtkType proj_type = 0;

	if (!proj_type) {
		static const GtkTypeInfo proj_info = {
			"GBRunProject",
			sizeof (GBRunProject),
			sizeof (GBRunProjectClass),
			(GtkClassInitFunc) gbrun_project_class_init,
			(GtkObjectInitFunc) gbrun_project_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		proj_type = gtk_type_unique (GB_TYPE_OBJECT, &proj_info);
	}

	return proj_type;
}

static GBRunObjectClass *
get_class (const char *opt_name, const GBParseData *pd)
{
	GBRunObjectClass *klass;
	GtkType           type;

	g_return_val_if_fail (pd != NULL, NULL);

	if (pd->form) {
#ifdef HAVE_FORM_CODE
		type = gbrun_object_subclass_simple (
			gbrun_form_get_type (), pd->form->name);
#else
		g_warning ("No form code");
		return NULL;
#endif
	} else
		type = gbrun_object_subclass_simple (
			gbrun_object_get_type (), opt_name);
	
	klass = gtk_type_class (type);

	return klass;
}

static GBRunObjectClass *
add_data_to_class (GBRunEvalContext  *ec,
		   GBRunObjectClass  *klass,
		   const GBParseData *pd)
{
	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (pd != NULL, NULL);
	g_return_val_if_fail (klass != NULL, NULL);

	/*
	 * FIXME: this is a hack; we need to be able to scope
	 * types, which could be really, really fun. This would
	 * seem to involve re-writing the gtk type system, or
	 * at least, hacking it in some clever ( ugly ) way.
	 * for now, a Private type is a pipe-dream, all are public.
	 */
	gbrun_register_types (ec, klass, pd->types);
	gbrun_object_add_routines (ec, klass, pd->routines);
	gbrun_object_add_variables (ec, klass, pd->variables);

	return klass;
}

const GBParseData *
parsed_load (GBRunEvalContext    *ec,
	     const char          *filename,
	     GBRunStreamProvider *provider,
	     gpointer             user_data,
	     GBParsingState       state)
{
	GBLexerStream     *ls;
	const GBParseData *pd;

	if (!(ls = provider (ec, filename, user_data)))
		return NULL;

	/* Set parser state before and then turn it off later in grammar */
	gb_lexer_stream_state_set (ls, state);

	pd = gb_parse_stream (GB_EVAL_CONTEXT (ec), ls);
	gtk_object_destroy (GTK_OBJECT (ls));

	return pd;
}

GBRunProject *
gbrun_project_new (GBRunEvalContext *ec, GBProject *p,
		   GBRunStreamProvider *provider,
		   gpointer user_data)
{
	GSList           *l;
	GBRunProject     *proj;
	GBRunProjectPriv *priv;
	char             *proj_name;

	g_return_val_if_fail (p != NULL, NULL);
	g_return_val_if_fail (provider != NULL, NULL);
	proj = GBRUN_PROJECT (gtk_type_new (GBRUN_TYPE_PROJECT));

	priv = proj->priv;

	proj_name = gbrun_eval_context_get_module (ec);

	priv->provider  = provider;
	priv->user_data = user_data;
	priv->gb_proj   = p;
	priv->classes   = g_hash_table_new (g_str_hash, g_str_equal);

	gbrun_eval_context_proj_push (ec, proj);

	for (l = p->modules; l; l = l->next) {
		GBProjectPair     *pp = l->data;
		const GBParseData *pd;
		GBRunObject       *obj;

		fprintf (stderr, "Loading module '%s'\n", pp->filename);

		gbrun_eval_context_set_module (ec, pp->filename);

		if (!(pd = parsed_load (ec, pp->filename, provider, user_data, GB_PARSING_BASIC)))
			return NULL;

		
		obj = gbrun_object_new (
			add_data_to_class (
				ec, get_class (pp->filename, pd), pd));

/*		gb_parse_data_destroy (pd);*/

		gbrun_project_register_object (proj, obj, pp->filename);
	}

	for (l = p->classes; l; l = l->next) {
		GBProjectPair     *pp = l->data;
		const GBParseData *pd;
		GBRunObject       *obj;

		fprintf (stderr, "Loading class '%s'\n", pp->filename);

		gbrun_eval_context_set_module (ec, pp->filename);

		if (!(pd = parsed_load (ec, pp->filename, provider, user_data, GB_PARSING_CLASS)))
			return NULL;

		obj = gbrun_object_new (
			add_data_to_class (
				ec, get_class (pp->filename, pd), pd));
/*		gb_parse_data_destroy (pd);*/
		
		g_hash_table_insert (priv->classes, (gpointer) pp->name, (gpointer) obj); 

	}
		
	for (l = p->forms; l; l = l->next) {
		char              *filename = l->data;
		const GBParseData *pd;
		GBRunObject       *obj;

		fprintf (stderr, "Loading form '%s'\n", (char *)l->data);

		gbrun_eval_context_set_module (ec, filename);

		if (!(pd = parsed_load (ec, filename, provider, user_data, GB_PARSING_FORM)))
			return NULL;

		obj = gbrun_object_new (
			add_data_to_class (
				ec, get_class ("Unused", pd), pd));

#ifdef HAVE_FORM_CODE	
		gbrun_project_register_object (proj, obj, pd->form->name);

		gbrun_form_init (ec, GBRUN_FORM (obj), pd);
		gbrun_form_invoke (ec, GBRUN_FORM (obj), "Form_Load");
#endif

/*		gb_parse_data_destroy (pd);*/
	}

	gbrun_eval_context_set_module (ec, proj_name);
	g_free (proj_name);

	gbrun_eval_context_proj_pop (ec);

	return proj;
}

gboolean
gbrun_project_execute (GBRunEvalContext *ec, GBRunProject *proj)
{
	const char *startup;
	gboolean    success;

	g_return_val_if_fail (GBRUN_IS_EVAL_CONTEXT (ec), FALSE);
	g_return_val_if_fail (GBRUN_IS_PROJECT (proj), FALSE);
	g_return_val_if_fail (proj->priv != NULL, FALSE);
	g_return_val_if_fail (proj->priv->gb_proj != NULL, FALSE);

	startup = gb_project_startup_get (proj->priv->gb_proj);

	if (!g_strncasecmp (startup, "Sub ", 4)) {
		char    *subname = g_strchug (g_strchomp (g_strdup (startup + 4)));
		GBValue *val;

/*		g_warning ("Project sub : '%s'", subname);*/
		
		val = gbrun_project_invoke (ec, proj, subname, NULL);
		gb_value_destroy (val);

		g_free (subname);

		success = !gbrun_eval_context_exception (ec);
	} else { /* A form */
/*		g_warning ("Project form : '%s'", startup);*/
		gtk_main ();
		success = TRUE;
	}

	return success;
}

void
gbrun_init (GBEvalContext *ec)
{
	gbrun_object_init (ec);
	if (gb_eval_exception (ec))
		return;

	libgba_register (ec);
	if (gb_eval_exception (ec))
		return;

	gbrun_objects_register (ec);
}


void
gbrun_shutdown (void)
{
	gbrun_objects_shutdown ();
	libgba_shutdown        ();
	gbrun_object_shutdown  ();
}

void
gbrun_project_register_object (GBRunProject *proj,
			       GBRunObject  *object,
			       const char   *name)
{
	g_return_if_fail (name != NULL);
	g_return_if_fail (GBRUN_IS_PROJECT (proj));

	if (!g_slist_find (proj->priv->objects, object)) {
		/* FIXME: disgusting hack; battery dying */
		gtk_object_set_data (GTK_OBJECT (object), "Hack",
				     g_strdup (name));
		proj->priv->objects = g_slist_append (proj->priv->objects, object);
	} else
		g_warning ("Registered project object '%s' twice", name);
}

void
gbrun_project_deregister_object (GBRunProject *proj,
				 GBRunObject  *object)
{
	g_return_if_fail (GBRUN_IS_PROJECT (proj));

	proj->priv->objects = g_slist_remove (proj->priv->objects, object);
}

GBValue *
gbrun_project_invoke (GBRunEvalContext *ec, GBRunProject *proj,
		      const char *name, GSList *args)
{
	GBObjRef ref;
	GSList  *l, *exprs = NULL;
	GBValue *val;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);

	gbrun_eval_context_proj_push (ec, proj);

	/* FIXME */
	ref.method = FALSE;
	ref.name   = name;

	/* FIXME: Amusingly inefficient */
	for (l = args; l; l = l->next) {
		gpointer expr = (gpointer) gb_expr_new_value (gbrun_value_copy (ec, l->data));
		exprs = g_slist_prepend (exprs, expr);
	}

	exprs = g_slist_reverse (exprs);
	ref.parms = exprs;
	ref.method = TRUE;

	val = gbrun_objref_deref (ec, NULL, &ref, TRUE);

	while (exprs) {
		gb_expr_destroy (exprs->data);
		exprs = g_slist_remove (exprs, exprs->data);
	}

	gbrun_eval_context_proj_pop (ec);

	return val;
}

GSList *
gbrun_project_fn_names (GBRunProject *proj)
{
	GSList *ans = NULL;
	GSList *l;

	g_return_val_if_fail (proj != NULL, NULL);
	g_return_val_if_fail (proj->priv != NULL, NULL);

	for (l = proj->priv->objects; l; l = l->next) {
		GBRunObject          *obj = l->data;
		GSList               *m, *i;

		m = gbrun_object_get_methods (GBRUN_OBJECT_GET_CLASS (obj));

		for (i = m; i; i = i->next) {
			GBRunObjMethod *method = i->data;

			ans = g_slist_prepend (ans, method->name);
		}
		
		g_slist_free (m);
	}

	return ans;
}

static const GBParseData *
parse_str (GBRunEvalContext *ec,
	   const char       *str,
	   gboolean          needs_eol,
	   GBParsingState    state)
{
	GBLexerStream *ls;
	char          *terminated;
	int            len;
	const GBParseData *pd;

	g_return_val_if_fail (str != NULL, NULL);

	len = strlen (str);
	if (needs_eol) {
		if (str [len] != '\n') {
			terminated = g_strconcat (str, "\n", NULL);
			len++;
		} else
			terminated = g_strdup (str);
	} else {
		if (str [len] == '\n')
			len--;

		terminated = g_strdup (str);
	}

/*	g_warning ("Trying to parse '%s'", terminated); */

	ls = gb_mmap_stream_new (terminated, len);

	gb_lexer_stream_state_set (ls, state);

	pd = gb_parse_stream (GB_EVAL_CONTEXT (ec), ls);
	gtk_object_destroy (GTK_OBJECT (ls));

	if (!pd)
		return NULL;

/*	g_warning ("Parsed ok"); */

	return pd;
}

void
gbrun_exec_str (GBRunEvalContext *ec,
		GBRunObject      *opt_object,
		const char       *basic_string)
{
	const GBParseData *pd;

	g_return_if_fail (GBRUN_IS_EVAL_CONTEXT (ec));
	g_return_if_fail (!opt_object || GBRUN_IS_OBJECT (opt_object));

	pd = parse_str (ec, basic_string, TRUE, GB_PARSING_STATEMENTS);
	if (!pd)
		return;

	if (pd->stmts) {
		if (opt_object)
			gbrun_eval_context_me_set (
				ec, GB_OBJECT (opt_object));

		gbrun_stmts_evaluate (ec, pd->stmts);
	}

	gb_parse_data_destroy (pd);
}

GBValue *
gbrun_eval_str (GBRunEvalContext *ec,
		GBRunObject      *opt_object,
		const char       *basic_string)
{
	const GBParseData *pd;
	GBValue           *val = NULL;

	g_return_val_if_fail (GBRUN_IS_EVAL_CONTEXT (ec), NULL);
	g_return_val_if_fail (!opt_object || GBRUN_IS_OBJECT (opt_object), NULL);

	pd = parse_str (ec, basic_string, FALSE, GB_PARSING_EXPR);
	if (!pd)
		return NULL;

	if (pd->expr) {
		if (opt_object)
			gbrun_eval_context_me_set (
				ec, GB_OBJECT (opt_object));

		val = gb_eval_context_eval (
			GB_EVAL_CONTEXT (ec), pd->expr);
	}

	gb_parse_data_destroy (pd);

	return val;
}
