
/*
 * gbrun-form-item.c
 *
 * Gnome Basic Interpreter Form functions.
 *
 * Authors:
 *	Frank Chiulli  <fc-linux@home.com>
 *      Michael Meeks  <michael@helixcode.com>
 */
#include "gbrun/gbrun-eval.h"
#include "gbrun-form-item.h"

#define ITEM_NAME "--GB.FormItem--"

static void copy (GBEvalContext *, GBObject *, GBObject *);
static void destruct (GBObject *);

static void
destruct (GBObject *object)
{
	g_warning ("Unimplemented %s destruct", ITEM_NAME);
}

GBObjectClass *
gbrun_form_item_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBRunObjectPrivClass p;
		gb_object_priv_class_init (&p.priv, ITEM_NAME,
					   sizeof (GBRunFormItem),
					   (GBObjectCopy *)copy,
					   (GBObjectDestructor *)destruct);
		gbrun_object_priv_class_init (&p);

		oc = gb_object_class_new_single (&p.priv, gbrun_object_class ());
		gbrun_object_register (ITEM_NAME, oc);
	}
		
	return oc;
}

#define PRIV(o) ((GBRunFormItem *)gb_object_get_priv ((o), gbrun_form_item_class ()))

static GBRunObjProperty *p_width    = NULL;
static GBRunObjProperty *p_height   = NULL;
static GBRunObjProperty *p_top      = NULL;
static GBRunObjProperty *p_left     = NULL;

static GBRunObjProperty *p_tabindex = NULL;

static GBRunObjProperty *p_forecolor   = NULL;
static GBRunObjProperty *p_backcolor   = NULL;
static GBRunObjProperty *p_bordercolor = NULL;
static GBRunObjProperty *p_fillcolor   = NULL;
static GBRunObjProperty *p_maskcolor   = NULL;

static void
copy (GBEvalContext  *ec,
      GBObject       *dest_obj,
      GBObject       *templ_obj)
{
	GBRunFormItem  *fi = PRIV (dest_obj);

	if (templ_obj)
		g_warning ("Copy unimplemented");

	fi->x = fi->y = 0;
	fi->form = NULL;
	fi->widget = NULL;
	fi->tabindex = -1;
}

void
form_item_set_arg (GBRunEvalContext   *ec,
		   GBRunObject        *object,
		   GBRunObjProperty   *property,
		   GBValue            *val) 
{
	GBRunFormItem *fi = PRIV (object);
	GBRunForm     *form = gb_object_get_priv (fi->form, gbrun_form_class ());

	if (property == p_width) {
		GtkWidget *w = fi->widget;
		w->allocation.width = GBRUN_FORM_TWIPS_TO_X (val->v.i);
		gtk_widget_set_usize (w, w->allocation.width, w->allocation.height);

	} else if (property == p_height) {
		GtkWidget *w = fi->widget;
		w->allocation.height = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_widget_set_usize (w, w->allocation.width, w->allocation.height);

	} else if (property == p_top) {
		fi->y = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_fixed_move (form->fixed, fi->widget, fi->x, fi->y);

	} else if (property == p_left) {
		fi->x = GBRUN_FORM_TWIPS_TO_X (val->v.i);
		gtk_fixed_move (form->fixed, fi->widget, fi->x, fi->y);

	} else if (property == p_tabindex)
		fi->tabindex = val->v.i;
		
	else if (property == p_forecolor)
		gbrun_form_widget_set_color (fi->widget, GBRUN_FORM_COLOR_FORE, val->v.l);

	else if (property == p_backcolor)
		gbrun_form_widget_set_color (fi->widget, GBRUN_FORM_COLOR_BACK, val->v.l);

	else if (property == p_fillcolor)
		gbrun_form_widget_set_color (fi->widget, GBRUN_FORM_COLOR_FILL, val->v.l);

	else if (property == p_maskcolor)
		gbrun_form_widget_set_color (fi->widget, GBRUN_FORM_COLOR_MASK, val->v.l);
		
	else if (property == p_bordercolor)
		gbrun_form_widget_set_color (fi->widget, GBRUN_FORM_COLOR_BORDER, val->v.l);
	else
		g_warning ("Bad property set %s", property->name);
}

GBValue *
form_item_get_arg (GBRunEvalContext *ec,
		   GBRunObject      *object,
		   GBRunObjProperty *property)
{
	GBRunFormItem *fi = PRIV (object);

	if (property == p_width) {
		guint i = fi->widget->allocation.width;
		return gb_value_new_int (GBRUN_FORM_X_TO_TWIPS (i));

	} else if (property == p_height) {
		guint i = fi->widget->allocation.height;
		return gb_value_new_int (GBRUN_FORM_Y_TO_TWIPS (i));

	} else if (property == p_top) {
		g_warning ("Unimplemented top");

	} else if (property == p_left) {
		g_warning ("Unimplemented left");

	} else if (property == p_tabindex) {
		return gb_value_new_int (fi->tabindex);

	} else
		g_warning ("Bad property get %s", property->name);

	return NULL;
}

void
gbrun_form_item_set_widget (GBRunObject *object, GtkWidget *w)
{
	GBRunFormItem *fi = PRIV (object);

	g_return_if_fail (w != NULL);
	g_return_if_fail (fi != NULL);
	g_return_if_fail (GB_IS_OBJECT (object));

	fi->widget = w;
	if (w) { /* FIXME: set containees data too ? */
		gtk_object_set_data (GTK_OBJECT (w), GBRUN_FORM_ITEM_KEY, object);
		gtk_widget_show (w);
	}

	fi->x = 0;
	fi->y = 0;
}

GtkWidget *
gbrun_form_item_get_widget (GBRunObject *object)
{
	GBRunFormItem *fi = PRIV (object);

	g_return_val_if_fail (fi != NULL, NULL);
       
	return fi->widget;
}

void
gbrun_form_item_invoke (GBRunEvalContext *ec,
			GtkWidget *w, char *suffix)
{
	char          *fn_name;
	GBRunFormItem *fi;
	GBObject      *object;

	g_return_if_fail (w != NULL);
	g_return_if_fail (ec != NULL);
	g_return_if_fail (suffix != NULL);

	object = gtk_object_get_data (GTK_OBJECT (w), GBRUN_FORM_ITEM_KEY);
	g_return_if_fail (object != NULL);

	fi = PRIV (object);
	g_return_if_fail (fi != NULL);

	fn_name = g_strconcat (fi->name, suffix, NULL);;

	gbrun_form_invoke (ec, fi->form, fn_name);

	g_free (fn_name);
}

static GBValue *
form_item_show (GBRunEvalContext *ec,
		GBRunObject      *object,
		GBValue         **args)
{
	GBRunFormItem *fi = PRIV (object);

	gtk_widget_show (fi->widget);

	return gb_value_new_empty ();
}

static GBValue *
form_item_hide (GBRunEvalContext *ec,
		GBRunObject      *object,
		GBValue         **args)
{
	GBRunFormItem *fi = PRIV (object);

	gtk_widget_hide (fi->widget);

	return gb_value_new_empty ();
}

void
gbrun_form_item_register (void)
{
	GBRunObjectPrivClass *klass;
	
	klass = gbrun_object_class_get_priv (gbrun_form_item_class ());

	g_return_if_fail (klass != NULL);

	klass->set_arg = form_item_set_arg;
	klass->get_arg = form_item_get_arg;

	p_width   = gbrun_object_add_property_val (klass, "Width",   GB_VALUE_INT);
	p_height  = gbrun_object_add_property_val (klass, "Height",  GB_VALUE_INT);
	p_top     = gbrun_object_add_property_val (klass, "Top",     GB_VALUE_INT);
	p_left    = gbrun_object_add_property_val (klass, "Left",    GB_VALUE_INT);

	p_tabindex    = gbrun_object_add_property_val (klass, "TabIndex",    GB_VALUE_INT);

	p_forecolor   = gbrun_object_add_property_val (klass, "ForeColor",   GB_VALUE_LONG);
	p_backcolor   = gbrun_object_add_property_val (klass, "BackColor",   GB_VALUE_LONG);
	p_bordercolor = gbrun_object_add_property_val (klass, "BorderColor", GB_VALUE_LONG);
	p_fillcolor   = gbrun_object_add_property_val (klass, "FillColor",   GB_VALUE_LONG);
	p_maskcolor   = gbrun_object_add_property_val (klass, "MaskColor",   GB_VALUE_LONG);

	gbrun_object_add_method_arg (klass, "sub;show;g", form_item_show);
	gbrun_object_add_method_arg (klass, "sub;hide;g", form_item_hide);
}

void
gbrun_form_item_shutdown (void)
{
}
