
/*
 * gbrun-cmdbutton.c
 *
 * Gnome Basic Interpreter Command Button functions.
 *
 * Authors:
 *	Frank Chiulli (fc-linux@home.com)
 *      Michael Meeks (mmeeks@gnu.org)
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include "gbrun-form-item.h"

#define ITEM_NAME "VB.CommandButton"

struct _GBRunCmdButton {
	GBRunObjectPriv priv;

	GtkLabel       *label;
	char           *real_caption;
};

static void copy     (GBEvalContext *, GBObject *, GBObject *);
static void destruct (GBObject *object);

static GBObjectClass *
cmdbutton_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBRunObjectPrivClass p;
		gb_object_priv_class_init (&p.priv, ITEM_NAME,
					   sizeof (GBRunCmdButton),
					   (GBObjectCopy *)copy,
					   (GBObjectDestructor *)destruct);
		gbrun_object_priv_class_init (&p);

		oc = gb_object_class_new_single (&p.priv, gbrun_form_item_class ());
		gbrun_object_register (ITEM_NAME, oc);
	}

	return oc;
}

#define PRIV(o) (gb_object_get_priv ((o), cmdbutton_class ()))


/*
 * _GBECmdButton defines a VB Command Button
 *
 * back_color		Specifies the command button's background color.
 * cancel	 	Determines whether the command button gets a Click
 *			event if the user presses ESC.
 * caption		The text that appears on the command button.
 * def_resp		Determines if the command button respond to an Enter
 *			keypress even if another control has the focus.
 * enabled		Determines whether the command button is active.
 * font		 	Caption's font name, style and size.
 * height		The height of button in twips.
 * left			The number of twips from the command button's left
 *			edge to the Form window's left edge.
 * mouse_ptr		The shape of the mouse cursor when the user moves the
 *			mouse over the command button.
 * picture		The name of icon graphic image that appears on the
 *			command button as long as the Style property is set to
 *			1-Graphical.
 * style		Determines whether the command button appears as a
 *			standard Windows command button (if set to 0-Standard)
 *			or a command button with a color and possible picture
 *			(if set to 1-Graphical).
 * tab_index		Specifies the order of the command button in the focus
 *			order.
 * tab_stop		Determines whether the command button can receive the
 *			focus.
 * tool_tip_text	Holds the text that appears as a ToolTip at runtime.
 * top			Holds the number of twips from the command button's
 *			top edge to the Form window's top edge.
 * visible		Determines whether the command button appears or is
 *			hidden from the user.
 * width		The width of the command button in twips.
 *
 * Definitions:
 * twip		1,440th of an inch (the smallest screen measurement)
 *
 */

static GBRunObjProperty *p_caption       = NULL;
static GBRunObjProperty *p_tool_tip_text = NULL;
static GBRunObjProperty *p_enabled       = NULL;
static GBRunObjProperty *p_default       = NULL;

/**
 * cmdbutton_click
 *   @button
 *   @ec
 *
 *   Callback for 'click'.
 *
 **/
static void
cmdbutton_click (GtkWidget *button, GBRunEvalContext *ec)
{
	gbrun_form_item_invoke (ec, button, "_Click");
}

static void
cmdbutton_mousedown (GtkWidget *button, GBRunEvalContext *ec)
{
	gbrun_form_item_invoke (ec, button, "_MouseDown");
}

static void
cmdbutton_mouseup (GtkWidget *button, GBRunEvalContext *ec)
{
	gbrun_form_item_invoke (ec, button, "_MouseUp");
}

static void
copy (GBEvalContext  *ec,
      GBObject       *dest_obj,
      GBObject       *templ_obj)
{
	GtkWidget      *w;
	GBRunCmdButton *dest = PRIV (dest_obj);

	if (templ_obj)
		g_warning ("Copy unimplemented");

	w = gtk_button_new ();
	gbrun_form_item_set_widget (dest_obj, w);
	dest->label = GTK_LABEL (gtk_label_new ("WIBBLE"));
	gtk_widget_show (GTK_WIDGET (dest->label));

	gtk_container_add (GTK_CONTAINER (w),
			   GTK_WIDGET    (dest->label));

	/* FIXME: more signals etc. & leak fix */
	gtk_signal_connect (GTK_OBJECT (w), "clicked",
			    (GtkSignalFunc) cmdbutton_click, ec);
	gtk_signal_connect (GTK_OBJECT (w), "pressed",
				(GtkSignalFunc) cmdbutton_mousedown, ec);
	gtk_signal_connect (GTK_OBJECT (w), "released",
				(GtkSignalFunc) cmdbutton_mouseup, ec);

}

static void
destruct (GBObject *object)
{
	GBRunCmdButton *button = PRIV (object);

	if (button->real_caption)
		g_free (button->real_caption);

	g_warning ("Unimplemented %s destruct", ITEM_NAME);
}

/**
 * cmdbutton_setarg:
 *   @ec
 *   @object
 *   @property
 *   @val
 **/
static void
cmdbutton_setarg (GBRunEvalContext *ec,
		  GBRunObject      *object,
		  GBRunObjProperty *property,
		  GBValue          *val)
{
	GBRunCmdButton *button = PRIV (object);
	GtkWidget      *bwidget = gbrun_form_item_get_widget (object);

	g_return_if_fail (button != NULL);
	g_return_if_fail (button->label != NULL);

	if      (property == p_caption) {
		char *txt;

		button->real_caption = g_strdup (val->v.s->str);

		txt = gbrun_form_un_shortcutify (button->real_caption, NULL);
		gtk_label_set_text (button->label, txt);
		g_free (txt);

	} else if (property == p_tool_tip_text) {
		GtkTooltips *t = gtk_tooltips_new();
		gtk_tooltips_set_tip (t, GTK_WIDGET (bwidget),
		                      val->v.s->str, NULL);

	} else if (property == p_enabled) {
		gtk_widget_set_sensitive (GTK_WIDGET (bwidget),
					  val->v.bool);

	} else if (property == p_default) {
		/* FIXME: Is this the right function ? */
		gtk_widget_set (GTK_WIDGET (bwidget), "can_default", val->v.bool, NULL);
		g_warning ("FIXME: hacked");
/*		if (val->v.bool)
		gtk_window_set_default (button->item.form->window, button->item.widget);*/

	} else
		g_warning ("Unhandled property '%s'", property->name);
}


/**
 * cmdbutton_getarg:
 *   @ec
 *   @object
 *   @property
 **/
static GBValue *
cmdbutton_getarg (GBRunEvalContext *ec,
		  GBRunObject      *object,
		  GBRunObjProperty *property)
{
	GBRunCmdButton *button = PRIV (object);
	GtkWidget      *bwidget = gbrun_form_item_get_widget (object);

	g_return_val_if_fail (button != NULL, NULL);
	g_return_val_if_fail (button->label != NULL, NULL);

	if (property == p_caption)
		return gb_value_new_string_chars (button->real_caption);

	else if (property == p_enabled)
		return gb_value_new_boolean (GTK_WIDGET_SENSITIVE (bwidget));

	else if (property == p_default)
		/* Is this the right macro to use ? */
		return gb_value_new_boolean (GTK_WIDGET_RECEIVES_DEFAULT (bwidget));

	else
		g_warning ("Unhandled property '%s'", property->name);

	return NULL;
}

/**
 * gbrun_cmdbutton_register:
 *
 **/
void
gbrun_cmdbutton_register ()
{
	GBRunObjectPrivClass *cmdbutton;

	cmdbutton = gbrun_object_class_get_priv (cmdbutton_class ());

	cmdbutton->set_arg = cmdbutton_setarg;
	cmdbutton->get_arg = cmdbutton_getarg;

	p_caption       = gbrun_object_add_property_val (cmdbutton, "caption",
						         GB_VALUE_STRING);
	p_tool_tip_text = gbrun_object_add_property_val (cmdbutton, "tooltiptext",
						         GB_VALUE_STRING);
	p_enabled       = gbrun_object_add_property_val (cmdbutton, "enabled",
						         GB_VALUE_BOOLEAN);
	p_default       = gbrun_object_add_property_val (cmdbutton, "default",
							 GB_VALUE_BOOLEAN);
}


/**
 *  gbrun_cmdbutton_shutdown:
 **/
void
gbrun_cmdbutton_shutdown ()
{
}
