
/*
 * GNOME Basic Values Manuipulation
 *
 * Author:
 *    Michael Meeks <mmeeks@gnu.org>
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include <gbrun/gbrun.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-object.h>
#include <gbrun/gbrun-array.h>
#include <gbrun/gbrun-stack.h>

GBValue *
gbrun_value_promote_name (GBRunEvalContext *ec,
			  GBValue          *v,
			  const char       *type_name)
{
	GBValueType  t;
	GBValue     *ret;

	g_return_val_if_fail (v != NULL, NULL);
	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (type_name != NULL, NULL);

	t = gb_value_type_from_name (type_name);

	if (t == GB_VALUE_EMPTY) {
		/* It's an object => unpromotable; check type instead */
		if (v->type == GB_VALUE_OBJECT) {
			const char *obj_name = gbrun_object_name (v->v.obj);

			if (g_strcasecmp (obj_name, type_name))
				ret = gbrun_exception_firev (ec, "Can't promote %s to %s",
							     obj_name, type_name);
			else
				ret = gbrun_value_copy (ec, v);
		} else {
			g_warning ("wierd empty type with non-object");
			ret = NULL;
		}
	} else
		ret = gb_value_promote (GB_EVAL_CONTEXT (ec), v, t);

	return ret;
}

static GBValue *
objref_stack_deref (GBRunEvalContext *ec,
		    const GBObjRef   *ref)
{
	GBValue **val;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (ref != NULL, NULL);
	g_return_val_if_fail (!ref->parms, NULL);

	val = gbrun_stack_get (ec, ref->name);

	if (!val)
		return gbrun_exception_firev (ec, "No such variable %s",
					      ref->name);

	return gbrun_value_copy (ec, *val);
}

GBValue *
gbrun_objref_deref  (GBRunEvalContext *ec,
		     GBRunObject      *object,
		     const GBObjRef   *ref)
{
	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (ref != NULL, NULL);

	if (!object &&
	    !ref->method &&
	    !ref->parms)
		return objref_stack_deref (ec, ref);
	else {
		if (ref->method || ref->parms) {
			GBValue **t = gbrun_stack_get (ec, ref->name);
			if (t && (*t)->type == GB_VALUE_OBJECT &&
			    (*t)->v.obj && GBRUN_IS_ARRAY ((*t)->v.obj))
				return gbrun_array_deref (ec, (*t)->v.obj, ref);
			else
				return gbrun_method_invoke (ec, object, ref);
		} else
			return gbrun_object_get_arg (ec, object,
						     ref->name);
	}
}

static GBRunObject *
eval_to_penultimate (GBRunEvalContext *ec,
		     const GSList     *objref)
{
	const GSList *l;
	GBRunObject  *object = NULL;

	for (l = objref; l && l->next; l = l->next) {
		GBValue *nxt = gbrun_objref_deref (ec, object, l->data);

		if (!nxt)
			return FALSE;

		else if (nxt->type != GB_VALUE_OBJECT) {
			gbrun_exception_firev (ec, "Duff object dereference %s",
					       ((GBObjRef *)l->data)->name);
			return FALSE;
		}

		object = nxt->v.obj;
		gb_object_ref    (GB_EVAL_CONTEXT (ec), GB_OBJECT (object));
		gb_value_destroy (nxt);
	}

	return object;
}

gboolean
gbrun_eval_assign (GBRunEvalContext *ec,
		   const GSList     *objref,
		   GBValue          *value)
{
	GBRunObject    *object = NULL;
	const GBObjRef *ref;

	g_return_val_if_fail (ec != NULL, FALSE);
	g_return_val_if_fail (objref != NULL, FALSE);

	/* FIXME: With ? */
	if (!objref->next) { /* Plain stack variable */
		ref = objref->data;

		if (ref->parms || ref->method) {
			GBValue **t;

			t = gbrun_stack_get (ec, ref->name);
			if (t && (*t)->type == GB_VALUE_OBJECT &&
			    (*t)->v.obj && GBRUN_IS_ARRAY ((*t)->v.obj))
				return gbrun_array_assign (ec, (*t)->v.obj, ref, value);

			gbrun_exception_firev (ec, "Can't assign to %s", ref->name);
			return FALSE;
		}

		gbrun_stack_set (ec, ref->name, value);
		return TRUE;
	}

	object = eval_to_penultimate (ec, objref);
	if (!object) {
		g_warning ("Nothing penultimate: wierd");
		return FALSE;
	}
	
	ref = g_slist_last ((GSList *)objref)->data;
	if (ref->method) {
		gbrun_exception_firev (ec, "Methods can't be lvalues");
		return FALSE;
	}

	if (ref->parms) {
		if (GBRUN_IS_ARRAY (object))
			return gbrun_array_assign (ec, object, ref, value);
		else {
			gbrun_exception_firev (ec, "Unimplemented lvalue object dereference");
			return FALSE;
		}
	} else {
		gbrun_object_set_arg (ec, object, ref->name, value);
		return TRUE;
	}

	gbrun_exception_firev (ec, "Unimplemented lvalue object dereference");
	return FALSE;
}

GBValue *
gbrun_eval_objref  (GBRunEvalContext *ec,
		    const GBExpr     *expr)
{
	GSList      *objref;
	GBRunObject *object;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (expr != NULL, NULL);
	g_return_val_if_fail (expr->type == GB_EXPR_OBJREF, NULL);

	objref = expr->parm.objref;
	g_return_val_if_fail (objref != NULL, NULL);

	/* FIXME: With ? */
	object = eval_to_penultimate (ec, objref);

	return gbrun_objref_deref (ec, object, g_slist_last (objref)->data);
}
