
/*
 * GNOME Basic Object runtime ...
 *
 * Authors:
 *    Michael Meeks (mmeeks@gnu.org)
 *
 * Copyright 2000, Helix Code Inc.
 */
#include <math.h>
#include <setjmp.h>
#include <ctype.h>

#include <gbrun/gbrun.h>
#include <gbrun/gbrun-object.h>
#include <gbrun/gbrun-array.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-stack.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-statement.h>
#include <gbrun/gbrun-project.h>

#undef VB_METHOD_DEBUG

#define PRIV(o)        ((GBRunObjectPriv *)gb_object_get_priv ((o), gbrun_object_class ()))

GHashTable     *gbrun_object_hash = NULL;
GList          *gbrun_global_objects = NULL;

GtkObjectClass *gbrun_object_parent = NULL;

static void gbrun_object_add_variables (GBRunEvalContext *ec,
					GBRunObject      *obj,
					GHashTable       *vars);
GBRunObjectClass *
gbrun_object_lookup (const char *name)
{
	GBRunObjectClass *tmp;

	g_return_val_if_fail (name != NULL, FALSE);

	tmp = (GBRunObjectClass *) g_hash_table_lookup (gbrun_object_hash,
							(gconstpointer) name);

	return tmp;
}

GBRunObject *
gbrun_object_new (GBRunEvalContext *ec,
		  const char       *vb_name)
{
	GBRunObjectClass     *klass;
	GBRunObjectPriv      *priv;
	GBRunObjectPrivClass *pc;
	GBObject             *obj;

	klass = gbrun_object_lookup (vb_name);
	
	obj = gb_object_new (GB_EVAL_CONTEXT (ec),
			     klass);

	priv = PRIV (obj);
	if (priv) { /* Builtin objects don't inherit from GBRunObject */
		/*
		 * FIXME: some idiot could stick vars on all other class' we only
		 * get the GBRunObject's
		 */
		pc = GBRUN_OBJECT_PRIV_GET_CLASS (priv);
		if (pc->vars)
			gbrun_object_add_variables (ec, obj, pc->vars);
	}

	return obj;
}

void
gbrun_object_priv_copy (GBEvalContext *ec,
			GBObject      *copy,
			GBObject      *templ)
{
	GBRunObjectPriv *priv = PRIV (copy);

	if (templ == NULL) {
		priv->vars = NULL;
	} else if (priv->vars)
		g_warning ("FIXME: we need to copy vars.");
}

void
gbrun_object_priv_destruct (GBObject *object)
{
	GBRunObjectPriv *priv = PRIV (object);

	if (priv->vars) {
		gbrun_stack_level_destroy (priv->vars);
		priv->vars = NULL;
	}
}

/**
 * gbrun_object_register:
 * 
 * Return value: the newly created class object or NULL on failure.
 **/
void
gbrun_object_register (const char *vb_name, GBObjectClass *klass)
{
	GBRunObjectPrivClass *pc;

	g_return_if_fail (klass != NULL);

	if (!vb_name) /* Internal object */
		return;

	/* Setup name */
	pc = gbrun_object_class_get_priv (klass);
	if (pc->name)
		printf ("Registering '%s' over '%s'\n", vb_name, pc->name);
	else if (vb_name)
		pc->name = g_strdup (vb_name);
	
	if (g_hash_table_lookup (gbrun_object_hash, vb_name))
		g_warning ("Object '%s' already registered", vb_name);
	else
		g_hash_table_insert (gbrun_object_hash, (gpointer)vb_name, klass);
}

GBObjectClass *
gbrun_object_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBRunObjectPrivClass p;
		gb_object_priv_class_init (&p.priv, "--GBRun.Object--",
					   sizeof (GBRunObjectPriv),
					   gbrun_object_priv_copy,
					   gbrun_object_priv_destruct);
		gbrun_object_priv_class_init (&p);

		oc = gb_object_class_new_single (&p.priv, NULL);
	}
		
	return oc;
}

static void
make_object_vars (GBRunObjectPriv *priv)
{
	g_return_if_fail (priv != NULL);

	if (!priv->vars)
		/* FIXME: a more descriptive stack name would be good */
		priv->vars = gbrun_stack_level_new ("Object vars");
}

void
gbrun_object_var_set (GBRunEvalContext *ec,
		      GBRunObject      *obj,
		      const char       *name,
		      GBValue          *value)
{
	GBValue        **val;
	GBRunObjectPriv *priv;

	g_return_if_fail (obj != NULL);
	priv = PRIV (obj);
	g_return_if_fail (priv != NULL);

	make_object_vars (priv);

	val = gbrun_stack_level_lookup (priv->vars, name);
	if (!val) {
		gbrun_object_var_add (ec, obj, name, value);
	} else {
		if (*val)
			gb_value_destroy (*val);

		*val = gbrun_value_copy (ec, value);
	}
}

GBValue **
gbrun_object_var_get (GBRunEvalContext *ec,
		      GBRunObject      *obj,
		      const char       *name)
{
	GBValue        **val;
	GBRunObjectPriv *priv;

	g_return_val_if_fail (obj != NULL, NULL);
	priv = PRIV (obj);
	g_return_val_if_fail (priv != NULL, NULL);

	make_object_vars (priv);

	val = gbrun_stack_level_lookup (priv->vars, name);

	return val;
}

void
gbrun_object_var_add (GBRunEvalContext *ec,
		      GBRunObject      *obj,
		      const char       *name,
		      GBValue          *value)
{
	GBRunObjectPriv *priv;

	g_return_if_fail (obj != NULL);
	priv = PRIV (obj);
	g_return_if_fail (priv != NULL);
	g_return_if_fail (name != NULL);
	g_return_if_fail (value != NULL);

	make_object_vars (priv);

	gbrun_stack_level_add (ec, priv->vars, name, value);
}

typedef struct {
	GBRunEvalContext *ec;
	GBRunObject      *obj;
} setup_vars_closure_t;

void
setup_vars (gpointer key, gpointer value, gpointer user_data)
{
	setup_vars_closure_t *c = user_data;
	GBVar                *var = value;
	GBValueType           t;
	GBValue              *val;

	if (var->is_array)
		val = gb_value_new_object (GB_OBJECT (gbrun_array_new (c->ec, var)));
	else {
		if (var->object) {
			GBObject *obj = gbrun_object_new (c->ec, var->type);
			val = gb_value_new_object (obj);
		} else {
			t   = gb_value_type_from_name (var->type);
			if (t == GB_VALUE_EMPTY)
				gb_eval_exception_firev (GB_EVAL_CONTEXT (c->ec),
							 "Non base type '%s'",
							 var->type);
			else
				val = gb_value_new_default (GB_EVAL_CONTEXT (c->ec), t);
		}
	}

/*	g_warning ("Adding variable '%s' to object %s", var->name,
	gbrun_object_name (c->obj));*/

	gbrun_object_var_add (c->ec, c->obj, var->name, val);
}

static void
gbrun_object_add_variables (GBRunEvalContext *ec,
			    GBRunObject      *obj,
			    GHashTable       *vars)
{
	setup_vars_closure_t c;

	g_return_if_fail (obj != NULL);

	if (!vars)
		return;

	c.ec  = ec;
	c.obj = obj;

	g_hash_table_foreach (vars, setup_vars, &c);
}

static void
add_routine (gpointer key, gpointer value, gpointer user_data)
{
	GBRunObjectPrivClass *cl = user_data;

/*	g_warning ("Adding '%s' to object\n", ((GBRoutine *)value)->name);*/

	gbrun_object_add_routine (cl, value);
}

void
gbrun_object_add_routines (GBRunEvalContext     *ec,
			   GBRunObjectPrivClass *klass,
			   GHashTable           *routines)
{
	g_return_if_fail (ec != NULL);
	g_return_if_fail (klass != NULL);

	if (!routines)
		return;

	g_hash_table_foreach (routines, add_routine, klass);
}

GBObjectClass *
gbrun_object_create (const char *vb_name, GSList *parents)
{
	GBObjectClass *oc;

	if (!parents)
		parents = g_slist_append (NULL, gbrun_object_class ());

	if ((oc = gbrun_object_lookup (vb_name))) {
		g_warning ("re-creating '%s'", vb_name);
		return oc;

	} else { /* FIXME: leak */
		GBRunObjectPrivClass *p = g_new0 (GBRunObjectPrivClass, 1);

		gb_object_priv_class_init (&p->priv, vb_name,
					   sizeof (GBRunObjectPriv),
					   NULL, NULL);
		gbrun_object_priv_class_init (p);
		oc = gb_object_class_new (&p->priv, parents);
		gbrun_object_register (vb_name, oc);
	}

	if (!g_slist_find (oc->parents, gbrun_object_class ()->primary))
		g_error ("A GBRun object not inheriting correctly");
		
	return oc;
}

GBObjectClass *
gbrun_object_create_single (const char    *vb_name,
			    GBObjectClass *parent)
{
	GSList        *l;
	GBObjectClass *ret;

	if (parent)
		l = g_slist_prepend (NULL, parent);
	else
		l = NULL;

	ret = gbrun_object_create (vb_name, l);

	g_slist_free (l);

	return ret;
}

GBRunObjectPriv *
gbrun_object_get_priv (GBObject *object)
{
	GBObjectClass *klass = GBRUN_OBJECT_GET_CLASS (object);

	return gb_object_get_priv (object, klass);
}

GBRunObjectPrivClass *
gbrun_object_class_get_priv (GBObjectClass *klass)
{
	return (GBRunObjectPrivClass *)klass->primary;
}

/*
 * FIXME: ugly, sluggish, tedious.
 */
GBValue *
parse_def (const char       *str,
	   GBValueType       to)
{
	GBValue          *def_val, *tmp;
	GBRunEvalContext *ec;
	
	tmp = gb_value_new_string_chars (str);
	ec  = gbrun_eval_context_new ("Default arg promotion", GBRUN_SEC_HARD);

	def_val = gb_value_promote (GB_EVAL_CONTEXT (ec), tmp, to);
	gb_value_destroy (tmp);

	if (gbrun_eval_context_exception (ec))
		return NULL;

	gtk_object_destroy (GTK_OBJECT (ec));

	return def_val;
}

inline static GBArgDesc *
parse_arg (const char *arg_txt)
{
	GBArgDesc    *arg;
	gchar       **txt;
	int           i, max;
	const GBExpr *def_expr = NULL;
	gboolean      by_val   = FALSE;
	gboolean      optional = FALSE;

	txt = g_strsplit (arg_txt, ",", -1);

/*
 * txt [0]: Name
 * txt [1]: Type
 * txt [2]: [byref]
 * txt [3]: [default value] ( implies optional )
 */

	for (i = 0; txt [i]; i++);
	max = i;
	
	if (max < 2)
		g_error ("We must have 'name,type' at minimum in '%s'", arg_txt);

	if (txt [2]) {
		if (!g_strcasecmp (txt [2], "byref"))
			by_val = FALSE;
		
		if (txt [3]) { /* Extract default */
			GBValueType t = gb_value_type_from_name (txt [1]);
			GBValue *def_value;
			
			if (t == GB_VALUE_EMPTY)
				g_error ("Invalid type for a default '%s'", txt [1]);

			if (g_strcasecmp (txt [3], "NULL")) {
				def_value = parse_def (txt [3], t);
				if (!def_value)
					g_error ("Syntax error in '%s', trying to promote "
						 "'%s' to type '%s'", arg_txt, txt [3], txt [1]);
				def_expr = gb_expr_new_value (def_value);
			}
			
			optional = TRUE;
		}
	}

	arg = gb_arg_desc_new (txt [0], txt [1], def_expr,
			       by_val, optional);

	g_strfreev (txt);

	return arg;
}

static GBRunSecurityFlag
parse_security (const char *txt)
{
	GBRunSecurityFlag f;

	while (*txt) {
		switch (*txt) {
		case 'n':
		case 'N':
			f = GBRUN_SEC_NONE;
			break;

		case 'x':
		case 'X':
			f |= GBRUN_SEC_EXEC;
			break;

		case 'g':
		case 'G':
			f |= GBRUN_SEC_GUI;
			break;

		case 'w':
		case 'W':
			f |= GBRUN_SEC_WINE;
			break;

		case 'i':
		case 'I':
			f |= GBRUN_SEC_IO;
			break;

		default:
			g_warning ("Unknown security type '%c'\n", *txt);
			break;
		}
		txt++;
	}
	
	return 0;
}

static void
parse_arg_desc (GBRunObjMethod *m, const char *arg_desc,
		gboolean vararg)
{
	gchar  **txt;
	int      i, max;

	g_return_if_fail (arg_desc != NULL);

	m->args = NULL;
	m->min_args = 0;
	m->max_args = 0;

	txt = g_strsplit (arg_desc, ";", -1);

	for (i = 0; txt [i]; i++);
	max = i;

	if (max < 3)
		g_error ("Must have at least 3 sections in '%s'", arg_desc);

	g_strchomp (txt [0]);
	if (!g_strcasecmp (txt [0], "sub")) {
		m->is_sub = TRUE;
	} else if (!g_strcasecmp (txt [0], "func")) {
		m->is_sub = FALSE;
	} else
		g_error ("Incorrect sub/func specifier '%s'", txt [0]);

	m->name = g_strdup (txt [1]);
	m->mask = parse_security (txt [max - 1]);

	for (i = max - (m->is_sub?1:2) - 1; i >= 2; i--) {
		GBArgDesc *arg;

		if (txt [i] [0] == '.')
			break;

		arg = parse_arg (txt [i]);
		
		m->args = g_slist_prepend (m->args, arg);
		if (!arg->optional)
			m->min_args++;
		m->max_args++;
	}

	if (vararg)
		m->max_args = G_MAXINT;	

	g_strfreev (txt);
}

/**
 * gbrun_object_add_method_var:
 * @klass:  the class to add it to.
 * @desc:   the method's description.
 * @method: Method pointer.
 * 
 *  Add a variable no. of arguments method to
 * an existing class.
 **/
void
gbrun_object_add_method_var (GBRunObjectPrivClass  *klass,
			     const char   *desc,
			     GBRunMethodVar *method)
{
	GBRunObjMethod *m;

	g_return_if_fail (desc != NULL);
	g_return_if_fail (klass != NULL);

	m              = g_new0   (GBRunObjMethod, 1);
	m->type        = GBRUN_METHOD_VAR;
	m->handler.var = method;
	parse_arg_desc (m, desc, TRUE);
	
	klass->methods = g_slist_prepend (klass->methods, m);
}

/**
 * gbrun_object_add_method_arg:
 * @klass:  the class to add it to.
 * @desc:   the method's description.
 * @method: Method pointer.
 * 
 * Add a method to an existing class.
 **/
void
gbrun_object_add_method_arg (GBRunObjectPrivClass  *klass,
			     const char   *desc,
			     GBRunMethodArg *method)
{
	GBRunObjMethod *m;

	g_return_if_fail (desc != NULL);
	g_return_if_fail (klass != NULL);

	m              = g_new0   (GBRunObjMethod, 1);
	m->type        = GBRUN_METHOD_ARG;
	m->handler.arg = method;
	parse_arg_desc (m, desc, FALSE);
	
	klass->methods = g_slist_prepend (klass->methods, m);
}

/**
 * gbrun_object_add_routine:
 * @klass:   the class to add it to.
 * @routine: the routine structure
 * 
 * Add a VB routine to an existing class.
 **/
void
gbrun_object_add_routine (GBRunObjectPrivClass  *klass,
			  GBRoutine    *routine)
{
	GBRunObjMethod *m;

	g_return_if_fail (klass != NULL);
	g_return_if_fail (routine != NULL);

	m              = g_new0   (GBRunObjMethod, 1);

	m->type        = GBRUN_METHOD_VB;
	m->name        = g_strdup (routine->name);
	m->args        = g_slist_copy (routine->args);
	m->handler.vb  = routine;
	/* FIXME: need to split parse_args code */
	m->max_args    = g_slist_length (routine->args);
	m->min_args    = m->max_args;
	/* No pure VB can have security implications */
	m->mask        = 0;
	m->is_sub      = !routine->is_function;
	
	klass->methods = g_slist_prepend (klass->methods, m);
}

GBRunObjProperty *
gbrun_object_add_property_val (GBRunObjectPrivClass *klass,
			       const char  *name,
			       GBValueType  type)
{
	GBRunObjProperty *p;

	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (klass != NULL, NULL);

	p              = g_new0   (GBRunObjProperty, 1);
	p->idx         = 0; /* FIXME */
	p->get_arg     = klass->get_arg;
	p->set_arg     = klass->set_arg;

	/* It helps if it is settable */
	g_assert (p->get_arg || p->set_arg);

	p->type        = GBRUN_PROP_VAL;
	p->name        = g_strdup (name);
	p->u.prop_type = type;
	
	klass->properties = g_list_prepend (klass->properties, p);

	return p;
}

GBRunObjProperty *
gbrun_object_add_property_obj (GBRunObjectPrivClass *klass,
			       const char  *name,
			       const char  *objname)
{
	GBRunObjProperty *p;

	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (klass != NULL, NULL);
	g_return_val_if_fail (objname != NULL, NULL);

	p            = g_new0   (GBRunObjProperty, 1);
	p->idx       = 0; /* FIXME */
	p->get_arg   = klass->get_arg;
	p->set_arg   = klass->set_arg;

	/* It helps if it is settable */
	g_assert (p->get_arg || p->set_arg);

	p->type      = GBRUN_PROP_OBJ;
	p->name      = g_strdup (name);
	p->u.objname = g_strdup (objname);
	
	klass->properties = g_list_prepend (klass->properties, p);

	return p;
}

GBRunObjMethod *
gbrun_object_get_method (GBRunObjectClass *klass,
			 const char       *name)
{
	GSList *l, *i;
	GSList *privs = g_slist_reverse (g_slist_copy (klass->parents));

	for (i = privs; i; i = i->next) {
		GBRunObjectPrivClass *pc = i->data;

		for (l = pc->methods; l; l = l->next) {
			GBRunObjMethod *m = l->data;
			
			if (!g_strcasecmp (m->name, name))
				return m;
		}
	}
	g_slist_free (privs);

	return NULL;
}

GSList *
gbrun_object_get_methods (GBObjectClass *klass)
{
	GSList *ans = NULL;
	GSList *l, *i;
	GSList *privs;

	g_return_val_if_fail (klass != NULL, NULL);

	privs = g_slist_reverse (g_slist_copy (klass->parents));

	for (i = privs; i; i = i->next) {
		GBRunObjectPrivClass *pc = i->data;

		for (l = pc->methods; l; l = l->next) {
			ans = g_slist_prepend (ans, l->data);
		}
	}

	return g_slist_reverse (ans);
}

static GBRunObjProperty *
get_class_prop (GBRunObjectPrivClass *pc,
		const char           *name)
{
	GList *l;
	
	for (l = pc->properties; l; l = l->next) {
		GBRunObjProperty *p = l->data;
		
		if (!g_strcasecmp (p->name, name))
			return p;
	}

	return NULL;	
}

GBRunObjProperty *
gbrun_object_get_property (GBRunObjectClass      *klass,
			   const char            *name)
{
	GSList *l, *privs = g_slist_reverse (g_slist_copy (klass->parents));

	for (l = privs; l; l = l->next) {
		GBRunObjectPrivClass *pc = l->data;
		GBRunObjProperty     *p;
		
		if ((p = get_class_prop (pc, name)))
			return p;
	} 

	return NULL;
}

void
gbrun_object_globalize (GBObject *klass)
{
	gbrun_global_objects = g_list_prepend (gbrun_global_objects, klass);
}

void
gbrun_object_priv_class_init (GBRunObjectPrivClass *pc)
{
	pc->vars       = NULL;

	pc->name       = NULL;
	pc->properties = NULL;
	pc->methods    = NULL;

	pc->set_arg    = NULL;
	pc->get_arg    = NULL;
}

/**
 * gbrun_object_global_func:
 * @name: 
 * 
 *  This returns both functions & naff cretinous unscoped
 * object methods such as 'Debug.print'.
 * 
 * Return value: 
 **/
GBRunObjMethod *
gbrun_object_global_func (const char *name)
{
	GBRunObjMethod *m = NULL;
	GList        *l;

	for (l = gbrun_global_objects; l; l = l->next) {
		GBRunObjectClass *klass;

		klass = GBRUN_OBJECT_GET_CLASS (l->data);
		if ((m = gbrun_object_get_method (klass, name)))
			break;
	}

	return m;
}

const char *
gbrun_object_name (GBRunObject *object)
{
	GBRunObjectClass     *klass;
	GBRunObjectPrivClass *pc;

	g_return_val_if_fail (GB_IS_OBJECT (object), "Non GB object");

	klass = GBRUN_OBJECT_GET_CLASS (object);

	pc = gbrun_object_class_get_priv (klass);

	return pc->name;
}

/*
 * End of Class maniupulators, start of object manipulators.
 */ 

void
gbrun_object_set_arg (GBRunEvalContext *ec,
		      GBRunObject      *object,
		      const char       *prop_name,
		      GBValue          *val)
{
	GBRunObjProperty     *prop;
	GBValue              *v;

	g_return_if_fail (ec != NULL);
	g_return_if_fail (val != NULL);
	g_return_if_fail (object != NULL);
	g_return_if_fail (prop_name != NULL);
	g_return_if_fail (GB_IS_OBJECT (object));

	prop = gbrun_object_get_property (GBRUN_OBJECT_GET_CLASS (object), prop_name);

	if (!prop || !prop->set_arg) {
		gbrun_exception_firev (ec, "No writeable property '%s' on object '%s'",
				       prop_name, gbrun_object_name (object));
		return;
	}

	if (prop->type == GBRUN_PROP_VAL) {
		v = gbrun_value_promote (ec, val, prop->u.prop_type);

		if (!v)
			return;

		prop->set_arg (ec, object, prop, v);

		gb_value_destroy (v);
	} else {
		g_warning ("FIXME: untested object setting ...");
		prop->set_arg (ec, object, prop, val);
	}
}

GBValue *
gbrun_object_get_arg (GBRunEvalContext *ec,
		      GBRunObject      *object,
		      const char       *prop_name)
{
	GBRunObjProperty     *prop;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (prop_name != NULL, NULL);
	g_return_val_if_fail (GB_IS_OBJECT (object), NULL);

	prop = gbrun_object_get_property (GBRUN_OBJECT_GET_CLASS (object), prop_name);

	if (!prop || !prop->get_arg)
		return gbrun_exception_firev (ec, "No reaable property '%s' on object '%s'",
					      prop_name, gbrun_object_name (object));
	
	return prop->get_arg (ec, object, prop);
}

static GBValue *
gbrun_method_invoke_arg (GBRunEvalContext *ec, GBRunObject      *obj,
			 GBRunObjMethod   *m,  const GBObjRef *func)
{
	GBValue         *val;
	GBValue        **vals;
	GSList          *lp, *lm;
	int              i = 0;
	int              num_opt;
	int              len;
	gboolean         exception;

	g_return_val_if_fail (m != NULL, NULL);
	g_return_val_if_fail (func != NULL, NULL);

	len = g_slist_length (func->parms);

	if (len < m->min_args)
		return gbrun_exception_firev (ec, "Too few args to %s %d expected %d given",
					    func->name, m->min_args, len);
	if (len > m->max_args)
		return gbrun_exception_firev (ec, "Too many args to %s %d expected %d given",
					    func->name, m->max_args, len);
	
	vals = g_new (GBValue *, m->max_args + 1);

	/*
	 * FIXME: see instr for something that will cock this up; we need to work
	 * out how many optional args to use, and use them as we go along.
	 */
	num_opt = len - m->min_args;
	exception = FALSE;
	lp = func->parms;
	for (lm = m->args; lm; i++, lm = lm->next) {
		GBArgDesc *arg = lm->data;
		GBExpr    *expr;

		if (lp) {
			expr = lp->data;
			lp   = lp->next;
		} else if (arg->optional)
			expr = arg->def_value;

		if (expr) {
			GBValue *v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), expr);
			
			if (!v)
				exception = TRUE;
			else {
				vals [i] = gbrun_value_promote_name (ec, v, arg->type_name);
				gb_value_destroy (v);
			}
		} else
			vals [i] = NULL;
	}
	vals [i] = NULL;
	
	if (!exception)
		val = m->handler.arg (ec, obj, vals);
	else
		val = NULL;
	
	i = 0;
	while (vals [i])
		gb_value_destroy (vals [i++]);
	
	g_free (vals);

	return val;
}

static GBValue *
gbrun_method_invoke_var (GBRunEvalContext *ec, GBRunObject      *obj,
			 GBRunObjMethod   *m,  const GBObjRef *func)
{
	int              len;

	g_return_val_if_fail (m != NULL, NULL);
	g_return_val_if_fail (func != NULL, NULL);

	len = g_slist_length (func->parms);

	if (len < m->min_args)
		return gbrun_exception_firev (ec, "Too few args to %s %d expected %d given",
					    func->name, m->min_args, len);

	return m->handler.var (ec, obj, func->parms); 
}

void
stack_setup_vars (gpointer key, gpointer value, gpointer user_data)
{
	GBRunEvalContext *ec  = user_data;
	GBVar          *var = value;
	GBValueType     t;
	GBValue        *val;

	if (var->is_array)
		val = gb_value_new_object (GB_OBJECT (gbrun_array_new (ec, var)));
	else {
		if (var->object) {
			GBObject *obj = gbrun_object_new (ec, var->type);
			val = gb_value_new_object (obj);
		} else {
			t   = gb_value_type_from_name (var->type);
			if (t == GB_VALUE_EMPTY)
				gb_eval_exception_firev (GB_EVAL_CONTEXT (ec),
							 "Non base type '%s'", var->type);
			else
				val = gb_value_new_default (GB_EVAL_CONTEXT (ec), t);
		}
	}

/*	g_warning ("Adding variable '%s' at stack level %d", var->name,
	g_list_length (ec->stack->level));*/
	gbrun_stack_set (ec, var->name, val);
}

static GBValue *
gbrun_method_invoke_vb (GBRunEvalContext *ec, GBRunObject      *obj,
			GBRunObjMethod   *m,  const GBObjRef *func)
{
	GBValue     *ans = NULL, **ans_var;
	GSList      *l;
	GSList      *def;
	GSList      *parms;
	GBRunObject *old_me;
	int          len;

	g_return_val_if_fail (m != NULL, NULL);
	g_return_val_if_fail (func != NULL, NULL);
	g_return_val_if_fail (func->name != NULL, NULL);

	len = g_slist_length (func->parms);

	if (len < m->min_args)
		return gbrun_exception_firev (ec, "Too few args to %s %d expected %d given",
					    func->name, m->min_args, len);
	if (len > m->max_args)
		return gbrun_exception_firev (ec, "Too many args to %s %d expected %d given",
					    func->name, m->max_args, len);

	/* Set 'Me' up */
	old_me = gbrun_eval_context_me_get (ec);
	gbrun_eval_context_me_set (ec, obj);

#ifdef VB_METHOD_DEBUG
	fprintf (stderr, "Calling %s (", func->name?func->name:"[Unknown]");
#endif
	/*
	 * Evaluate the arguments in the current stack frame.
	 */
	parms = NULL;
	for (l = func->parms; l; l = l->next) {
		GBValue   *val;

		val = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), l->data);
		if (!val)
			goto gb_cleanup;

		parms = g_slist_append (parms, val);
	}
	
	/*
	 * Create the callee's frame.
	 */
	gbrun_stack_call (ec, func->name);

	def = m->args;
	l   = parms;
	while (l && def) {
		GBArgDesc *arg = def->data;
		GBValue   *v;

		v = gbrun_value_promote_name (ec, l->data, arg->type_name);
		if (v) {
#ifdef VB_METHOD_DEBUG
			fprintf (stderr, "%s%s'%s'%s",arg->name?arg->name:"",
				 arg->name?"=":"", gb_value_get_as_string (v)->str,
				 l->next?", ":"");
#endif
			gbrun_stack_add (ec, arg->name, v, GBRUN_STACK_LOCAL);
			gb_value_destroy (v);
		} else
			goto gb_cleanup;

		l = l->next;
		def = def->next;
	}
	
#ifdef VB_METHOD_DEBUG
	fprintf (stderr, ")\n");
#endif

	/* Setup the functions variables */
	if (m->handler.vb->variables)
		g_hash_table_foreach (m->handler.vb->variables,
				      stack_setup_vars, ec);
	if (gbrun_eval_context_exception (ec))
		goto gb_cleanup;

	if (!gbrun_stmts_evaluate (ec, m->handler.vb->body))
		goto gb_cleanup;

	/* Get the return value */
	ans_var = gbrun_stack_get (ec, func->name);
	if (!ans_var || !*ans_var) {
		if (m->is_sub)
			ans = gb_value_new_empty ();
		else
			gbrun_exception_firev (ec, "No return value for function '%s'",
					       func->name);
	} else
		ans = gbrun_value_copy (ec, *ans_var);

#ifdef VB_METHOD_DEBUG
	if (ans && ans->type != GB_VALUE_EMPTY)
		fprintf (stderr, "%s returns '%s'\n", func->name,
			 gb_value_get_as_string (ans)->str);
#endif

gb_cleanup:	
	for (l = parms; l; l = g_slist_remove (l, l->data))
		gb_value_destroy (l->data);

	gbrun_stack_return (ec);
	gbrun_eval_context_me_set (ec, old_me);

	return ans;
}

static GBRunObjMethod *
get_method (GBRunEvalContext *ec, GBRunObject **obj, const char *name)
{
	GBRunObjMethod *m = NULL;
	GBRunProject   *proj;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (obj != NULL, NULL);

	if (!*obj)
		*obj = gbrun_eval_context_me_get (ec);

	if (*obj)
		m = gbrun_object_get_method (GBRUN_OBJECT_GET_CLASS (*obj), name);

	if (m)
		return m;

	if ((proj = gbrun_eval_context_proj_get (ec))) {

		GSList *l = gbrun_project_get_modules (proj);

		while (l) {
			*obj = l->data;

			m = gbrun_object_get_method (GBRUN_OBJECT_GET_CLASS (*obj), name);
			if (m)
				return m;

			l = l->next;
		}
	}

	*obj = NULL;

	return gbrun_object_global_func (name);
}

GBValue *
gbrun_method_invoke (GBRunEvalContext *ec, GBRunObject *obj,
		     const GBObjRef *func)
{
	GBValue        *val = NULL;
	GBRunObjMethod *m;

	if (obj && GBRUN_IS_ARRAY (obj))
		return gbrun_array_deref (ec, obj, func);

	if (!(m = get_method (ec, &obj, func->name)))
		return gbrun_exception_firev (ec, "No '%s' method / function",
					    func->name);

	if (ec->flags & m->mask)
		return gbrun_exception_firev (ec, "Security block on function '%s'",
					    func->name);

	switch (m->type) {
	case GBRUN_METHOD_ARG:
		val = gbrun_method_invoke_arg (ec, obj, m, func);
		break;

	case GBRUN_METHOD_VAR:
		val = gbrun_method_invoke_var (ec, obj, m, func);
		break;

	case GBRUN_METHOD_VB:
		val = gbrun_method_invoke_vb  (ec, obj, m, func);
		break;
	}

	return val;
}

static gint
g_str_case_equal (gconstpointer v, gconstpointer v2)
{
	  return g_strcasecmp ((const gchar*) v, (const gchar*)v2) == 0;
}

static guint
g_str_case_hash (gconstpointer v)
{
	const char *s = (char*)v;
	const char *p;
	guint h=0, g;
	
	for (p = s; *p != '\0'; p += 1) {
		h = ( h << 4 ) + tolower (*p);
		if ( ( g = h & 0xf0000000 ) ) {
			h = h ^ (g >> 24);
			h = h ^ g;
		}
	}
	
	return h /* % M */;
}

void
gbrun_object_init (GBEvalContext *ec)
{
	gbrun_object_hash = g_hash_table_new (g_str_case_hash, g_str_case_equal);
}

void 
gbrun_object_shutdown ()
{
	/* FIXME: For now, just do this */
	/* Later, each object's destructor would have to be called, I guess */

	g_hash_table_destroy (gbrun_object_hash);
}
