
/*
 * GNOME Basic Array support
 *
 * Authors:
 *    Michael Meeks (mmeeks@gnu.org)
 *
 * Copyright 2000, Helix Code Inc.
 */
#include <math.h>
#include <setjmp.h>

#include <gbrun/gbrun.h>
#include <gbrun/gbrun-array.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-stack.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-statement.h>

#undef ARRAY_DEBUG

#define PRIV(o) ((GBRunArray *)gb_object_get_priv ((o), gbrun_array_get_class ()))

typedef struct {
	GBLong min, max;
} GBRunARange;

static void
destruct (GBObject *o)
{
	g_warning ("Array destruction unimplemented");
}

static GBValue *
array_deref (GBRunEvalContext *ec,
	     GBObject         *obj,
	     const GBObjRef   *ref,
	     const GBValue    *assign)
{
	gpointer   *data;
	GBValue   **pos = NULL;
	GSList     *i, *offset;
	GBRunArray *a;

	g_return_val_if_fail (obj != NULL, NULL);
	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (ref != NULL, NULL);
	g_return_val_if_fail (GBRUN_IS_ARRAY (obj), NULL);

	a = PRIV (obj);

	if (g_slist_length (a->indicees) !=
	    g_slist_length (ref->parms))
		return gbrun_exception_firev (ec, "Too many / few array indicees");

	i      = a->indicees;
	data   = a->data;
	offset = ref->parms;

#ifdef ARRAY_DEBUG
	fprintf (stderr, "Array index : %s (", (ref->name)?(ref->name):"[Unknown]");
#endif

	while (i && offset) {
		GBRunARange *r = i->data;
		GBValue *v;
		int      into;
		
		v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), offset->data);
		if (!v)
			return NULL;
		into = gb_value_get_as_int (v);
		gb_value_destroy (v);
		
		if (into < r->min ||
		    into > r->max)
			return gbrun_exception_firev (ec, "Out of bounds array index "
						      "%d !<= %d !<= %d", r->min, into, r->max);

		pos    = (GBValue **)&(data [into - r->min]);
		data   = data [into - r->min];
#ifdef ARRAY_DEBUG
		fprintf (stderr, "%d%s", into, i->next?", ":"");
#endif
		offset = offset->next;
		i      = i->next;
	}

#ifdef ARRAY_DEBUG
	if (assign)
		fprintf (stderr, ") := '%s'\n",
			 gb_value_get_as_string (assign)->str);
	else
		fprintf (stderr, ") == '%s'\n",
			 gb_value_get_as_string ((GBValue *)data)->str);
#endif

	if (assign) {
		if (!pos)
			return gbrun_exception_firev (ec, "Wierd, nowhere  to assign");
		if (*pos)
			gb_value_destroy (*pos);

		*pos = gb_value_copy (GB_EVAL_CONTEXT (ec), assign);

		return gb_value_new_empty ();
	}

	if (((GBValue *)data)->type > GB_VALUE_TYPE_MAX)
		return gbrun_exception_firev (ec, "Serious internal error in array dereference");

	return gb_value_copy (GB_EVAL_CONTEXT (ec), (GBValue *)data);
}

/*
static void
gbrun_array_class_init (GBRunArrayClass *klass)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*) klass;
	object_class->destroy = gbrun_array_destroy_real;

	klass->dereference = array_deref;
}

GtkType
gbrun_array_get_type (void)
{
	static GtkType gbrun_array_type = 0;

	if (!gbrun_array_type) {
		static const GtkTypeInfo gbrun_array_info = {
			"GBRunArray",
			sizeof (GBRunArray),
			sizeof (GBRunArrayClass),
			(GtkClassInitFunc) gbrun_array_class_init,
			NULL, NULL, NULL,
			(GtkClassInitFunc) NULL,
		};

		gbrun_array_type = gtk_type_unique (GB_TYPE_OBJECT,
						    &gbrun_array_info);
	}
	
	return gbrun_array_type;	
}*/

static gboolean
get_as_long (GBRunEvalContext *ec,
	     const GBExpr     *expr,
	     GBLong           *ans)
{
	gboolean ret;
	GBValue *v, *i;

	v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), expr);
	if (!v) {
		gb_value_destroy (v);
		return FALSE;
	}
	
	i = gbrun_value_promote (ec, v, GB_VALUE_LONG);
	if (!i || i->type != GB_VALUE_LONG)
		ret = FALSE;
	else
		ret = TRUE;

	*ans = i->v.l;

	gb_value_destroy (v);
	gb_value_destroy (i);

	return ret;
}

static GBRunARange *
range_create (GBRunEvalContext *ec,
	      GBIndex          *idx)
{
	GBRunARange *r = g_new (GBRunARange, 1);

	if (!get_as_long (ec, idx->min, &r->min) ||
	    !get_as_long (ec, idx->max, &r->max)) {
		g_free (r);
		return NULL;
	}

	if (r->min > r->max) {
		GBLong t = r->min;
		g_warning ("Testme: swapping indicees");
		r->min = r->max;
		r->max = t;
	}

	return r;
}

static gpointer
alloc_array (GBRunEvalContext *ec,
	     GSList           *l,
	     GBRunArray       *a)
{
	GBRunARange *r;
	GBLong       size, i;
	gpointer    *data;

	if (!l) {
		if (a->type == GBRUN_ARRAY_OBJECT)
			return gb_value_new_empty ();
		else
			return gb_value_new_default (GB_EVAL_CONTEXT (ec),
						     a->t.value);
	}
	r = l->data;

	size = r->max - r->min + 1;
	if (size < 0)
		size = -size;

	data = g_new (gpointer, size);

	for (i = 0; i < size; i++)
		data [i] = alloc_array (ec, l->next, a);

	return data;
}

GBObject *
gbrun_array_new (GBRunEvalContext *ec,
		 const GBVar      *var)
{
	GBObject   *obj;
	GBRunArray *a;
	GSList     *l;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (var != NULL, NULL);
	g_return_val_if_fail (GB_IS_EVAL_CONTEXT (ec), NULL);

	obj = gbrun_object_newc (ec, gbrun_array_get_class ());

	a = PRIV (obj);

	if (var->object) {
		a->type = GBRUN_ARRAY_OBJECT;
		a->t.object = g_strdup (var->type);
	} else {
		a->type = GBRUN_ARRAY_VALUE;
		a->t.value = gb_value_type_from_name (var->type);
		if (a->t.value == GB_VALUE_EMPTY) { /* FIXME: leak */
			gbrun_exception_firev (ec, "Invalid type '%s'", var->type);
			return NULL;
		}
	}

	/* Evaluate indicees */
	a->indicees = NULL;
	for (l = var->indicees; l; l = l->next) {
		GBRunARange *r = range_create (ec, l->data);

		if (!r) /* FIXME: leak */
			return NULL;

		a->indicees = g_slist_append (a->indicees, r);
	}

	a->data = alloc_array (ec, a->indicees, a);

	return obj;
}

GBObject *
gbrun_array_new_vals (GBRunEvalContext *ec,
		      GSList           *values)
{
	GBObject    *obj;
	GBRunArray  *a;
	GBRunARange *r;
	GBValue     *v;
	GBValue    **data;
	int          i;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (values != NULL, NULL);
	g_return_val_if_fail (values->data != NULL, NULL);

	obj = gbrun_object_newc (ec, gbrun_array_get_class ());

	a = PRIV (obj);

	r = g_new0 (GBRunARange, 1);
	r->min = 0;
	r->max = g_slist_length (values) - 1;

	a->indicees = g_slist_append (NULL, r);

	v = values->data;

	if (v->type == GB_VALUE_OBJECT) {
		a->type = GBRUN_ARRAY_OBJECT;
		a->t.object = g_strdup (gbrun_object_name (v->v.obj));
	} else {
		a->type = GBRUN_ARRAY_VALUE;
		a->t.value = v->type;
	}	

	data = g_new (GBValue *, r->max + 1);
	a->data = (gpointer)data;

	for (i = 0; i < r->max + 1; i++) {
		data [i] = gbrun_value_promote (ec, values->data, a->t.value);
		values = values->next;
	}

	return obj;
}

GBValue *
gbrun_array_deref (GBRunEvalContext *ec,
		   GBObject         *obj,
		   const GBObjRef   *func)
{
	GBRunArrayClass *klass;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (obj != NULL, NULL);
	g_return_val_if_fail (func != NULL, NULL);
	g_return_val_if_fail (GBRUN_IS_ARRAY (obj), NULL);

	klass = (GBRunArrayClass *) GB_OBJECT_PRIV_GET_CLASS (PRIV (obj));
	
	return klass->dereference (ec, obj, func, NULL);
}

gboolean
gbrun_array_assign (GBRunEvalContext *ec,
		    GBRunObject      *obj,
		    const GBObjRef   *func,
		    const GBValue    *value)
{
	GBValue    *ans;
	GBRunArrayClass *klass;

	g_return_val_if_fail (ec != NULL, FALSE);
	g_return_val_if_fail (obj != NULL, FALSE);
	g_return_val_if_fail (func != NULL, FALSE);
	g_return_val_if_fail (value != NULL, FALSE);
	g_return_val_if_fail (GBRUN_IS_ARRAY (obj), FALSE);

	if (value->type > GB_VALUE_TYPE_MAX) {
		gbrun_exception_firev (ec, "Assigning non value to array");
		return FALSE;
	}

	klass = (GBRunArrayClass *) GB_OBJECT_PRIV_GET_CLASS (PRIV (obj));
	
	ans = klass->dereference (ec, obj, func, value);

	if (!ans)
		return FALSE;

	if (ans->type > GB_VALUE_TYPE_MAX) {
		gbrun_exception_firev (ec, "Serious internal error in array assignment");
		return FALSE;
	}

	gb_value_destroy (ans);

	return TRUE;
}

GBObjectClass *
gbrun_array_get_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBRunArrayClass p;
		gb_object_priv_class_init (&p.priv, "--GB.Array--",
					   sizeof (GBRunArray),
					   NULL,
/*					   (GBObjectCopy *)copy,*/
					   (GBObjectDestructor *)destruct);
		p.dereference = array_deref;
		oc = gb_object_class_new_single (&p.priv, NULL);
	}
		
	return oc;
}
