/*
 * Gnome Basic Object system test program.
 *
 * Author:
 *     Michael Meeks (mmeeks@gnu.org)
 *
 * Copyright 2000, Helix Code, Inc.
 */
#include "config.h"

#include <sys/types.h>
#include <sys/mman.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <gb/libgb.h>
#include <gb/gb-object.h>

typedef struct {
	GBObjectPriv priv;

	char *name;
} MethodA;

typedef struct {
	GBObjectPriv priv;

	int  num;
} MethodB;

static GBObjectClass *method_a_class (void);
static GBObjectClass *method_b_class (void);

static void
a_cc (GBEvalContext *ec,
      GBObject      *dest,
      GBObject      *templ)
{
	MethodA *a = gb_object_get_priv (dest, method_a_class ());

	if (templ) {
		MethodA *b = gb_object_get_priv (templ, method_a_class ());

		a->name = g_strdup (b->name);
	} else
		a->name = NULL;
}

static void
a_destr (GBObject *object)
{
	MethodA *a = gb_object_get_priv (object, method_a_class ());

	if (a->name)
		g_free (a->name);
}

GBObjectClass *
method_a_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBObjectPrivClass p;
		gb_object_priv_class_init (&p, "Method A", sizeof (MethodA),
					   a_cc, a_destr);

		oc = gb_object_class_new_single (&p, NULL);
		printf ("Init A's class to %p\n", &p);
	}
		
	return oc;
}

GBObject *
method_a_new (GBEvalContext *ec, const char *name)
{
	GBObject *o;
	MethodA  *a;

	o = gb_object_new (ec, method_a_class ());

	a = gb_object_get_priv (o, method_a_class ());

	a->name = g_strdup (name);
	
	return o;
}

GBObjectClass *
method_b_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBObjectPrivClass p;
		gb_object_priv_class_init (&p, "Method B", sizeof (MethodA),
					   NULL, NULL);

		oc = gb_object_class_new_single (&p, NULL);
		printf ("Init B's class to %p\n", &p);
	}
		
	return oc;
}

GBObject *
method_b_new (GBEvalContext *ec, int num)
{
	GBObject *o;
	MethodB  *b;

	o = gb_object_new (ec, method_b_class ());

	b = gb_object_get_priv (o, method_b_class ());

	b->num = num;
	
	return o;
}

static void
method_a_dump (GBObject *o)
{
	MethodA *a = gb_object_get_priv (o, method_a_class ());
	GBObjectPrivClass *klass;

	if (!a)
		printf ("Not A type\n");
	else {
		printf ("A contains '%s'\n", a->name);

		klass = GB_OBJECT_PRIV_GET_CLASS (a);
		printf ("A's class is %p\n", klass);
	}
}

static void
method_b_dump (GBObject *o)
{
	MethodB *b = gb_object_get_priv (o, method_b_class ());
	GBObjectPrivClass *klass;

	if (!b)
		printf ("Not B type\n");
	else {
		printf ("B contains %d\n", b->num);

		klass = GB_OBJECT_PRIV_GET_CLASS (b);
		printf ("B's class is %p\n", klass);
	}
}

GBObjectClass *
method_ab_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBObjectPrivClass p;
		GSList *l;

		gb_object_priv_class_init (&p, "Method A&B", 0,
					   NULL, NULL);
		
		l = g_slist_prepend (NULL, method_a_class ());
		l = g_slist_prepend (l, method_b_class ());

		oc = gb_object_class_new (&p, l);
	}

	return oc;
}


GBObject *
method_ab_new (GBEvalContext *ec, char *name, int num)
{
	GBObject *o;
	MethodA  *a;
	MethodB  *b;

	o = gb_object_new (ec, method_ab_class ());

	b = gb_object_get_priv (o, method_b_class ());
	b->num = num;
	
	a = gb_object_get_priv (o, method_a_class ());
	a->name = g_strdup (name);

	return o;
}

int
main (int argc, char **argv)
{
	GBEvalContext *ec;
	GBObject *ma, *mb, *mab, *cpy;

	gtk_init (&argc, &argv);
	gb_init ();
	
	ec = gb_eval_context_new ();

	printf ("Test\n");

	ma = method_a_new (ec, "Hello there");
	mb = method_b_new (ec, 3);
	mab = method_ab_new (ec, "What !?", 7);

	printf ("\nPure A\n");
	method_a_dump (ma);
	method_b_dump (ma);
	printf ("\nPure B\n");
	method_a_dump (mb);
	method_b_dump (mb);
	printf ("\nA & B\n");
	method_a_dump (mab);
	method_b_dump (mab);

	printf ("\ncopy of A & B\n");
	cpy = gb_object_copy (ec, mab);

	gb_object_unref (mab);

	method_a_dump (cpy);
	method_b_dump (cpy);

	gb_object_unref (cpy);

	gb_object_unref (ma);
	gb_object_unref (mb);

	gtk_object_unref (GTK_OBJECT (ec));

	gb_shutdown ();

	return 0;
}
