/*
 * GNOME Object
 *
 * Author:
 *    Michael Meeks <michael@imaginator.com>
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include <math.h>
#include <setjmp.h>

#include <gb/gb.h>
#include <gb/gb-object.h>
#include <gb/gb-eval.h>

static GtkObjectClass *gb_object_parent = NULL;

/* Group the nasties at the end */
static void      gb_object_destroy_do     (GBObject      *obj);

static void
gb_object_destroy (GtkObject *object)
{
	gb_object_destroy_do (GB_OBJECT (object));
	gb_object_parent->destroy (object);
}

static void
gb_object_class_init (GBObjectClass *klass)
{
	GtkObjectClass *object_class;

	gb_object_parent = gtk_type_class (GTK_TYPE_OBJECT);

	object_class = (GtkObjectClass*) klass;
	object_class->destroy = gb_object_destroy;

	klass->parents = NULL;
	klass->size    = 0;
	klass->offsets = g_hash_table_new (NULL, NULL);
	klass->primary = NULL;
}

static void
gb_object_init (GBObject *ec)
{
}

static GtkType
gb_object_sub_type_new (const char *name)
{
	GtkType type;

	static GtkTypeInfo object_info = {
		NULL,
		sizeof (GBObject),
		sizeof (GBObjectClass),
		(GtkClassInitFunc)  gb_object_class_init,
		(GtkObjectInitFunc) gb_object_init,
		/* reserved_1 */ NULL,
		/* reserved_2 */ NULL,
		(GtkClassInitFunc) NULL,
	};
	object_info.type_name = (char *)name;

	type = gtk_type_unique (GB_TYPE_OBJECT, &object_info);

	return type;	
}

GtkType
gb_object_get_type (void)
{
	static GtkType object_type = 0;

	if (!object_type) {
		static const GtkTypeInfo object_info = {
			"GBObject",
			sizeof (GBObject),
			sizeof (GBObjectClass),
			(GtkClassInitFunc)  gb_object_class_init,
			(GtkObjectInitFunc) gb_object_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		object_type = gtk_type_unique (GTK_TYPE_OBJECT, &object_info);
		gtk_type_class (object_type);
	}

	return object_type;	
}

/*
 * The difference of 4 is a nasty NULL = 0 assumption.
 */ 
static int
get_priv_offset (GBObjectClass *klass, GBObjectPrivClass *pc)
{
	gpointer ans;

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (klass != NULL, -1);

	ans = g_hash_table_lookup (klass->offsets, pc);
	if (ans == NULL)
		return -1;

	return GPOINTER_TO_INT (ans - 4);
}

static void
set_priv_offset (GBObjectClass *klass, GBObjectPrivClass *pc,
		 int offset)
{
	g_hash_table_insert (klass->offsets, pc,
			     GINT_TO_POINTER (offset + 4));
}

static GBObject *
do_cc (GBEvalContext *ec, GBObjectClass *klass, GBObject *templ)
{
	GBObject *o;
	GSList   *l;
	int       offset;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (klass != NULL, NULL);
	g_return_val_if_fail (klass->size > 0, NULL);

	o = GB_OBJECT (gtk_type_new (GTK_OBJECT_CLASS (klass)->type));

	o->priv = g_malloc (klass->size);

	offset = 0;
	for (l = klass->parents; l; l = l->next) {
		GBObjectPrivClass *pc = l->data;
		GBObjectPriv      *p;
		
		if (!pc) /* You must alloc / static it */
			g_error ("Stacked Priv record");
		
		/* I suffer from paranoid delusions */
		g_assert (offset == get_priv_offset (klass, pc));

		p = o->priv + offset;
		p->klass = pc;

		if (pc->copy)
			pc->copy (ec, o, templ);
		
		offset += pc->priv_size;
	}

	return o;	
}

GBObject *
gb_object_copy (GBEvalContext *ec, GBObject *templ)
{
	GBObjectClass *klass;

	g_return_val_if_fail (templ != NULL, NULL);

	klass = GB_OBJECT_CLASS (GTK_OBJECT (templ)->klass);

	return do_cc (ec, klass, templ);
}

GBObject *
gb_object_new (GBEvalContext *ec, GBObjectClass *klass)
{
	return do_cc (ec, klass, NULL);
}

GBObject *
gb_object_ref (GBEvalContext *ec, GBObject *object)
{
	gtk_object_ref (GTK_OBJECT (object));

	return object;
}

void
gb_object_unref (GBObject *object)
{
	gtk_object_unref (GTK_OBJECT (object));
}

/*
 * Inheritance handling routines...
 */
int
gb_object_priv_get_type (void)
{
	static int type = 0;

	if (!type) {
		static const GtkTypeInfo priv_info = {
			"GBObjectPrivClass",
			sizeof (GBObjectPrivClass),
			0, 0, 0, 0, 0, 0
		};

		type = gtk_type_unique (0, &priv_info);
	}

	return type;
}

void
gb_object_priv_class_init (GBObjectPrivClass  *priv,
			   const char         *name,
			   int                 size,
			   GBObjectCopy       *copy,
			   GBObjectDestructor *destructor)
{
	g_return_if_fail (priv != NULL);

	priv->type        = GB_TYPE_OBJECT_PRIV;
	priv->name        = g_strdup (name);
	priv->priv_size   = size;
	priv->copy        = copy;
	priv->destructor  = destructor;
}

gpointer
gb_object_get_priv (GBObject      *object,
		    GBObjectClass *pc)
{
	GBObjectClass *klass;
	int            offset;

	g_return_val_if_fail (pc != NULL, NULL);
	g_return_val_if_fail (object != NULL, NULL);

	klass = GB_OBJECT_CLASS (object->object.klass);

	offset = get_priv_offset (klass, pc->primary);

	if (offset < 0)
		return NULL;
	else
		return object->priv + offset;
}

gboolean
gb_object_implements (GBObject      *object,
		      GBObjectClass *pc)
{
	return (gb_object_get_priv (object, pc) != NULL);
}

static void
gb_object_destroy_do (GBObject *obj)
{
	GSList        *l, *cpy;
	GBObjectClass *klass;
	
	g_return_if_fail (obj != NULL);
	g_return_if_fail (obj->priv != NULL);
	g_return_if_fail (GB_IS_OBJECT (obj));

	klass = GB_OBJECT_CLASS (obj->object.klass);
	g_return_if_fail (klass != NULL);

	/*
	 * Destroy base classes last.
	 */
	cpy = g_slist_reverse (g_slist_copy (klass->parents));
	for (l = cpy; l; l = l->next) {
		GBObjectPrivClass *pc = l->data;
	
		if (pc->destructor)
			pc->destructor (obj);
	}
	g_slist_free (cpy);

	g_free (obj->priv);
	obj->priv = NULL;
}

GBObjectClass *
gb_object_class_new (GBObjectPrivClass *priv,
		     const GSList *parents)
		     
{
	GSList        *l;
	GSList        *i = NULL;
	GtkType        type;
	GBObjectClass *klass;
	int            offset;

	g_return_val_if_fail (priv != NULL, NULL);
	g_return_val_if_fail (priv->name != NULL, NULL);
	g_return_val_if_fail (GB_IS_OBJECT_PRIV_CLASS (priv), NULL);

	i = g_slist_prepend (NULL, priv);

	for (l = (GSList *)parents; l; l = l->next) {
		GBObjectClass *pc = l->data;

		g_assert (GB_IS_OBJECT_CLASS (pc));
			
		i = g_slist_concat (g_slist_copy (pc->parents), i);
	}

	/* slow sanity checks */
	for (l = i; l; l = l->next) {
		GBObjectPrivClass *pc = l->data;

		g_assert (pc != NULL);
		g_assert (GB_IS_OBJECT_PRIV_CLASS (pc));

		if (g_slist_find (l->next, pc))
			g_error ("Duplicate inheritance nodes / base clases");
	}

	/* We need a better type system */
	type = gb_object_sub_type_new (priv->name);
	klass = gtk_type_class (type);

	/*
	 * Calculate offsets;
	 */
	offset = 0;
	for (l = i; l; l = l->next) {
		GBObjectPrivClass *pc = l->data;
		
		set_priv_offset (klass, pc, offset);
		offset += pc->priv_size;
	}

	klass->parents = i;
	klass->size    = offset;
	klass->primary = priv;

	return klass;
}

GBObjectClass *
gb_object_class_new_single (GBObjectPrivClass  *priv,
			    GBObjectClass      *parent)
			    
{
	GSList        *l;
	GBObjectClass *ret;

	if (parent)
		l = g_slist_prepend (NULL, parent);
	else
		l = NULL;

	ret = gb_object_class_new (priv, l);

	g_slist_free (l);

	return ret;
}



