/*
 * Gnome Basic controller.
 *
 * Authors:
 *     Jody Goldberg (jgoldberg@home.com)
 *     Michael Meeks (mmeeks@gnu.org)
 *
 * Copyright 2000, Helix Code, Inc.
 */
#include "config.h"

#include <sys/types.h>
#include <sys/mman.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <gnome.h>
#include <gb/libgb.h>
#include <gb/gb-mmap-lex.h>
#include <gbrun/libgbrun.h>
#include <gbrun/objects/libgbobj.h>

int gb_debug = 0;
poptContext ctx;

const struct poptOption gb_popts [] = {
	{ "debug", '\0', POPT_ARG_INT, &gb_debug, 0,
	  N_("Enables some debugging functions"), N_("LEVEL") },

	{ NULL, '\0', 0, NULL, 0 }
};

GBLexerStream *
file_to_stream (const char *filename)
{
	guint8            *data, *datacpy;
	struct stat        sbuf;
	int                fd, len;
	GBLexerStream     *ans;

	fd = open (filename, O_RDONLY);
	if (fd < 0 || fstat (fd, &sbuf) < 0) {
		fprintf (stderr, "gb: %s : %s\n", filename, strerror (errno));

		if (fd >= 0)
			close (fd);
		return NULL;
	}

	if ((len = sbuf.st_size) <= 0) {
		fprintf (stderr, "%s : empty file\n", filename);
		close (fd);
		return NULL;
	}

	data = mmap (0, len, PROT_READ, MAP_PRIVATE, fd, 0);
	if (data == MAP_FAILED) {

		fprintf (stderr, "%s : unable to mmap\n", filename);
		close (fd);
		return NULL;
	}

	if (!isspace (data [len - 1])) {
		fprintf (stderr, "File must end in whitespace");

		return NULL;
	}

	datacpy = g_new (guint8, len);
	memcpy (datacpy, data, len);

	ans = gb_mmap_stream_new (datacpy, len);

	munmap ((char *)data, len);
	close (fd);

	return ans;
}

const GBParseData *
parse_file (const char *filename)
{
	GBLexerStream     *ls;
	const GBParseData *module;
	GBEvalContext     *ec;

	if (!(ls = file_to_stream (filename)))
		return NULL;

	ec = gb_eval_context_new ();
	module = gb_parse_stream (ec, ls);
	if (!module) {
  		fputs (gb_eval_context_get_text (ec), stderr);
  		fputs ("\n", stderr);
  	}

	gtk_object_destroy (GTK_OBJECT (ec));
	gtk_object_destroy (GTK_OBJECT (ls));

	return module;
}

GBLexerStream *
stream_provider (GBRunEvalContext *ec,
		 const char       *name,
		 gpointer          user_data)
{
	GBLexerStream *ls;

	ls = file_to_stream (name);
	if (!ls) {
		gb_eval_exception_firev (GB_EVAL_CONTEXT (ec),
					 "no such file '%s'", name);
		return NULL;
	}

	return ls;
}

static void
exec_project (const char *filename)
{
	GBLexerStream    *ls;
	GBProject        *gb_proj;
	GBRunEvalContext *ec;
	GBRunProject     *proj;

	if (!(ls = file_to_stream (filename)))
		return;

	ec = gbrun_eval_context_new (filename, GBRUN_SEC_NONE);
	gb_proj = gb_project_new (GB_EVAL_CONTEXT (ec), ls);
	if (!gb_proj) {
  		fputs (gb_eval_context_get_text (GB_EVAL_CONTEXT (ec)), stderr);
  		fputs ("\n", stderr);
  	}

	proj = gbrun_project_new (ec, gb_proj, stream_provider, NULL);

	if (!gbrun_project_execute (ec, proj))
		g_warning ("An exception occured\n%s",
			   gb_eval_context_get_text (GB_EVAL_CONTEXT (ec)));

	gtk_object_destroy (GTK_OBJECT (proj));
	gtk_object_destroy (GTK_OBJECT (ec));
	gtk_object_destroy (GTK_OBJECT (ls));
}

int
main (int argc, char * argv[])
{
	const char **args;

	gnome_init_with_popt_table (PACKAGE, VERSION, argc, argv,
				    gb_popts, 0, &ctx);
	gb_init ();

	args = poptGetArgs (ctx);
	if (args) {
		GBEvalContext *ec;
		int i;

		ec = gb_eval_context_new ();

		gbrun_init (ec);

		if (gb_eval_exception (ec)) {
			g_warning ("%s", gb_eval_context_get_text (ec));
			gtk_object_unref (GTK_OBJECT (ec));
			return -1;
		}
		gtk_object_unref (GTK_OBJECT (ec));

		for (i = 0; args [i]; i++) {
#if 0
			const GBParseData *pd;
			GBRunEvalContext  *ec;
			GBRunObject       *obj;
#endif
			char              *s;

			if ((s = strrchr (args [i], '.')) &&
			    !g_strcasecmp (s + 1, "vbp"))
				exec_project (args [i]);
			else {
			        if (s) { /* s already == strrchr (args [i], '.') */
				        int diff_len = strlen (args [i]) - strlen (s);
					char *str1   = g_strndup (args [i], diff_len);
					char *str2   = g_strconcat (str1, ".vbp", NULL);
					
					exec_project (str2);
					
					g_free (str2);
					g_free (str1);
				} else {
					char *str1 = g_strconcat (args [i], ".vbp", NULL);
					
					exec_project (str1);
					
					g_free (str1);
				}
			}

#if 0			
			pd = parse_file (args [i]);
			if (!pd)
				continue;

			ec  = gbrun_eval_context_new (args [i], GBRUN_SEC_NONE);
			obj = gbrun_object_from_data (ec, "Module", pd);

			if (args [i + 1]) {
				GBValue *val =
					gbrun_routine (ec, obj, args [i + 1], NULL);
				if (!val)
					g_warning ("An exception occured\n%s",
						   gb_eval_context_get_text (GB_EVAL_CONTEXT (ec)));
				else {
					fprintf (stdout, "%s returned ", args [i + 1]);
					gb_value_print (stdout, val);
					fprintf (stdout, "\n");

					gb_value_destroy (val);
				}
				i++;
			} else {
				if (obj && GBRUN_IS_FORM (obj))
					gbrun_context_exec (ec, obj);
				else
					g_warning ("Can't decide what to do with this object");
			}

			gtk_object_destroy (GTK_OBJECT (ec));
			gb_object_unref (obj);
			gb_parse_data_destroy (pd);
			
			if (!args [i])
				break;
#endif
		}

		gbrun_shutdown ();
	}

	poptFreeContext (ctx);

	gb_shutdown ();

	return 0;
}
