
/*
 * gbrun-form.c
 *
 * Gnome Basic Interpreter Form functions.
 *
 * Authors:
 *	Frank Chiulli  <fc-linux@home.com>
 *      Michael Meeks  <michael@helixcode.com>
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include <gbrun/gbrun-stack.h>

#include "gbrun-form.h"
#include "gbrun-form-item.h"

#define ITEM_NAME "VB.Form"

static void copy     (GBEvalContext *, GBObject *, GBObject *);
static void destruct (GBObject *);

GBObjectClass *
gbrun_form_class (void)
{
	static GBObjectClass *oc = NULL;

	if (!oc) {
		static GBRunObjectPrivClass p;
		gb_object_priv_class_init (&p.priv, ITEM_NAME,
					   sizeof (GBRunForm),
					   (GBObjectCopy *)copy,
					   (GBObjectDestructor *)destruct);
		gbrun_object_priv_class_init (&p);

		oc = gb_object_class_new_single (&p.priv, gbrun_object_class ());
		gbrun_object_register (ITEM_NAME, oc);
	}
		
	return oc;
}

#define PRIV(o) ((GBRunForm *)gb_object_get_priv ((o), gbrun_form_class ()))

/*
 *  temporary.
 */
static gint
delete_event_cb (GtkWidget *window, GdkEventAny *e, gpointer data)
{
	gtk_main_quit ();
	return TRUE;
}

static void
copy (GBEvalContext  *ec,
      GBObject       *dest,
      GBObject       *templ)
{
	GBRunForm *form = PRIV (dest);

	if (templ)
		g_warning ("Copy unimplemented");

	form->window = GTK_WINDOW (gtk_window_new (GTK_WINDOW_TOPLEVEL));
	gtk_signal_connect (GTK_OBJECT (form->window),
  		            "delete_event",
		            GTK_SIGNAL_FUNC (delete_event_cb),
		            dest);

	form->fixed = GTK_FIXED (gtk_fixed_new ()); 

	gtk_container_add (GTK_CONTAINER (form->window),
			   GTK_WIDGET (form->fixed));
}

static void
destruct (GBObject *obj)
{
	GBRunForm *form = PRIV (obj);

	if (form->window)
		gtk_widget_destroy (GTK_WIDGET (form->window));
	form->window = NULL;
}

/*
 * _GBEFormFont defines a Microsoft font.
 */
/*
struct _GBEFormFont {
	gchar		*name;
	gfloat		 size;
	guint8		 charset;
	guint16		 weight;
	gboolean	 underline;
	gboolean	 italic;
	gboolean	 strike_thru;
};
*/

/*
 * _GBEForm defines a VB Form
 *
 * appearance
 * auto_redraw
 * back_color		Specifies the forms's background color.
 * border_style		Determines whether a single-line border appears
 *			around the form.
 * caption		The text that appears on the form.
 * clip_controls
 * control_box
 * draw_mode
 * draw_style
 * draw_width
 * enabled
 * fill_color
 * fill_style
 * font		 	Caption's font name, style and size.
 * font_transparent
 * fore_color
 * has_dc
 * height		The height of button in twips.
 * help_context_id
 * icon
 * key_preview
 * left			The distance between the internal left edge of an 
 *			object and the left edge of its container.
 * link_mode
 * link_topic
 * max_button
 * mdi_child
 * min_button
 * mouse_icon
 * mouse_ptr		The shape of the mouse cursor when the user moves the 
 *			mouse over the command button.
 * moveable
 * negotiate_menus
 * ole_drop_mode
 * palette
 * palette_mode
 * picture		The name of icon graphic image that appears on the 
 *			command button as long as the Style property is set to
 *			1-Graphical.
 * right_to_left
 * scale_height		The number of units for the vertical measurement of an
 *			object's interior
 * scale_left		The horizontal coordinates for the left edge of an 
 *			object.
 * scale_mode
 * scale_top
 * scale_width
 * show_in_taskbar
 * start_up_position
 * tag
 * top			Holds the number of twips from the command button's
 *			top edge to the Form window's top edge.
 * visible		Determines whether the command button appears or is
 *			hidden from the user.
 * whats_this_button
 * whats_this_help
 * width		The width of an object
 * window_state
 *
 * cmd_buttons
 * labels
 * text_boxes
 *
 * window
 * fixed
 *
 * Definitions:
 * twip		1,440th of an inch (the smallest screen measurement)
 */
/*	guint8		 appearance;
	gboolean	 auto_redraw;
	guint32		 back_color;
	guint		 border_style;
	gchar		*caption;
	gboolean	 clip_controls;
	gboolean	 control_box;
	guint8		 draw_mode;
	guint8		 draw_style;
	guint8		 draw_width;
	gboolean	 enabled;
	guint32		 fill_color;
	guint8		 fill_style;
	GBEFormFont	 font;
	gboolean	 font_transparent;
	guint32		 fore_color;
	gboolean	 has_dc;
	guint16		 height;
	guint8		 help_context_id;
	gchar		*icon;
	gboolean	 key_preview;
	guint16		 left;
	guint8		 link_mode;
	gchar		*link_topic;
	gboolean	 max_button;
	gboolean	 mdi_child;
	gboolean	 min_button;
	gchar		*mouse_icon;
	guint8		 mouse_ptr;
	gboolean	 moveable;
	gboolean	 negotiate_menus;
	guint8		 ole_drop_mode;
	guint16		 palette;
	guint8		 palette_mode;
	gchar		*picture;
	gboolean	 right_to_left;
	guint16		 scale_height;
	guint16		 scale_left;
	guint8		 scale_mode;
	guint16		 scale_top;
	guint16		 scale_width;
	gboolean	 show_in_taskbar;
	guint8		 start_up_position;
	gchar		*tag;
	guint16		 top;
	gboolean	 visible;
	gboolean	 whats_this_button;
	gboolean	 whats_this_help;
	guint16		 width;
	guint8		 window_State; */

/*
 * Form properties
 * p_caption		The text that appears on the form.
 * p_height		The height of a form.
 * p_left		The distance between the internal left edge of an
 *			object and the left edge of its container.
 * p_scale_height	The number of units for the vertical measurement of an
 *			object's interior.
 * p_scale_width	The number of units for the horizontal measurement of
 *			an object's interior.
 * p_top		The distance between the internal top edge of an object
 *			and the top edge of its container.
 * p_width		The width of a form.
 *
 */

static GBRunObjProperty *p_caption      = NULL;
static GBRunObjProperty *p_height       = NULL;
static GBRunObjProperty *p_left         = NULL;
static GBRunObjProperty *p_scale_height = NULL;
static GBRunObjProperty *p_scale_width  = NULL;
static GBRunObjProperty *p_top          = NULL;
static GBRunObjProperty *p_width        = NULL;

/**
 * form_setarg
 *   @ec
 *   @object
 *   @property
 *   @val
 *
 *   Internal function.
 *   This function processes the form properties and calls the appropriate
 * GTK function to set the appropriate properties of a GTK object.
 **/
static void
form_setarg (GBRunEvalContext *ec,
	     GBRunObject      *object,
	     GBRunObjProperty *property,
	     GBValue          *val)
{
	GBRunForm *form = PRIV (object);

	g_return_if_fail (form != NULL);
	g_return_if_fail (form->window != NULL);

	if (property == p_caption)
		gtk_window_set_title (form->window, val->v.s->str);

	else if (property == p_height) {
		GtkWidget *w = GTK_WIDGET (form->window);
		w->requisition.height = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_window_set_default_size (form->window, w->requisition.width,
				             w->requisition.height);

	} else if (property == p_left) {
		GtkWidget *w = GTK_WIDGET (form->window);
		w->allocation.x = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_widget_set_uposition (w, w->allocation.x, w->allocation.y);

	} else if (property == p_top) {
		GtkWidget *w = GTK_WIDGET (form->window);
		w->allocation.y = GBRUN_FORM_TWIPS_TO_Y (val->v.i);
		gtk_widget_set_uposition (w, w->allocation.x, w->allocation.y);
	
	} else if (property == p_width) {
		GtkWidget *w = GTK_WIDGET (form->window);
		w->requisition.width = GBRUN_FORM_TWIPS_TO_X (val->v.i);
 		gtk_window_set_default_size (form->window, w->requisition.width, 
		                             w->requisition.height);
	}
}


/**
 * form_getarg
 *   @ec
 *   @object
 *   @property
 *
 *   Internal function.
 **/
static GBValue *
form_getarg (GBRunEvalContext *ec,
	     GBRunObject        *object,
	     GBRunObjProperty   *property)
{
	GBRunForm *form = PRIV (object);

	g_return_val_if_fail (form != NULL, NULL);
	g_return_val_if_fail (form->window != NULL, NULL);

	if (property == p_caption)
		return gb_value_new_string_chars (form->window->title);

	else if (property == p_width) {
		guint i = GTK_WIDGET (form->window)->allocation.width;
		return gb_value_new_int (GBRUN_FORM_X_TO_TWIPS (i));

	} else if (property == p_height) {
		guint i = GTK_WIDGET (form->window)->allocation.height;
		return gb_value_new_int (GBRUN_FORM_Y_TO_TWIPS (i));
	}

	return NULL;
}

static GBValue *
form_show (GBRunEvalContext *ec,
	   GBRunObject      *object,
	   GBValue         **args)
{
	GBRunForm *form = PRIV (object);

	/* FIXME: loads if not already loaded !? */
	if (args [0] || args [1])
		g_warning ("Modality & owner unimplemented");

	gtk_widget_show (GTK_WIDGET (form->window));

	return gb_value_new_empty ();
}

static GBValue *
form_hide (GBRunEvalContext *ec,
	   GBRunObject      *object,
	   GBValue         **args)
{
	GBRunForm *form = PRIV (object);

	gtk_widget_hide (GTK_WIDGET (form->window));

	return gb_value_new_empty ();
}

/**
 * gbrun_form_register
 **/
void
gbrun_form_register (void)
{
	GBRunObjectPrivClass *klass;
	
	klass = gbrun_object_class_get_priv (gbrun_form_class ());

	klass->set_arg = form_setarg;
	klass->get_arg = form_getarg;

	p_caption      = gbrun_object_add_property_val (klass, "caption",      GB_VALUE_STRING);
	p_height       = gbrun_object_add_property_val (klass, "clientheight", GB_VALUE_INT);
	p_left         = gbrun_object_add_property_val (klass, "clientleft",   GB_VALUE_INT);
	p_scale_height = gbrun_object_add_property_val (klass, "scaleheight",  GB_VALUE_INT);
	p_scale_width  = gbrun_object_add_property_val (klass, "scalewidth",   GB_VALUE_INT);
	p_top          = gbrun_object_add_property_val (klass, "clienttop",    GB_VALUE_INT);
	p_width        = gbrun_object_add_property_val (klass, "clientwidth",  GB_VALUE_INT);

	gbrun_object_add_method_arg (klass, "sub;show;modal,integer,byval,0;ownerform,variant,byref,0;g",
				     form_show);
	gbrun_object_add_method_arg (klass, "sub;hide;g",
				     form_hide);
}


/**
 * gbrun_form_shutdown
 **/
void
gbrun_form_shutdown (void)
{	
}


/**
 * gbrun_form_pass_properties:
 *   @ec: 
 *   @obj: 
 *   @item: 
 * 
 *   This function takes the properties described by the form description,
 * promotes their values to the correct types as expected by gbrun_object_set_arg
 * etc.  It then calls gbrun_object_set_arg. Ideally we must move the promotion 
 * into gbrun_object_set_arg.
 **/
static void
gbrun_form_pass_properties (GBRunEvalContext *ec, GBRunObject *obj,
			    GBFormItem *item)
{
	GSList *l;
	
	for (l = item->properties; l; l = l->next) {
		GBFormProperty *prop = l->data;

		if (gbrun_object_get_property (GBRUN_OBJECT_GET_CLASS (obj),
					       prop->name))
			gbrun_object_set_arg (ec, obj,
					      prop->name, prop->value);
		else
			g_warning ("Missing property '%s' on '%s' named '%s'", 
			           prop->name, gbrun_object_name (obj),
				   item->name);
	}
}


/**
 * gbrun_form_show:
 *   @obj: 
 **/
void
gbrun_form_show (GBRunObject *obj)
{
	GBRunForm *form;

	g_return_if_fail (obj != NULL);

	form = PRIV (obj);
	g_return_if_fail (form != NULL);
	
	gtk_widget_show_all (GTK_WIDGET (form->window));
}

static void
gbrun_form_add (GBObject          *form,
		GBObject          *obj,
		const char        *name,
		const GBParseData *module)
{
	GBRunFormItem *fi;

	g_return_if_fail (obj != NULL);
	g_return_if_fail (name != NULL);
	g_return_if_fail (form != NULL);

	fi = gb_object_get_priv (obj, gbrun_form_item_class ());
	g_return_if_fail (fi != NULL);

	fi->form = form;
	fi->name = g_strdup (name);

	gtk_fixed_put (PRIV (form)->fixed, fi->widget, 0, 0);
}

gboolean
gbrun_form_invoke (GBRunEvalContext *ec,
		   GBRunObject      *form,
		   const char       *method)
{
	gboolean ret = TRUE;

	g_return_val_if_fail (form != NULL, FALSE);
	
	if (gbrun_object_get_method (GBRUN_OBJECT_GET_CLASS (form),
				     method)) {
		GBObjRef  ref;
		GBValue  *ignore;
		
		ref.method = FALSE;
		ref.name   = method;
		ref.parms  = NULL;

		ignore = gbrun_method_invoke (ec, form, &ref);

		if (ignore)
			gb_value_destroy (ignore);
		else {
			g_warning ("Error invoking '%s' : '%s", method,
				   gb_eval_context_get_text (GB_EVAL_CONTEXT (ec)));
			ret = FALSE;
		}
	} else {
		g_warning ("No '%s' handler on form", method);
		ret = FALSE;
	}

	return ret;
}

/**
 * gbrun_form_new:
 *   @ec: 
 *   @item: 
 * 
 **/
void
gbrun_form_init (GBRunEvalContext *ec, GBRunObject *obj,
		 const GBParseData *pd)
{
	GSList        *l;
	GBRunForm     *form;
	GBFormItem    *item;

	g_return_if_fail (ec != NULL);
	g_return_if_fail (pd != NULL);
	g_return_if_fail (obj != NULL);

	form = PRIV (obj);
	g_return_if_fail (form != NULL);

	item = pd->form;
	gbrun_form_pass_properties (ec, obj, item);

	for (l = item->children; l; l = l->next) {
		GBFormItem       *i = l->data;
		GBRunObjectClass *klass;
		GBRunObject      *object;

		klass = gbrun_object_lookup (i->type);
		if (!klass) {
			g_warning ("Unknown sub-form type '%s'", i->type);
			continue;
		}

		object = gbrun_object_new (ec, i->type);
		if (gb_object_implements (object, gbrun_form_item_class ()))
			gbrun_form_add (obj, object, i->name, pd);
		else
			g_warning ("Non form item '%s' put into form", i->type);
		
		gbrun_form_pass_properties (ec, object, i);
		
		/* Add it to the stack */
		gbrun_object_ref (ec, object);
		gbrun_stack_add  (ec, i->name, gb_value_new_object (object),
				  GBRUN_STACK_MODULE);
	}
	gbrun_form_show (obj);
}

void
gbrun_form_widget_set_color (GtkWidget         *widget,
			     GBRunFormColorType type,
			     GBLong             color)
{
	GdkColor  col;
	GtkStyle *style;
	GdkColor *array = NULL;

	col.red   = ((color >>  0) & 0xff) * 255;
	col.green = ((color >>  8) & 0xff) * 255;
	col.blue  = ((color >> 16) & 0xff) * 255;

	style = gtk_style_copy (widget->style);

/*	g_warning ("Setting %d color on widget to 0x%x (= %d, %d, %d)", type,
	color, col.red, col.green, col.blue);*/
	/*
	 * FIXME: should recurse down containment hierarchy perhaps ?
	 * at least until we hit a widget with the FormItemKey
	 */

	/* FIXME: these need testing / rationalizing */
	switch (type) {

	case GBRUN_FORM_COLOR_BACK:
		array = style->base;
		break;

	case GBRUN_FORM_COLOR_BORDER:
		array = style->base;
		break;

	case GBRUN_FORM_COLOR_FILL:
		array = style->bg;
		break;

	case GBRUN_FORM_COLOR_FORE:
		array = style->fg;
		break;

	case GBRUN_FORM_COLOR_MASK:
	default:
		g_warning ("Unknown color type");
		break;
	};

	if (array) {
		int i;
		for (i = GTK_STATE_NORMAL; i <= GTK_STATE_INSENSITIVE; i++)
			array [i] = col;
	}

	gtk_widget_set_style  (widget, style);
	gtk_widget_queue_draw (widget);
}

GBLong
gbrun_form_widget_get_color (GtkWidget         *widget,
			     GBRunFormColorType type,
			     GBLong             color)
{
	g_warning ("Unimplemented");

	return 0;
}

char *
gbrun_form_un_shortcutify (const char *txt, char *shortcut)
{
	char *ans;
	int   i;

	g_return_val_if_fail (txt != NULL, NULL);

	ans = g_strdup (txt);

	/* FIXME: some sort of escaping must happen */
	for (i = 0; ans [i]; i++)
		if (ans [i] == '&') {
			if (shortcut)
				*shortcut = ans [i + 1];
			ans [i] = ' ';
		}

	return ans;
}
