
/*
 * GNOME Basic Enterpreter Function registration
 *
 * Authors:
 *    Michael Meeks (mmeeks@gnu.org)
 *    Thomas Meeks  (meekte95@christs-hospital.org.uk)
 *    Ariel Rios    (ariel@arcavia.com)
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gbrun/gbrun.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-array.h>
#include <gbrun/gbrun-object.h>
#include <gbrun/gbrun-file.h>

#include <gb/gb-constants.h>

#include "gba-func.h"


static GBValue *
gbrun_func_vartype (GBRunEvalContext *ec,
		  GBObject         *object,
		  GBValue         **args)
{
	if (!args [0])
		return gbrun_exception_firev (ec, "No argument");

	return gb_value_new_int (args [0]->type);
}


static GBValue *
gbrun_func_filelen (GBRunEvalContext *ec,
		  GBObject         *object,
		  GBValue         **args)
{
	char       *name;
	struct stat buf;
	
	GB_IS_VALUE (ec, args [0], GB_VALUE_STRING);

	name = args [0]->v.s->str;

	if (stat (name, &buf))
		return gbrun_exception_firev (ec, "FileLen error on %s", name);
	
	return gb_value_new_long (buf.st_size);
}


static GBValue *
gbrun_sub_print (GBRunEvalContext *ec,
	       GBObject         *object,
	       GSList           *expr)
{
	for (; expr; expr = expr->next) {
		GBValue *v;

		v = gbrun_eval_as (ec, expr->data, GB_VALUE_STRING);
		if (!v)
			return NULL;

		printf ("%s", v->v.s->str);

		gb_value_destroy (v);
	}
	printf ("\n");

	return gb_value_new_empty ();
}

static GBValue *
gbrun_sub_array (GBRunEvalContext *ec,
	       GBObject         *object,
	       GSList           *expr)
{
	GBObject *a;
	GSList   *vals = NULL;

	for (; expr; expr = expr->next) {
		GBValue *v;

		v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), expr->data);
		if (!v)
			return NULL;

		vals = g_slist_append (vals, v);
	}

	a = gbrun_array_new_vals (ec, vals);

	while (vals) {
		gb_value_destroy (vals->data);
		vals = g_slist_remove (vals, vals->data);
	}
	
	return gb_value_new_object (a);
}


static GBValue *
gbrun_func_isnumeric (GBRunEvalContext *ec,
		    GBObject         *object,
		    GBValue        **args)
{
	if (!args [0])
		return gb_value_new_boolean (FALSE);

	switch (args [0]->type) {

	case GB_VALUE_CURRENCY:
	case GB_VALUE_NUMBER:
		return gb_value_new_boolean (TRUE);

	default:
		return gb_value_new_boolean (FALSE);
	}
}

static GBValue *
gbrun_func_isnull (GBRunEvalContext *ec,
		    GBObject         *object,
		    GBValue        **args)
{
	if (!args [0] ||
	    args [0]->type != GB_VALUE_NULL)
		return gb_value_new_boolean (FALSE);

	return gb_value_new_boolean (TRUE);
}


static GBValue *
gbrun_func_isobject (GBRunEvalContext *ec,
		    GBObject         *object,
		    GBValue        **args)
{
	if (!args [0] ||
	    args [0]->type != GB_VALUE_OBJECT)
		return gb_value_new_boolean (FALSE);
	
	return gb_value_new_boolean (TRUE);
}

void
gba_func_register (GBEvalContext *ec)
{
	GBObjectClass        *gba_object, *debug_object;
	GBRunObjectPrivClass *gba, *debug;

	gba_object = gbrun_object_create ("vba", NULL);
	gba = gbrun_object_class_get_priv (gba_object);


	gbrun_object_add_method_arg (gba, "func;vartype;a,variant;integer;n",
				     gbrun_func_vartype);
        
	gbrun_object_add_method_arg (gba, "func;filelen;pathname,string;string;i",
				     gbrun_func_filelen);

	gbrun_object_add_method_arg (gba, "func;eof;filenumber,integer;boolean;i",
				     gbrun_func_eof);

	gbrun_object_add_method_arg (gba, "func;seek;filenumber,integer;long;i",
				     gbrun_func_seek);

	gbrun_object_add_method_arg (gba, "func;loc;filenumber,integer;long;i",
				     gbrun_func_loc);

	gbrun_object_add_method_arg (gba, "func;lof;filenumber,integer;long;i",
				     gbrun_func_lof);

	gbrun_object_add_method_arg (gba, "func;freefile;rangenumber,integer,byval,NULL;integer;i",
				     gbrun_func_freefile);

        gbrun_object_add_method_arg (gba, "func;isnumeric;expression,variant;boolean;n",
				     gbrun_func_isnumeric);

	gbrun_object_add_method_arg (gba, "func;isnull;expression,variant;boolean;n",
				     gbrun_func_isnull);

	gbrun_object_add_method_arg (gba, "func;isobject;expression,variant;boolean;n",
				     gbrun_func_isobject);

        gbrun_object_add_method_var (gba, "sub;array;...;n",  gbrun_sub_array);

	gbrun_object_globalize      (gb_object_new (ec, gba_object));


	debug_object = gbrun_object_create ("Debug", NULL);
	debug = gbrun_object_class_get_priv (debug_object);

	gbrun_object_add_method_var (debug, "sub;print;...;n",  gbrun_sub_print);

	gbrun_object_globalize      (gb_object_new (ec, debug_object));
}

void
gba_func_shutdown (void)
{
}
