
/*
 * GNOME Basic Enterpreter library main.
 *
 * Author:
 *    Michael Meeks <mmeeks@gnu.org>
 *
 * Copyright 2000, Helix Code, Inc
 */

#include <gbrun/gbrun.h>
#include <gb/libgb.h>
#include <gb/gb-main.h>
#include <gb/gb-form.h>
#include <gbrun/libgbrun.h>
#include <gbrun/gbrun-stack.h>
#include <gbrun/gbrun-statement.h>
#include <gbrun/objects/libgbobj.h>
#include <gbrun/objects/gba/libgba.h>
#include <gbrun/objects/gbrun-objects.h>

struct _GBRunProjectPriv {
	GBRunStreamProvider *provider;
	gpointer             user_data;
	GBProject           *gb_proj;
	GSList              *modules;

	GHashTable          *classes;
};

static void
project_destroy (GtkObject *obj)
{
	GBRunProject     *proj = GBRUN_PROJECT (obj);
	GBRunProjectPriv *priv = proj->priv;

	gb_project_destroy (priv->gb_proj);
	priv->gb_proj = NULL;

	while (priv->modules) {
		GBObject *obj = priv->modules->data;

		gb_object_unref (obj);
		priv->modules = g_slist_remove (priv->modules, obj);
	}

	g_free (priv);
}

static void
gbrun_project_class_init (GtkObjectClass *klass)
{
	GBRunProjectClass *proj_class;
	
	proj_class = GBRUN_PROJECT_CLASS (klass);

	klass->destroy = project_destroy;
}

static void
gbrun_project_init (GBRunProject *proj)
{
	proj->priv = g_new0 (GBRunProjectPriv, 1);
}

GtkType
gbrun_project_get_type (void)
{
	static GtkType proj_type = 0;

	if (!proj_type) {
		static const GtkTypeInfo proj_info = {
			"GBRunProject",
			sizeof (GBRunProject),
			sizeof (GBRunProjectClass),
			(GtkClassInitFunc) gbrun_project_class_init,
			(GtkObjectInitFunc) gbrun_project_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		proj_type = gtk_type_unique (GTK_TYPE_OBJECT, &proj_info);
	}

	return proj_type;
}


static GBRunObject *
object_from_data (GBRunEvalContext  *ec,
		  const char        *name,
		  const GBParseData *pd)
{
	GBRunObject          *obj;
	GBObjectClass        *klass;
	GBRunObjectPrivClass *pc;
	const char           *objname;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (pd != NULL, NULL);

	
	if (pd->form) {
		objname = pd->form->name;
		klass = gbrun_object_create_single (objname,
						    gbrun_form_class ());
	} else {
		objname = name;
		klass = gbrun_object_create_single (objname,
						    gbrun_object_class ());
	}

	{
		GSList *l;
		/* FIXME: Ugly hack to setup data */
		for (l = klass->parents; l; l = l->next) {
			pc = (GBRunObjectPrivClass *)l->data;
			if (!g_strcasecmp (pc->priv.name,
 					   "--GBRun.Object--"))
				pc->vars = pd->variables;
		}
	}
	obj = gbrun_object_new (ec, objname);
	
	if (pd->form)
		gbrun_form_init (ec, obj, pd);

	pc = (GBRunObjectPrivClass *)GB_OBJECT_PRIV_GET_CLASS (gbrun_object_get_priv (obj));
	gbrun_object_add_routines  (ec, pc, pd->routines);

	if (pd->form)
		gbrun_form_invoke (ec, obj, "Form_Load");

	return obj;
}

const GBParseData *
parsed_load (GBRunEvalContext    *ec,
	     const char          *filename,
	     GBRunStreamProvider *provider,
	     gpointer             user_data,
	     GBParsingState       state)
{
	GBLexerStream     *ls;
	const GBParseData *pd;

	if (!(ls = provider (ec, filename, user_data)))
		return NULL;

	/* Set parser state before and then turn it off later in grammar */
	gb_lexer_stream_state_set (ls, state);

	pd = gb_parse_stream (GB_EVAL_CONTEXT (ec), ls);
	gtk_object_destroy (GTK_OBJECT (ls));

	return pd;
}

GBRunProject *
gbrun_project_new (GBRunEvalContext *ec, GBProject *p,
		   GBRunStreamProvider *provider,
		   gpointer user_data)
{
	GSList           *l;
	GBRunProject     *proj;
	GBRunProjectPriv *priv;

	g_return_val_if_fail (p != NULL, NULL);
	g_return_val_if_fail (provider != NULL, NULL);
	proj = GBRUN_PROJECT (gtk_type_new (GBRUN_TYPE_PROJECT));

	priv = proj->priv;

	priv->provider  = provider;
	priv->user_data = user_data;
	priv->gb_proj   = p;

	for (l = p->modules; l; l = l->next) {
		GBProjectPair     *pp = l->data;
		const GBParseData *pd;
		GBObject          *obj;

		fprintf (stderr, "Loading module '%s'\n", pp->filename);

		if (!(pd = parsed_load (ec, pp->filename, provider, user_data, GB_PARSING_BASIC)))
			return NULL;

		obj = object_from_data (ec, pp->filename, pd);
		gb_parse_data_destroy (pd);

		priv->modules = g_slist_append (priv->modules, obj);
	}

/*	for (l = p->classes; l; l = l->next) {
		GBProjectPair     *pp = l->data;
		const GBParseData *pd;
		GBObject          *obj;

		fprintf (stderr, "Loading class '%s'\n", pp->filename);

		if (!(pd = parsed_load (ec, pp->filename, provider, user_data)))
			return NULL;

		obj = object_from_data (ec, pp->name, pd);
		gb_parse_data_destroy (pd);
*//* FIXME: leak object *//*
     }*/

	for (l = p->forms; l; l = l->next) {
		char              *filename = l->data;
		const GBParseData *pd;
		GBObject          *obj;

		fprintf (stderr, "Loading form '%s'\n", (char *)l->data);

		if (!(pd = parsed_load (ec, filename, provider, user_data, GB_PARSING_FORM)))
			return NULL;

		obj = object_from_data (ec, "Unused", pd);
		gb_parse_data_destroy (pd);
	}

	return proj;
}

gboolean
gbrun_project_execute (GBRunEvalContext *ec, GBRunProject *proj)
{
	const char *startup;
	gboolean    success;

	g_return_val_if_fail (GBRUN_IS_EVAL_CONTEXT (ec), FALSE);
	g_return_val_if_fail (GBRUN_IS_PROJECT (proj), FALSE);
	g_return_val_if_fail (proj->priv != NULL, FALSE);
	g_return_val_if_fail (proj->priv->gb_proj != NULL, FALSE);

	startup = gb_project_startup_get (proj->priv->gb_proj);

	if (!g_strncasecmp (startup, "Sub ", 4)) {
		char    *subname = g_strchug (g_strchomp (g_strdup (startup + 4)));
		GBValue *val;

/*		g_warning ("Project sub : '%s'", subname);*/
		
		val = gbrun_project_invoke (ec, proj, subname, NULL);
		gb_value_destroy (val);

		g_free (subname);

		success = !gbrun_eval_context_exception (ec);
	} else { /* A form */
/*		g_warning ("Project form : '%s'", startup);*/
		gtk_main ();
		success = TRUE;
	}

	return success;
}

void
gbrun_init (GBEvalContext *ec)
{
	gbrun_object_init (ec);
	if (gb_eval_exception (ec))
		return;

	libgba_register (ec);
	if (gb_eval_exception (ec))
		return;

	gbrun_objects_register (ec);
}


void
gbrun_shutdown (void)
{
	gbrun_objects_shutdown ();
	libgba_shutdown        ();
	gbrun_object_shutdown  ();
}

GSList *
gbrun_project_get_modules (GBRunProject *proj)
{
	g_return_val_if_fail (proj != NULL, NULL);
	g_return_val_if_fail (proj->priv != NULL, NULL);

	return proj->priv->modules;
}

GBValue *
gbrun_project_invoke (GBRunEvalContext *ec, GBRunProject *proj,
		      const char *name, GSList *args)
{
	GBObjRef ref;
	GSList  *l, *exprs = NULL;
	GBValue *val;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);

	gbrun_eval_context_proj_push (ec, proj);

	/* FIXME */
	ref.method = FALSE;
	ref.name   = name;

	/* FIXME: Amusingly inefficient */
	for (l = args; l; l = l->next) {
		gpointer expr = (gpointer) gb_expr_new_value (gbrun_value_copy (ec, l->data));
		exprs = g_slist_prepend (exprs, expr);
	}

	exprs = g_slist_reverse (exprs);
	ref.parms = exprs;

	val = gbrun_method_invoke (ec, NULL, &ref);

	while (exprs) {
		gb_expr_destroy (exprs->data);
		exprs = g_slist_remove (exprs, exprs->data);
	}

	gbrun_eval_context_proj_pop (ec);

	return val;
}

GSList *
gbrun_project_fn_names (GBRunProject *proj)
{
	GSList *ans = NULL;
	GSList *l;

	g_return_val_if_fail (proj != NULL, NULL);
	g_return_val_if_fail (proj->priv != NULL, NULL);

	for (l = proj->priv->modules; l; l = l->next) {
		GBRunObject          *obj = l->data;
		GSList               *m, *i;

		m = gbrun_object_get_methods (GBRUN_OBJECT_GET_CLASS (obj));

		for (i = m; i; i = i->next) {
			GBRunObjMethod *method = i->data;

			ans = g_slist_prepend (ans, method->name);
		}
		
		g_slist_free (m);
	}

	return ans;
}
