
/*
 * GNOME Basic File I/O statements handler
 *
 * Authors:
 *    Ravi Pratap   (ravi_pratap@email.com)
 *    Michael Meeks (michael@helixcode.com)
 *
 * Copyright 2000, Helix Code, Inc.
 */

#include <gbrun/gbrun.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-stack.h>
#include <gbrun/gbrun-statement.h>
#include <gbrun/gbrun-object.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "gbrun-file.h"

#define FILE_STMT_SEC_CHECK(ec,txt) \
	if (ec->flags & GBRUN_SEC_IO) { \
		gbrun_exception_firev (ec, "Insufficient privilage to %s file", \
				       txt); \
		return FALSE; \
	}

typedef struct {
        /* VB related */
	int       fileno;
	OpenMode  mode;
	guint     recordlen;

	/* Internal */
	FILE     *file;
} GBRunFileHandle;

static GBRunFileHandle *
internal_handle_from_gb_no (GBRunEvalContext *ec, int fileno)
{
	GSList *l;

	for (l = ec->open_files; l != NULL; l = l->next) {
		GBRunFileHandle *h = l->data;

		if (h->fileno == fileno) 
			return h;
	}
		
	gbrun_exception_firev (ec, "Invalid filenumber %d", fileno);

	return NULL;
}

/********** File I/O functions ************/

GBValue *
gbrun_func_eof (GBRunEvalContext *ec,
		GBObject         *object,
		GBValue         **args)
{
	GBRunFileHandle *h;	

	GB_IS_VALUE (ec, args [0], GB_VALUE_INT);

	h = internal_handle_from_gb_no (ec, args [0]->v.i);
	
	if (feof (h->file))
		return gb_value_new_boolean (GBTrue);
	else
		return gb_value_new_boolean (GBFalse);
}

GBValue *
gbrun_func_freefile (GBRunEvalContext *ec,
		     GBObject         *object,
		     GBValue         **args)
{
	GSList *l;
	int     max = -1;

	/*
	 * FIXME: obey the range argument.
	 */
	if (args [0])
		g_warning ("FreeFile: Rangenumber unimplemented");

	/*
	 * FIXME: need to test whether back to back calls to freefile
	 * return the same number, or to avoid races, reserve the ID.
	 */
	for (l = ec->open_files; l != NULL; l = l->next) {
		GBRunFileHandle *h = l->data;

		/*
		 * FIXME: this will die after too few file opens, we
		 * need to sort by fileno and find non-adjacent ids.
		 */

		if (h->fileno > max)
			max = h->fileno;
	}

	if (max < 0)
		max = 1;

	return gb_value_new_int (max);
}

GBValue *
gbrun_func_loc (GBRunEvalContext *ec,
		GBObject         *object,
		GBValue         **args)
{
	GBRunFileHandle *h;
	GBValue         *res;
	long             pos;
	
	GB_IS_VALUE (ec, args [0], GB_VALUE_INT);
	
	h = internal_handle_from_gb_no (ec, args [0]->v.i);

	pos = ftell (h->file) + 1;

	if (h->mode == GB_OPEN_RANDOM)
		res = gb_value_new_long (pos / h->recordlen);
	else if (h->mode == GB_OPEN_INPUT  || 
		 h->mode == GB_OPEN_OUTPUT ||
		 h->mode == GB_OPEN_APPEND)
		res = gb_value_new_long (pos / 128);
	else if (h->mode == GB_OPEN_BINARY)
		res = gb_value_new_long (pos);

	return res;
}

GBValue *
gbrun_func_lof (GBRunEvalContext *ec,
		GBObject         *object,
		GBValue         **args)
{
	GBRunFileHandle *h;
	GBValue         *res;
	long             pos;
	
	GB_IS_VALUE (ec, args[0], GB_VALUE_INT);
	h = internal_handle_from_gb_no (ec, args [0]->v.i);
	
        /* Save current position */
	pos = ftell (h->file); 
	
	/*
	 * This small roundabout way of getting the size of an
	 * open file is necessary as we have ostensibly, no func
	 * which does that for us on an open file using the FILE * !
	 */
	if (fseek (h->file, 0, SEEK_END)) {
		gbrun_exception_fire (ec, "Lof can't seek to EOF!");
		return NULL;
	}
	
	res = gb_value_new_long (ftell (h->file));
	if (fseek (h->file, pos, SEEK_SET)) {
		gbrun_exception_fire (ec, "Lof can't seek to EOF!");
		return NULL;
	}

	return res;
}

GBValue *
gbrun_func_seek (GBRunEvalContext *ec,
		 GBObject         *object,
		 GBValue         **args)
{
	GBRunFileHandle *h;
	GBValue         *res;
	long             pos;
	
	GB_IS_VALUE (ec, args [0], GB_VALUE_INT);
	
	h = internal_handle_from_gb_no (ec, args [0]->v.i);

	pos = ftell (h->file) + 1;

	if (h->mode == GB_OPEN_RANDOM)
		res = gb_value_new_long (pos / h->recordlen);
	else 
		res = gb_value_new_long (pos);
	
	return res;
}

gboolean
gbrun_files_clean (GBRunEvalContext *ec)
{
	GSList *l;

	for (l = ec->open_files; l; l = l->next) {
		GBRunFileHandle *h = l->data;

		if (h->file)
			fclose (h->file);
		h->file = NULL;
		g_free (h);
	}
	g_slist_free (ec->open_files);
	ec->open_files = NULL;

	return TRUE;
}

gboolean 
gbrun_stmt_open (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GBRunFileHandle  *h = g_new (GBRunFileHandle, 1);

	GBValue  *filename  = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), stmt->parm.open.filename);
	GBValue  *handle    = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), stmt->parm.open.handle);
	GBValue  *reclen;
	FILE     *fd;

	FILE_STMT_SEC_CHECK (ec, "open");

	if (filename->type != GB_VALUE_STRING &&
	    handle->type   != GB_VALUE_INT) {
		gbrun_exception_fire (ec, "Filename and filenumber are of incorrect types");
		return FALSE;
	}

	switch (stmt->parm.open.mode) {
		
	case GB_OPEN_OUTPUT:
		fd = fopen (filename->v.s->str, "w");
		break;

	case GB_OPEN_INPUT:
		fd = fopen (filename->v.s->str, "r");
		break;

	case GB_OPEN_APPEND:
		fd = fopen (filename->v.s->str, "a");
		break;
	       
	case GB_OPEN_BINARY: case GB_OPEN_RANDOM:
		/*
		 * Small workaround to create file if it doesn't
		 * exist. Apparently, directly using ab+ doesn't allow
		 * repositioning of stream : behaves funnily 
		 */
		fd = fopen (filename->v.s->str, "ab+");
		fclose (fd); 
		fd = fopen (filename->v.s->str, "rb+");
		break;
		
	default:
		g_warning ("Unhandled OPEN mode");
		break;
	}

	if (!fd) {
		gbrun_exception_fire (ec, "Open failed");
		return FALSE;
	}
	
	h->fileno = handle->v.i;
	h->mode   = stmt->parm.open.mode;
	h->file   = fd;

	if (stmt->parm.open.recordlen) {
		reclen = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), stmt->parm.open.recordlen);
		h->recordlen = reclen->v.i;
	} else 
		h->recordlen = 1;   /* Nothing specified for record length */

	if (h->mode == GB_OPEN_BINARY)
		h->recordlen = 1;  /* Ignore recordlen for Binary mode */

	/* Rewind file pointer to beginning always */
	fseek (h->file, 0, SEEK_SET);

	ec->open_files = g_slist_prepend (ec->open_files, (gpointer) h);
	
	return TRUE;
}

gboolean 
gbrun_stmt_close (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GSList *l;
	
	FILE_STMT_SEC_CHECK (ec, "close");

	/* If no arguments are passed, close all open files */
	if (!stmt->parm.close.handles)
		return gbrun_files_clean (ec);

	for (l = stmt->parm.close.handles; l != NULL; l = l->next) {
		GBRunFileHandle *h;
		GBExpr          *handle = l->data;
		GBValue         *fileno;

		fileno = gbrun_eval_as (ec, handle,
					GB_VALUE_INT);

		h = internal_handle_from_gb_no (ec, fileno->v.i);
		gb_value_destroy (fileno);

		if (h) {
			ec->open_files = g_slist_remove (ec->open_files, h);

			fclose (h->file);
			h->file = NULL;
			g_free (h);
		} else 
			return FALSE;
	}
		
	return TRUE;
}

static GBValue *
read_string (GBRunEvalContext *ec, GBRunFileHandle *h, gboolean allow_space)
{
	GArray  *chars;
	GBValue *val;
	char     chr;

	chars = g_array_new (FALSE, FALSE, sizeof (char));

	while (!feof (h->file)) {
		int  c;

		c = fgetc (h->file);
		if (c < 0)
			break;

		if (c == '\n')
			break;

		if (!allow_space && c == ' ')
			break;

		chr = (char)c;
		g_array_append_val (chars, chr);
	}
	
	chr = '\0';
	g_array_append_val (chars, chr);

	val = gb_value_new_string_chars (chars->data);

	g_array_free (chars, TRUE);

	return val;
}

gboolean 
gbrun_stmt_line_input (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GBValue         *handle;
	GBValue         *val;
	const GBExpr    *objref = stmt->parm.line_input.objref;
	GBRunFileHandle *h;
	
	FILE_STMT_SEC_CHECK (ec, "line input from");

	handle = gbrun_eval_as (ec, stmt->parm.line_input.handle,
				GB_VALUE_INT);

	h = internal_handle_from_gb_no (ec, handle->v.i);
	gb_value_destroy (handle);

	if (objref->type != GB_EXPR_OBJREF) {
		gbrun_exception_fire (ec, "Need an object ref");
		return FALSE;
	} else if (h->mode != GB_OPEN_INPUT  && 
		   h->mode != GB_OPEN_RANDOM &&
		   h->mode != GB_OPEN_BINARY) {
		gbrun_exception_fire (ec, "Can't read from a file not opened for input/random/binary");
		return FALSE;
	}

	val = read_string (ec, h, FALSE);
	if (!val)
		return FALSE;

	if (!gbrun_eval_assign (ec, objref->parm.objref, val)) {
		gbrun_exception_fire (ec, "Assignment of value to objectref failed");
		return FALSE;
	}
	
	g_free (val);
		
	return TRUE;
}

gboolean
gbrun_stmt_input (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GBValue         *handle;
	GBExprList      *objrefs = stmt->parm.input.objrefs;
	GBExprList      *list;
	GBRunFileHandle *h;

	FILE_STMT_SEC_CHECK (ec, "input from");

	handle = gbrun_eval_as (ec, stmt->parm.input.handle,
				GB_VALUE_INT);

	h = internal_handle_from_gb_no (ec, handle->v.i);
	gb_value_destroy (handle);

	if (h->mode != GB_OPEN_INPUT  && 
	    h->mode != GB_OPEN_RANDOM &&
	    h->mode != GB_OPEN_BINARY) {
		gbrun_exception_fire (ec, "Can't read from a file not opened for input/random/binary");
		return FALSE;
	}

	for (list = objrefs; list != NULL; list = list->next) {
		GBValue *val;
		GBExpr  *objref = (GBExpr *)list->data;

		if (objref->type != GB_EXPR_OBJREF) {
			gbrun_exception_fire (ec, "Need an object ref");
			return FALSE;
		}

		val = read_string (ec, h, TRUE);
		if (!val)
			return FALSE;

		if (!gbrun_eval_assign (ec, objref->parm.objref, val)) {
			gbrun_exception_fire (ec, "Assignment of value to objectref failed");
			return FALSE;
		}
	}

	return TRUE;
}

gboolean 
gbrun_stmt_get (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GBValue         *handle;
	GBRunFileHandle *h;
	GBValue         *recordnum = NULL;
	
	FILE_STMT_SEC_CHECK (ec, "get from");

	if (stmt->parm.get.recordnum)
		recordnum = gbrun_eval_as (ec, stmt->parm.get.recordnum,
					   GB_VALUE_LONG);

	handle = gbrun_eval_as (ec, stmt->parm.get.handle,
				GB_VALUE_INT);

	h = internal_handle_from_gb_no (ec, handle->v.i);
	gb_value_destroy (handle);

	if (h->mode != GB_OPEN_RANDOM &&
	    h->mode != GB_OPEN_BINARY) {
		gbrun_exception_fire (ec, "Can't Get from a file not opened in random/binary");
		return FALSE;
	}

	if (h->mode == GB_OPEN_RANDOM)
		g_warning ("Get: Can't handle Random mode correctly yet.");

	/*
	 * FIXME: need to check bounds on recordnum ideally.
	 */
	if (h->recordlen && recordnum) 
		fseek (h->file, (h->recordlen) * (recordnum->v.l - 1), SEEK_SET);

	if (!feof (h->file)) {
		GBValue      *val;
		char         *str = g_strdup ("");
		char         *tmpstr = g_strdup ("");
		const GBExpr *objref = stmt->parm.get.objref;
		int          i;
		
		for (i = 0; i < h->recordlen; ++i) {
			tmpstr[0] = (char) fgetc (h->file);
			tmpstr[1] = '\0';
			str = g_strconcat (str, tmpstr, NULL);
		}
		
		val = gb_value_new_string_chars (str);

		if (!gbrun_eval_assign (ec, objref->parm.objref, val)) {
			gbrun_exception_fire (ec, "Assignment of value to objref failed");
			return FALSE;
		}
		
		gb_value_destroy (val);
		g_free (str);
		g_free (tmpstr);
	} 
	
	return TRUE;
}

gboolean
gbrun_stmt_put (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GBValue         *handle;
	GBRunFileHandle *h;
	GBValue         *recordnum = NULL;
	GBValue         *tmp, *str;

	FILE_STMT_SEC_CHECK (ec, "put to");

	if (stmt->parm.put.recordnum) 
		recordnum = gbrun_eval_as (ec, stmt->parm.put.recordnum,
					   GB_VALUE_LONG);
	
	handle = gbrun_eval_as (ec, stmt->parm.put.handle,  
				GB_VALUE_INT);

	h = internal_handle_from_gb_no (ec, handle->v.i);
	gb_value_destroy (handle);

	if (h->mode != GB_OPEN_RANDOM &&
	    h->mode != GB_OPEN_BINARY) {
		gbrun_exception_fire (ec, "Can't Put to a file not opened in random/binary");
		return FALSE;
	}
	
	if (h->mode == GB_OPEN_RANDOM)
		g_warning ("Put: Can't handle Random mode correctly yet.");

	/* FIXME: do we need to check ranges here ? */
	if (h->recordlen && recordnum)
		fseek (h->file, (h->recordlen) * (recordnum->v.l - 1), SEEK_SET);
	
        tmp = gbrun_eval_objref (ec, stmt->parm.put.objref);
	str = gb_value_promote (GB_EVAL_CONTEXT (ec), tmp, GB_VALUE_STRING);
	gb_value_destroy (tmp);

	if ((fputs (str->v.s->str, h->file)) == EOF) {
		gbrun_exception_fire (ec, "Error while putting to file");
		return FALSE;
	}

	gb_value_destroy (str);
	
	return TRUE;
}

gboolean 
gbrun_stmt_seek (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GBValue          *handle;
	GBRunFileHandle  *h;
	GBValue          *pos;

	FILE_STMT_SEC_CHECK (ec, "seek within");

	handle = gbrun_eval_as (ec, stmt->parm.seek.handle,  
				GB_VALUE_INT);

	h = internal_handle_from_gb_no (ec, handle->v.i);
	gb_value_destroy (handle);

	pos = gbrun_eval_as (ec, stmt->parm.seek.pos,
			     GB_VALUE_LONG);
	if (pos->v.l <= 0) {
		gbrun_exception_fire (ec, "Can't seek to a non-positive position");
		return FALSE;
	}
		
	if (h->mode != GB_OPEN_RANDOM) {
		if (fseek (h->file, pos->v.l - 1, SEEK_SET)) {
			gbrun_exception_fire (ec, "Unable to seek");
			return FALSE;
		}
	} else {
		if (fseek (h->file, (pos->v.l - 1) * h->recordlen, SEEK_SET)) {
			gbrun_exception_fire (ec, "Unable to seek");
			return FALSE;
		}
	}
	
	return TRUE;

}
