# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import cStringIO
import xml.dom
from xml.sax.saxutils import escape

import gobject
import gtk

from gazpacho.choice import enum_to_string, flags_to_string
from gazpacho.loader import tags
from gazpacho.placeholder import Placeholder
from gazpacho.properties import prop_registry
from gazpacho.widget import Widget

def write_xml(file, xml_node, indent=0):
    if xml_node.nodeType == xml_node.TEXT_NODE:
        file.write(xml_node.data)
        return
    elif xml_node.nodeType == xml_node.CDATA_SECTION_NODE:
        file.write('<![CDATA[%s]]>' % xml_node.data)
        return

    file.write(' '*indent)
    
    file.write('<%s' % xml_node.tagName)
    if len(xml_node.attributes) > 0:
        attr_string = ' '.join(['%s="%s"' % (n, v)
                                    for n, v in xml_node.attributes.items()])
        file.write(' ' + attr_string)

    children = [a for a in xml_node.childNodes 
                    if a.nodeType != a.ATTRIBUTE_NODE]
    if children:
        has_text_child = False
        for child in children:
            if child.nodeType in (child.TEXT_NODE,
                                  child.CDATA_SECTION_NODE):
                has_text_child = True
                break

        if has_text_child:
            file.write('>')
        else:
            file.write('>\n')
        for child in children:
            write_xml(file, child, indent+4)

        if not has_text_child:
            file.write(' '*indent)
        file.write('</%s>\n' % xml_node.tagName)
    else:
        file.write('/>\n')

class XMLWriter:
    def __init__(self, document=None, project=None):
        if not document:
            dom = xml.dom.getDOMImplementation()
            document = dom.createDocument(None, None, None)
        self._doc = document
        self._project = project
        
    def write(self, path, widgets, uim):
        self._write_root(widgets, uim)

        self.write_node(path, self._doc.documentElement)
        
    def write_node(self, path, node):
        f = file(path, 'w')
        f.write('<?xml version="1.0" ?>\n')
        write_xml(f, node)
        f.close()

    def serialize_node(self, widget):
        element = self._doc.createElement(tags.XML_TAG_PROJECT)
        root = self._doc.appendChild(element)

        libglade_module = widget.adaptor.library.libglade_module
        if libglade_module:
            requires = self._doc.createElement(tags.XML_TAG_REQUIRES)
            requires.setAttribute(tags.XML_TAG_LIB, libglade_module)
            root.appendChild(requires)

        # save the UI Manager if needed
        widget.project.uim.save_widget(widget, root, self._doc)
        root.appendChild(self._write_widget(widget))

        return self._doc.documentElement
        
    def serialize(self, widget):
        fp = cStringIO.StringIO()
        node = self.serialize_node(widget)
        write_xml(fp, node)
        
        fp.seek(0)
        
        return fp.read()

    def serialize_widgets(self, widgets, uim):
        fp = cStringIO.StringIO()
        self._write_root(widgets, uim)
        
        write_xml(fp, self._doc.documentElement)
        
        fp.seek(0)
        
        return fp.read()

    def _get_requirements(self, widgets):
        # check what modules are the widgets using
        modules = []
        for gtk_widget in widgets:
            widget = Widget.from_widget(gtk_widget)
            
            libglade_module = widget.adaptor.library.libglade_module
            if libglade_module and libglade_module not in modules:
                modules.append(libglade_module)

        return modules
    
    def _write_root(self, widgets, uim):
        project_node = self._doc.createElement(tags.XML_TAG_PROJECT)
        node = self._doc.appendChild(project_node)

        for module in self._get_requirements(widgets):
            n = self._doc.createElement(tags.XML_TAG_REQUIRES)
            n.setAttribute(tags.XML_TAG_LIB, module)
            node.appendChild(n)

        self._write_widgets(node, widgets, uim)

        return node
    
    def _write_widgets(self, node, widgets, uim):

        # Append uimanager
        ui_widgets = [Widget.from_widget(w) 
                          for w in widgets
                               if isinstance(w, (gtk.Toolbar,
                                                 gtk.MenuBar))]
        if ui_widgets:
            ui_node = uim.save(self._doc, ui_widgets)
            if ui_node:
                node.appendChild(ui_node)
        
        # Append toplevel widgets. Each widget then takes care of
        # appending its children
        for widget in widgets:
            gwidget = Widget.from_widget(widget)
            if not gwidget.is_toplevel():
                continue
            
            wnode = self._write_widget(gwidget)
            node.appendChild(wnode)
    
    def _write_widget(self, widget):
        """Serializes this widget into a XML node and returns this node"""

        widget.maintain_gtk_properties = True
        
        widget.adaptor.save(widget.project.context, widget)

        # otherwise use the default saver
        node = self._write_basic_information(widget)
        
        self._write_properties(widget, node, child=False)
        self._write_signals(widget, node)
        
        # Children
        gtk_widget = widget.gtk_widget
        if not isinstance(gtk_widget, gtk.Container):
            return node

        # We're not writing children when we have a constructor set
        if widget.constructor:
            return node
        
        for child_widget in gtk_widget.get_children():
            child = self._write_child(child_widget)
            if child:
                node.appendChild(child)

        widget.maintain_gtk_properties = False
                    
        return node

    def _write_basic_information(self, widget):
        assert widget.adaptor.type_name
        assert widget.name, 'widget %r is nameless' % widget.gtk_widget
        node = self._doc.createElement(tags.XML_TAG_WIDGET)
        node.setAttribute(tags.XML_TAG_CLASS, widget.adaptor.type_name)
        node.setAttribute(tags.XML_TAG_ID, widget.name)
        if widget.constructor:
            node.setAttribute('constructor', widget.constructor)
        return node
        
    def _write_properties(self, widget, widget_node, child):
        # Sort them properties, to have a stable order
        gtk_widget = widget.gtk_widget
        properties = prop_registry.list(gobject.type_name(gtk_widget),
                                        gtk_widget.get_parent())
        properties.sort(lambda a, b: cmp(a.name, b.name))

        # write the properties
        for prop_type in properties:
            # don't save the name property since it is saved in the id
            # attribute of the tag
            if prop_type.name == 'name':
                continue

            # Do not save non-editable gobject properties, eg
            # GtkWindow::screen
            # hack?
            if (prop_type.base_type == gobject.GObject.__gtype__ and
                not prop_type.editable):
                continue
            
            # child properties are saved later
            if prop_type.child != child:
                continue

            prop = widget.get_prop(prop_type.name)

            child_node = self._write_property(prop)
            if child_node:
                widget_node.appendChild(child_node)

    def _write_signals(self, widget, widget_node):
        for signal_name, handlers in widget.signals.items():
            for handler in handlers:
                child = self._doc.createElement(tags.XML_TAG_SIGNAL)
                child.setAttribute(tags.XML_TAG_NAME, handler['name'])
                child.setAttribute(tags.XML_TAG_HANDLER, handler['handler'])
                child.setAttribute(tags.XML_TAG_AFTER,
                                   handler['after'] \
                                   and tags.TRUE or tags.FALSE)
                widget_node.appendChild(child)
                
    def _write_child(self, gtk_widget):
        child_tag = self._doc.createElement(tags.XML_TAG_CHILD)
        if isinstance(gtk_widget, Placeholder):
            child = self._doc.createElement(tags.XML_TAG_PLACEHOLDER)
            child_tag.appendChild(child)
            # we need to write the packing properties of the placeholder.
            # otherwise the container gets confused when loading its
            # children
            packing = self._write_placeholder_properties(gtk_widget)
            if packing is not None:
                child_tag.appendChild(packing)
            return child_tag

        child_widget = Widget.from_widget(gtk_widget)
        if child_widget is None:
            # if there is no GazpachoWidget for this child
            # we don't save it. If your children are not being
            # saved you should create a GazpachoWidget for them
            # in your Adaptor
            return
            
        if child_widget.internal_name is not None:
            child_tag.setAttribute(tags.XML_TAG_INTERNAL_CHILD,
                                   child_widget.internal_name)

        child = self._write_widget(child_widget)
        child_tag.appendChild(child)

        # Append the packing properties
        packing = self._doc.createElement(tags.XML_TAG_PACKING)
        self._write_properties(child_widget, packing, child=True)

        if packing.childNodes:
            child_tag.appendChild(packing)

        return child_tag

    def _write_placeholder_properties(self, placeholder):
        parent = placeholder.get_parent()
        # get the non default packing properties
        packing_list = []
        props = gtk.container_class_list_child_properties(parent)
        for prop in props:
            v = parent.child_get_property(placeholder, prop.name)
            if v != prop.default_value:
                packing_list.append((prop, v))
        
        if not packing_list:
            return
        
        packing_node = self._doc.createElement(tags.XML_TAG_PACKING)
        for prop, value in packing_list:
            prop_node = self._doc.createElement(tags.XML_TAG_PROPERTY)
            prop_name = prop.name.replace('-', '_')
            prop_node.setAttribute(tags.XML_TAG_NAME, prop_name)

            if prop.value_type == gobject.TYPE_ENUM:
                v = enum_to_string(value, prop)
            elif prop.value_type == gobject.TYPE_FLAGS:
                v = flags_to_string(value, prop)
            else:
                v = escape(str(value))

            text = self._doc.createTextNode(v)
            prop_node.appendChild(text)
            packing_node.appendChild(prop_node)

        return packing_node
        
    def _write_property(self, prop):
        value = prop.save()
        
        # None means it doesn't need to be saved, it already contains
        # the default value or shouldn't be saved
        if value == None:
            return
        
        node = self._doc.createElement(tags.XML_TAG_PROPERTY)

        # We should change each '-' by '_' on the name of the property
        # put the name="..." part on the <property ...> tag
        node.setAttribute(tags.XML_TAG_NAME, prop.name.replace('-', '_'))

        # Only write context and comment if translatable is
        # enabled, to mimic glade-2
        if prop.is_translatable():
            node.setAttribute(tags.XML_TAG_TRANSLATABLE, tags.YES)
            if prop.has_i18n_context:
                node.setAttribute(tags.XML_TAG_CONTEXT, tags.YES)
            if prop.i18n_comment:
                node.setAttribute(tags.XML_TAG_COMMENT, prop.i18n_comment)

        text = self._doc.createTextNode(escape(value))
        node.appendChild(text)
        return node
    
