/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gul-tb-button.h"
#include "gul-gobject-misc.h"
#include "gul-gui.h"
#include "galeon-marshal.h"
#include "prefs-strings.h"
#include "eel-gconf-extensions.h"

#include <libgnome/gnome-i18n.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtktoolbar.h>
#include <bonobo/bonobo-ui-toolbar.h>
#include <string.h>

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GulTbButtonPrivate 
{
	gchar *label;
	GtkWidget *image;
	GtkWidget *widget;
	gboolean use_stock;
	gchar *tooltip_text;
	GtkMenu *menu;
	GtkWidget *arrow_widget;
	gboolean sensitive;

	GtkWidget *button;
	GtkWidget *align;
	GtkBox *button_box;
	GtkLabel *label_wid;
	gboolean priority;
	gboolean always_image;
	gboolean show_button;
	gboolean toggle_button;

	gboolean in_bonobo_toobar;

	GtkReliefStyle button_relief;
	GtkOrientation orientation;
	GtkToolbarStyle	style_gtk;
	BonoboUIToolbarStyle style_bonobo;
	GtkIconSize icon_size;	// TODO
	gboolean show_tooltips;
	GtkTooltips *tooltips;
};

/**
 * Private functions, only availble from this file
 */
static void		gul_tb_button_class_init		(GulTbButtonClass *klass);
static void		gul_tb_button_init			(GulTbButton *b);
static void		gul_tb_button_finalize_impl		(GObject *o);
static void		gul_tb_button_build			(GulTbButton *b);
static void		gul_tb_button_parent_set_cb		(GtkWidget *widget, GtkObject *old_parent,
								 GulTbButton *tb);
static void		gul_tb_button_gtk_orientation_changed_cb (GtkToolbar *toolbar, GtkOrientation orientation,
								  GulTbButton *b);
static void		gul_tb_button_gtk_style_changed_cb	(GtkToolbar *toolbar, GtkToolbarStyle style,
								 GulTbButton *b);
static void		gul_tb_button_bonobo_set_orientation_cb	(BonoboUIToolbar *toolbar, GtkOrientation orientation,
								 GulTbButton *b);
static void		gul_tb_button_bonobo_set_style_cb	(BonoboUIToolbar *toolbar, GulTbButton *b);
static gboolean		gul_tb_button_arrow_key_press_event_cb	(GtkWidget *widget, GdkEventKey *event, 
								 GulTbButton *b);
static gboolean		gul_tb_button_arrow_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
								   GulTbButton *b);
static gboolean		gul_tb_button_button_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
								    GulTbButton *b);
static void 		gul_tb_button_button_popup_menu_cb	(GtkWidget *w, GulTbButton *b);
static void 		gul_tb_button_menu_deactivated_cb	(GtkMenuShell *ms, GulTbButton *b);


static gpointer gtk_hbox_class;

enum GulTbButtonSignalsEnum {
	GUL_TB_BUTTON_MENU_ACTIVATED,
	GUL_TB_BUTTON_LAST_SIGNAL
};
static gint GulTbButtonSignals[GUL_TB_BUTTON_LAST_SIGNAL];

/**
 * TbButton object
 */

MAKE_GET_TYPE (gul_tb_button, "GulTbButton", GulTbButton, gul_tb_button_class_init, 
	       gul_tb_button_init, GTK_TYPE_HBOX);

static void
gul_tb_button_class_init (GulTbButtonClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gul_tb_button_finalize_impl;
	gtk_hbox_class = g_type_class_peek_parent (klass);

	GulTbButtonSignals[GUL_TB_BUTTON_MENU_ACTIVATED] = g_signal_new (
		"menu-activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GulTbButtonClass, menu_activated), 
		NULL, NULL, 
		galeon_marshal_VOID__VOID,
		G_TYPE_NONE, 0);
}

static void 
gul_tb_button_init (GulTbButton *tb)
{
	GulTbButtonPrivate *p = g_new0 (GulTbButtonPrivate, 1);
	tb->priv = p;
	p->label = g_strdup ("");
	p->tooltip_text = g_strdup ("");

	p->button_relief = GTK_RELIEF_NORMAL;
	p->orientation = GTK_ORIENTATION_HORIZONTAL;
	p->style_gtk = GTK_TOOLBAR_BOTH_HORIZ;
	p->style_bonobo = BONOBO_UI_TOOLBAR_STYLE_PRIORITY_TEXT;
	p->icon_size =  GTK_ICON_SIZE_LARGE_TOOLBAR;
	p->show_tooltips = TRUE;
	p->show_button = TRUE;

	g_signal_connect (tb, "parent-set", 
			  G_CALLBACK (gul_tb_button_parent_set_cb), tb);
}

GulTbButton *
gul_tb_button_new (void)
{
	GulTbButton *ret = g_object_new (GUL_TYPE_TB_BUTTON, NULL);
	return ret;
}

static void
gul_tb_button_finalize_impl (GObject *o)
{
	GulTbButton *it = GUL_TB_BUTTON (o);
	GulTbButtonPrivate *p = it->priv;

	if (p->image)
	{
		g_object_unref (p->image);
	}

	if (p->widget)
	{
		g_object_unref (p->widget);
	}

	if (p->tooltips)
	{
		g_object_unref (p->tooltips);
	}

	if (p->menu)
	{
		g_object_unref (p->menu);
	}

	if (p->arrow_widget)
	{
		g_object_unref (p->arrow_widget);
	}

	g_free (p);
	
	DEBUG_MSG (("GulTbButton finalized\n"));
	
	G_OBJECT_CLASS (gtk_hbox_class)->finalize (o);
}

void
gul_tb_button_set_label (GulTbButton *b, const gchar *text)
{
	GulTbButtonPrivate *p = b->priv;
	g_free (p->label);
	p->label = g_strdup (text);
	DEBUG_MSG (("GulTbButton label set to '%s'\n", p->label));
	if (!p->label_wid || p->use_stock)
	{
		gul_tb_button_build (b);
	}
	else
	{
		gtk_label_set_text (p->label_wid, p->label);
	}
}

void
gul_tb_button_set_tooltip_text (GulTbButton *b, const gchar *text)
{
	GulTbButtonPrivate *p = b->priv;
	g_free (p->tooltip_text);
	p->tooltip_text = g_strdup (text);

	if (!p->tooltips || !p->button)
	{
		gul_tb_button_build (b);
	}
	else
	{
		gtk_tooltips_set_tip (p->tooltips, p->button,
				      p->tooltip_text, p->tooltip_text);
	}
}

/* this function comes directly from gtktoolbar.c */
static gchar * 
elide_underscores (const gchar *original)
{
	gchar *q, *result;
	const gchar *p;
	gboolean last_underscore;
	
	q = result = g_malloc (strlen (original) + 1);
	last_underscore = FALSE;
	
	for (p = original; *p; p++)
	{
		if (!last_underscore && *p == '_')
		{
			last_underscore = TRUE;
		}
		else
		{
			last_underscore = FALSE;
			*q++ = *p;
		}
	}
	
	*q = '\0';
	
	return result;
}

static void
maybe_reparent (GtkWidget *w, GtkWidget *p)
{
	if (w->parent && w->parent != p)
	{
		gtk_widget_reparent (w, p);
	}
	else if (!w->parent)
	{
		gtk_container_add (GTK_CONTAINER (p), w);
	}
}

static void
gul_tb_button_build (GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	GtkWidget *image;
	GtkStockItem stock_item;
	gboolean really_use_stock = p->use_stock && gtk_stock_lookup (p->label, &stock_item);
	gboolean show_image = p->always_image || p->label[0] == '\0'
		|| (p->in_bonobo_toobar && p->style_bonobo != BONOBO_UI_TOOLBAR_STYLE_TEXT_ONLY)
		|| (!p->in_bonobo_toobar && p->style_gtk != GTK_TOOLBAR_TEXT);
	gboolean show_label = !show_image
		|| (p->priority && ((p->in_bonobo_toobar && p->style_bonobo == BONOBO_UI_TOOLBAR_STYLE_PRIORITY_TEXT)
				 || (!p->in_bonobo_toobar && p->style_gtk == GTK_TOOLBAR_BOTH_HORIZ)))
		|| (!p->in_bonobo_toobar 
		    && (p->style_gtk == GTK_TOOLBAR_BOTH 
			|| p->style_gtk == GTK_TOOLBAR_TEXT
			/* CHECK: what about GTK_TOOLBAR_BOTH_HORIZ? */ ))
		|| (p->in_bonobo_toobar
		    && (p->style_bonobo == BONOBO_UI_TOOLBAR_STYLE_ICONS_AND_TEXT
			|| p->style_bonobo == BONOBO_UI_TOOLBAR_STYLE_TEXT_ONLY));
	
	if (!p->button && p->show_button)
	{
		p->button = (p->toggle_button) ? gtk_toggle_button_new() : gtk_button_new ();
		g_object_ref (p->button);
		gtk_box_pack_start_defaults (GTK_BOX (b), p->button);
	}

	if (!p->align)
	{
		p->align = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
	}
	gtk_widget_show (p->align);

	if (p->button)
	{
		if (p->show_button)
		{
			gtk_widget_show (p->button);
		}
		else
		{
			gtk_widget_hide (p->button);
		}
		gtk_button_set_relief (GTK_BUTTON (p->button), p->button_relief);
		gtk_widget_set_sensitive (p->button, p->sensitive);

		g_signal_connect (p->button, "button_press_event",
				  G_CALLBACK (gul_tb_button_button_button_press_event_cb), b);
		g_signal_connect (p->button, "popup_menu", 
				  G_CALLBACK (gul_tb_button_button_popup_menu_cb), b);
		if (p->tooltips)
		{
			gtk_tooltips_set_tip (p->tooltips, p->button,
					      p->tooltip_text, p->tooltip_text);
		}
	}
	
	if (p->show_button && p->button)
	{
		maybe_reparent (p->align, p->button);
	}
	else
	{
		maybe_reparent (p->align, GTK_WIDGET (b));
	}

	if (p->widget)
	{
		if (p->tooltips)
		{
			gtk_tooltips_set_tip (p->tooltips, p->widget,
					      p->tooltip_text, p->tooltip_text);
		}
	}

	if ((p->in_bonobo_toobar && p->style_bonobo == BONOBO_UI_TOOLBAR_STYLE_ICONS_AND_TEXT)
	    || (!p->in_bonobo_toobar && p->style_gtk == GTK_TOOLBAR_BOTH)
	    || p->orientation == GTK_ORIENTATION_VERTICAL)
	{
		if (p->button_box && !GTK_IS_VBOX (p->button_box))
		{
			gtk_container_remove (GTK_CONTAINER (p->align), GTK_WIDGET (p->button_box));
			p->button_box = NULL;
		}
		if (!p->button_box)
		{
			p->button_box = GTK_BOX (gtk_vbox_new (FALSE, 2));
		}
	}
	else
	{
		if (p->button_box && !GTK_IS_HBOX (p->button_box))
		{
			
			gtk_container_remove (GTK_CONTAINER (p->align), GTK_WIDGET (p->button_box));
			p->button_box = NULL;
		}
		if (!p->button_box)
		{
			p->button_box = GTK_BOX (gtk_hbox_new (FALSE, 2));
		}
	}
	g_object_ref (p->button_box);
	gtk_widget_show (GTK_WIDGET (p->button_box));
	maybe_reparent (GTK_WIDGET (p->button_box), p->align);
	
	if (p->widget)
	{
		gtk_widget_show (p->widget);
		maybe_reparent (p->widget, GTK_WIDGET (p->button_box));
	}

	if (!p->image && really_use_stock && show_image)
	{
		image = gtk_image_new_from_stock (p->label, p->icon_size);
		if (image)
		{
			p->image = g_object_ref (image);
			gtk_object_sink (GTK_OBJECT (image));
		}
	}
	else
	{
		image = p->image;
	}

	if (image)
	{
		if (show_image)
		{
			maybe_reparent (image, GTK_WIDGET (p->button_box));
			gtk_widget_show (image);
		}
	}
	else if (!p->widget)
	{
		show_label = TRUE;
	}

	if (p->label_wid && GTK_WIDGET (p->label_wid)->parent == GTK_WIDGET (p->button_box))
	{
		gtk_container_remove (GTK_CONTAINER (p->button_box), GTK_WIDGET (p->label_wid));
		}
	if (p->label_wid) 
	{
		g_object_unref (p->label_wid);
		p->label_wid = NULL;
	}

	if (show_label)
	{
		if (p->label_wid && GTK_WIDGET (p->label_wid)->parent == GTK_WIDGET (p->button_box))
		{
			gtk_container_remove (GTK_CONTAINER (p->button_box), GTK_WIDGET (p->label_wid));
		}
		if (p->label_wid) 
		{
			g_object_unref (p->label_wid);
		}
		p->label_wid = GTK_LABEL (gtk_label_new (p->label));
		g_object_ref (p->label_wid);
		
		if (really_use_stock)
		{
			gchar *l = elide_underscores (stock_item.label);
			gtk_label_set_text (p->label_wid, l);
			g_free (l);
		}
		
		gtk_widget_show (GTK_WIDGET (p->label_wid));
		gtk_box_pack_end_defaults (p->button_box, GTK_WIDGET (p->label_wid));
	}

	DEBUG_MSG (("GulTbButton built, label='%s'\n", p->label));
}

void
gul_tb_button_set_priority (GulTbButton *b, gboolean priority)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->priority != priority)
	{
		p->priority = priority;
		gul_tb_button_build (b);
	}
}

void
gul_tb_button_set_always_image (GulTbButton *b, gboolean value)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->always_image != value)
	{
		p->always_image = value;
		gul_tb_button_build (b);
	}
}

void
gul_tb_button_set_image (GulTbButton *b, GtkWidget *image)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->image)
	{
		g_object_unref (p->image);
	}
	p->image = image ? g_object_ref (image) : NULL;
	gul_tb_button_build (b);
}

void
gul_tb_button_set_widget (GulTbButton *b, GtkWidget *widget)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->widget)
	{
		g_object_unref (p->widget);
	}
	p->widget = widget ? g_object_ref (widget) : NULL;
	gul_tb_button_build (b);
}

void
gul_tb_button_set_show_arrow (GulTbButton *b, gboolean value)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->arrow_widget && !value)
	{
		if (p->arrow_widget->parent == GTK_WIDGET (b))
		{
			gtk_container_remove (GTK_CONTAINER (b), p->arrow_widget);
		}
		g_object_unref (p->arrow_widget);
		p->arrow_widget = NULL;
	}
	else if (!p->arrow_widget && value)
	{
		p->arrow_widget = gtk_toggle_button_new ();
		gtk_button_set_relief (GTK_BUTTON (p->arrow_widget), GTK_RELIEF_NONE);
		
		gtk_container_add (GTK_CONTAINER (p->arrow_widget),
				   gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT));
		
		g_object_ref (p->arrow_widget);
		gtk_object_sink (GTK_OBJECT (p->arrow_widget));

		g_signal_connect (p->arrow_widget, "key_press_event",
				  G_CALLBACK (gul_tb_button_arrow_key_press_event_cb),
				  b);

		g_signal_connect  (p->arrow_widget, "button_press_event",
				   G_CALLBACK (gul_tb_button_arrow_button_press_event_cb),
				   b);

		gtk_widget_show_all (p->arrow_widget);
		gtk_box_pack_end_defaults (GTK_BOX (b), p->arrow_widget);
		gtk_widget_set_sensitive (p->arrow_widget, value);
	}
}

void
gul_tb_button_set_enable_menu (GulTbButton *b, gboolean value)
{
	GulTbButtonPrivate *p = b->priv;
	if (value && !p->menu)
	{
		p->menu = GTK_MENU (gtk_menu_new ());
		g_object_ref (p->menu);
		gtk_object_sink (GTK_OBJECT (p->menu));
		g_signal_connect (p->menu, "deactivate", 
				  G_CALLBACK (gul_tb_button_menu_deactivated_cb), b);
	}
	else if (!value && p->menu)
	{
		g_object_unref (p->menu);
		p->menu = FALSE;
	}

}

GtkMenuShell *
gul_tb_button_get_menu (GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	return p->menu ? GTK_MENU_SHELL (p->menu) : NULL;
}

GtkButton *
gul_tb_button_get_button (GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	if (!p->button)
	{
		gul_tb_button_build (b);
	}
	return GTK_BUTTON (p->button);
}

void
gul_tb_button_set_use_toggle_button (GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	p->toggle_button = TRUE;

	if (p->button && ! GTK_IS_TOGGLE_BUTTON (p->button))
	{
		g_object_unref (p->button);
		p->button = 0;
	}
	if ( !p->button )
	{
		gul_tb_button_build (b);
	}
}


void
gul_tb_button_set_use_stock (GulTbButton *b, gboolean value)
{
	GulTbButtonPrivate *p = b->priv;
	if (value != p->use_stock)
	{
		p->use_stock = value;
		gul_tb_button_build (b);
	}
}


void
gul_tb_button_set_show_button (GulTbButton *b, gboolean value)
{
	GulTbButtonPrivate *p = b->priv;
	if (value != p->show_button)
	{
		p->show_button = value;
		gul_tb_button_build (b);
	}
}

static void
gul_tb_button_parent_set_cb (GtkWidget *widget, GtkObject *old_parent, GulTbButton *tb)
{
	GulTbButtonPrivate *p = tb->priv;
	GtkWidget *new_parent = widget->parent;

	DEBUG_MSG (("GulTbButton parent changed (widget=%p, button=%p, old=%p, new=%p)\n", 
		    widget, tb, old_parent, new_parent));

	if (new_parent)
	{
		GtkToolbar *gtktb = NULL;
		BonoboUIToolbar *btb = NULL;
		while (new_parent && !gtktb && !btb)
		{
			DEBUG_MSG (("new_parent ia a %s\n", g_type_name_from_instance ((void *) new_parent)));

			if (GTK_IS_TOOLBAR (new_parent))
			{
				gtktb = GTK_TOOLBAR (new_parent);
			}
			else if (BONOBO_IS_UI_TOOLBAR (new_parent))
			{
				btb = BONOBO_UI_TOOLBAR (new_parent);
			}
			else
			{
				g_signal_connect (new_parent, "parent_set", 
						  G_CALLBACK (gul_tb_button_parent_set_cb), tb);
			}
			new_parent = new_parent->parent;
		}

		if (gtktb)
		{
			DEBUG_MSG (("GulTbButton getting style from a GtkToolbar (%p)\n", gtktb));
			p->in_bonobo_toobar = FALSE;

			gtk_widget_ensure_style (GTK_WIDGET (gtktb));
			gtk_widget_style_get (GTK_WIDGET (gtktb), "button_relief", &p->button_relief, NULL);

			p->orientation = gtk_toolbar_get_orientation (gtktb);
			p->style_gtk = gtk_toolbar_get_style (gtktb);
			p->icon_size = gtk_toolbar_get_icon_size (gtktb);
			p->show_tooltips = gtk_toolbar_get_tooltips (gtktb);

			if (p->tooltips)
			{
				g_object_unref (p->tooltips);
			}
			p->tooltips = gtk_tooltips_new ();
			if (p->show_tooltips)
			{
				gtk_tooltips_enable (p->tooltips);
			}
			else
			{
				gtk_tooltips_disable (p->tooltips);
			}
			g_object_ref (p->tooltips);
			gtk_object_sink (GTK_OBJECT (p->tooltips));

			g_signal_connect (gtktb, "orientation-changed", 
					  G_CALLBACK (gul_tb_button_gtk_orientation_changed_cb), tb);
			g_signal_connect (gtktb, "style-changed", 
					  G_CALLBACK (gul_tb_button_gtk_style_changed_cb), tb);

			gul_tb_button_build (tb);
		}

		if (btb)
		{
			DEBUG_MSG (("GulTbButton getting style from a BonoboUIToolbar (%p)\n", btb));
			p->in_bonobo_toobar = TRUE;

			p->button_relief = GTK_RELIEF_NONE;

			p->orientation = bonobo_ui_toolbar_get_orientation (btb);
			p->style_bonobo = bonobo_ui_toolbar_get_style (btb);
			//p->icon_size = ???;
			p->show_tooltips = TRUE;

			if (p->tooltips)
			{
				g_object_unref (p->tooltips);
			}
			p->tooltips = bonobo_ui_toolbar_get_tooltips (btb);
			g_object_ref (p->tooltips);

			g_signal_connect (btb, "set-orientation",
					  G_CALLBACK (gul_tb_button_bonobo_set_orientation_cb), tb);
			g_signal_connect (btb, "set-style",
					  G_CALLBACK (gul_tb_button_bonobo_set_style_cb), tb);

			gul_tb_button_build (tb);
		}
	}
	else
	{
		while (old_parent)
		{
			g_signal_handlers_disconnect_matched (old_parent, G_SIGNAL_MATCH_DATA, 
							      0, 0, NULL, NULL, tb);
			if (GTK_IS_WIDGET (old_parent))
			{
				old_parent = GTK_WIDGET (old_parent)->parent 
					? GTK_OBJECT (GTK_WIDGET (old_parent)->parent)
					: NULL;
			}
			else
			{
				old_parent = NULL;
			}
		}
	}
}

static void 
gul_tb_button_gtk_orientation_changed_cb (GtkToolbar *toolbar, GtkOrientation orientation, GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->orientation != orientation)
	{
		p->orientation = orientation;
		gul_tb_button_build (b);
	}
}

static void 
gul_tb_button_gtk_style_changed_cb (GtkToolbar *toolbar, GtkToolbarStyle style, GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->style_gtk != style)
	{
		p->style_gtk = style;
		gul_tb_button_build (b);
	}
}

static void 
gul_tb_button_bonobo_set_orientation_cb (BonoboUIToolbar *toolbar, GtkOrientation orientation, GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->orientation != orientation)
	{
		p->orientation = orientation;
		gul_tb_button_build (b);
	}
}

static void 
gul_tb_button_bonobo_set_style_cb (BonoboUIToolbar *toolbar, GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	BonoboUIToolbarStyle style = bonobo_ui_toolbar_get_style (toolbar);
	if (style != p->style_bonobo)
	{
		p->style_bonobo = style;
		gul_tb_button_build (b);
	}
}

static void 
gul_tb_button_popup_menu_under_arrow (GulTbButton *b, GdkEventButton *event)
{
	GulTbButtonPrivate *p = b->priv;

	if (p->menu)
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p->arrow_widget), TRUE);
		g_signal_emit (b, GulTbButtonSignals[GUL_TB_BUTTON_MENU_ACTIVATED], 0);
		gtk_menu_popup (p->menu, NULL, NULL, gul_gui_menu_position_under_widget, p->arrow_widget, 
				event ? event->button : 0, 
				event ? event->time : gtk_get_current_event_time ());
	}
}

static void 
gul_tb_button_menu_deactivated_cb (GtkMenuShell *ms, GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;
	if (p->arrow_widget)
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p->arrow_widget), FALSE);
	}
}

static gboolean
gul_tb_button_arrow_button_press_event_cb  (GtkWidget *widget, GdkEventButton *event, GulTbButton *b)
{
	gul_tb_button_popup_menu_under_arrow (b, event);
	return TRUE;
}

static gboolean
gul_tb_button_arrow_key_press_event_cb (GtkWidget *widget, GdkEventKey *event, GulTbButton *b)
{
	if (event->keyval == GDK_space
	    || event->keyval == GDK_KP_Space
	    || event->keyval == GDK_Return
	    || event->keyval == GDK_KP_Enter
	    || event->keyval == GDK_Menu)
	{
		gul_tb_button_popup_menu_under_arrow (b, NULL);
	}

	return FALSE;
}

static gboolean
gul_tb_button_button_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
					    GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;

	if (event->button == 3 && p->menu)
	{
		g_signal_emit (b, GulTbButtonSignals[GUL_TB_BUTTON_MENU_ACTIVATED], 0);
		gtk_menu_popup (p->menu, NULL, NULL, NULL, b, 
				event ? event->button : 0, 
				event ? event->time : gtk_get_current_event_time ());
		return TRUE;
	}
	
	return FALSE;
}

static void 
gul_tb_button_button_popup_menu_cb (GtkWidget *w, GulTbButton *b)
{
	GulTbButtonPrivate *p = b->priv;

	if (p->menu)
	{
		g_signal_emit (b, GulTbButtonSignals[GUL_TB_BUTTON_MENU_ACTIVATED], 0);
		gtk_menu_popup (p->menu, NULL, NULL, 
				gul_gui_menu_position_under_widget, b, 0, gtk_get_current_event_time ());
	}
}

void
gul_tb_button_set_sensitivity (GulTbButton *b, gboolean value)
{
	GulTbButtonPrivate *p = b->priv;

	p->sensitive = value;

	if (!p->button)
	{
		gul_tb_button_build (b);
	}
	else
	{
		gtk_widget_set_sensitive (p->button, value);
		if (p->arrow_widget)
		{
			gtk_widget_set_sensitive (p->arrow_widget, value);
		}
	}
}


