/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "gul-glade.h"
#include "bookmarks-single-editor.h"
#include "gul-gobject-misc.h"
#include "gul-string.h"
#include "eel-gconf-extensions.h"
#include "prefs-strings.h"
#include <gtk/gtkspinbutton.h>
#include <libgnomeui/gnome-pixmap-entry.h>

/**
 * Private data
 */
struct _GbSingleEditorPrivate {
	GbBookmark *bm;

	guint no_update;

	GtkWidget *window;
	GtkWidget *name_label;
	GtkWidget *name_entry;
	GtkWidget *smarturl_label;
	GtkWidget *smarturl_entry;
	GtkWidget *date_added_label;
	GtkWidget *date_added_value_label;
	GtkWidget *date_modified_label;
	GtkWidget *date_modified_value_label;
	GtkWidget *date_visited_label;
	GtkWidget *date_visited_value_label;
	GtkWidget *smart_entry_width_label;
	GtkWidget *smart_entry_width_spinbutton;
	GtkWidget *parameter_encoding_label;
	GtkWidget *parameter_encoding_combo;
	GtkWidget *location_label;
	GtkWidget *location_entry;
	GtkWidget *notes_label;
	GtkWidget *notes_textview;
	GtkWidget *nicks_label;
	GtkWidget *nicks_entry;
	GtkWidget *create_toolbar_checkbox;
	GtkWidget *add_to_context_checkbox;
	GtkWidget *image_label;
	GtkWidget *image_pixmapentry;

};

/**
 * Private functions, only availble from this file
 */
static void		gb_single_editor_class_init		(GbSingleEditorClass *klass);
static void		gb_single_editor_init			(GbSingleEditor *e);
static void		gb_single_editor_init_widgets		(GbSingleEditor *e);
static void		gb_single_editor_bookmark_changed_cb	(GbBookmark *b, GbSingleEditor *e);
static void		gb_single_editor_yield_close		(GObject *o);
static void		gb_single_editor_finalize_impl		(GObject *o);
static void		gb_single_editor_update_controls	(GbSingleEditor *e);


static void		gb_single_editor_name_entry_changed_cb	(GtkWidget *edited,
								 GbSingleEditor *e);
static void		gb_single_editor_smarturl_entry_changed_cb (GtkWidget *edited,
								    GbSingleEditor *e);
static void		gb_single_editor_smart_entry_width_spinbutton_value_changed_cb (GtkWidget *edited, 
											GbSingleEditor *e);
static void		gb_single_editor_location_entry_changed_cb (GtkWidget *edited,
								    GbSingleEditor *e);
static void		gb_single_editor_notes_text_buffer_changed_cb (GtkWidget *edited,
								       GbSingleEditor *e);
static void		gb_single_editor_nicks_entry_changed_cb	(GtkWidget *edited,
								 GbSingleEditor *e);
static void		gb_single_editor_create_toolbar_checkbox_toggled_cb (GtkWidget *edited,
									     GbSingleEditor *e);
static void		gb_single_editor_add_to_context_checkbox_toggled_cb (GtkWidget *edited,
									     GbSingleEditor *e);
static void 		gb_single_editor_image_pixmapentry_changed_cb (GtkWidget *edited,
								       GbSingleEditor *e);
static void		gb_single_editor_parameter_encoding_combo_changed_cb (GtkWidget *edited,
									      GbSingleEditor *e);


static gpointer g_object_class;

/**
 * SingleEditor object
 */

MAKE_GET_TYPE (gb_single_editor, "GbSingleEditor", GbSingleEditor, gb_single_editor_class_init, 
	       gb_single_editor_init, G_TYPE_OBJECT);

static void
gb_single_editor_yield_close (GObject *o)
{
	/* need to flush all events before closing the editor */

	while (gtk_events_pending())
		gtk_main_iteration();
	g_object_unref(o);
	return;
}

static void
gb_single_editor_class_init (GbSingleEditorClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gb_single_editor_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);
}

static void 
gb_single_editor_init (GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = g_new0 (GbSingleEditorPrivate, 1);
	e->priv = p;
	
	gb_single_editor_init_widgets (e);
	
}

static void
gb_single_editor_finalize_impl (GObject *o)
{
	GbSingleEditor *e = GB_SINGLE_EDITOR (o);
	GbSingleEditorPrivate *p = e->priv;

	gb_single_editor_set_bookmark (e, NULL);
	
	g_free (p);
	
	G_OBJECT_CLASS (g_object_class)->finalize (o);

}

static void
gb_single_editor_init_widgets (GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	WidgetLookup lookups[] = {
		{ "bookmarks_properties_window", &p->window },
		{ "bookmarks_properties_name_label", &p->name_label },
		{ "bookmarks_properties_name_entry", &p->name_entry },
		{ "bookmarks_properties_smarturl_label", &p->smarturl_label },
		{ "bookmarks_properties_smarturl_entry", &p->smarturl_entry },
		{ "bookmarks_properties_date_added_label", &p->date_added_label },
		{ "bookmarks_properties_date_added_value_label", &p->date_added_value_label },
		{ "bookmarks_properties_date_modified_label", &p->date_modified_label },
		{ "bookmarks_properties_date_modified_value_label", &p->date_modified_value_label },
		{ "bookmarks_properties_date_visited_label", &p->date_visited_label },
		{ "bookmarks_properties_date_visited_value_label", &p->date_visited_value_label },
		{ "bookmarks_properties_smart_entry_width_label", &p->smart_entry_width_label },
		{ "bookmarks_properties_smart_entry_width_spinbutton", &p->smart_entry_width_spinbutton },
		{ "bookmarks_properties_parameter_encoding_label", &p->parameter_encoding_label },
		{ "bookmarks_properties_parameter_encoding_combo", &p->parameter_encoding_combo },
		{ "bookmarks_properties_location_label", &p->location_label },
		{ "bookmarks_properties_location_entry", &p->location_entry },
		{ "bookmarks_properties_notes_label", &p->notes_label },
		{ "bookmarks_properties_notes_textview", &p->notes_textview },
		{ "bookmarks_properties_nicks_label", &p->nicks_label },
		{ "bookmarks_properties_nicks_entry", &p->nicks_entry },
		{ "bookmarks_properties_create_toolbar_checkbox", &p->create_toolbar_checkbox },
		{ "bookmarks_properties_add_to_context_checkbox", &p->add_to_context_checkbox },
		{ "bookmarks_properties_image_label", &p->image_label },
		{ "bookmarks_properties_image_pixmapentry", &p->image_pixmapentry },
		{ NULL, NULL }
	};
	GladeXML *gxml = gul_glade_widget_new ("bookmarks-editor.glade", "bookmarks_properties_window",
					       NULL, e);
	GtkTextBuffer *buffer;
	gul_glade_lookup_widgets (gxml, lookups);
	g_object_unref (gxml);

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (p->notes_textview));

	g_signal_connect_swapped (p->window, "delete_event",
				  G_CALLBACK(gb_single_editor_yield_close), e);
	g_signal_connect (p->name_entry, "changed",
			  G_CALLBACK (gb_single_editor_name_entry_changed_cb), e);
	g_signal_connect (p->smarturl_entry, "changed",
			  G_CALLBACK (gb_single_editor_smarturl_entry_changed_cb), e);
	g_signal_connect (p->smart_entry_width_spinbutton, "value-changed",
			  G_CALLBACK (gb_single_editor_smart_entry_width_spinbutton_value_changed_cb), e);
	g_signal_connect (p->location_entry, "changed",
			  G_CALLBACK (gb_single_editor_location_entry_changed_cb), e);
	g_signal_connect (buffer, "changed",
			  G_CALLBACK (gb_single_editor_notes_text_buffer_changed_cb), e);
	g_signal_connect (p->nicks_entry, "changed", 
			  G_CALLBACK (gb_single_editor_nicks_entry_changed_cb), e);
	g_signal_connect (p->create_toolbar_checkbox, "toggled",
			  G_CALLBACK (gb_single_editor_create_toolbar_checkbox_toggled_cb), e);
	g_signal_connect (p->add_to_context_checkbox, "toggled",
			  G_CALLBACK (gb_single_editor_add_to_context_checkbox_toggled_cb), e);
	g_signal_connect (p->image_pixmapentry, "changed",
			  G_CALLBACK (gb_single_editor_image_pixmapentry_changed_cb), e);
	g_signal_connect (GTK_COMBO (p->parameter_encoding_combo)->entry, "changed",
			  G_CALLBACK (gb_single_editor_parameter_encoding_combo_changed_cb), e);
}

GbSingleEditor *
gb_single_editor_new (void)
{
	GbSingleEditor *ret = g_object_new (GB_TYPE_SINGLE_EDITOR, NULL);
	return ret;
}

void
gb_single_editor_show (GbSingleEditor *editor)
{
	gtk_widget_show (GTK_WIDGET (editor->priv->window));
}

void
gb_single_editor_set_bookmark (GbSingleEditor *e, GbBookmark *b)
{

	if (e->priv->bm)
	{
		g_signal_handlers_disconnect_matched (e->priv->bm, G_SIGNAL_MATCH_DATA, 0, 0, 
						      NULL, NULL, e);
		g_object_unref (e->priv->bm);
	}

	e->priv->bm = b;

	if (b)
	{
		g_object_ref (b);
		g_signal_connect (b, "modified", G_CALLBACK (gb_single_editor_bookmark_changed_cb), e);
		gb_single_editor_update_controls (e);
	}

}

static void
gb_single_editor_update_controls (GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	GtkTextBuffer *buffer;
	gchar *s;

	p->no_update++;

	gtk_entry_set_text (GTK_ENTRY (p->name_entry), p->bm->name);
	gtk_entry_set_text (GTK_ENTRY (p->nicks_entry), p->bm->nick);

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (p->notes_textview));
	gtk_text_buffer_set_text (buffer, p->bm->notes, -1);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p->add_to_context_checkbox), 
				      p->bm->add_to_context_menu);

	s = gul_string_time_to_string (p->bm->time_added);
	gtk_label_set_text (GTK_LABEL (p->date_added_value_label), s);
	g_free (s);

	s = gul_string_time_to_string (p->bm->time_modified);
	gtk_label_set_text (GTK_LABEL (p->date_modified_value_label), s);
	g_free (s);

	gnome_file_entry_set_filename (GNOME_FILE_ENTRY (p->image_pixmapentry), 
				       p->bm->pixmap_file);
	
	if (GB_IS_SITE (p->bm))
	{
		gtk_entry_set_text (GTK_ENTRY (p->location_entry), GB_SITE (p->bm)->url);
		gtk_widget_show (p->location_entry);
		gtk_widget_show (p->location_label);

		s = gul_string_time_to_string (GB_SITE (p->bm)->time_visited);
		gtk_label_set_text (GTK_LABEL (p->date_visited_value_label), s);
		g_free (s);
	}
	else
	{
		gtk_widget_hide (p->location_entry);
		gtk_widget_hide (p->location_label);
		gtk_widget_hide (p->date_visited_value_label);
		gtk_widget_hide (p->date_visited_label);
	}

	if (GB_IS_SMART_SITE (p->bm))
	{
		guint width = gb_smart_site_get_entry_size (GB_SMART_SITE (p->bm), 0);
		gchar *encoding = gb_smart_site_get_encoding (GB_SMART_SITE (p->bm));
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (p->smart_entry_width_spinbutton), width);
		gtk_widget_show (p->smart_entry_width_spinbutton);
		gtk_widget_show (p->smart_entry_width_label);

		gtk_entry_set_text (GTK_ENTRY (p->smarturl_entry), GB_SMART_SITE (p->bm)->smarturl);
		gtk_widget_show (p->smarturl_entry);
		gtk_widget_show (p->smarturl_label);

		gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (p->parameter_encoding_combo)->entry), encoding);
		gtk_widget_show (p->parameter_encoding_combo);
		gtk_widget_show (p->parameter_encoding_label);

		g_free (encoding);
	}
	else
	{
		gtk_widget_hide (p->smart_entry_width_spinbutton);
		gtk_widget_hide (p->smart_entry_width_label);
		gtk_widget_hide (p->smarturl_entry);
		gtk_widget_hide (p->smarturl_label);
		gtk_widget_hide (p->parameter_encoding_combo);
		gtk_widget_hide (p->parameter_encoding_label);
	}

	if (GB_IS_FOLDER (p->bm))
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p->create_toolbar_checkbox), 
					      GB_FOLDER (p->bm)->create_toolbar);
		gtk_widget_show (p->create_toolbar_checkbox);
	}
	else
	{
		gtk_widget_hide (p->create_toolbar_checkbox);
	}

	p->no_update--;

}

static void
gb_single_editor_bookmark_changed_cb (GbBookmark *b, GbSingleEditor *e)
{
	g_return_if_fail (e->priv->bm == b);
	
	gb_single_editor_update_controls (e);
}

static void
gb_single_editor_name_entry_changed_cb	(GtkWidget *edited,
					 GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gchar *s;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_BOOKMARK (p->bm));

	s = gtk_editable_get_chars (GTK_EDITABLE (p->name_entry), 0, -1);
	gb_bookmark_set_name (p->bm, s);
	g_free (s);
}

static void
gb_single_editor_smarturl_entry_changed_cb (GtkWidget *edited,
					    GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gchar *s;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_SMART_SITE (p->bm));

	s = gtk_editable_get_chars (GTK_EDITABLE (p->smarturl_entry), 0, -1);
	gb_smart_site_set_smarturl (GB_SMART_SITE (p->bm), s);
	g_free (s);
}

static void
gb_single_editor_location_entry_changed_cb (GtkWidget *edited,
					    GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gchar *s;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_SITE (p->bm));

	s = gtk_editable_get_chars (GTK_EDITABLE (p->location_entry), 0, -1);
	gb_site_set_url (GB_SITE (p->bm), s);
	g_free (s);
}

static void
gb_single_editor_smart_entry_width_spinbutton_value_changed_cb (GtkWidget *spinbutton, GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	guint width;

	if (p->no_update > 0) return;

	g_return_if_fail (GB_IS_SMART_SITE (p->bm));

	width = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (p->smart_entry_width_spinbutton));

	gb_smart_site_set_entry_size (GB_SMART_SITE (p->bm), 0, width);
}

static void
gb_single_editor_notes_text_buffer_changed_cb (GtkWidget *edited,
					       GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	GtkTextBuffer *b;
	GtkTextIter begin, end;
	gchar *notes;

	if (p->no_update > 0) return;

	g_return_if_fail (GB_IS_BOOKMARK (p->bm));

	b = gtk_text_view_get_buffer (GTK_TEXT_VIEW (p->notes_textview));
	gtk_text_buffer_get_start_iter (b, &begin);
	gtk_text_buffer_get_end_iter (b, &end);
	notes = gtk_text_buffer_get_text (b, &begin, &end, TRUE);
	gb_bookmark_set_notes (p->bm, notes);
	g_free (notes);

}

static void
gb_single_editor_nicks_entry_changed_cb	(GtkWidget *edited,
					 GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gchar *s;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_BOOKMARK (p->bm));

	s = gtk_editable_get_chars (GTK_EDITABLE (p->nicks_entry), 0, -1);
	gb_bookmark_set_nick (p->bm, s);
	g_free (s);

}

static void
gb_single_editor_create_toolbar_checkbox_toggled_cb (GtkWidget *edited,
						     GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gboolean val;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_FOLDER (p->bm));

	val = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (p->create_toolbar_checkbox));
	gb_folder_set_create_toolbar (GB_FOLDER (p->bm), val);

}

static void
gb_single_editor_add_to_context_checkbox_toggled_cb (GtkWidget *edited,
						     GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gboolean val;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_BOOKMARK (p->bm));

	val = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (p->add_to_context_checkbox));
	gb_bookmark_set_add_to_context_menu (p->bm, val);

}

static void
gb_single_editor_image_pixmapentry_changed_cb (GtkWidget *edited,
					       GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gchar *s;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_BOOKMARK (p->bm));

	s = gtk_editable_get_chars (GTK_EDITABLE (p->image_pixmapentry), 0, -1);
	gb_bookmark_set_pixmap (p->bm, s);
	g_free (s);

}

static void
gb_single_editor_parameter_encoding_combo_changed_cb (GtkWidget *edited,
						      GbSingleEditor *e)
{
	GbSingleEditorPrivate *p = e->priv;
	gchar *s;

	if (p->no_update > 0) return;
	
	g_return_if_fail (GB_IS_SMART_SITE (p->bm));

	s = gtk_editable_get_chars (GTK_EDITABLE (GTK_COMBO (p->parameter_encoding_combo)->entry), 0, -1);
	gb_smart_site_set_encoding (GB_SMART_SITE (p->bm), s);
	g_free (s);
}

