/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

#include <libgnome/gnome-i18n.h>
#include "bookmarks-bonoboui-tb.h"
#include "bookmarks-tb-widget.h"
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-bonobo-extensions.h"
#include "gul-string.h"

#define REBUILD_TIMEOUT 200

/**
 * Private data
 */
struct _GbBonoboUIToolbarPrivate {
	GbBookmarkSet *set;
	BonoboUIComponent *uic;
	gboolean visible;
	GHashTable *bookmark_to_buiti;
	GHashTable *bookmark_to_main_buiti;
	guint rebuid_timeout;

	GbLocationSource *location_source;
};

typedef struct _BuitItem BuitItem;
struct _BuitItem {
	GbBonoboUIToolbar *buit;
	GbBookmark *bm;
	GbTbWidget *widget;
	gchar *path;
	gboolean is_main : 1;
	gboolean dirty : 1;
};

/**
 * Private functions, only availble from this file
 */
static void		gb_bonobo_ui_toolbar_class_init	(GbBonoboUIToolbarClass *klass);
static void		gb_bonobo_ui_toolbar_init		(GbBonoboUIToolbar *e);
static void		gb_bonobo_ui_toolbar_finalize_impl (GObject *o);

static BuitItem *	buit_item_new			(GbBonoboUIToolbar *buit, GbBookmark *b,
							 gboolean is_main);
static void		buit_item_destroy		(BuitItem *bit, gboolean remove_from_hashtable);
static void		buit_item_destroy_wrapper	(gpointer key, gpointer value,
							 gpointer user_data);
static void		gb_bonobo_ui_toolbar_build	(GbBonoboUIToolbar *buit, GbFolder *f);
static void		gb_bonobo_ui_toolbar_rebuild	(GbBonoboUIToolbar *buit, BuitItem *buiti);
static void		gb_bonobo_ui_toolbar_destroy	(GbBonoboUIToolbar *buit, GbFolder *f);
static void		gb_bonobo_ui_toolbar_build_item (GbBonoboUIToolbar *buit, GbBookmark *item, 
							 gint index, gchar *path);
static void		gb_bonobo_ui_toolbar_bookmark_activated_cb (GtkWidget *w, 
								    GbBookmarkEventActivated *ev,
								    GbBonoboUIToolbar *buit);
static void		gb_bonobo_ui_toolbar_toolbar_cb (GbBookmarkSet *set, 
							 GbFolder *f, GbBonoboUIToolbar *buit);
static void		gb_bonobo_ui_toolbar_child_added_cb 	(GbFolder *f, GbBookmark *c,
								 guint pos, BuitItem *buiti);
static void		gb_bonobo_ui_toolbar_child_removed_cb 	(GbFolder *f, GbBookmark *c,
								 guint pos, BuitItem *buiti);
static void		gb_bonobo_ui_toolbar_widget_destroy_cb	(GtkWidget *w, BuitItem *buiti);
static gboolean		gb_bonobo_ui_toolbar_rebuild_timeout_cb (gpointer data);
static void		gb_bonobo_ui_toolbar_rebuild_with_timeout (BuitItem *buiti);
static void 		buit_item_rebuild			(gpointer key,
								 gpointer value,
								 gpointer user_data);
static void		gb_bonobo_ui_toolbar_set_location_source_buiti (gpointer key, 
									gpointer item, 
									gpointer data);
static void		gb_bonobo_ui_toolbar_hash_to_slist 	(gpointer key, gpointer data, 
								 gpointer user_data);


static gpointer g_object_class;

enum GbBonoboUIToolbarSignalsEnum {
	GB_BONOBO_UI_TOOLBAR_BOOKMARK_ACTIVATED,
	GB_BONOBO_UI_TOOLBAR_LAST_SIGNAL
};
static gint GbBonoboUIToolbarSignals[GB_BONOBO_UI_TOOLBAR_LAST_SIGNAL];

/**
 * BonoboUIToolbar object
 */

MAKE_GET_TYPE (gb_bonobo_ui_toolbar, "GbBonoboUIToolbar", GbBonoboUIToolbar, gb_bonobo_ui_toolbar_class_init,
	       gb_bonobo_ui_toolbar_init, G_TYPE_OBJECT);

static void
gb_bonobo_ui_toolbar_class_init (GbBonoboUIToolbarClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = gb_bonobo_ui_toolbar_finalize_impl;
	g_object_class = g_type_class_peek_parent (klass);

	GbBonoboUIToolbarSignals[GB_BONOBO_UI_TOOLBAR_BOOKMARK_ACTIVATED] = g_signal_new (
		"bookmark-activated", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GbBonoboUIToolbarClass, gb_bonobo_ui_toolbar_bookmark_activated), 
		NULL, NULL, 
		g_cclosure_marshal_VOID__POINTER,
		G_TYPE_NONE, 1, G_TYPE_POINTER);
}

static void 
gb_bonobo_ui_toolbar_init (GbBonoboUIToolbar *m)
{
	GbBonoboUIToolbarPrivate *p = g_new0 (GbBonoboUIToolbarPrivate, 1);

	m->priv = p;
	m->priv->visible = TRUE;
}

static void
gb_bonobo_ui_toolbar_finalize_impl (GObject *o)
{
	GbBonoboUIToolbar *buit = GB_BONOBO_UI_TOOLBAR (o);
	GbBonoboUIToolbarPrivate *p = buit->priv;
	GSList *l = NULL;
	GSList *li;

	gb_bonobo_ui_toolbar_set_location_source (buit, NULL);

	g_object_unref (G_OBJECT (p->uic));

	if (p->rebuid_timeout)
	{
		g_source_remove (p->rebuid_timeout);
		p->rebuid_timeout = 0;
	}

	g_hash_table_foreach (p->bookmark_to_buiti, gb_bonobo_ui_toolbar_hash_to_slist, &l);
	for (li = l; li; li = li->next)
	{
		BuitItem *buiti = li->data;
		g_object_unref (buiti->widget);
	}
	g_slist_free (l);

	if (g_hash_table_size (p->bookmark_to_buiti) > 0)
	{
		g_warning ("There are leaks in GbBonoboUIToolbar: %d items\n", 
			   g_hash_table_size (p->bookmark_to_buiti));
	}

	g_hash_table_foreach (p->bookmark_to_main_buiti, buit_item_destroy_wrapper, buit);
	g_hash_table_destroy (p->bookmark_to_buiti);
	g_hash_table_destroy (p->bookmark_to_main_buiti);

	g_signal_handlers_disconnect_matched (p->set, G_SIGNAL_MATCH_DATA, 0, 0, 
					      NULL, NULL, buit);

	g_object_unref (G_OBJECT (p->set));
	g_free (p);

	G_OBJECT_CLASS (g_object_class)->finalize (o);
}

GbBonoboUIToolbar *
gb_bonobo_ui_toolbar_new (GbBookmarkSet *set, BonoboUIComponent *uic)
{
	GbBonoboUIToolbar *ret = g_object_new (GB_TYPE_BONOBO_UI_TOOLBAR, NULL);
	GbBonoboUIToolbarPrivate *p = ret->priv;

	p->set = set;
	g_object_ref (G_OBJECT (set));

	p->uic = uic;
	g_object_ref (G_OBJECT (uic));

	p->bookmark_to_buiti = g_hash_table_new (NULL, NULL);
	p->bookmark_to_main_buiti = g_hash_table_new (NULL, NULL);
	
	gb_bonobo_ui_toolbar_set_visible (ret, TRUE);
	p->rebuid_timeout = 0;

	g_signal_connect (set, "toolbar", G_CALLBACK (gb_bonobo_ui_toolbar_toolbar_cb), ret);
	
	return ret;
}

static void 
buit_item_destroy_wrapper (gpointer key,
			   gpointer value,
			   gpointer user_data)
{
	BuitItem *buit = value;
	buit_item_destroy (buit, FALSE);
}

static void
buit_item_destroy (BuitItem *bit, gboolean remove_from_hashtable)
{
	if (remove_from_hashtable)
	{
		if (bit->is_main)
		{
			g_hash_table_remove (bit->buit->priv->bookmark_to_main_buiti, bit->bm);
		}
		else
		{
			g_hash_table_remove (bit->buit->priv->bookmark_to_buiti, bit->bm);
		}
	}

	g_signal_handlers_disconnect_matched (bit->bm, G_SIGNAL_MATCH_DATA, 0, 0, 
					      NULL, NULL, bit);

	g_free (bit->path);
	g_free (bit);
}

static BuitItem *
buit_item_new (GbBonoboUIToolbar *buit, GbBookmark *b, gboolean is_main)
{
	BuitItem *ret = g_new0 (BuitItem, 1);
	ret->buit = buit;
	ret->bm = b;
	ret->is_main = is_main;
	if (is_main)
	{
		g_hash_table_insert (buit->priv->bookmark_to_main_buiti, b, ret);
	}
	else
	{
		g_hash_table_insert (buit->priv->bookmark_to_buiti, b, ret);
	}
	return ret;
}

static void
gb_bonobo_ui_toolbar_build (GbBonoboUIToolbar *buit, GbFolder *f)
{
	GbBonoboUIToolbarPrivate *p = buit->priv;
	BuitItem *buiti;
	/* 5 is a random band number to make sure it's added after
	 * the other toolbars. This sucks, but I don't know a better fix */
	static int band_number_hack = 5;

	g_return_if_fail (GB_IS_BONOBO_UI_TOOLBAR (buit));
	g_return_if_fail (GB_IS_FOLDER (f));

	buiti = g_hash_table_lookup (p->bookmark_to_main_buiti, f);
	if (!buiti) 
	{
		gchar *tbname = g_strdup_printf ("BookmarkToolbar%20s%ld", 
						 GB_BOOKMARK (f)->name, 
						 GB_BOOKMARK (f)->id);
		buiti = buit_item_new (buit, GB_BOOKMARK (f), TRUE);
		buiti->path = gul_bonobo_add_dockitem (p->uic, tbname, ++band_number_hack);
		g_free (tbname);

		g_signal_connect (f, "child-added", 
				  G_CALLBACK (gb_bonobo_ui_toolbar_child_added_cb), buiti);
		g_signal_connect (f, "child-removed", 
				  G_CALLBACK (gb_bonobo_ui_toolbar_child_removed_cb), buiti);
	}
	gb_bonobo_ui_toolbar_rebuild (buit, buiti);

}

static void
gb_bonobo_ui_toolbar_rebuild_with_timeout (BuitItem *buiti)
{
	GbBonoboUIToolbar *buit = buiti->buit;
	GbBonoboUIToolbarPrivate *p = buit->priv;
	g_return_if_fail (buiti->is_main);
	buiti->dirty = TRUE;
	
	if (p->rebuid_timeout)
	{
		g_source_remove (p->rebuid_timeout);
	}

	p->rebuid_timeout = g_timeout_add (REBUILD_TIMEOUT, 
					   gb_bonobo_ui_toolbar_rebuild_timeout_cb, buiti->buit);
}

static gboolean
gb_bonobo_ui_toolbar_rebuild_timeout_cb (gpointer data)
{
	GbBonoboUIToolbar *buit = data;
	GbBonoboUIToolbarPrivate *p = buit->priv;

	g_hash_table_foreach (p->bookmark_to_main_buiti, buit_item_rebuild, NULL);

	p->rebuid_timeout = 0;
	return FALSE;
}

static void 
buit_item_rebuild (gpointer key,
		   gpointer value,
		   gpointer user_data)
{
	BuitItem *buiti = value;
	g_return_if_fail (buiti->is_main);
	if (buiti->dirty)
	{
		gb_bonobo_ui_toolbar_rebuild (buiti->buit, buiti);
	}
}

static void
gb_bonobo_ui_toolbar_child_added_cb (GbFolder *f, GbBookmark *c, guint pos, BuitItem *buiti)
{
	gb_bonobo_ui_toolbar_rebuild_with_timeout (buiti);
}

static void
gb_bonobo_ui_toolbar_child_removed_cb (GbFolder *f, GbBookmark *c, guint pos, BuitItem *buiti)
{
	gb_bonobo_ui_toolbar_rebuild_with_timeout (buiti);
}

static void
gb_bonobo_ui_toolbar_rebuild (GbBonoboUIToolbar *buit, BuitItem *buiti)
{
	GbBonoboUIToolbarPrivate *p = buit->priv;
	GbBookmark *i;
	gint index = 0;
	GbFolder *f = GB_FOLDER (buiti->bm);
	gul_bonobo_clear_path (p->uic, buiti->path);
	for (i = f->child; i != NULL; i = i->next)
	{
		gb_bonobo_ui_toolbar_build_item (buit, i, index, buiti->path);
		index++;
	}
	buiti->dirty = FALSE;
}

static void
gb_bonobo_ui_toolbar_destroy (GbBonoboUIToolbar *buit, GbFolder *f)
{
	GbBonoboUIToolbarPrivate *p = buit->priv;
	BuitItem *buiti = g_hash_table_lookup (p->bookmark_to_main_buiti, f);
	if (buiti)
	{
		bonobo_ui_component_rm (p->uic, buiti->path, NULL);
		buit_item_destroy (buiti, TRUE);
	}
}

static void
gb_bonobo_ui_toolbar_build_item (GbBonoboUIToolbar *buit, GbBookmark *item, gint index, gchar *path)
{
	GbBonoboUIToolbarPrivate *p = buit->priv;
	BuitItem *buiti;
	GbTbWidget *b = gb_bookmark_create_toolbar_widget (item);
	gtk_widget_show (GTK_WIDGET (b));
	buiti = buit_item_new (buit, item, FALSE);
	buiti->widget = b;
	gul_bonobo_add_numbered_widget (p->uic, GTK_WIDGET (b), index, path);
	buiti->path = gul_bonobo_get_numbered_menu_item_path (p->uic, path, index);

	g_signal_connect (b, "bookmark-activated", 
			  G_CALLBACK (gb_bonobo_ui_toolbar_bookmark_activated_cb), buit);
	g_signal_connect (b, "destroy",
			  G_CALLBACK (gb_bonobo_ui_toolbar_widget_destroy_cb), buiti);

	if (p->location_source)
	{
		gb_tb_widget_set_location_source (b, p->location_source);
	}
}

static void
gb_bonobo_ui_toolbar_widget_destroy_cb (GtkWidget *w, BuitItem *buiti)
{
	buit_item_destroy (buiti, TRUE);
}

static void
gb_bonobo_ui_toolbar_bookmark_activated_cb (GtkWidget *w, 
					    GbBookmarkEventActivated *ev,
					    GbBonoboUIToolbar *buit)
{
	g_signal_emit (buit, GbBonoboUIToolbarSignals[GB_BONOBO_UI_TOOLBAR_BOOKMARK_ACTIVATED], 0, ev);

}

void
gb_bonobo_ui_toolbar_set_visible (GbBonoboUIToolbar *buit, gboolean visible)
{
	GSList *l;
	GSList *li;
	GbBonoboUIToolbarPrivate *p = buit->priv;
	p->visible = visible;

	l = gb_bookmark_set_get_toolbars (p->set);
	for (li = l; li != NULL; li = li->next)
	{
		if (visible)
		{
			gb_bonobo_ui_toolbar_build (buit, li->data);
		}
		else
		{
			gb_bonobo_ui_toolbar_destroy (buit, li->data);
		}
	}
	g_slist_free (l);
}

static void
gb_bonobo_ui_toolbar_toolbar_cb (GbBookmarkSet *set, GbFolder *f, GbBonoboUIToolbar *buit)
{
	if (buit->priv->visible)
	{
		if (f->create_toolbar)
		{
			gb_bonobo_ui_toolbar_build (buit, f);
		}
		else
		{
			gb_bonobo_ui_toolbar_destroy (buit, f);
		}
	}
}

static void
gb_bonobo_ui_toolbar_set_location_source_buiti (gpointer key, gpointer item, gpointer data)
{
	BuitItem *it = item;

	g_return_if_fail (GB_IS_TB_WIDGET (it->widget));

	gb_tb_widget_set_location_source (it->widget, data);
}

void
gb_bonobo_ui_toolbar_set_location_source (GbBonoboUIToolbar *buit, GbLocationSource *src)
{
	GbBonoboUIToolbarPrivate *p = buit->priv;
	
	if (p->location_source)
	{
		g_object_remove_weak_pointer (G_OBJECT (p->location_source),
					      (gpointer *) &p->location_source);
	}

	p->location_source = src;

	if (p->location_source)
	{
		g_object_add_weak_pointer (G_OBJECT (p->location_source), 
					   (gpointer *) &p->location_source);
	}

	g_hash_table_foreach (p->bookmark_to_buiti, 
			      gb_bonobo_ui_toolbar_set_location_source_buiti, src);
}

static void
gb_bonobo_ui_toolbar_hash_to_slist (gpointer key, gpointer data, 
				    gpointer user_data)
{
	GSList **l = user_data;
	*l = g_slist_prepend (*l, data);
}

