/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

GtkWidget* toolbar_create_main (GaleonBrowser *gb);
void toolbar_create_location_widgets (GaleonBrowser *gb, GtkWidget *hbox, GtkToolbarStyle go_toolbar_style);
GtkWidget* toolbar_create_location (GaleonBrowser *gb);
GtkWidget* toolbar_button_append (GtkWidget *window, GtkWidget *toolbar, char *label, char* tip, char* pixmap);
GtkWidget * toolbar_add_docked (GaleonBrowser *gb, GtkWidget *hbox, char *name);
void toolbar_add_drag_event_box (GaleonBrowser *gb, GtkWidget *hbox, GtkWidget *widget);

/* from browser_callbacks.c */
void window_location_gnomeentry_popwin_cb(GtkWidget *widget, GaleonBrowser *browser);

extern GtkTargetEntry url_drag_types[];
extern gint url_drag_types_num_items;

/*
 * toolbar_create: create toolbars
 */
void
toolbar_create (GaleonBrowser *gb)
{
	/* create main toolbar */
	gb->main_dockitem = toolbar_add_docked (gb, toolbar_create_main (gb), "main");

	/* create location toolbar if needed */
	if (gnome_config_get_int ("/galeon/Appearance/url_location=1")==0)
	{
		gb->location_dockitem = toolbar_add_docked (gb, toolbar_create_location (gb), "location");
	}
	else
	{
		gb->location_dockitem = NULL;
	}
}

/*
 * toolbar_add_docked: add an horizontal box to the toolbar dock
 */
GtkWidget *
toolbar_add_docked (GaleonBrowser *gb, GtkWidget *hbox, char *name)
{
	static int num = 1;
	gboolean toolbar_relief;

	toolbar_relief = gnome_preferences_get_toolbar_relief ();

	gnome_app_add_docked (GNOME_APP (gb->WMain), hbox, name, 
			      GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
			      GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL,
			      GNOME_DOCK_TOP, 
			      num++, 0, 0);
	gtk_container_set_border_width (GTK_CONTAINER(hbox->parent), 2);
	gnome_dock_item_set_shadow_type 
		(GNOME_DOCK_ITEM (hbox->parent), 
		 toolbar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	gtk_widget_show (hbox);

	return hbox->parent;
}

/*
 * toolbar_destroy: destroy toolbars
 */
void toolbar_destroy (GaleonBrowser *gb)
{	
	if (gb->spinner)
	{
		gnome_animator_stop(GNOME_ANIMATOR(gb->spinner));
	}

	gtk_widget_destroy (GTK_WIDGET(gb->main_dockitem));

	if (gb->location_dockitem)
	{
		gtk_widget_destroy (GTK_WIDGET(gb->location_dockitem));
	}

	bookmarks_remove_menu_global();
	bookmarks_remove_tb_global();
}

/*
 * toolbar_recreate: recreate toolbars
 */
void toolbar_recreate (GaleonBrowser *gb)
{
	gchar *url;

	// save the current contents of the location entry
	url = gtk_editable_get_chars(GTK_EDITABLE(gb->toolbar_entry), 0, -1);
	gtk_editable_delete_text(GTK_EDITABLE(gb->toolbar_entry), 0, -1);
	toolbar_destroy (gb);
	toolbar_create (gb);

	// restore the saved contents
	gtk_entry_set_text(GTK_ENTRY(gb->toolbar_entry), url);
	g_free(url);

	gtk_widget_show (GTK_WIDGET(gb->main_dockitem));
	if (gb->location_dockitem)
	{
		gtk_widget_show (GTK_WIDGET(gb->location_dockitem));
	}

	bookmarks_create_menu_global();
	bookmarks_create_tb_global();
}

/*
 * toolbar_create: recreate toolbars of all galeon windows
 */
void toolbar_global_recreate (void)
{
	GList *l;

	for (l = all_windows; l != NULL; l = l->next)
	{
		toolbar_recreate ((GaleonBrowser *) l->data);
	}
}

/*
 * toolbar_create_main: create main toolbar (buttons)
 */
GtkWidget*
toolbar_create_main (GaleonBrowser *gb)
{
	GtkWidget *hbox;
	GtkWidget *toolbar;
	gboolean toolbar_button_relief;
	int url_location;
	GtkToolbarStyle style;

	/* read preferences */
	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();
	url_location = 
		gnome_config_get_int ("/galeon/Appearance/url_location=1");
	style = gnome_config_get_int ("/galeon/Appearance/toolbar_type=2");

	hbox = gtk_hbox_new (FALSE, 0);

	/* create buttons toolbar */

	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), style);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar),
				       toolbar_button_relief ? GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);

	gb->BBack = toolbar_button_append (gb->WMain, toolbar, _("Back"), _("Go Back - Right button: History"), GNOME_STOCK_PIXMAP_BACK);
	gtk_signal_connect (GTK_OBJECT(gb->BBack), "clicked", window_back_button_clicked_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->BBack), "pressed", window_back_button_pressed_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->BBack), "button_press_event", (GtkSignalFunc)window_back_forward_button_press_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->BBack), "leave",window_back_button_leave_cb,gb);

	gb->BForward = toolbar_button_append (gb->WMain, toolbar, _("Forward"), _("Go Forward - Right button: History"), GNOME_STOCK_PIXMAP_FORWARD);
	gtk_signal_connect (GTK_OBJECT(gb->BForward), "clicked", window_forward_button_clicked_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->BForward), "pressed", window_forward_button_pressed_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->BForward), "leave", window_forward_button_leave_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->BForward), "button_press_event", (GtkSignalFunc)window_back_forward_button_press_cb, gb);

	gb->BRefresh = toolbar_button_append (gb->WMain, toolbar, _("Reload"), _("Reload current page"), GNOME_STOCK_PIXMAP_REFRESH);
	gtk_signal_connect (GTK_OBJECT(gb->BRefresh), "clicked", window_refresh_button_clicked_cb, gb);

	gb->BStart = toolbar_button_append (gb->WMain, toolbar, _("Home"), _("Go to your Home Page"), GNOME_STOCK_PIXMAP_HOME);
	gtk_signal_connect (GTK_OBJECT(gb->BStart), "clicked", window_start_button_clicked_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->BStart), "button_press_event", (GtkSignalFunc)window_start_button_press_cb, gb);

	gb->BStop = toolbar_button_append (gb->WMain, toolbar, _("Stop"), _("Stop current data transfer"), GNOME_STOCK_PIXMAP_STOP);
	gtk_signal_connect (GTK_OBJECT(gb->BStop), "clicked", window_stop_button_clicked_cb, gb);

	gtk_box_pack_start (GTK_BOX(hbox), toolbar, FALSE, FALSE, 1);
	gtk_widget_show (GTK_WIDGET(toolbar));

	/* create zoom spin entry */

	gb->zoom_spin = gtk_spin_button_new (GTK_ADJUSTMENT(gtk_adjustment_new (100, 1, 999, 10, 25, 10)), 1 ,0);
	gtk_box_pack_start (GTK_BOX(hbox), gb->zoom_spin, FALSE, FALSE, 1);
	gtk_widget_show (GTK_WIDGET(gb->zoom_spin));
	gtk_signal_connect (GTK_OBJECT(gb->zoom_spin), "changed",
			    window_zoom_spin_changed_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->zoom_spin), "key_press_event",
			    (GtkSignalFunc)window_zoom_spin_key_press_cb, gb);

	/* create location related widgets if needed */
	
	if (url_location==1)
	{
		toolbar_create_location_widgets (gb, hbox, style);
		gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), style);
	}

	/* create the spinner */
	if (gnome_config_get_bool ("/galeon/General/show_spinner=true"))
	{
		gb->spinner = spinner_create (hbox);
		if (gb->spinner != NULL)
		{
			gtk_signal_connect 
				(GTK_OBJECT(gb->spinner->parent), "clicked",
				 GTK_SIGNAL_FUNC(window_spinner_button_clicked_cb),
				 (gpointer)gb);
		}
	}
	else
	{
		gb->spinner = NULL;
	}

	/* show the box */

	gtk_widget_show (GTK_WIDGET(hbox));

	return hbox;
}

void
toolbar_add_drag_event_box (GaleonBrowser *gb, GtkWidget *hbox, GtkWidget *widget)
{
	GtkWidget *event_box = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(event_box), widget);
	gtk_box_pack_start (GTK_BOX(hbox), event_box, FALSE, FALSE, 1);
	gtk_signal_connect (GTK_OBJECT(event_box), "drag_data_get",
			    window_drag_pixmap_drag_data_get_cb, gb);
	gtk_drag_source_set (GTK_WIDGET (event_box),
			     GDK_BUTTON1_MASK, url_drag_types, url_drag_types_num_items,
			     GDK_ACTION_COPY|GDK_ACTION_LINK);
	gtk_widget_show_all (event_box);
}

/*
 * toolbar_create_location_widgets: create location related widgets
 */
void
toolbar_create_location_widgets (GaleonBrowser *gb, GtkWidget *hbox, GtkToolbarStyle go_toolbar_style)
{
	GtkWidget *go_button;
	gboolean toolbar_button_relief;

	/* read preferences */
	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();

	gb->drag_pixmap = gnome_pixmap_new_from_file (SHARE_DIR "/drag-url-icon.png");
	toolbar_add_drag_event_box (gb, hbox, gb->drag_pixmap);

	if (gnome_config_get_bool ("/galeon/Appearance/show_location_label=TRUE"))
	{
		gb->location_label = gtk_label_new (_("Location:"));
		toolbar_add_drag_event_box (gb, hbox, gb->location_label);
	}
	else
	{
		gb->location_label = NULL;
	}

	/* create location entry */
	
	gb->toolbar_gnomeentry = gnome_entry_new ("location_history");
	gnome_entry_load_history (GNOME_ENTRY(gb->toolbar_gnomeentry));
	/* connect the popwin handlers */
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(
	                   gb->toolbar_gnomeentry)->popwin),
	                   "show",
	                   GTK_SIGNAL_FUNC(window_location_gnomeentry_popwin_cb), gb);
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(
	                   gb->toolbar_gnomeentry)->popwin),
	                   "hide",
	                   GTK_SIGNAL_FUNC(window_location_gnomeentry_popwin_cb), gb);
	gb->toolbar_entry = gnome_entry_gtk_entry (GNOME_ENTRY (gb->toolbar_gnomeentry));
	gtk_box_pack_start (GTK_BOX(hbox), gb->toolbar_gnomeentry, TRUE, TRUE, 1);
	gtk_widget_show (GTK_WIDGET(gb->toolbar_gnomeentry));
	gtk_signal_connect_after (GTK_OBJECT(gb->toolbar_entry), "key_press_event",
				  (GtkSignalFunc)window_location_entry_key_press_cb, gb);
	gtk_signal_connect (GTK_OBJECT(gb->toolbar_entry), "drag_data_received",
			   window_location_entry_drag_data_received_cb, gb);

	/* create go button toolbar */

	if (gnome_config_get_bool ("/galeon/Appearance/show_go_button=TRUE"))
	{
		gb->go_toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
		go_button = toolbar_button_append (gb->WMain, gb->go_toolbar, _("Go"), "", GNOME_STOCK_PIXMAP_JUMP_TO);
		gtk_signal_connect (GTK_OBJECT(go_button), "clicked", window_go_button_clicked_cb, gb);
		gtk_box_pack_start (GTK_BOX(hbox), gb->go_toolbar, FALSE, FALSE, 1);
		gtk_toolbar_set_style (GTK_TOOLBAR (gb->go_toolbar), go_toolbar_style);
		gtk_toolbar_set_button_relief (GTK_TOOLBAR (gb->go_toolbar),
					       toolbar_button_relief ? GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
		gtk_widget_show (GTK_WIDGET(gb->go_toolbar));
	}
	else
	{
		gb->go_toolbar = NULL;
	}
}

/*
 * toolbar_create_location: create location toolbar
 */
GtkWidget*
toolbar_create_location (GaleonBrowser *gb)
{
	GtkWidget *hbox;
	GtkWidget *toolbar;

	hbox = gtk_hbox_new (FALSE, 0);

	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
	toolbar_create_location_widgets (gb, hbox, GTK_TOOLBAR_TEXT);

	gtk_widget_show (GTK_WIDGET(toolbar));

	gtk_widget_show (GTK_WIDGET(hbox));

	return hbox;
}

/*
 * toolbar_button_append: append a button to a toolbar
 */
GtkWidget*
toolbar_button_append (GtkWidget *window, GtkWidget *toolbar, char *label, char* tip, char* pixmap)
{
	GtkWidget *icon = gnome_stock_pixmap_widget (window, pixmap);
	return gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
					   GTK_TOOLBAR_CHILD_BUTTON,
					   NULL,
					   label,
					   tip, NULL,
					   icon, NULL, NULL);
}
