/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/*
 *  Session code 
 *
 *  Copyright (C) 2000 Matthew Aubury
 *
 *  This code implements the idea of "sessions", meaning the state of
 *  all the open browsers with respect to the URLs they're pointed at.
 *
 *  This should be useful in two ways:
 *    -- You need to quit X but have got lots of windows open
 *       with stuff you want to read, but you don't want to
 *       bookmark it all, just come back and see it later.
 *    -- By autosaving sessions we should offer a degree of 
 *       automatic crash recovery. This should mitigate some
 *       of the Mozilla/Galeon instability problems.
 *
 *  NOTE: this code won't do the right thing with frames yet. This is
 *  an ongoing problem with frames (they can't easily be referenced by
 *  URL) which is shared with bookmarks.
 *
 *  This code is quite derived from the history code,
 *  kudos to whoever did that...
 *
 *  At the moment we only do crash recovery saves, as I haven't yet
 *  gotten around to doing the dialogs for generic session saving.
 *
 *  -- MattA
 */

/* local function prototypes */
static void autoresume_cb(gint reply, xmlDocPtr doc);
static void autoresume_delete_cb(GtkWidget *widget, GdkEventAny *event, 
				 gpointer data);
static xmlDocPtr session_load(gchar *filename);
static void session_resume(xmlDocPtr doc, gboolean override_behaviour);

/**
 * session_autosave: save to the "crash recovery" session file
 */
void
session_autosave(void)
{
	gchar *filename;
	gint autosave;

	/* check we're configured to autosave */
	autosave = gnome_config_get_int("/galeon/Advanced/crash_recovery=1");
	if (autosave == FALSE)
		return;

	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (),
				"/.galeon/session.xml", NULL);

	/* save it out */
	session_save(filename);

	/* free allocated string */
	g_free(filename);
}

/**
 * session_autoresume: check to see if there's a crashed session and load it
 */
gint
session_autoresume(void)
{
	gint crashed, saved;
	gchar *filename;
	AutoResume autoresume;
	GtkWidget *dialog;
	xmlDocPtr session;

	/* see if there's a crashed or saved session to resume */
	crashed = gnome_config_get_int ("/galeon/General/crashed=0");
	saved = gnome_config_get_int ("/galeon/General/session_saved=0");
	gnome_config_set_int ("/galeon/General/session_saved", 0);
	gnome_config_sync ();
	if (crashed == FALSE && saved == FALSE)
		return FALSE;

	/* see if we're configured to autoresume */
	autoresume = gnome_config_get_int("/galeon/Advanced/crash_recovery=1");
	if (autoresume == RESUME_NEVER && !saved)
		return FALSE;

	/* see if there's a session file to resume from */
	if (crashed)
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session.xml", NULL);
	}
	else
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session_saved.xml", NULL);
	}
	if (access (filename, F_OK) != 0)
	{
		g_free(filename);
		return FALSE;
	}
	
	/* load it */
	session = session_load(filename);
	g_free(filename);
	if (session == NULL)
	{
		return FALSE;
	}

	/* abort if no windows in session */
	if (session->root->childs == NULL)
	{
		return FALSE;
	}

	/* resume saved session */
	if (saved)
	{
		/* resume and free */
		session_resume (session, TRUE);
		xmlFreeDoc (session);
		return TRUE;
	}

	/* do the appropriate thing */
	switch (autoresume)
	{
	case RESUME_NEVER:
		/* never reached: this is here to stop a warning */
		g_assert_not_reached ();
		return FALSE;

	case RESUME_ASK:
		dialog = gnome_question_dialog_modal
			(_("Galeon appears to have crashed or "
			   "been killed last time it was run\n\n"
			   "Would you like to recover the state "
			   "of your last browsing session?"),
			 (GnomeReplyCallback) autoresume_cb, session);
		gtk_signal_connect (GTK_OBJECT (dialog), "delete-event",
				    autoresume_delete_cb, NULL);
		return TRUE;

	case RESUME_ALWAYS:
		/* resume and free */
		session_resume (session, FALSE);
		xmlFreeDoc (session);
		return TRUE;
	}

	/* never reached */
	g_assert_not_reached ();
	return FALSE;
}

/**
 * autoresume_cb: handle the result of the resume question dialog box
 */
static void
autoresume_cb(gint reply, xmlDocPtr session)
{
	/* callback paranoia */
	g_assert(session != NULL);

        /* resume it that was what was requested */
        if (reply == GNOME_YES)
	{
		session_resume (session, FALSE);
	}
	else
	{
		/* create a default browser if none already created */
		if (g_list_length (all_browsers) == 0)
		{
			browser_create_default (NULL);
		}
	}

	/* free the session whether it was restored or not */
	xmlFreeDoc (session);
}

/**
 * autoresume_delete_cb: cope if the user destroys the autoresume question
 * dialog -- this is necessary as we don't have any other windows open
 * at the time
 */
static void
autoresume_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	/* create a default browser */
	browser_create_default (NULL); 
}

/**
 * session_save: record the session state to the standard location
 */
void
session_save(gchar *filename)
{
	GList *w;
	xmlNodePtr root_node;
	xmlNodePtr item_node;
	xmlDocPtr doc;
	gchar *url;

	/* version doesn't really make sense, but... */
        doc = xmlNewDoc ("1.0");

	/* create and set the root node for the session */
        root_node = xmlNewDocNode (doc, NULL, "session", NULL);
        xmlDocSetRootElement (doc, root_node);

	/* iterate through all the browsers */
	for (w = all_browsers; w != NULL; w = g_list_next(w))
	{
		/* get this item */
		GaleonBrowser *gb = (GaleonBrowser *)(w->data);

		/* make a new XML node */
		item_node = xmlNewDocNode (doc, NULL, "item", NULL);

		/* get the current URL */
		url = gtk_moz_embed_get_location (gb->mozEmbed);

		/* store it in the node */
		xmlSetProp (item_node, "url", url);

		/* insert node into the tree */
		if (strlen(url) > 0)
		{
			xmlAddChild (root_node, item_node);
		}

		/* free allocated string */
		g_free (url);
	}

	/* save it all out to disk */
        xmlSaveFile (filename, doc);
	xmlFreeDoc (doc);
}

/**
 * session_load: load a session into a new XML document
 */
static xmlDocPtr
session_load(gchar *filename)
{
	xmlDocPtr doc;

	/* read the session file */
        doc = xmlParseFile (filename);
        if (doc == NULL) 
	{
		g_warning("unable to parse session file `%s'", filename);
		/* NOTE: fall through to return NULL */
	}

	return doc;
}

/**
 * session_resume: open browsers to resume the session state
 */
static void
session_resume(xmlDocPtr doc, gboolean override_behaviour)
{
	xmlNodePtr item_node;
	gchar *url;
	gint behaviour = gnome_config_get_int (CONF_ADVANCED_CRASH_BEHAVIOUR);
	BookMarkItem *cat = NULL; 

	/* so we don't open saved sessions into temporary bookmarks) */
	if (override_behaviour)
	{
		behaviour = 0;
	}

	if (behaviour == 1) {
		browser_show_temporary_bookmarks (NULL);
		cat = add_temp_bookmark (CATEGORY, _("URLs opened before crashing"),
	                                 NULL, NULL);
	}

	/* iterate over session items */
        for (item_node = doc->root->childs;
	     item_node != NULL;
	     item_node = item_node->next)
	{
		/* get the browser URL */
		url = xmlGetProp (item_node, "url");

		if (behaviour == 1) {
			add_temp_bookmark (SITE, url, url, cat);
		} else {
			/* open a browser pointing to this URL */
			browser_create_from_url(url);
		}

		/* free allocated string */
		xmlFree (url);
	}

	/* add a blank page bookmark to the temp bookmarks window */
	if (behaviour == 1)
		add_temp_bookmark(SITE, N_("Empty browser window"),
				  "about:blank", cat);
}
