/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <galeon.h>

void prefs_ctree_tree_select_row_cb (GtkWidget       *ctree,
				    GtkCTreeNode	*node,
				    GaleonBrowser *browser);
void prefs_ctree_tree_unselect_row_cb (GtkWidget       *ctree,
				      GList		*node,
				      gint             column,
				      GaleonBrowser   *browser);
void prefs_ok_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_apply_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_cancel_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_mime_add_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_mime_delete_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_mime_edit_cb (GtkWidget *w, GaleonBrowser *browser);
void prefs_mime_list_select_row_cb (GtkCList *clist, 
				    gint row, 
				    gint column, 
				    GdkEventButton *event, 
				    GaleonBrowser *browser);
void prefs_mime_list_unselect_row_cb (GtkCList *clist,
				      gint row,
				      gint column,
				      GdkEventButton *event,
				      GaleonBrowser *browser);
void prefs_ctree_tree_expand_cb (GtkCTree      *ctree,
                                GList         *node,
                                GaleonBrowser *browser);
void prefs_download_dir_ok_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_download_dir_cancel_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_homepage_my_portal_button_clicked_cb (GtkButton *button,
					   GaleonBrowser *browser);
void prefs_homepage_current_button_clicked_cb (GtkButton *button,
					 GaleonBrowser *browser);
void prefs_lang_add_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_lang_remove_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_lang_up_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_lang_down_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_clear_cache_button_clicked_cb (GtkButton *button, GaleonBrowser *browser);
void prefs_clear_location_history_button_clicked_cb (GtkButton *button,
                                               GaleonBrowser *browser);
void prefs_clear_history_button_clicked_cb (GtkButton *button,
				      GaleonBrowser *browser);
void prefs_clear_history_question_cb (gint reply, gpointer data);
void prefs_clear_location_history_question_cb (gint reply, gpointer data);
gboolean prefs_history_item_free_cb (gpointer key, gpointer value, gpointer user_data);
void prefs_option_selected_cb (GtkMenuShell *menushell, gpointer data);
void prefs_browse_clicked_cb (GnomeFileEntry *fileentry, gpointer data);
void prefs_picker_clicked_cb (GtkWidget *widget, gpointer picker);

/* from prefs.c */
void mouse_wheel_controls_set_sensitive (GtkWidget *widget, gint key,
					 gboolean sensitive);

static MimeItem *selected_mime;
static gboolean enable_edit_mime = FALSE;

/**
 * prefs_ok_button_clicked_cb: preferences dialog ok button clicked
 */
void
prefs_ok_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkWidget *prefs = glade_lookup_widget(GTK_WIDGET(button), "Preferences");
	preferences_set_config_to_dialog (prefs);
	preferences_apply ();
	gtk_widget_destroy (prefs);
}

/**
 * prefs_ok_button_clicked_cb: preferences dialog ok button clicked
 */
void
prefs_clear_cache_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	gchar *confdir = g_strconcat(g_get_home_dir(),"/.galeon",NULL);
	gchar *cachedir = g_strconcat(confdir,"/cache",NULL);

	remove_directory (cachedir, TRUE);

	g_free (confdir);
	g_free (cachedir);
}


/**
 * prefs_apply_button_clicked_cb: preferences dialog apply button clicked
 */
void
prefs_apply_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkWidget *prefs = glade_lookup_widget (GTK_WIDGET(button), "Preferences");
	preferences_set_config_to_dialog (prefs);
	preferences_apply ();
}


void
prefs_cancel_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	// This is ok for now, but we also need a 'revert' feature
	GtkWidget *prefs = glade_lookup_widget(GTK_WIDGET(button), "Preferences");
	gtk_widget_destroy(prefs);
}

/**
 * on_prefs_add_button_clicked: mime types add button clicked
 */
void
prefs_mime_add_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	MimeItem *newitem = g_new0 (MimeItem, 1);
	gchar *text[3];
	GtkWidget *clist = glade_lookup_widget
		(GTK_WIDGET (button), "preferences_mime_list");
	gint row;
	newitem->name = g_strdup (_("New mime item"));
	newitem->action = UNKNOWN;
	text[0] = newitem->name;
	text[1] = N_("Unknown");
	text[2] = NULL;
	row = gtk_clist_append (GTK_CLIST (clist), text);
	gtk_clist_set_row_data (GTK_CLIST (clist), row, newitem);
	gtk_clist_select_row (GTK_CLIST (clist), row, 0);
}


/**
 * on_prefs_delete_button_clicked: mime types delete button clicked
 */
void
prefs_mime_delete_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkWidget *clist = glade_lookup_widget
		(GTK_WIDGET (button), "preferences_mime_list");
	gint row;
	
	if (!selected_mime)
		return;

	row = gtk_clist_find_row_from_data (GTK_CLIST (clist), selected_mime);
	gtk_clist_remove (GTK_CLIST (clist), row);
	selected_mime = NULL;
}


/**
 * prefs_mime_list_select_row_cb: mime type selected
 */
void prefs_mime_list_select_row_cb (GtkCList *clist, 
				    gint row, 
				    gint column, 
				    GdkEventButton *event, 
				    GaleonBrowser *browser)
{
	GtkWidget *name_entry = glade_lookup_widget
		(GTK_WIDGET(clist), "prefs_mime_name_entry");
	GtkWidget *action_optionmenu = glade_lookup_widget
		(GTK_WIDGET(clist), "prefs_mime_action_optionmenu");
	gint act_num = 0;

	enable_edit_mime = FALSE;
	selected_mime = (MimeItem*) gtk_clist_get_row_data(clist, row);
		
	switch (selected_mime->action) {
	case SAVE_TO_DISK:
		act_num = 0; /* save */
		break;
	case LEAVE_TO_MOZILLA:
		act_num = 1; /* Render in Galeon */
		break;
	case ASK_ACTION:
		act_num = 2; /* Ask */
		break;
	case RUN_PROGRAM: /* not yet done */
	case UNKNOWN:
		act_num = 2; /* default to ask */
		break;
	}

	gtk_option_menu_set_history(GTK_OPTION_MENU(action_optionmenu), act_num);
	gtk_entry_set_text(GTK_ENTRY(name_entry), selected_mime->name);
	gtk_widget_set_sensitive (name_entry, TRUE);
	/* doesn't seem to work right ...*/
	/*gtk_widget_set_sensitive (action_optionmenu, TRUE);*/
	enable_edit_mime = TRUE;
}

/**
 * prefs_mime_list_unselect_row_cb: mime type unselected
 */
void prefs_mime_list_unselect_row_cb   (GtkCList *clist,
					gint row,
					gint column,
					GdkEventButton *event,
					GaleonBrowser *browser)
{
	GtkWidget *name_entry = glade_lookup_widget
		(GTK_WIDGET(clist), "prefs_mime_name_entry");
	GtkWidget *action_optionmenu = glade_lookup_widget
		(GTK_WIDGET(clist), "prefs_mime_action_optionmenu");

	selected_mime = NULL;
	enable_edit_mime = FALSE;

	/* The default is Ask*/
	gtk_option_menu_set_history(GTK_OPTION_MENU(action_optionmenu), 2); 
	gtk_entry_set_text(GTK_ENTRY(name_entry), "");
	gtk_widget_set_sensitive (name_entry, FALSE);
	/* doesn't seem to work right */
	/*gtk_widget_set_sensitive (action_optionmenu, FALSE);*/

}

void prefs_mime_edit_cb (GtkWidget *w, GaleonBrowser *browser)
{
	GtkWidget *name_entry = glade_lookup_widget
		(w, "prefs_mime_name_entry");
	GtkWidget *action_optionmenu = glade_lookup_widget
		(w, "prefs_mime_action_optionmenu");
	GtkWidget *clist = glade_lookup_widget
		(w, "preferences_mime_list");
	gchar *text[3];
	gint row;
        GtkWidget *action_menu;
        GList *l;
        gpointer item;
        gint action_index;
	
        /* extract the action menu index */
        action_menu = GTK_OPTION_MENU (action_optionmenu)->menu;
        l = GTK_MENU_SHELL (action_menu)->children;
        item = gtk_menu_get_active (GTK_MENU (action_menu));
        action_index = g_list_index (l, item);

	if (!selected_mime) return;
	if (!enable_edit_mime) return;

	if (selected_mime->name) g_free (selected_mime->name);
	selected_mime->name = gtk_editable_get_chars (GTK_EDITABLE (name_entry), 0, -1);
	switch (action_index) {
	case 0:
		selected_mime->action = SAVE_TO_DISK;
		break;
	case 1:
		selected_mime->action = LEAVE_TO_MOZILLA;
		break;
	case 2:
		selected_mime->action = ASK_ACTION;
		break;
	default:
		selected_mime->action = UNKNOWN;
		break;
	}

	row = gtk_clist_find_row_from_data (GTK_CLIST (clist), selected_mime);

	text[0] = selected_mime->name;
	switch (selected_mime->action) {
	case LEAVE_TO_MOZILLA:
		text[1] = N_("Render with Galeon");
		break;
	case SAVE_TO_DISK:
		text[1] = N_("Save to disk");
		break;
	case RUN_PROGRAM:  
		/* Should run the program associted to the mime type by gnome */
		/* not yet done */
	case UNKNOWN:
		text[1] = N_("Unknown");
		break;
	case ASK_ACTION:
		text[1] = N_("Ask the user");
		break;
	}
	gtk_clist_set_text (GTK_CLIST(clist), row, 0, text[0]);
	gtk_clist_set_text (GTK_CLIST(clist), row, 1, text[1]);
}

/**
 * on_dirselect2_ok_clicked: download directory dialog ok clicked (prefs dialog)
 */
void
prefs_download_dir_ok_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkWidget *dirselect = glade_lookup_widget(GTK_WIDGET(button), "dirselection2");
	char *download_dir = gtk_file_selection_get_filename(GTK_FILE_SELECTION(dirselect));
	char *url = gtk_object_get_data(GTK_OBJECT(dirselect), "url");
	
	gnome_config_push_prefix("/galeon/Downloading/");
	
	gnome_config_set_string("download_dir", download_dir);
	gnome_config_pop_prefix();
	gnome_config_sync();
	
	if (gnome_config_get_int (CONF_DOWNLOADING_FTP_PROGRAM) == 0) {
	  save_url_with_gtm (url);
	} else {
		save_url_with_command_line (url);
	}
	
	g_free(url);
	
	gtk_widget_destroy(GTK_WIDGET(dirselect));
}

/**
 * on_dirselect2_cancel_clicked: download directory dialog cancel clicked (prefs dialog)
 */
void
prefs_download_dir_cancel_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
  GtkWidget *dirselect = glade_lookup_widget(GTK_WIDGET(button), "dirselection2");
  gtk_widget_destroy(dirselect);
}


/**
 * prefs_ctree_tree_select_row_cb: select a prefs category
 */
void
prefs_ctree_tree_select_row_cb          (GtkWidget       *ctree,
                                        GtkCTreeNode	*node,
					GaleonBrowser *browser)
{
  GtkWidget *notebook = glade_lookup_widget(ctree, "notebook1");
  int row = 0;

  row = (gint)gtk_ctree_node_get_row_data(GTK_CTREE(ctree), GTK_CTREE_NODE(node));
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook), row);
}


/**
 * prefs_ctree_tree_unselect_row_cb: unselect a prefs category
 */
void
prefs_ctree_tree_unselect_row_cb        (GtkWidget       *ctree,
                                        GList		*node,
					gint             column,
                                        GaleonBrowser   *browser)
{
  GtkWidget *notebook = glade_lookup_widget(ctree, "notebook1");
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 0);
}


/**
 * automatically scroll the prefs tree window if clicking on either
 * of the last two items has made the scrollbars appear
 */
void 
prefs_ctree_tree_expand_cb              (GtkCTree        *ctree,
                                        GList           *node,
                                        GaleonBrowser   *browser)
{
	GtkWidget *window;
	GtkAdjustment *vadj;
	gint newpos;
	GtkCTreeNode *last_child;

	window = glade_lookup_widget(GTK_WIDGET(ctree), "scrolledwindow9");

	last_child = GTK_CTREE_ROW(node)->children;
	while (GTK_CTREE_ROW(last_child)->sibling != NULL)
		last_child = GTK_CTREE_ROW(last_child)->sibling;

	if (gtk_ctree_node_is_visible(ctree, last_child) != GTK_VISIBILITY_FULL)
	{
		vadj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(window));
		newpos = vadj->upper - vadj->lower - vadj->page_size;
		gtk_adjustment_set_value(vadj, newpos);
		gtk_adjustment_value_changed(vadj);
	}
}


/**
 * prefs_homepage_my_portal_button_clicked_cb: set homepage URL to "My Portal"
 */
void
prefs_homepage_my_portal_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
  GtkWidget *entry;

  entry = glade_lookup_widget(GTK_WIDGET(button), "startpage_entry");
  gtk_entry_set_text(GTK_ENTRY(entry), MYPORTAL_URL);
}


/**
 * prefs_homepage_current_button_clicked_cb: set homepage URL to current page
 */
void
prefs_homepage_current_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
  GtkWidget *entry;
  gchar *url;

  url = history_get_last_url();
  entry = glade_lookup_widget(GTK_WIDGET(button), "startpage_entry");
  gtk_entry_set_text(GTK_ENTRY(entry), url);
  g_free(url);
}

void
prefs_lang_add_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkList *list = GTK_LIST(glade_lookup_widget (GTK_WIDGET(button),"lang_list"));
	GtkWidget *entry = glade_lookup_widget (GTK_WIDGET(button),"lang_entry");
	GList *items = NULL;
	GtkWidget *list_item;
	char *entry_text;

	entry_text = gtk_editable_get_chars(GTK_EDITABLE(entry),0,-1);
	
	list_item = gtk_list_item_new_with_label(entry_text);
	gtk_object_set_data(GTK_OBJECT(list_item), "string", entry_text);

	items = g_list_append (items, list_item);
	gtk_list_append_items(list, items);
	gtk_widget_show(list_item);
}

void
prefs_lang_remove_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkList *list = GTK_LIST(glade_lookup_widget (GTK_WIDGET(button),"lang_list"));
	GList *selected;

	if (list->selection)
	{
		selected = g_list_copy(list->selection);
		gtk_list_remove_items (list, selected); 
	}
}

void
prefs_lang_up_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkList *list = GTK_LIST(glade_lookup_widget (GTK_WIDGET(button),"lang_list"));
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy(list->selection);
		pos = gtk_list_child_position(list, GTK_WIDGET(selected->data));
		if (pos!=0)
		{
			gtk_list_remove_items_no_unref (list, selected); 
			pos --;
			gtk_list_insert_items (list,selected,pos);
			gtk_list_select_item (list,pos);
		}
	}
}

void
prefs_lang_down_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	GtkList *list = GTK_LIST(glade_lookup_widget (GTK_WIDGET(button),"lang_list"));
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy(list->selection);
		pos = gtk_list_child_position(list, GTK_WIDGET(selected->data));
		gtk_list_remove_items_no_unref (list, selected); 
		pos ++;
		gtk_list_insert_items (list, selected, pos);
		gtk_list_select_item (list, pos);
	}
	
}

void 
prefs_clear_location_history_button_clicked_cb (GtkButton *button,
                                          GaleonBrowser *browser)
{
	GtkWidget *dialog;

	dialog = gnome_question_dialog_modal_parented(
			_("This will delete all items in your location dropdown menu.\n"
			  "Are you sure you want to do this?"),
			prefs_clear_location_history_question_cb,
			NULL,
			GTK_WINDOW(prefs));

	window_set_layer(dialog);
	gnome_dialog_run(GNOME_DIALOG(dialog));
}

void
prefs_clear_location_history_question_cb (gint reply, gpointer data)
{
	GList *w;

	if (reply)
		return;

	for (w = all_windows; w != NULL; w = g_list_next(w))
	{
		GaleonBrowser *gb = (GaleonBrowser *)(w->data);
		GnomeEntry *ge = GNOME_ENTRY(gb->toolbar_gnomeentry);

		gnome_entry_set_history_id(ge, "_dummy");
		gnome_entry_load_history(ge);
		gnome_entry_set_history_id(ge, "location_history");

		if (!g_list_next(w))
			gnome_entry_save_history(ge);
	}
}

void
prefs_clear_history_button_clicked_cb (GtkButton *button,
				 GaleonBrowser *browser)
{
	GtkWidget *dialog;

	dialog = gnome_question_dialog_modal_parented(
			_("This will delete all items stored in your history.\n"
			  "Are you sure you want to do this?"),
			prefs_clear_history_question_cb,
			NULL,
			GTK_WINDOW(prefs));

	window_set_layer(dialog);
	gnome_dialog_run(GNOME_DIALOG(dialog));
}

void
prefs_clear_history_question_cb (gint reply, gpointer data)
{
	extern GHashTable *history;
	extern GtkCList *clHistory;
	extern GCompletion *URLCompletion;

	if (reply)
		return;

	/* clear the history list */
	gtk_clist_freeze(clHistory);
	gtk_clist_clear(clHistory);
	gtk_clist_thaw(clHistory);

	/* clear the hash table */
	g_hash_table_freeze(history);
	g_hash_table_foreach_remove(history, (GHRFunc) prefs_history_item_free_cb, NULL);
	g_hash_table_thaw(history);

	/* clear the autocompletion history */
	g_completion_clear_items(URLCompletion);

	/* clear the file */
	history_save();
}

gboolean
prefs_history_item_free_cb (gpointer key, gpointer value, gpointer user_data)
{
	if (value)
		g_free(value);
	return TRUE;
}

void
prefs_option_selected_cb (GtkMenuShell *menushell, gpointer key)
{
	gint mod_key = GPOINTER_TO_INT(key);
	gint item_index;
	GtkWidget *active_item;

	active_item = gtk_menu_get_active(GTK_MENU(menushell));
	item_index = g_list_index(menushell->children, active_item);

	if (item_index == 0)
		mouse_wheel_controls_set_sensitive(active_item, mod_key, TRUE);
	else
		mouse_wheel_controls_set_sensitive(active_item, mod_key, FALSE);
}

void
prefs_browse_clicked_cb (GnomeFileEntry *fileentry, gpointer data)
{
	if (fileentry->fsw)
		window_set_layer(fileentry->fsw);
}

void
prefs_picker_clicked_cb (GtkWidget *widget, gpointer picker)
{
	if (GNOME_IS_FONT_PICKER(picker))
		window_set_layer(GNOME_FONT_PICKER(picker)->font_dialog);
	else if (GNOME_IS_COLOR_PICKER(picker))
		window_set_layer(GNOME_COLOR_PICKER(picker)->cs_dialog);
}
