/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* from prefs_callbacks.c */
void prefs_option_selected_cb (GtkMenuShell *menushell, gpointer data);

/* local types */

typedef enum
{
	NOKEY,
	ALT,
	CTRL,
	SHIFT
} wheel_mod_key;

typedef struct
{ 
	gint value;
	gchar *name;
	GtkCTreeNode **node_ref;
	gboolean expand;
} preferences_tree_item;

typedef struct
{
	GtkWidget *mime_enable_check;
	GtkWidget *mime_list;
	GtkWidget *auto_download_optionmenu;
	GtkWidget *ask_download_dir_optionmenu;
	GtkWidget *location_optionmenu;
	GtkWidget *toolbar_optionmenu;
	GtkWidget *startpage_entry;
	GtkWidget *startpage_optionmenu;
	GtkWidget *new_optionmenu;
	GtkWidget *bookmarks_optionmenu;
	GtkWidget *autobookmarks_enabled_check;
	GtkWidget *autobookmarks_count_spin;
	GtkWidget *autobookmarks_shorten_spin;
	GtkWidget *java_optionmenu;
	GtkWidget *javascript_optionmenu;
	GtkWidget *download_dir_entry;
	GtkWidget *tabbed_enable_check;
	GtkWidget *tabbed_autojump_check;
	GtkWidget *tabbed_position_optionmenu;
	GtkWidget *tabbed_shorten_spin;
	GtkWidget *completion_enable_optionmenu;
	GtkWidget *completion_sort_optionmenu;
	GtkWidget *show_spinner_check;
	GtkWidget *http_proxy_entry;
	GtkWidget *ftp_proxy_entry;
	GtkWidget *http_proxy_port_spin;
	GtkWidget *ftp_proxy_port_spin;
        GtkWidget *no_proxies_for_entry;
	GtkWidget *disk_cache_entry;
	GtkWidget *mem_cache_entry;
	GtkWidget *serif_font_fpicker;
	GtkWidget *sansserif_font_fpicker;
	GtkWidget *monospace_font_fpicker;
	GtkWidget *size_font_spin;
	GtkWidget *default_font_optionmenu;
	GtkWidget *history_expire_spin;
	GtkWidget *use_provided_fonts_optionmenu;
	GtkWidget *image_optionmenu;
	GtkWidget *ftp_optionmenu;
	GtkWidget *ftp_program_optionmenu;
	GtkWidget *ftp_command_entry;
	GtkWidget *menu_ctree;
	GtkWidget *show_location_label_check;
	GtkWidget *show_go_button_check;
	GtkWidget *use_external_source_viewer_check;
        GtkWidget *external_source_viewer_entry;
	GtkWidget *syntax_highlighting_checkbutton;
	GtkWidget *crash_optionmenu;
	GtkWidget *newwindows_optionmenu;
	GtkWidget *windows_title_entry;
	GtkWidget *fullscreen_stacking_checkbutton;
	GtkWidget *myportal_stylesheet_entry;
	GtkWidget *spinner_dir_entry;
	GtkWidget *autodetect_charset_optionmenu;
	GtkWidget *default_charset_entry;
	GtkWidget *lang_list;
	GtkWidget *background_color;
	GtkWidget *text_color;
	GtkWidget *unvisited_link_color;
	GtkWidget *visited_link_color;
	GtkWidget *use_provided_colors_optionmenu;
	GtkWidget *underline_links_checkbutton;
	GtkWidget *http_version_optionmenu;
	GtkWidget *proxy_keepalive_checkbutton;
	GtkWidget *min_doc_context_checkbutton;
	GtkWidget *history_popup_checkbutton;
	GtkWidget *middle_button_optionmenu;
	GtkWidget *right_button_optionmenu;
	GtkWidget *cookie_behavior_optionmenu;
	GtkWidget *warn_cookie_checkbutton;
	GtkWidget *wheel_nokey_action_optionmenu;
	GtkWidget *wheel_nokey_step_spinbutton;
	GtkWidget *wheel_nokey_default_step_checkbutton;
	GtkWidget *wheel_alt_action_optionmenu;
	GtkWidget *wheel_alt_step_spinbutton;
	GtkWidget *wheel_alt_default_step_checkbutton;
	GtkWidget *wheel_ctrl_action_optionmenu;
	GtkWidget *wheel_ctrl_step_spinbutton;
	GtkWidget *wheel_ctrl_default_step_checkbutton;
	GtkWidget *wheel_shift_action_optionmenu;
	GtkWidget *wheel_shift_step_spinbutton;
	GtkWidget *wheel_shift_default_step_checkbutton;
	GtkWidget *crash_behaviour_optionmenu;
	GtkWidget *links_drag_checkbutton;
} preferences_widgets;

/* local function prototypes */
static void preferences_load_mime(void);
static void preferences_save_mime(void);
static void preferences_lookup_widgets(GtkWidget *widget,
				       preferences_widgets *pw);
static void build_tree(GtkWidget *ctree, GtkCTreeNode *parent,
		       preferences_tree_item *item);
static void set_settings_menu_global (void);
static void set_config_from_editable(GtkWidget *editable, gchar *config_name);
static void set_config_from_optionmenu(GtkWidget *optionmenu,
				       gchar *config_name);
static void set_config_from_fontpicker(GtkWidget *fontpicker,
				       gchar *config_name);
static void set_config_from_spin_button(GtkWidget *spinbutton,
					gchar *config_name);
static void set_config_from_togglebutton(GtkWidget *togglebutton,
					 gchar *config_name);
static void set_config_from_int_entry(GtkWidget *entry, gchar *config_name);
static void set_config_from_color (GtkWidget *colorpicker, gchar *config_name);
static void set_editable_from_config(GtkWidget *editable, gchar *config_name);
static void set_optionmenu_from_config(GtkWidget *optionmenu,
				       gchar *config_name);
static void set_fontpicker_from_config(GtkWidget *fontpicker,
				       gchar *config_name);
static void set_spin_button_from_config(GtkWidget *spinbutton,
					gchar *config_name);
static void set_int_entry_from_config(GtkWidget *entry, gchar *config_name);
static void set_togglebutton_from_config(GtkWidget *togglebutton,
					 gchar *config_name);
static void set_list_from_config(GtkWidget *list, gchar *config_name);
static void set_config_from_list(GtkWidget *list, gchar *config_name);
static void set_color_from_config (GtkWidget *colorpicker, gchar *config_name);
static void preferences_mime_set_config_to_dialog(preferences_widgets pw);
static void preferences_mime_set_dialog_to_config(preferences_widgets pw);
void prefs_mime_edit_cb (GtkWidget *w, GaleonBrowser *browser);
static int optionmenu_get_curr_index (GtkWidget *optionmenu);
void mouse_wheel_controls_set_sensitive (GtkWidget *widget, wheel_mod_key key,
					 gboolean sensitive);

/**
 * preferences_load: load Galeon preferences needed at startup
 */
void
preferences_load(void)
{
	preferences_load_mime();
}

/**
 * preferences_save: save all Galeon preferences
 */
void
preferences_save(void)
{
	preferences_save_mime();
	gnome_config_sync();
}

/**
 * preferences_apply: make any global changes required for system state
 * to match config state
 */
void
preferences_apply (void)
{
	autobookmarks_generate ();
	mozilla_prefs_set ();
	toolbar_global_recreate ();  
	set_settings_menu_global ();

	if (tabbed_mode && master_notebook != NULL)
	{
		gint tab_pos;

		tab_pos = gnome_config_get_int
			(CONF_APPEARANCE_TABBED_POSITION);
		gtk_notebook_set_tab_pos (GTK_NOTEBOOK (master_notebook),
					  tab_pos);
	}
}

/* default MIME types */
struct { gchar *name; MimeAction action; } default_mimetypes[] =
{
	{ "application/x-bzip",       SAVE_TO_DISK },
	{ "audio/mp3",                SAVE_TO_DISK },
	{ "video/mpeg",               SAVE_TO_DISK },
	{ "application/x-tar",        SAVE_TO_DISK },
	{ "application/x-gzip",       SAVE_TO_DISK },
	{ "application/zip",          SAVE_TO_DISK },	
	{ "application/x-rpm",        SAVE_TO_DISK },
	{ NULL, UNKNOWN } /* terminator -- must be last */
};

/**
 * preferences_load_mime: load MIME types and details of their handlers
 */
static void
preferences_load_mime(void)
{
	gint i, count;
	gchar *mime_path;
	gchar **mime_type;
	MimeAction action;

	/* free up any existing list of MIME types */
	if (mime_types != NULL)
	{
		g_list_foreach(mime_types, (GFunc) mime_item_free, NULL);
		g_list_free(mime_types);
		mime_types = NULL;
	}

	/* working in the galeon/mime configuration hierarchy */
	gnome_config_push_prefix(CONF_MIME);
     
	/* find out the number of configured MIME types */
	count = gnome_config_get_int("count=-1");

	/* if this isn't yet set, copy in the defaults */
	if (count == -1)
	{
		for (i = 0; default_mimetypes[i].name != NULL; i++)
		{
			mime_set_action (default_mimetypes[i].name, 
					 default_mimetypes[i].action);
		}
	}

	/* otherwise (count >= 0) read from configuration structure */
	for (i = 0; i < count; i++)
	{
		gint c;

		/* generate the key */
		mime_path = g_strdup_printf("mime%d", i);

		/* lookup the data */
		gnome_config_get_vector(mime_path, &c, &mime_type);
		if (c != 2) { /* Probably a galeon <= 0.8.0 config file */
			int j;
			g_warning ("Strange MIME info in prefs, ignoring");
			g_print ("If you have recently updated your version of Galeon,\n"
				 "you can safely ignore the previous message.\n"
				 "Galeon will install the default MIME actions now.");
			/* Remove all MIME prefs */
			gnome_config_set_int ("count", -1);
			gnome_config_sync ();
			/* cleanup */
			g_free (mime_path);
			for (j = 0; j < 2; j++) 
				g_free (mime_type[j]);
			g_free (mime_type);
			gnome_config_pop_prefix ();
			/* call ourselves again */
			preferences_load_mime ();
			return;

		}
		if (!strcmp(mime_type[1], "SAVE_TO_DISK"))
			action = SAVE_TO_DISK;
		else if (!strcmp(mime_type[1], "RUN_PROGRAM"))
			action = RUN_PROGRAM;
		else if (!strcmp(mime_type[1], "ASK_ACTION"))
			action = ASK_ACTION;
		else if (!strcmp(mime_type[1], "LEAVE_TO_MOZILLA"))
			action = LEAVE_TO_MOZILLA;
		else {
			g_warning("preferences_load_mime: unknown mime action");
			action = UNKNOWN;
		}

		/* build the mime type */
		mime_set_action (mime_type[0], action);

		/* free allocated strings */
		g_free(mime_path);
		g_free (mime_type[0]);
		g_free (mime_type[1]);
		g_free (mime_type);
	}

	/* ! CONF_MIME */
	gnome_config_pop_prefix();
}

/**
 * preferences_save_mime: save MIME types and handlers 
 */
void
preferences_save_mime(void)
{
	GList *l;
	gchar *mime_type[2];
	gchar *mime_path;
	gint i, count;
  
	/* working in the galeon/mime configuration hierarchy */
	gnome_config_push_prefix(CONF_MIME);

	/* write out the number of MIME types */
	count = g_list_length(mime_types);
	gnome_config_set_int("count",  count);

	/* write out each MIME type entry as string vectors */
	for (l = mime_types, i = 0; l != NULL; l = g_list_next(l), i++)
	{
		/* get the MIME type */
		MimeItem *m = (MimeItem *)(l->data);
		if (m->action == UNKNOWN) continue; /* do not save this */

		/* fill the entries into the vector */
		mime_type[0] = m->name;
		switch (m->action)
		{
		case SAVE_TO_DISK : mime_type[1] = "SAVE_TO_DISK"; break;
		case RUN_PROGRAM  : mime_type[1] = "RUN_PROGRAM";  break;
		case ASK_ACTION   : mime_type[1] = "ASK_ACTION";   break;
		case LEAVE_TO_MOZILLA : mime_type[1] = "LEAVE_TO_MOZILLA"; break;
		case UNKNOWN : mime_type[1] = "UNKNOWN"; break;
		}

		/* write the vector out to the config */
		mime_path = g_strdup_printf("mime%d", i);
		gnome_config_set_vector(mime_path, 2,
					(const gchar **)mime_type);

		/* free allocated strings */
		g_free(mime_path);
	}

	/* ! CONF_MIME */
	gnome_config_pop_prefix();
}

/**
 * preferences_set_config_to_dialog: read all the preferences values 
 * out of the dialog box widget and store them in the configuration space
 */
void
preferences_set_config_to_dialog(GtkWidget *widget)
{
        preferences_widgets pw;

	/* lookup all the widgets */
	preferences_lookup_widgets(widget, &pw);

	/* set each of the general configuration options */
	gnome_config_push_prefix("/galeon/General/");
	set_config_from_editable(pw.startpage_entry, "startpage");
	set_config_from_optionmenu(pw.startpage_optionmenu, "startpage_type");
	set_config_from_optionmenu(pw.new_optionmenu, "newpage_type");
	set_config_from_optionmenu(pw.bookmarks_optionmenu, "bookmark_title");
	set_config_from_togglebutton(pw.show_spinner_check, "show_spinner");
	set_config_from_optionmenu(pw.newwindows_optionmenu, "newwindows");
	gnome_config_pop_prefix();	

	/* autobookmarks preferences */
	gnome_config_push_prefix("/galeon/AutoBookmarks/");
	set_config_from_togglebutton(pw.autobookmarks_enabled_check,"enabled");
	set_config_from_spin_button(pw.autobookmarks_count_spin, "count");
	set_config_from_spin_button(pw.autobookmarks_shorten_spin, "shorten");
	gnome_config_pop_prefix();	
	
	/* process downloading preferences */
	gnome_config_push_prefix("/galeon/Downloading/");
	set_config_from_optionmenu(pw.auto_download_optionmenu,
				   "auto_download");
	set_config_from_optionmenu(pw.ask_download_dir_optionmenu,
				   "ask_for_download_dir");
	set_config_from_optionmenu(pw.ftp_program_optionmenu, 
				   "ftp_program");
	set_config_from_editable(pw.ftp_command_entry, "ftp_command");
	set_config_from_editable(pw.download_dir_entry, "download_dir");
	gnome_config_pop_prefix();
	
	/* process appearance preferences */
	gnome_config_push_prefix("/galeon/Appearance/");
	set_config_from_togglebutton(pw.tabbed_enable_check, "tabbed");
	set_config_from_togglebutton(pw.tabbed_autojump_check, 
				     "tabbed_autojump");
	set_config_from_optionmenu(pw.tabbed_position_optionmenu, 
				   "tabbed_position");
	set_config_from_spin_button(pw.tabbed_shorten_spin, "tabbed_shorten");
	set_config_from_optionmenu(pw.location_optionmenu, "url_location");
	set_config_from_optionmenu(pw.toolbar_optionmenu, "toolbar_type");
	set_config_from_fontpicker(pw.serif_font_fpicker, "serif_font");
	set_config_from_fontpicker(pw.sansserif_font_fpicker,
				   "sansserif_font");
	set_config_from_fontpicker(pw.monospace_font_fpicker,
				   "monospace_font");
	set_config_from_spin_button(pw.size_font_spin, "size_font");
	set_config_from_optionmenu(pw.use_provided_fonts_optionmenu, 
				   "use_provided_fonts");
 	set_config_from_optionmenu(pw.default_font_optionmenu, "default_font");
	set_config_from_togglebutton(pw.show_location_label_check,
				     "show_location_label");
	set_config_from_togglebutton(pw.show_go_button_check,
				     "show_go_button");
	set_config_from_editable(pw.windows_title_entry, "windows_title");
	set_config_from_togglebutton(pw.fullscreen_stacking_checkbutton,
				     "fullscreen_stacking");
	set_config_from_editable(pw.myportal_stylesheet_entry,
				 "myportal_stylesheet");
	set_config_from_editable(pw.spinner_dir_entry, "spinner_dir");
	set_config_from_color (pw.background_color, "background_color");
	set_config_from_color (pw.text_color, "text_color");
	set_config_from_color (pw.unvisited_link_color, "unvisited_link_color");
	set_config_from_color (pw.visited_link_color, "visited_link_color");
	set_config_from_optionmenu (pw.use_provided_colors_optionmenu, 
				    "use_provided_colors");
	set_config_from_togglebutton (pw.underline_links_checkbutton, 
				      "underline_links");
	set_config_from_togglebutton (pw.min_doc_context_checkbutton, 
				      "min_doc_context_checkbutton");
	gnome_config_pop_prefix();

	/* process mouse options */
	gnome_config_push_prefix("/galeon/Mouse/");
	set_config_from_togglebutton (pw.history_popup_checkbutton,
				      "history_popup");
	set_config_from_optionmenu (pw.middle_button_optionmenu,
				    "middle_button_action");
	set_config_from_optionmenu (pw.right_button_optionmenu,
				    "right_button_action");
	set_config_from_togglebutton (pw.links_drag_checkbutton,
				      "links_drag");
	gnome_config_pop_prefix();

	/* process completion preferences */
	gnome_config_push_prefix("/galeon/completion/");
	set_config_from_optionmenu(pw.completion_enable_optionmenu,
				   "autocompletion");
	set_config_from_optionmenu(pw.completion_sort_optionmenu,
				   "sorting_type");
	gnome_config_pop_prefix();

        /* process handlers preferences */
	gnome_config_push_prefix("/galeon/Handlers/");
	set_config_from_optionmenu(pw.ftp_optionmenu, "ftp");
	set_config_from_togglebutton (pw.use_external_source_viewer_check,
				      "use_external_source_viewer");
	set_config_from_editable (pw.external_source_viewer_entry, 
				  "external_source_viewer");
	set_config_from_togglebutton (pw.syntax_highlighting_checkbutton,
				      "enable_syntax_highlighting");
	set_config_from_togglebutton (pw.mime_enable_check, "mime_enable");
	gnome_config_pop_prefix();
 
        /* process advanced preferences */
	gnome_config_push_prefix("/galeon/Advanced/");
	set_config_from_editable(pw.http_proxy_entry, "http_proxy");
	set_config_from_editable(pw.ftp_proxy_entry, "ftp_proxy");
	set_config_from_editable(pw.no_proxies_for_entry, "no_proxies_for");
	set_config_from_spin_button(pw.http_proxy_port_spin,
				    "http_proxy_port");
	set_config_from_spin_button(pw.ftp_proxy_port_spin, "ftp_proxy_port");
	set_config_from_int_entry(pw.disk_cache_entry, "disk_cache_size");
	set_config_from_int_entry(pw.mem_cache_entry, "mem_cache_size");
	set_config_from_optionmenu(pw.java_optionmenu, "java_enabled");
	set_config_from_optionmenu(pw.javascript_optionmenu,
				   "javascript_enabled");
	set_config_from_spin_button(pw.history_expire_spin, "history_expire");
	set_config_from_optionmenu(pw.image_optionmenu, "image_loading_type");
	set_config_from_optionmenu(pw.crash_optionmenu, "crash_recovery");
	set_config_from_optionmenu(pw.autodetect_charset_optionmenu,
				   "autodetect_charset");
	set_config_from_editable (pw.default_charset_entry, "default_charset");
	set_config_from_list (pw.lang_list, "lang");
	set_config_from_optionmenu(pw.http_version_optionmenu, 
				   "http_version"); 
	set_config_from_togglebutton(pw.proxy_keepalive_checkbutton, 
				     "proxy_keepalive");
	set_config_from_togglebutton(pw.warn_cookie_checkbutton, 
				     "cookie_warn");
	set_config_from_optionmenu(pw.cookie_behavior_optionmenu, 
				   "cookie_behavior");
	set_config_from_optionmenu(pw.wheel_nokey_action_optionmenu,
				   "wheel_nokey_action"); 
	set_config_from_optionmenu(pw.wheel_ctrl_action_optionmenu,
				   "wheel_ctrl_action"); 
	set_config_from_optionmenu(pw.wheel_shift_action_optionmenu,
				   "wheel_shift_action");
	set_config_from_optionmenu(pw.wheel_alt_action_optionmenu,
				   "wheel_alt_action");  
	set_config_from_spin_button(pw.wheel_nokey_step_spinbutton,
				    "wheel_nokey_step");
	set_config_from_spin_button(pw.wheel_ctrl_step_spinbutton,
				    "wheel_ctrl_step");
	set_config_from_spin_button(pw.wheel_shift_step_spinbutton,
				    "wheel_shift_step");
	set_config_from_spin_button(pw.wheel_alt_step_spinbutton,
				    "wheel_alt_step");
	set_config_from_togglebutton(pw.wheel_nokey_default_step_checkbutton,
				     "wheel_nokey_default_step");
	set_config_from_togglebutton(pw.wheel_ctrl_default_step_checkbutton,
				     "wheel_ctrl_default_step");
	set_config_from_togglebutton(pw.wheel_shift_default_step_checkbutton,
				     "wheel_shift_default_step");
	set_config_from_togglebutton(pw.wheel_alt_default_step_checkbutton,
				     "wheel_alt_default_step");
	set_config_from_optionmenu(pw.crash_behaviour_optionmenu,
				   "crash_behaviour");
	gnome_config_pop_prefix();

	preferences_mime_set_config_to_dialog (pw);

	/* save all these preferences to disk */
	preferences_save();
}

void
preferences_set_dialog_to_config(GtkWidget *widget)
{
	preferences_widgets pw;
	GtkCTreeNode *browsing_node;
	GtkCTreeNode *user_interface_node;
	GtkCTreeNode *handlers_node;
	GtkCTreeNode *rendering_node;
	GtkCTreeNode *advanced_node;
  
	preferences_tree_item main_tree[] =
	{
		{  0, _("Browsing"),       &browsing_node,       TRUE  },
		{  0, _("User Interface"), &user_interface_node, TRUE  },
		{  0, _("Handlers"),       &handlers_node,       TRUE  },
		{  0, _("Rendering"),      &rendering_node,      TRUE  },
		{  0, _("Advanced"),       &advanced_node,       FALSE },
		{ -1, NULL, NULL, FALSE } /* terminator */
	};

	preferences_tree_item browsing_tree[] =
	{
		{  1, _("Homepage"),        NULL, FALSE },
		{  2, _("Bookmarks"),       NULL, FALSE },
		{  3, _("History"),         NULL, FALSE },
		{  4, _("Java"),            NULL, FALSE },
		{  5, _("Filtering"),       NULL, FALSE },
		{ -1, NULL, NULL, FALSE } /* terminator */
        };

	preferences_tree_item user_interface_tree[] =
	{
		{  6, _("Layout"),          NULL, FALSE },
		{  7, _("Tabbed Browsing"), NULL, FALSE },
		{  8, _("Spinner"),         NULL, FALSE },
		{  9, _("Mouse/Context"),   NULL, FALSE },
		{ 10, _("Completion"),      NULL, FALSE },
		{ 11, _("Crash Recovery"),  NULL, FALSE },
		{ -1, NULL, NULL, FALSE } /* terminator */
        };

	preferences_tree_item handlers_tree[] =
	{
		{ 12, _("Downloading"),     NULL, FALSE },
		{ 13, _("View Source"),     NULL, FALSE },

/* removed whilst we experiment with other approaches to
   handling mime types, may be removed altogether eventually - MattA 2/10/00 */

/* readded Ricardo -- 18/11/2000 */ 

/* D'Oh! These settings refer to the "Galeon MIME types" -- in actual
 * fact the currently active code uses "GNOME MIME types", so this
 * preference is irrelevant :-) The only reason I haven't removed it
 * yet is because when we have the proper callbacks in place we'll need
 * to be able to choose between a default of ask/open/view/edit...
 * at which point we might be able to reuse some of the associated code.
 *       -- Removed again. 19/11/200. MattA. :-)
 */

/* I think we should use this instead of gnome mime types. Gnome mime types
 * are not meant for a web browser... -- Ricardo agian (btw: must we use
 * the source code to talk about this? ;) )
 */		

/* Heh. Well we've been using the gnome mime types since the beginning of
 * October, when the old mime handling was commented out. Why don't you
 * think these are suitable for a web browser? I think they should
 * be ideal. -- MattA again (ps: got any better ideas on communication? :-) )
 */

/* Finally, added again. This will be the last time */

		{ 14, _("MIME Types"),      NULL, FALSE }, 

		{ -1, NULL, NULL, FALSE } /* terminator */
	};

	preferences_tree_item rendering_tree[] =
	{
		{ 15, _("Fonts"),           NULL, FALSE },
		{ 16, _("Colors"),          NULL, FALSE },
		{ 17, _("Languages"),       NULL, FALSE },

		{ -1, NULL, NULL, FALSE } /* terminator */
	};

	preferences_tree_item advanced_tree[] =
	{
		{ 18, _("Proxy"),           NULL, FALSE },
		{ 19, _("Cache"),           NULL, FALSE },
		{ 20, _("Network"),         NULL, FALSE },
		{ 21, _("Cookies"),         NULL, FALSE },
		{ 22, _("Mouse Wheel"),     NULL, FALSE },

		{ -1, NULL, NULL } /* terminator */
	};

	/* lookup all the widgets */
	preferences_lookup_widgets(widget, &pw);

	/* build the preferences tree */
	build_tree(pw.menu_ctree, NULL, main_tree);
	build_tree(pw.menu_ctree, browsing_node, browsing_tree);
	build_tree(pw.menu_ctree, user_interface_node, user_interface_tree);
	build_tree(pw.menu_ctree, handlers_node, handlers_tree);
	build_tree(pw.menu_ctree, rendering_node, rendering_tree);
	build_tree(pw.menu_ctree, advanced_node, advanced_tree);

	/* connect signals to mouse wheel optionmenus */
	gtk_signal_connect(
	   GTK_OBJECT(GTK_OPTION_MENU(pw.wheel_nokey_action_optionmenu)->menu),
	   "deactivate", GTK_SIGNAL_FUNC(prefs_option_selected_cb),
	   GINT_TO_POINTER(NOKEY));
	gtk_signal_connect(
	   GTK_OBJECT(GTK_OPTION_MENU(pw.wheel_alt_action_optionmenu)->menu),
	   "deactivate", GTK_SIGNAL_FUNC(prefs_option_selected_cb),
	   GINT_TO_POINTER(ALT));
	gtk_signal_connect(
	   GTK_OBJECT(GTK_OPTION_MENU(pw.wheel_ctrl_action_optionmenu)->menu),
	   "deactivate", GTK_SIGNAL_FUNC(prefs_option_selected_cb),
	   GINT_TO_POINTER(CTRL));
	gtk_signal_connect(
	   GTK_OBJECT(GTK_OPTION_MENU(pw.wheel_shift_action_optionmenu)->menu),
	   "deactivate", GTK_SIGNAL_FUNC(prefs_option_selected_cb),
	   GINT_TO_POINTER(SHIFT));

	/* fill general */
	gnome_config_push_prefix("/galeon/General/");
	set_editable_from_config(pw.startpage_entry,
				 "startpage=www.gnome.org");
	set_optionmenu_from_config(pw.startpage_optionmenu,
				   "startpage_type=0");
	set_optionmenu_from_config(pw.new_optionmenu, "newpage_type=0");
	set_optionmenu_from_config(pw.bookmarks_optionmenu, 
				   "bookmark_title=0");
	set_togglebutton_from_config(pw.show_spinner_check, "show_spinner=1");
	set_optionmenu_from_config(pw.newwindows_optionmenu, "newwindows=0");
	gnome_config_pop_prefix();

	/* autobookmarks preferences */
	gnome_config_push_prefix("/galeon/AutoBookmarks/");
	set_togglebutton_from_config(pw.autobookmarks_enabled_check,
				     "enabled=TRUE");
	set_spin_button_from_config(pw.autobookmarks_count_spin, "count=15");
	set_spin_button_from_config(pw.autobookmarks_shorten_spin, 
				    "shorten=25");
	gnome_config_pop_prefix();
	
	/* process downloading preferences */
	gnome_config_push_prefix("/galeon/Downloading/");
	set_optionmenu_from_config(pw.auto_download_optionmenu,
				   "auto_download=0");
	set_optionmenu_from_config(pw.ask_download_dir_optionmenu,
				   "ask_for_download_dir=0");
	set_editable_from_config(pw.download_dir_entry, "download_dir=");
	set_optionmenu_from_config(pw.ftp_program_optionmenu, "ftp_program=0");
	set_editable_from_config(pw.ftp_command_entry, "ftp_command=");
	gnome_config_pop_prefix();
	
	/* process appearance preferences */
	gnome_config_push_prefix("/galeon/Appearance/");
	set_togglebutton_from_config(pw.tabbed_enable_check, "tabbed=FALSE");
	set_togglebutton_from_config(pw.tabbed_autojump_check, 
				     "tabbed_autojump=FALSE");
	set_optionmenu_from_config(pw.tabbed_position_optionmenu, 
				   "tabbed_position=2");
	set_spin_button_from_config(pw.tabbed_shorten_spin,
				    "tabbed_shorten=30");
	set_optionmenu_from_config(pw.location_optionmenu, "url_location=1");
	set_optionmenu_from_config(pw.toolbar_optionmenu, "toolbar_type=2");
	set_fontpicker_from_config(pw.serif_font_fpicker,
	    "serif_font=-*-times-medium-r-normal-*-14-*-*-*-*-*-*-*");
	set_fontpicker_from_config(pw.sansserif_font_fpicker,
            "sansserif_font=-*-helvetica-medium-r-normal-*-14-*-*-*-*-*-*-*");
	set_fontpicker_from_config(pw.monospace_font_fpicker,
	    "monospace_font=-*-courier-medium-r-normal-*-14-*-*-*-*-*-*-*");
	set_spin_button_from_config(pw.size_font_spin, "size_font=14");
	set_optionmenu_from_config(pw.use_provided_fonts_optionmenu, 
				   "use_provided_fonts=1");
 	set_optionmenu_from_config(pw.default_font_optionmenu,
				   "default_font=1");
	set_togglebutton_from_config(pw.show_location_label_check,
				     "show_location_label=TRUE");
	set_togglebutton_from_config(pw.show_go_button_check,
				     "show_go_button=TRUE");
	set_editable_from_config(pw.windows_title_entry,
				 "windows_title=Galeon - %s");
	set_togglebutton_from_config(pw.fullscreen_stacking_checkbutton,
				     "fullscreen_stacking=TRUE");
	set_editable_from_config(pw.myportal_stylesheet_entry,
				 "myportal_stylesheet=");
	set_editable_from_config(pw.spinner_dir_entry,
				 "spinner_dir=" SHARE_DIR);
	set_color_from_config (pw.background_color, "background_color=#FFFFFF");
	set_color_from_config (pw.text_color, "text_color=#000000");
	set_color_from_config (pw.unvisited_link_color, "unvisited_link_color=#0000FF");
	set_color_from_config (pw.visited_link_color, "visited_link_color=#FF0000");
	set_optionmenu_from_config (pw.use_provided_colors_optionmenu,
				    "use_provided_colors=0");
	set_togglebutton_from_config (pw.underline_links_checkbutton, 
				      "underline_links=TRUE");
	set_togglebutton_from_config (pw.min_doc_context_checkbutton, 
				      "min_doc_context_checkbutton=FALSE");
	gnome_config_pop_prefix();

	/* process mouse options */
	gnome_config_push_prefix("/galeon/Mouse/");
	set_togglebutton_from_config (pw.history_popup_checkbutton,
				      "history_popup=FALSE");
	set_optionmenu_from_config (pw.middle_button_optionmenu,
				    "middle_button_action=0");
	set_optionmenu_from_config (pw.right_button_optionmenu,
				    "right_button_action=0");
	set_togglebutton_from_config (pw.links_drag_checkbutton,
				      "links_drag=FALSE");
	gnome_config_pop_prefix();

	/* process completion preferences */
	gnome_config_push_prefix("/galeon/completion/");
	set_optionmenu_from_config(pw.completion_enable_optionmenu,
				   "autocompletion=0");
	set_optionmenu_from_config(pw.completion_sort_optionmenu,
				   "sorting_type=0");
	gnome_config_pop_prefix();
	
        /* process handlers preferences */
	gnome_config_push_prefix("/galeon/Handlers/");
	set_optionmenu_from_config(pw.ftp_optionmenu, "ftp=0");
	set_togglebutton_from_config (pw.use_external_source_viewer_check, 
				      "use_external_source_viewer=FALSE");
	set_editable_from_config (pw.external_source_viewer_entry, 
				  "external_source_viewer=gnome-edit");
	set_togglebutton_from_config (pw.mime_enable_check, "mime_enable=TRUE");
	set_togglebutton_from_config (pw.syntax_highlighting_checkbutton,
				      "enable_syntax_highlighting=FALSE");
	gnome_config_pop_prefix();
 
        /* process advanced preferences */
	gnome_config_push_prefix("/galeon/Advanced/");
	set_editable_from_config(pw.http_proxy_entry, "http_proxy=");
	set_editable_from_config(pw.ftp_proxy_entry, "ftp_proxy=");
	set_editable_from_config(pw.no_proxies_for_entry, "no_proxies_for=");
	set_spin_button_from_config(pw.http_proxy_port_spin,
				    "http_proxy_port=8080");
	set_spin_button_from_config(pw.ftp_proxy_port_spin,
				    "ftp_proxy_port=8080");
	set_optionmenu_from_config(pw.http_version_optionmenu,
				   "http_version=1.1");
	set_togglebutton_from_config(pw.proxy_keepalive_checkbutton,
				     "proxy_keepalive=TRUE");
	/* these defaults seem a bit tight, even for a 
	 * slimline browser such as Galeon! -- MattA */
	set_int_entry_from_config(pw.disk_cache_entry, "disk_cache_size=7680");
	set_int_entry_from_config(pw.mem_cache_entry, "mem_cache_size=1024");
	set_optionmenu_from_config(pw.java_optionmenu, "java_enabled=1");
	set_optionmenu_from_config(pw.javascript_optionmenu,
				   "javascript_enabled=1");
	set_spin_button_from_config(pw.history_expire_spin,
				    "history_expire=100");
	set_optionmenu_from_config(pw.image_optionmenu,
				   "image_loading_type=0");
	set_optionmenu_from_config(pw.crash_optionmenu, "crash_recovery=1");
	set_optionmenu_from_config(pw.autodetect_charset_optionmenu,
				   "autodetect_charset=0");
 	set_editable_from_config (pw.default_charset_entry,
				  "default_charset=Western (ISO-8859-1)"); 
 	set_list_from_config (pw.lang_list,"lang=English\\ (en)"); 
	set_togglebutton_from_config(pw.warn_cookie_checkbutton,
				     "cookie_warn=FALSE");
	set_optionmenu_from_config(pw.cookie_behavior_optionmenu,
				   "cookie_behavior=0");
	set_optionmenu_from_config(pw.wheel_nokey_action_optionmenu,
				   "wheel_nokey_action=0"); 
	set_optionmenu_from_config(pw.wheel_ctrl_action_optionmenu,
				   "wheel_ctrl_action=2"); 
	set_optionmenu_from_config(pw.wheel_shift_action_optionmenu,
				   "wheel_shift_action=0");
	set_optionmenu_from_config(pw.wheel_alt_action_optionmenu,
				   "wheel_alt_action=0");  
	if (optionmenu_get_curr_index(pw.wheel_nokey_action_optionmenu) == 0)
		mouse_wheel_controls_set_sensitive(widget, NOKEY, TRUE);
	else
		mouse_wheel_controls_set_sensitive(widget, NOKEY, FALSE);
	if (optionmenu_get_curr_index(pw.wheel_alt_action_optionmenu) == 0)
		mouse_wheel_controls_set_sensitive(widget, ALT, TRUE);
	else
		mouse_wheel_controls_set_sensitive(widget, ALT, FALSE);
	if (optionmenu_get_curr_index(pw.wheel_ctrl_action_optionmenu) == 0)
		mouse_wheel_controls_set_sensitive(widget, CTRL, TRUE);
	else
		mouse_wheel_controls_set_sensitive(widget, CTRL, FALSE);
	if (optionmenu_get_curr_index(pw.wheel_shift_action_optionmenu) == 0)
		mouse_wheel_controls_set_sensitive(widget, SHIFT, TRUE);
	else
		mouse_wheel_controls_set_sensitive(widget, SHIFT, FALSE);

	set_spin_button_from_config(pw.wheel_nokey_step_spinbutton,
				    "wheel_nokey_step=1");
	set_spin_button_from_config(pw.wheel_ctrl_step_spinbutton,
				    "wheel_ctrl_step=1");
	set_spin_button_from_config(pw.wheel_shift_step_spinbutton,
				    "wheel_shift_step=1");
	set_spin_button_from_config(pw.wheel_alt_step_spinbutton,
				    "wheel_alt_step=1");
	set_togglebutton_from_config(pw.wheel_nokey_default_step_checkbutton,
				     "wheel_nokey_default_step=TRUE");
	set_togglebutton_from_config(pw.wheel_ctrl_default_step_checkbutton,
				     "wheel_ctrl_default_step=TRUE");
	set_togglebutton_from_config(pw.wheel_shift_default_step_checkbutton,
				     "wheel_shift_default_step=TRUE");
	set_togglebutton_from_config(pw.wheel_alt_default_step_checkbutton,
				     "wheel_alt_default_step=TRUE");
	set_optionmenu_from_config(pw.crash_behaviour_optionmenu,
				   "crash_behaviour=0");
	gnome_config_pop_prefix();

        /* process mime types */
	preferences_mime_set_dialog_to_config(pw);
}

static void
preferences_mime_set_config_to_dialog(preferences_widgets pw)
{
	int i;
	/* free up any existing list of MIME types */
	if (mime_types != NULL) {
		g_list_foreach(mime_types, (GFunc) mime_item_free, NULL);
		g_list_free(mime_types);
		mime_types = NULL;
	}

	for (i = 0; i < GTK_CLIST (pw.mime_list)->rows; i++) {
		MimeItem *item = gtk_clist_get_row_data (GTK_CLIST (pw.mime_list), i);
		mime_types = g_list_append (mime_types, mime_item_clone (item));
	}
}

static void
preferences_mime_set_dialog_to_config(preferences_widgets pw)
{
	GList *l;
	gchar *text[3] = { NULL, NULL, NULL};
	GtkWidget *clist = pw.mime_list;
	GtkWidget *action_optionmenu = glade_lookup_widget 
		(pw.mime_list, "prefs_mime_action_optionmenu");
	gint row;
        GtkWidget *menuitem;
	GtkWidget *action_menu;
        
        /* Connect the action menu signals, I can't connect them in a better way :( */
        action_menu = GTK_OPTION_MENU (action_optionmenu)->menu;
        for (l = GTK_MENU_SHELL (action_menu)->children; l != NULL; l = l->next) {
		menuitem = GTK_WIDGET (l->data);
		gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
				    prefs_mime_edit_cb, NULL);
	}

	/* freeze and empty the list */
	gtk_clist_freeze (GTK_CLIST(clist));
	gtk_clist_clear (GTK_CLIST(clist));

	/* re-insert every element */
	for (l = mime_types; l != NULL; l = g_list_next(l))
	{
		MimeItem *m = (MimeItem *)(l->data);

		/* insert the row */
		text[0] = m->name;
		switch (m->action) {
		case LEAVE_TO_MOZILLA:
			text[1] = N_("Render with Galeon");
			break;
		case SAVE_TO_DISK:
			text[1] = N_("Save to disk");
			break;
		case RUN_PROGRAM:  
			/* Should run the program associted to the mime type by gnome */
			/* not yet done */
		case UNKNOWN:
			text[1] = N_("Unknown");
			break;
		case ASK_ACTION:
			text[1] = N_("Ask the user");
			break;
		}
		row = gtk_clist_append (GTK_CLIST(clist), text);

		gtk_clist_set_row_data (GTK_CLIST(clist), row, mime_item_clone (m));
	}

	/* thaw the list -- make changes visible */
	gtk_clist_thaw (GTK_CLIST(clist));	
}

static void 
set_settings_menu_global (void)
{
	GList *l;

	for (l = all_windows; l != NULL; l = l->next)
	{
		set_settings_menu_window((GaleonBrowser *) l->data);
	}
}

void
set_settings_menu_window (GaleonBrowser *browser)
{
	gint load_images = gnome_config_get_int (CONF_ADVANCED_IMAGE_LOADING_TYPE);
	gboolean use_own_fonts = 
		(gnome_config_get_int (CONF_APPEARANCE_USE_PROVIDED_FONTS) == 0);
	gboolean use_own_colors =
		(gnome_config_get_int (CONF_APPEARANCE_USE_PROVIDED_COLORS) == 1);
	gboolean enable_java = gnome_config_get_int (CONF_ADVANCED_JAVA_ENABLED);
	gboolean enable_javascript = 
		gnome_config_get_int (CONF_ADVANCED_JAVASCRIPT_ENABLED);
	gboolean enable_mime = 
		gnome_config_get_bool (CONF_HANDLERS_MIME_ENABLE);
	
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (browser->use_own_fonts),
					use_own_fonts);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (browser->use_own_colors), 
					use_own_colors);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (browser->enable_java), 
					enable_java);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (browser->enable_javascript),
					enable_javascript);
	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM (browser->use_galeon_mime_handling), enable_mime);
	
	switch (load_images) {
	case 0:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (browser->load_images_always), TRUE);
		break;
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM 
			 (browser->load_images_from_current_server_only), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (browser->load_images_never), TRUE);
		break;
	default:
		g_warning ("strange option for " CONF_ADVANCED_IMAGE_LOADING_TYPE);
	}
}

static void
preferences_lookup_widgets(GtkWidget *widget, preferences_widgets *retpw)
{
	/* this "unusual" technique ensures all elements are initialised */
#define Q(name) glade_lookup_widget(widget, name)
	preferences_widgets pw =
	{
mime_list:                           Q("preferences_mime_list"),
mime_enable_check:                   Q("mime_enable_check"),
auto_download_optionmenu:            Q("auto_download_optionmenu"),
ask_download_dir_optionmenu:         Q("ask_download_dir_optionmenu"),
location_optionmenu:                 Q("location_optionmenu"),
toolbar_optionmenu:                  Q("toolbar_optionmenu"),
startpage_entry:                     Q("startpage_entry"),
startpage_optionmenu:                Q("startpage_optionmenu"),
new_optionmenu:                      Q("new_optionmenu"),
bookmarks_optionmenu:                Q("bookmarks_optionmenu"),
autobookmarks_enabled_check:         Q("autobookmarks_enabled_check"),
autobookmarks_count_spin:            Q("autobookmarks_count_spin"),
autobookmarks_shorten_spin:          Q("autobookmarks_shorten_spin"),
java_optionmenu:                     Q("java_optionmenu"),
javascript_optionmenu:               Q("javascript_optionmenu"),
download_dir_entry:                  Q("download_dir_entry"),
tabbed_enable_check:                 Q("tabbed_enable_check"),
tabbed_autojump_check:               Q("tabbed_autojump_check"),
tabbed_position_optionmenu:          Q("tabbed_position_optionmenu"),
tabbed_shorten_spin:                 Q("tabbed_shorten_spin"),
completion_enable_optionmenu:        Q("completion_enable_optionmenu"),
completion_sort_optionmenu:          Q("completion_sort_optionmenu"),
show_spinner_check:                  Q("show_spinner_checkbutton"),
http_proxy_entry:                    Q("http_proxy_entry"),
ftp_proxy_entry:                     Q("ftp_proxy_entry"),
no_proxies_for_entry:                Q("no_proxies_for_entry"),
http_proxy_port_spin:                Q("http_proxy_port_spin"),
ftp_proxy_port_spin:                 Q("ftp_proxy_port_spin"),
disk_cache_entry:                    Q("disk_cache_entry"),
mem_cache_entry:                     Q("mem_cache_entry"),
serif_font_fpicker:                  Q("serif_font_fpicker"),
sansserif_font_fpicker:              Q("sansserif_font_fpicker"),
monospace_font_fpicker:              Q("monospace_font_fpicker"),
size_font_spin:                      Q("size_font_spin"),
use_provided_fonts_optionmenu:       Q("use_provided_fonts_optionmenu"),
default_font_optionmenu:             Q("default_font_optionmenu"),
history_expire_spin:                 Q("history_expire_spin"),
image_optionmenu:                    Q("image_optionmenu"),
ftp_optionmenu:                      Q("ftp_optionmenu"),
ftp_program_optionmenu:              Q("ftp_program_optionmenu"),
ftp_command_entry:                   Q("ftp_command_entry"),
menu_ctree:                          Q("menu_ctree"),
show_location_label_check:           Q("show_location_label_check"),
show_go_button_check:                Q("show_go_button_check"),
use_external_source_viewer_check:    Q("use_external_source_viewer_check"),
syntax_highlighting_checkbutton:     Q("syntax_highlighting_checkbutton"),
external_source_viewer_entry:        Q("external_source_viewer_entry"),
crash_optionmenu:                    Q("crash_optionmenu"),
newwindows_optionmenu:               Q("newwindows_optionmenu"),
windows_title_entry:                 Q("windows_title_entry"),
fullscreen_stacking_checkbutton:     Q("fullscreen_stacking_checkbutton"),
myportal_stylesheet_entry:           Q("myportal_stylesheet_entry"),
spinner_dir_entry:                   Q("spinner_dir_entry"),
autodetect_charset_optionmenu:       Q("autodetect_charset_optionmenu"),
default_charset_entry:               Q("default_charset_entry"),
lang_list:                           Q("lang_list"),
background_color:                    Q("background_color"),
text_color:                          Q("text_color"),
unvisited_link_color:                Q("unvisited_link_color"),
visited_link_color:                  Q("visited_link_color"),
use_provided_colors_optionmenu:      Q("use_provided_colors_optionmenu"),
underline_links_checkbutton:         Q("underline_links_checkbutton"),
proxy_keepalive_checkbutton:         Q("proxy_keepalive_checkbutton"),
http_version_optionmenu:             Q("http_version_optionmenu"),
min_doc_context_checkbutton:         Q("min_doc_context_checkbutton"),
history_popup_checkbutton:           Q("history_popup_checkbutton"),
middle_button_optionmenu:            Q("middle_button_optionmenu"),
right_button_optionmenu:             Q("right_button_optionmenu"),
cookie_behavior_optionmenu:          Q("cookie_behavior_optionmenu"),
warn_cookie_checkbutton:             Q("warn_cookie_checkbutton"),
wheel_nokey_action_optionmenu:       Q("wheel_nokey_action_optionmenu"),
wheel_nokey_step_spinbutton:         Q("wheel_nokey_step_spinbutton"),
wheel_nokey_default_step_checkbutton:Q("wheel_nokey_default_step_checkbutton"),
wheel_alt_action_optionmenu:         Q("wheel_alt_action_optionmenu"),
wheel_alt_step_spinbutton:           Q("wheel_alt_step_spinbutton"),
wheel_alt_default_step_checkbutton:  Q("wheel_alt_default_step_checkbutton"),
wheel_ctrl_action_optionmenu:        Q("wheel_ctrl_action_optionmenu"),
wheel_ctrl_step_spinbutton:          Q("wheel_ctrl_step_spinbutton"),
wheel_ctrl_default_step_checkbutton: Q("wheel_ctrl_default_step_checkbutton"),
wheel_shift_action_optionmenu:       Q("wheel_shift_action_optionmenu"),
wheel_shift_step_spinbutton:         Q("wheel_shift_step_spinbutton"),
wheel_shift_default_step_checkbutton:Q("wheel_shift_default_step_checkbutton"),
crash_behaviour_optionmenu:          Q("crash_behaviour_optionmenu"),
links_drag_checkbutton:              Q("links_drag_checkbutton")
	};
#undef Q

	/* copy the structure to the return value */
	*retpw = pw;
}

static void
build_tree(GtkWidget *ctree, GtkCTreeNode *parent, preferences_tree_item *item)
{
	GtkCTreeNode *node;
	gchar *text[] = { NULL, NULL };

	for (; item->name != NULL; item++)
	{
		text[0] = item->name;
		node = gtk_ctree_insert_node(GTK_CTREE(ctree), parent,
					     NULL, text, 0, NULL,
					     NULL, NULL, NULL,
					     FALSE, item->expand);

		gtk_ctree_node_set_row_data(GTK_CTREE(ctree), node,
					    (gpointer)(item->value));

		if (item->value == 0)
		{
			gtk_ctree_node_set_selectable(GTK_CTREE(ctree),
						      GTK_CTREE_NODE(node),
						      FALSE);
		}

		if (item->node_ref != NULL)
		{
			*(item->node_ref) = node;
		}
	}
}

static void
set_config_from_editable(GtkWidget *editable, gchar *config_name)
{
	gchar *value;

	/* get all the text into a new string */
	value = gtk_editable_get_chars (GTK_EDITABLE(editable), 0, -1);

	/* set it in the configuration space */
	gnome_config_set_string (config_name, value);

	/* free the allocated string */
	g_free (value);
}

static void
set_config_from_optionmenu(GtkWidget *optionmenu, gchar *config_name)
{
	GtkWidget *menu;
	GList *list;
	gpointer item;
	gint index;

	/* extract the selection */
	menu = GTK_OPTION_MENU(optionmenu)->menu;
	list = GTK_MENU_SHELL(menu)->children;
	item = gtk_menu_get_active (GTK_MENU(menu));
	index = g_list_index (list, item);
	
	/* set the configuration value */
	gnome_config_set_int (config_name, index);
}

static void
set_config_from_fontpicker(GtkWidget *fontpicker, gchar *config_name)
{
	gchar *font;

	/* get the name of the font */
	font = gnome_font_picker_get_font_name (GNOME_FONT_PICKER(fontpicker));

	/* set the configuration value */
	gnome_config_set_string (config_name, font);
}

static void
set_config_from_spin_button(GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* read the value as an integer */
	value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(spinbutton));

	/* set the configuration value */
	gnome_config_set_int (config_name, value);
}

static void
set_config_from_togglebutton(GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* read the value */
	value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(togglebutton));

	/* set the configuration value */
	gnome_config_set_bool (config_name, value);
}

static void
set_config_from_int_entry(GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the string */
	value_string = gtk_entry_get_text (GTK_ENTRY(entry));

	/* convert it into an integer */
	/* FIXME: atoi is a rubbish function to use, it
	 * doesn't detect any errors. We should fix this -
	 * perhaps by using spin buttons for these things? -- MattA */
	value = atoi(value_string);

	/* set the configuration value */
	gnome_config_set_int (config_name, value);
}

static void
set_config_from_list(GtkWidget *list, gchar *config_name)
{
	GList *children = g_list_copy (GTK_LIST(list)->children);
	guint size = g_list_length (children);
	char **strings = g_malloc (size*sizeof(char*));
	int x = 0;

	while (children) 
	{
		GtkObject       *list_item;
		gchar           *item_data_string;
        
		list_item=GTK_OBJECT(children->data);
		item_data_string=gtk_object_get_data(list_item, "string");
						     
		strings[x] = item_data_string;
			
		children=children->next;
		x++;
	}
	
	/* set the configuration value */
	gnome_config_set_vector (config_name, size, (const char**)strings);

	g_free (children);
	g_free (strings);
}

static void
set_config_from_color (GtkWidget *colorpicker, gchar *config_name)
{
	guint8 r, g, b, a;
	gchar *color_string;
	
	gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (colorpicker), &r, &g, &b, &a);
	color_string = g_strdup_printf ("#%02X%02X%02X", r, g, b);

	/* set the configuration value */
	gnome_config_set_string (config_name, color_string);
}

static void
set_editable_from_config(GtkWidget *editable, gchar *config_name)
{
	gchar *value;

	/* get the current value from the configuration space */
	value = gnome_config_get_string (config_name);

	/* set this string value in the widget */
	gtk_entry_set_text(GTK_ENTRY(editable), value);

	/* free the allocated string */
	g_free (value);
}

static void
set_optionmenu_from_config(GtkWidget *optionmenu, gchar *config_name)
{
	gint index;

	/* get the current value from the configuration space */
	index = gnome_config_get_int (config_name);

	/* set this option value in the widget */
	gtk_option_menu_set_history(GTK_OPTION_MENU(optionmenu), index);
}

static void
set_fontpicker_from_config(GtkWidget *fontpicker, gchar *config_name)
{
	gchar *font;

	/* get the current value from the configuration space */
	font = gnome_config_get_string (config_name);

	/* get the name of the font */
	gnome_font_picker_set_font_name (GNOME_FONT_PICKER(fontpicker), font);

	/* free the allocated string */
	g_free (font);
}

static void
set_spin_button_from_config(GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* get the current value from the configuration space */
	value = gnome_config_get_int (config_name);

	/* set this option value in the widget */
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinbutton), value);
}

static void
set_togglebutton_from_config(GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* get the current value from the configuration space */
	value = gnome_config_get_bool (config_name);

	/* set this option value in the widget */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (togglebutton), value);
}

static void
set_int_entry_from_config(GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the current value from the configuration space */
	value = gnome_config_get_int (config_name);

	/* convert to a string */
	value_string = g_strdup_printf("%d", value);

	/* set this string value in the widget */
	gtk_entry_set_text(GTK_ENTRY(entry), value_string);

	/* free the allocated string */
	g_free (value_string);
}

static void
set_list_from_config(GtkWidget *list, gchar *config_name)
{
	int size;
	gchar **strings;
	GList *items = NULL;
	int k;

	gnome_config_get_vector (config_name, &size, &strings);
	
	if (!strings) return;
	if (!strcmp(strings[0],"")) return;

	for (k=0; k<size ; k++)
	{	
		GtkWidget *list_item;
		char *label;
		
		label = strdup(strings[k]);	
		list_item = gtk_list_item_new_with_label(label);
		gtk_object_set_data(GTK_OBJECT(list_item), "string", label);
		items = g_list_append (items, list_item);
		gtk_widget_show(list_item);
	}

	if (items)
		gtk_list_append_items(GTK_LIST(list), items);
}

static void
set_color_from_config (GtkWidget *colorpicker, gchar *config_name)
{
	guint ir, ig, ib;
	guint8 r, g, b;
	gchar *color_string = gnome_config_get_string (config_name);
	
	sscanf (color_string, "#%2X%2X%2X", &ir, &ig, &ib);
	r = ir;
	g = ig;
	b = ib;
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (colorpicker), r, g, b, 0);

	/* set the configuration value */
	gnome_config_set_string (config_name, color_string);
}

void
preferences_set_use_provided_colors (gboolean value)
{
	int new_value = value ? 0 : 1;
	int old_value = gnome_config_get_int (CONF_APPEARANCE_USE_PROVIDED_COLORS);
	if (new_value != old_value) {
		gnome_config_set_int (CONF_APPEARANCE_USE_PROVIDED_COLORS, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); /* we need a less general way to do this... */
		set_settings_menu_global (); /* update the menu in other windows */
	}
}

void 
preferences_set_enable_javascript (gboolean value)
{
	int new_value = value ? 1 : 0;
	int old_value = gnome_config_get_int (CONF_ADVANCED_JAVASCRIPT_ENABLED);
	if (new_value != old_value) {
		gnome_config_set_int (CONF_ADVANCED_JAVASCRIPT_ENABLED, new_value);
		gnome_config_sync ();
		mozilla_prefs_set ();
		set_settings_menu_global (); 
	}
}

void 
preferences_set_enable_java (gboolean value)
{
 	int new_value = value ? 1 : 0;
 	int old_value = gnome_config_get_int (CONF_ADVANCED_JAVA_ENABLED);
  	if (new_value != old_value) {
		gnome_config_set_int (CONF_ADVANCED_JAVA_ENABLED, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void
preferences_set_use_provided_fonts (gboolean value)
{	
	int new_value = value ? 1 : 0;
	int old_value = gnome_config_get_int (CONF_APPEARANCE_USE_PROVIDED_FONTS);
	if (new_value != old_value) {
		gnome_config_set_int (CONF_APPEARANCE_USE_PROVIDED_FONTS, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void 
preferences_set_image_loading (ImageLoadingType value)
{
	int new_value = value;
	int old_value = gnome_config_get_int (CONF_ADVANCED_IMAGE_LOADING_TYPE);
	if (new_value != old_value) {
		gnome_config_set_int (CONF_ADVANCED_IMAGE_LOADING_TYPE, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void
preferences_set_handlers_enable_mime (gboolean value)
{
	gboolean new_value = value;
	gboolean old_value = gnome_config_get_bool (CONF_HANDLERS_MIME_ENABLE);
	if (new_value != old_value) {
		gnome_config_set_int (CONF_HANDLERS_MIME_ENABLE, new_value);
		gnome_config_sync ();
		set_settings_menu_global (); 
	}
}

static int
optionmenu_get_curr_index (GtkWidget *optionmenu)
{
	GtkWidget *menu, *active_item;
	gint item_index;
  
	menu = GTK_OPTION_MENU(optionmenu)->menu;
	active_item = gtk_menu_get_active(GTK_MENU(menu));
	item_index = g_list_index(GTK_MENU_SHELL(menu)->children, active_item);

	return(item_index);
}

void
mouse_wheel_controls_set_sensitive (GtkWidget *widget, wheel_mod_key key,
				    gboolean sensitive)
{
	GtkWidget *spinbutton = NULL, *checkbutton = NULL;
	
	switch (key)
	{
	case NOKEY:
		spinbutton = glade_lookup_widget(widget,
						 "wheel_nokey_step_spinbutton");
		checkbutton = glade_lookup_widget(widget,
						  "wheel_nokey_default_step_checkbutton");
		break;
	case ALT:
		spinbutton = glade_lookup_widget(widget,
						 "wheel_alt_step_spinbutton");
		checkbutton = glade_lookup_widget(widget,
						  "wheel_alt_default_step_checkbutton");
		break;
	case CTRL:
		spinbutton = glade_lookup_widget(widget,
						 "wheel_ctrl_step_spinbutton");
		checkbutton = glade_lookup_widget(widget,
						  "wheel_ctrl_default_step_checkbutton");
		break;
	case SHIFT:
		spinbutton = glade_lookup_widget(widget,
						 "wheel_shift_step_spinbutton");
		checkbutton = glade_lookup_widget(widget,
						  "wheel_shift_default_step_checkbutton");
		break;
	default:
		break;
	}

	if (spinbutton && checkbutton)
	{
		gtk_widget_set_sensitive(spinbutton, sensitive);
		gtk_widget_set_sensitive(checkbutton, sensitive);
	}
}
