/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <galeon.h>
#include <gtkmozembed_internal.h>

#include "regex.h"
#include "nsIPref.h"
#include "nsIDOMDocument.h"
#include "nsIGlobalHistory.h"
#include "nsIDocShellHistory.h"
#include "nsIDocShellTreeItem.h"
#include "nsIDocShellTreeOwner.h"
#include "nsIWebBrowser.h"
#include "nsIDiskDocument.h"
#include "nsIDocument.h"
#include "nsFileSpec.h"
#include "nsCOMPtr.h"
#include "nsIWebNavigation.h"
#include "nsISHistory.h"
#include "nsISHEntry.h"
#include "nsIURI.h"
#include "nsIURL.h"
#include "nsITextServicesDocument.h"
#include "nsIPresShell.h"
#include "nsTextServicesCID.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIDOMMouseEvent.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMHTMLElement.h"
#include "nsIDOMNamedNodeMap.h"
#include "nsIDOMEventReceiver.h"
#include "nsIDOMDragListener.h"
#include "nsIFocusController.h"
#include "nsIScriptGlobalObject.h"
#include "nsPIDOMWindow.h"
#include "nsIDOMWindowInternal.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIContentViewerFile.h"
#include "nsMPFileLocProvider.h"

/* local function prototypes */
static gboolean mozilla_find_setup(nsITextServicesDocument *aDoc, 
				   PRInt32 *outBlockOffset,
				   gboolean search_backwards, gboolean start);
static gchar *mozilla_get_attribute (nsIDOMNode *node, gchar *attribute);
static nsIDocShell *mozilla_get_primary_docshell (GtkMozEmbed *b);
static nsIDocShell *mozilla_get_event_docshell (GtkMozEmbed *b, 
						nsIDOMMouseEvent *event);
static gboolean mozilla_resolve_url (char *base, char *relative,
				     char **resolved);

static NS_DEFINE_CID(kCTextServicesDocumentCID, NS_TEXTSERVICESDOCUMENT_CID);
static NS_DEFINE_CID(kStandardURLCID, NS_STANDARDURL_CID);
static NS_DEFINE_CID(kPrefCID, NS_PREF_CID);

#define PREF_ID NS_PREF_CONTRACTID

class GaleonEmbed
{
public:
	GaleonEmbed();
	~GaleonEmbed();

	nsresult Init (GaleonBrowser *aBrowser);
	nsresult Destroy (void);
	nsresult AddListener (void);
	nsresult RemoveListener (void);
	nsresult AddEventListener (void);
	nsresult RemoveEventListener (void);

	nsresult SetZoom (float aTextZoom);

	nsresult SaveMainDocument (const char* filename);
	nsresult SaveDocument (nsIDOMDocument *aDomDocument, const char* filename);

	nsresult Print ();

	nsresult FindInMainDocument (const char *exp, PRBool matchcase, 
				     PRBool search_backwards, PRBool start);
	nsresult FindInDocument (nsIDOMDocument *aDomDocument, const char *exp, PRBool matchcase, 
				 PRBool search_backwards, PRBool start);

	nsresult GetDocumentUrl (nsIDOMDocument *aDOMDocument, char **url);

	nsresult ReloadDocument (nsIDOMDocument *aDOMDocument);

	nsresult GetSHInfo (PRInt32 *count, PRInt32 *index);
	nsresult GetSHTitleAtIndex (PRInt32 index, PRUnichar **title);

	nsresult GetKey (nsIDOMKeyEvent *keyEvent, PRUint32 *keycode, PRInt16 *modifier);
	nsresult GetMouseEventContext (nsIDOMMouseEvent *event, int *context, char **img, 
				       nsIDOMDocument **document, char **link);

	nsresult  CopyHistoryTo (GaleonEmbed *embed);

	nsresult SetViewSourceMode (PRInt32 mode);

	nsresult GoToHistoryIndex (PRInt16 index);

	nsCOMPtr<nsIWebBrowser>           mWebBrowser;
	nsCOMPtr<nsIDOMEventListener>        mEventListener;
	nsCOMPtr<nsIWebNavigation>        mChromeNav;
	GtkMozEmbed *mGtkMozEmbed;
	GaleonBrowser *browser;
	PRBool attachListener;
private:
	nsresult SetupFind (nsITextServicesDocument *aDoc, PRInt32 *outBlockOffset, 
			    PRBool search_backwards, PRBool start);
	nsresult GetDOMAttribute (nsIDOMNode *node, char *tag, char **attribute);
	nsresult GetDOMDocument (nsIDOMDocument **aDOMDocument);
	nsresult GetDocShell (nsIDocShell **aDocShell);
	nsresult GetSHistory (nsISHistory **aSHistory);
};

class GaleonEventListener : public nsIDOMDragListener
{
public:
	GaleonEventListener();
	virtual ~GaleonEventListener();

	void Init (GaleonEmbed *embed);
  
	NS_DECL_ISUPPORTS

	// nsIDOMEventListener
	
	NS_IMETHOD HandleEvent(nsIDOMEvent* aEvent);

	// nsIDOMDragListener

	NS_IMETHOD DragEnter(nsIDOMEvent* aMouseEvent);
	NS_IMETHOD DragExit(nsIDOMEvent* aMouseEvent);
	NS_IMETHOD DragDrop(nsIDOMEvent* aMouseEvent);
	NS_IMETHOD DragGesture(nsIDOMEvent* aMouseEvent);
	NS_IMETHOD DragOver(nsIDOMEvent* aMouseEvent);

private:
	GaleonEmbed *embed;
};

GaleonEventListener::GaleonEventListener(void)
{
  NS_INIT_REFCNT();
  embed = NULL;
}

GaleonEventListener::~GaleonEventListener(void)
{
}


NS_IMPL_ADDREF(GaleonEventListener)
NS_IMPL_RELEASE(GaleonEventListener)
NS_INTERFACE_MAP_BEGIN(GaleonEventListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsIDOMEventListener, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY(nsIDOMDragListener)
NS_INTERFACE_MAP_END

void
GaleonEventListener::Init(GaleonEmbed *galeon_embed)
{
	embed = galeon_embed;
}

NS_IMETHODIMP
GaleonEventListener::HandleEvent(nsIDOMEvent* aEvent)
{
	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragEnter(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragExit(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragDrop(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragGesture(nsIDOMEvent* aMouseEvent)
{
	gchar *link;
	gpointer ds;
	gchar *img;
	int type;
	gchar *old_dragging_link;

	type = mozilla_get_event_context(embed->browser, aMouseEvent, &img, &ds, &link);

	if (type & CONTEXT_LINK)
	{
		if ( ! link_drag_types_tl)
			link_drag_types_tl = gtk_target_list_new 
				(link_drag_types, link_drag_types_num_items);

		old_dragging_link = (gchar*)gtk_object_get_data (GTK_OBJECT (embed->mGtkMozEmbed), "dragging_link");
		/* FIXME better to use gtkmozembed widget to drag ? maybe the freeze is related ? */
		if (old_dragging_link) g_free (old_dragging_link);
		gtk_object_set_data (GTK_OBJECT(embed->mGtkMozEmbed), "dragging_link", g_strdup (link));
		gtk_drag_begin (GTK_WIDGET(embed->mGtkMozEmbed), link_drag_types_tl, 
				(GdkDragAction)(GDK_ACTION_COPY | GDK_ACTION_LINK), 1, NULL);
	}

	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragOver(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}

GaleonEmbed::GaleonEmbed ()
{
}

GaleonEmbed::~GaleonEmbed ()
{
}

nsresult GaleonEmbed::Init (GaleonBrowser *aBrowser)
{
	nsresult result;

	mGtkMozEmbed = aBrowser->mozEmbed;
	browser = aBrowser;

	gtk_moz_embed_get_nsIWebBrowser (mGtkMozEmbed, getter_AddRefs(mWebBrowser));
	if (!mWebBrowser) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShellHistory> dsHistory = do_QueryInterface (DocShell);
	if (!dsHistory) return NS_ERROR_FAILURE;

	NS_WITH_SERVICE(nsIGlobalHistory, history, NS_GLOBALHISTORY_CONTRACTID, &result);  
	if (NS_FAILED(result) || !history) return NS_ERROR_FAILURE;
	dsHistory->SetGlobalHistory(history);
	
  	mChromeNav = do_QueryInterface(mWebBrowser);
	if (!mChromeNav) return NS_ERROR_FAILURE;

	if (attachListener)
	return AddListener ();
}

nsresult GaleonEmbed::Print ()
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> ContentViewer;
	result = DocShell->GetContentViewer(getter_AddRefs(ContentViewer));
	if (NS_FAILED(result) || !ContentViewer) return NS_ERROR_FAILURE;
                
	nsCOMPtr<nsIContentViewerFile> ContentViewerFile = do_QueryInterface(ContentViewer);
	if (!ContentViewerFile) return NS_ERROR_FAILURE;

	return ContentViewerFile->Print(PR_TRUE, nsnull);
}

nsresult GaleonEmbed::GetSHistory (nsISHistory **aSHistory)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell, &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = ContentNav->GetSessionHistory (getter_AddRefs (SessionHistory));
	if (!SessionHistory) return NS_ERROR_FAILURE;

	*aSHistory = SessionHistory.get();
	NS_IF_ADDREF (*aSHistory);
}

nsresult GaleonEmbed::GetDocShell (nsIDocShell **aDocShell)
{
	nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
	browserAsItem = do_QueryInterface(mWebBrowser);
	if (!browserAsItem) return NS_ERROR_FAILURE;

	// get the owner for that item
	nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
	browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
	if (!treeOwner) return NS_ERROR_FAILURE;

	// get the primary content shell as an item
	nsCOMPtr<nsIDocShellTreeItem> contentItem;
	treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
	if (!contentItem) return NS_ERROR_FAILURE;

	// QI that back to a docshell
	nsCOMPtr<nsIDocShell> DocShell;
	DocShell = do_QueryInterface(contentItem);
	if (!DocShell) return NS_ERROR_FAILURE;

	*aDocShell = DocShell.get();

	NS_IF_ADDREF(*aDocShell);
}

nsresult GaleonEmbed::Destroy ()
{
	if (attachListener)
		return RemoveListener ();
      	mWebBrowser = nsnull;
	mChromeNav = nsnull;
}

nsresult GaleonEmbed::AddListener ()
{
	nsresult rv = NS_OK;

	nsCOMPtr <nsIDOMDocument> domDoc;
	mChromeNav->GetDocument(getter_AddRefs(domDoc));
	if (!domDoc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventReceiver> eventReceiver;
	eventReceiver = do_QueryInterface(domDoc);
	if (!eventReceiver) return NS_ERROR_FAILURE;

	GaleonEventListener *newListener = new GaleonEventListener();
	NS_ADDREF(newListener);
	if (!newListener) return NS_ERROR_FAILURE;

	newListener->Init (this);
	nsCOMPtr<nsIDOMEventListener> mEventListener = do_QueryInterface(NS_STATIC_CAST(nsISupports *,
								       NS_STATIC_CAST(nsIDOMDragListener *,
									 newListener)));
	if (!mEventListener) return NS_ERROR_FAILURE;
	NS_RELEASE(newListener);

	rv = eventReceiver->AddEventListenerByIID(mEventListener,
						  NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
}

nsresult GaleonEmbed::RemoveListener ()
{
	nsresult rv = NS_OK;

	if (!mEventListener) return NS_ERROR_FAILURE;

	nsCOMPtr <nsIDOMDocument> domDoc;
	mChromeNav->GetDocument(getter_AddRefs(domDoc));
	if (!domDoc) return NS_ERROR_FAILURE;;

	nsCOMPtr<nsIDOMEventReceiver> eventReceiver;
	eventReceiver = do_QueryInterface(domDoc);
	if (!eventReceiver) return NS_ERROR_FAILURE;

	rv = eventReceiver->RemoveEventListenerByIID(mEventListener,
						     NS_GET_IID(nsIDOMDragListener));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	mEventListener = NULL;
}

nsresult GaleonEmbed::GoToHistoryIndex (PRInt16 index)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell, &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	return  ContentNav->GotoIndex (index);
}

nsresult GaleonEmbed::SetViewSourceMode (PRInt32 mode)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	return DocShell->SetViewMode (mode);
}

nsresult GaleonEmbed::SetZoom (float aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer, &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->SetTextZoom (aZoom);
}

nsresult GaleonEmbed::GetDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;
	
	return contentViewer->GetDOMDocument (getter_AddRefs(aDOMDocument));
}

nsresult GaleonEmbed::SaveMainDocument (const char* filename)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument)
	{
		return NS_ERROR_FAILURE;
	}

	return SaveDocument (aDOMDocument, filename);
}

nsresult GaleonEmbed::SaveDocument (nsIDOMDocument *aDOMDocument, const char* filename)
{
	nsresult result;


#ifdef CVS_SOURCE	
	/* write the document on the disk */
/* FIXME it does compile but it does not work, I've not again been able to figure out why.*/

	nsString mime_type;
	nsString charset;
	mime_type.AssignWithConversion("text/html");
	charset.AssignWithConversion("");

	nsCOMPtr<nsILocalFile> file;
	file = do_CreateInstance(NS_LOCAL_FILE_CONTRACTID);
	result = file->InitWithPath(filename);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDiskDocument> diskDoc = do_QueryInterface(aDOMDocument);
	result = diskDoc->SaveFile(file, PR_TRUE, PR_TRUE, mime_type.ToNewUnicode(), charset.ToNewUnicode(), 0, 80);
	if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
#else
	/* write the document on the disk */
	nsString mime_type;
	nsString charset;

	/* FIXME should these be freed ? */
	mime_type.AssignWithConversion("text/html");
	charset.AssignWithConversion("");

	nsCOMPtr<nsIDiskDocument> diskDoc = do_QueryInterface(aDOMDocument);
	if (!diskDoc) return NS_ERROR_FAILURE;
        
	nsFileSpec fileSpec(filename, PR_FALSE);
	result = diskDoc->SaveFile(&fileSpec, PR_TRUE, PR_TRUE, mime_type, charset, 0);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;
#endif
}

nsresult GaleonEmbed::GetSHInfo (PRInt32 *count, PRInt32 *index)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	SessionHistory->GetCount (count);
	SessionHistory->GetIndex (index);	
}

nsresult GaleonEmbed::GetSHTitleAtIndex (PRInt32 index, PRUnichar **title)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE, getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	result = he->GetTitle (title);
	if (!NS_SUCCEEDED(result) || (!title)) return NS_ERROR_FAILURE;
}

nsresult GaleonEmbed::FindInMainDocument (const char *exp, PRBool matchcase, 
			     PRBool search_backwards, PRBool start)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> PresShell;
	result = DocShell->GetPresShell (getter_AddRefs(PresShell));
	if (NS_FAILED(result) || !PresShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> Document;
	result = PresShell->GetDocument (getter_AddRefs(Document));
	if (NS_FAILED(result) || !Document) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIScriptGlobalObject> ourGlobal;
	result = Document->GetScriptGlobalObject(getter_AddRefs(ourGlobal));
	if (NS_FAILED(result) || !ourGlobal) return NS_ERROR_FAILURE;

	nsCOMPtr<nsPIDOMWindow> ourWindow = do_QueryInterface(ourGlobal);
	if (!ourWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIFocusController> focusController;
	result = ourWindow->GetRootFocusController(getter_AddRefs(focusController));
	if (NS_FAILED(result) || !focusController) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindowInternal> Window;
	result = focusController->GetFocusedWindow(getter_AddRefs(Window));
	if (NS_FAILED(result) || !Window) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMDocument> domDoc;
	Window->GetDocument (getter_AddRefs(domDoc));
	if (NS_FAILED(result) || !domDoc) return NS_ERROR_FAILURE;

	return FindInDocument (domDoc, exp,matchcase, search_backwards, start);
}

nsresult GaleonEmbed::FindInDocument (nsIDOMDocument *aDOMDocument, const char *exp,  PRBool matchcase, 
				      PRBool search_backwards, PRBool start)
{
	nsresult result;

	/* create nsIServiceDocument */

	nsCOMPtr<nsITextServicesDocument> textDoc;
	result = nsComponentManager::CreateInstance(kCTextServicesDocumentCID,nsnull,
						    NS_GET_IID(nsITextServicesDocument),
						    getter_AddRefs(textDoc));
	if (NS_FAILED(result) || !textDoc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> presShell = getter_AddRefs(doc->GetShellAt(0));
	if(!presShell) return NS_ERROR_FAILURE;

	/* init nsIServiceDocument */

	result = textDoc->InitWithDocument(aDOMDocument, presShell);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;
	
	PRInt32 outBlockOffset = -1;

	SetupFind(textDoc,&outBlockOffset,search_backwards,start);

	PRBool atExtremum = PR_FALSE;     

	regex_t preg;
	regmatch_t match[1];
	int res;
	int eflags = 0;
	
	if (!matchcase) eflags = REG_ICASE;

	regcomp(&preg,exp,eflags);

	while ( NS_SUCCEEDED(textDoc->IsDone(&atExtremum)) && !atExtremum )
	{
		nsString str;
		char *cstr, *tempcstr;

		result = textDoc->GetCurrentTextBlock(&str);
		if (NS_FAILED(result)) return result;
		cstr = str.ToNewCString();
		tempcstr = cstr;

		match[0].rm_so = -1;
		match[0].rm_eo = -1;

		if (!search_backwards)
		{
			PRInt32 offset = 0;
			if (outBlockOffset!=-1) offset = outBlockOffset;
			res = regexec(&preg,tempcstr+offset,1,match,0);

			if (res==0) {

				textDoc->SetSelection(match[0].rm_so+offset,match[0].rm_eo-match[0].rm_so);
				textDoc->ScrollSelectionIntoView();
				regfree(&preg);
				return NS_OK;
			}

		}
		else
		{
			gboolean not_found = TRUE;
			PRInt32 eo = 0;
			PRInt32 old_rm_eo = 0;
			if (outBlockOffset!=-1) cstr[outBlockOffset] = 0;
			while ((res = regexec(&preg,tempcstr,1,match,0))==0)
			{
				eo += old_rm_eo;
				old_rm_eo = match[0].rm_eo;
				tempcstr += match[0].rm_eo;
				not_found = FALSE;
			}
			
			if (!not_found) {

				textDoc->SetSelection(match[0].rm_so+eo,match[0].rm_eo-match[0].rm_so);
				textDoc->ScrollSelectionIntoView();
				regfree(&preg);
				return NS_OK;
			}
		}

		g_free(cstr);


		if (!search_backwards)
		{
			result = textDoc->NextBlock(); 
		}
		else
		{
			result = textDoc->PrevBlock();
		}

		if (NS_FAILED(result)) return NS_ERROR_FAILURE;

		outBlockOffset = -1;
	}

	regfree(&preg);
	return NS_ERROR_FAILURE;
}

nsresult GaleonEmbed::ReloadDocument (nsIDOMDocument *aDOMDocument)
{
	nsresult result;
	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> presShell = getter_AddRefs(doc->GetShellAt(0));
	if(!presShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresContext> presContext;
	presShell->GetPresContext(getter_AddRefs(presContext));
	if(!presContext) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISupports> pcContainer;
	presContext->GetContainer(getter_AddRefs(pcContainer));
	if(!pcContainer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> docshell(do_QueryInterface(pcContainer));
	if(!docshell) return NS_ERROR_FAILURE;


	nsCOMPtr<nsIWebNavigation> wn = do_QueryInterface (docshell, &result);
	if (!wn || !NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	result = wn->Reload (nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE | 
			     nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;
}

nsresult GaleonEmbed::SetupFind (nsITextServicesDocument *aDoc, PRInt32 *outBlockOffset, PRBool search_backwards, PRBool start)
{
	nsresult rv;
	nsITextServicesDocument::TSDBlockSelectionStatus blockStatus;
	PRInt32 selOffset;
	PRInt32 selLength;

	rv = aDoc->LastSelectedBlock(&blockStatus, &selOffset, &selLength);

	if (!search_backwards)
	{
		if (NS_SUCCEEDED(rv) && (!start))
		{
			*outBlockOffset = selOffset+selLength;
		}
		else 
		{
			rv = aDoc->FirstBlock();
		}
	}
	else
	{
		if (NS_SUCCEEDED(rv) && (!start))
		{
			*outBlockOffset = selOffset;
		}
		else 
		{
			rv = aDoc->LastBlock();
		}
	
	}
}

nsresult GaleonEmbed::GetDocumentUrl (nsIDOMDocument *aDOMDocument, char **url)
{
	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsIURI *uri = doc->GetDocumentURL();
	uri->GetSpec (url);
}

nsresult GaleonEmbed::GetKey (nsIDOMKeyEvent *keyEvent, PRUint32 *keycode, PRInt16 *modifier)
{
	PRBool mod_key;
	nsresult res;

	/* if we are in a text area do not process keys */
	nsCOMPtr<nsIDOMEventTarget>  targetNode;
	res = keyEvent->GetTarget(getter_AddRefs(targetNode));
	if (targetNode)
	{
		nsCOMPtr<nsIDOMNode> node = do_QueryInterface(targetNode);
		if (node)
		{
	
			nsCOMPtr<nsIDOMHTMLElement> element;
	
			element = do_QueryInterface(node);
			if (element)
			{
				nsAutoString tag;
				element->GetTagName(tag);
				if (!tag.EqualsWithConversion("html", PR_TRUE))
					return NS_ERROR_FAILURE;
			}
		}
	}

	*modifier = 0;

	keyEvent->GetCharCode(keycode);

	if (*keycode==0) 
	{
		keyEvent->GetKeyCode(keycode);
		*modifier |= KEY_CODE;
	}

	keyEvent->GetAltKey(&mod_key);
	if (mod_key) *modifier |= ALT_KEY;

	keyEvent->GetShiftKey(&mod_key);
	if (mod_key) *modifier |= SHIFT_KEY;

	keyEvent->GetMetaKey(&mod_key);
	if (mod_key) *modifier |= META_KEY;
	
	keyEvent->GetCtrlKey(&mod_key);
	if (mod_key) *modifier |= CTRL_KEY;
}

nsresult GaleonEmbed::GetDOMAttribute (nsIDOMNode *node, char *tag, char **attribute)
{
	nsresult result;
	
	nsCOMPtr<nsIDOMNamedNodeMap> attributes;
	result = node->GetAttributes(getter_AddRefs (attributes));
	if (!NS_SUCCEEDED (result) || !attributes) return NS_ERROR_FAILURE;

	nsAutoString attr; 

	attr.AssignWithConversion (tag);
                                        
	nsCOMPtr<nsIDOMNode> attrNode;
	result = attributes->GetNamedItem (attr, getter_AddRefs (attrNode));
	if (!NS_SUCCEEDED (result) || !attrNode) return NS_ERROR_FAILURE;

	nsAutoString nodeValue;
			
	result = attrNode->GetNodeValue (nodeValue);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;
                                                
	char *cstr = nodeValue.ToNewCString();
	*attribute = g_strdup (cstr);
			
	nsMemory::Free (cstr);
}

nsresult GaleonEmbed::GetMouseEventContext (nsIDOMMouseEvent *event, int *context, char **img, 
					    nsIDOMDocument **document, char **link)
{
	nsresult result;
	*img = *link = NULL;
	nsIDOMMouseEvent *aMouseEvent = 	(nsIDOMMouseEvent*)event;

	nsCOMPtr<nsIDOMEventTarget> EventTarget;
	result = aMouseEvent->GetTarget(getter_AddRefs(EventTarget));
	if (NS_FAILED(result) || !EventTarget) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(EventTarget);
	if (!node) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMDocument> domDoc;
	result = node->GetOwnerDocument(getter_AddRefs(domDoc));
	if (!NS_SUCCEEDED (result) || !domDoc) return NS_ERROR_FAILURE;

	*document = domDoc;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
	if(!doc) return NS_ERROR_FAILURE;

	nsIURI *baseURI;
	result = doc->GetBaseURL(baseURI);
	if (NS_FAILED(result) || !baseURI) return NS_ERROR_FAILURE;
   
	nsString mime;  
	doc->GetContentType(mime);  
	if (strcmp (mime.ToNewCString(),"text/xul") == 0) return NS_ERROR_FAILURE;

	*context = CONTEXT_NONE;
	nsCOMPtr<nsIDOMHTMLElement> element;
	do {
		PRUint16 type;
		node->GetNodeType(&type);

		element = do_QueryInterface(node);
		if (element)
		{
			nsAutoString tag;
			element->GetTagName(tag);

			if (tag.EqualsWithConversion("input", PR_TRUE))
			{
				*context |= CONTEXT_INPUT;
			}
			else if (tag.EqualsWithConversion("img", PR_TRUE))
			{
				*context |= CONTEXT_IMAGE;

				char *src;
				result = GetDOMAttribute (node, g_strdup("src"), &src);
				if (NS_FAILED(result) || !src)
					return NS_ERROR_FAILURE;
			
				result = baseURI->Resolve(src, img);
				if (NS_FAILED(result) || !img)
				return NS_ERROR_FAILURE;

			}
			else
			{
				*context |= CONTEXT_OTHER;
			}

			nsCOMPtr<nsIDOMNamedNodeMap> attributes;
			node->GetAttributes(getter_AddRefs(attributes));
			if (attributes)
			{
				nsCOMPtr<nsIDOMNode> hrefNode;
				nsAutoString href; 
				href.AssignWithConversion("href");
				attributes->GetNamedItem(href, getter_AddRefs(hrefNode));
				if (hrefNode)
				{
					*context |= CONTEXT_LINK;

					char *href;
					result = GetDOMAttribute (node, g_strdup("href"), &href);
					if (NS_FAILED(result) || !href)
						return NS_ERROR_FAILURE;
				
					result = baseURI->Resolve(href, link);
					if (NS_FAILED(result) || !link)
						return NS_ERROR_FAILURE;
					break;
				}
			}

		}
			nsCOMPtr<nsIDOMNode> parentNode;
			node->GetParentNode(getter_AddRefs(parentNode));
		
			if (!parentNode)
			{
				node = nsnull;
				*context |= CONTEXT_DOCUMENT;
				break;
			}
			node = parentNode;
	} while (node);

}

nsresult  GaleonEmbed::CopyHistoryTo (GaleonEmbed *embed)
{
	nsresult result;
	int count,index;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_src = do_QueryInterface (DocShell, &result);
	if (!wn_src) return FALSE;
	
	nsCOMPtr<nsISHistory> h_src;
	result = wn_src->GetSessionHistory (getter_AddRefs (h_src));
	if (!NS_SUCCEEDED (result) || (!h_src)) return FALSE;

	nsCOMPtr<nsIDocShell> destDocShell;
	result = embed->GetDocShell (getter_AddRefs(destDocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_dest = do_QueryInterface (destDocShell, &result);
	if (!wn_dest) return FALSE;
	
	nsCOMPtr<nsISHistory> h_dest;
	result = wn_dest->GetSessionHistory (getter_AddRefs (h_dest));
	if (!NS_SUCCEEDED (result) || (!h_dest)) return FALSE;

	h_src->GetCount (&count);
	h_src->GetIndex (&index);

	if (count) {
		nsCOMPtr<nsISHEntry> he;

		for (PRInt32 i = 0; i < count; i++) {

			result = h_src->GetEntryAtIndex (i, PR_FALSE, getter_AddRefs (he));
			if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

			result = h_dest->AddEntry(he, PR_TRUE);
			if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;
		}

		result = wn_dest->GotoIndex(index);
		if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
	}
}

extern "C" gpointer 
mozilla_embed_init (GaleonBrowser *browser, gboolean attach_listener)
{
	GaleonEmbed *embed = new GaleonEmbed ();
	embed->attachListener = attach_listener;
	embed->Init (browser);
	return embed;
}

extern "C" gboolean
mozilla_embed_destroy (GaleonBrowser *browser)
{
	nsresult result;

	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;
	result = embed->Destroy();
	delete embed;
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}


extern "C" gboolean
mozilla_preference_set(const char *preference_name, const char *new_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  g_return_val_if_fail (new_value != NULL, FALSE);
  nsCOMPtr<nsIPref> pref = do_CreateInstance(PREF_ID);

  if (pref)
    {
      nsresult rv = pref->SetCharPref (preference_name, new_value);            
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

extern "C" gboolean
mozilla_preference_set_boolean (const char        *preference_name,
				gboolean        new_boolean_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  
  nsCOMPtr<nsIPref> pref = do_CreateInstance(PREF_ID);
  
  if (pref)
    {
      nsresult rv = pref->SetBoolPref (preference_name,
				       new_boolean_value ? PR_TRUE : PR_FALSE);
      
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

extern "C" gboolean
mozilla_preference_set_int (const char        *preference_name,
				int        new_int_value)
{
  g_return_val_if_fail (preference_name != NULL, FALSE);
  
  nsCOMPtr<nsIPref> pref = do_CreateInstance(PREF_ID);
  
  if (pref)
    {
      nsresult rv = pref->SetIntPref (preference_name,
				       new_int_value);
      
      return NS_SUCCEEDED (rv) ? TRUE : FALSE;
    }
  
  return FALSE;
}

extern "C" gboolean
mozilla_save (GaleonBrowser *browser, const char *fname, gpointer target)
{
	nsresult result;
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	if (target)
	{
		result = embed->SaveDocument ((nsIDOMDocument*)target, fname);
	}
	else
	{
		result = embed->SaveMainDocument (fname);
	}

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * Gets the session history and current position of a GtkMozEmbed. On return, *titles 
 * will be an array of char * wich must be freed.
 */
extern "C" gboolean
mozilla_session_history (GaleonBrowser *browser, char **titles[], int *count, int *index)
{
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;
	PRUnichar *title;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	embed->GetSHInfo (count, index);

	char **t = g_new(char *, *count);
	nsresult result;
	for (PRInt32 i = 0; i < *count; i++) {

		result = embed->GetSHTitleAtIndex(i, &title);

		if (NS_FAILED(result))
		{
			return NS_OK;
		}

		// The title is in 16-bit unicode, we should make it 8bit (UTF)
		// FIXME: Do this properly
		char *s;
		int j, length;
	
		/* compute the string length */
		for (length = 0; title[length] != 0; length++);
		
		/* allocate an 8-bit string for handling it */
		s = g_new (char, length + 1);

		/* copy 16-bit -> 8-bit, substituting 0's */
		for (j = 0; j < length; j++)
		{
			s[j] = title[j] ? title[j] : ' ';
		}

		/* zero terminate the string */
		s[length] = 0;
		
		t[i] = s;
	}
	*titles = t;

	return TRUE;
}

/**
 * Goes to the SHEntry at the given index
 */
extern "C" gboolean
mozilla_session_history_go (GaleonBrowser *browser, int index)
{
	nsresult result;
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	result = embed->GoToHistoryIndex (index);
	
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_find: find a word in the document
 */
extern "C" gboolean
mozilla_find (GaleonBrowser *browser,const char *exp,gboolean matchcase,gboolean search_backwards,gboolean start, gpointer target)
{
	nsresult result;
	GaleonEmbed *embed;

	g_return_val_if_fail (browser != NULL, FALSE);
	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	embed = (GaleonEmbed *)browser->embed;

	if (target)
	{
		result = embed->FindInDocument ((nsIDOMDocument*)target, exp, matchcase, search_backwards, start);
	}
	else
	{
		result = embed->FindInMainDocument (exp, matchcase, search_backwards, start);
	}

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * Sets the zoom factor of a browser
 */
extern "C" gboolean
mozilla_set_zoom (GaleonBrowser *browser, float f)
{
	GaleonEmbed *embed;

	g_return_val_if_fail (browser != NULL, FALSE);
	g_return_val_if_fail (browser->embed != NULL, FALSE); 
	embed = (GaleonEmbed *)browser->embed;

	nsresult result = embed->SetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_print: print a document
 */
extern "C" gboolean
mozilla_print (GaleonBrowser *browser)
{
	nsresult result;
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	result = embed->Print();
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * Turn a browser source view mode
 */
extern "C" gboolean 
mozilla_view_source (GaleonBrowser *browser, gboolean view_source)
{
	nsresult result;
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	result = embed->SetViewSourceMode (view_source ? nsIDocShell::viewSource : nsIDocShell::viewNormal);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_mouse_button: get pressed mouse button from DOM mouse event
 */
extern "C" gint 
mozilla_get_mouse_button (gpointer event)
{
	nsresult result;
	PRUint16 button;
	nsIDOMMouseEvent *mouseevent = ((nsIDOMMouseEvent *)event);

	result = mouseevent->GetButton(&button);

	return NS_SUCCEEDED (result) ? button : 0;
}

/**
 * mozilla_get_key_pressed: get pressed key button from a DOM key event
 */
extern "C" glong 
mozilla_get_key_pressed (GaleonBrowser *browser, gpointer event, int *modifier)
{
	PRUint32 key;
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 
	g_return_val_if_fail (event != NULL, FALSE);

	embed->GetKey ((nsIDOMKeyEvent *)event, &key, (PRInt16*)modifier);
	return key;
}

/**
 * mozilla_get_context_menu_type: get in what context the mouse event is occurred
 */
extern "C" gint
mozilla_get_event_context (GaleonBrowser *browser, gpointer event, gchar **img, gpointer *target, gchar **link)
{
	nsresult result;
	int context;
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 
	g_return_val_if_fail (event != NULL, FALSE);

	result = embed->GetMouseEventContext ((nsIDOMMouseEvent*)event, &context, img, (nsIDOMDocument**)target, link);

	if (NS_FAILED(result))
	{
		return CONTEXT_NONE;
	}
	
	return context;
}

extern "C" char* 
mozilla_get_eventtarget_url (GaleonBrowser *browser, gpointer target)
{
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;
	char *url;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	embed->GetDocumentUrl ((nsIDOMDocument *)target, &url);
	return url;
}

extern "C" gboolean
mozilla_reload_docshell (GaleonBrowser *browser, gpointer target)
{
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;

	g_return_val_if_fail (browser->embed != NULL, FALSE); 

	nsresult result = embed->ReloadDocument ((nsIDOMDocument *)target);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" void
mozilla_prefs_init (char *dir)
{
/*
#ifndef USE_PROFILE_API
	nsresult rv;	

	nsILocalFile* file = nsnull;
 	nsComponentManager::CreateInstance(NS_LOCAL_FILE_CONTRACTID, nsnull, NS_GET_IID(nsILocalFile), (void**)&file);
	if (!file) return;
	file->InitWithPath(dir);

	nsMPFileLocProvider *locationProvider = new nsMPFileLocProvider;
	if (!locationProvider) return;
	rv = locationProvider->Initialize(file, "galeon");   
	if (!NS_SUCCEEDED(rv)) return;

	NS_WITH_SERVICE(nsIPref, prefs, kPrefCID, &rv);
	if (!prefs) return;

	prefs->ResetPrefs();
	prefs->ReadUserPrefs();
#endif
*/
}

extern "C" gboolean
mozilla_copy_session_history (GaleonBrowser *browser, GaleonBrowser *dest)
{
	g_return_val_if_fail (browser != NULL, FALSE);
	g_return_val_if_fail (dest != NULL, FALSE);
	GaleonEmbed *embed = (GaleonEmbed *)browser->embed;
	GaleonEmbed *dest_embed = (GaleonEmbed *)dest->embed;
	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (dest_embed != NULL, FALSE);
	nsresult result = embed->CopyHistoryTo (dest_embed);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}
