/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* local function prototypes */
static void check_first_time(void);
static void first_time_bookmarks_cb(gint reply, gpointer data);
static void first_time_prefs_cb(gint reply, gpointer data);
#ifdef CORBA_INTERFACE
static void activate_corba_server(CORBA_Environment *ev,
				CORBA_ORB *orb,
				CORBA_Object *galeon_servant,
				PortableServer_POA *root_poa,
				PortableServer_POAManager *pm);

Galeon_Browser
impl_Galeon_Browser__create (PortableServer_POA poa, CORBA_Environment *ev);
#endif

/* global variables */
gboolean tabbed_mode = FALSE;

/**
 * main: main entry point
 */
int
main (int argc, char *argv[])
{
	int i, bc = 0;
	CORBA_Environment ev;
	CORBA_ORB orb;
	GnorbaInitFlags gnorba_flags = 0;
	int existing = FALSE;  /* load in existing win? */
	gchar *bookmark_url = NULL; /* the temp bookmark to add */
	GaleonBrowser *browser;

	struct poptOption popt_options[] =
	{
		{ "existing", 'x', POPT_ARG_NONE, &existing, 0,
		  N_("Attempt to load URL in existing Galeon window"),
		  NULL },
		{ "add-temp-bookmark", 't', POPT_ARG_STRING, &bookmark_url, 0,
		  N_("Add a temp bookmark (don't open any window)"),
		  N_("URL")},
		/* terminator, must be last */
		{ NULL, 0, 0, NULL, 0, NULL, NULL }
	};

	char **urls = NULL;
	poptContext ctx;
	char fullurl[PATH_MAX];
	

#ifdef CORBA_INTERFACE
	/* CORBA server variables */
//	GnomeClient *client;
	CORBA_Object galeon_servant;
	PortableServer_POA root_poa;
	PortableServer_POAManager pm;
	Galeon_Browser galeon_browser_client;
	gnorba_flags |= GNORBA_INIT_SERVER_FUNC;
#endif

	 /* Make stdout line buffered - we only use it for debug info */
	setvbuf (stdout, NULL, _IOLBF, 0);

	/* initialise localisation */
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);

	/* initialise GNOME */
	CORBA_exception_init (&ev);

	orb = gnome_CORBA_init_with_popt_table ( "galeon", VERSION,
						&argc, argv,
						popt_options, 0, &ctx,
						gnorba_flags,
						&ev );
	
	/* load arguments that aren't regular options (urls to load) */
	if (ctx)
		urls = (char **) poptGetArgs(ctx);

	/* set config */
	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);

#ifdef CORBA_INTERFACE
	/* attempt to connect to existing CORBA server */
	galeon_browser_client = goad_server_activate_with_repo_id(NULL,
						"IDL:galeon/browser:1.0",
						GOAD_ACTIVATE_EXISTING_ONLY,
						NULL);

	if (galeon_browser_client)  /* galeon CORBA server already running? */
	{
		g_print("Galeon already running, using existing process\n");
		/* FIXME: allow command line option to override this behavior */

		if (bookmark_url) {
			Galeon_Browser_addTempBookmark 
				(galeon_browser_client, bookmark_url, bookmark_url, &ev);
			exit (0);
		}
		
		if (urls)
			for (i = 0; urls[i]; i++)
			{
				fullurl[0] = '\0';

				if (g_str_is_url(urls[i]))
					strncpy(fullurl, urls[i], PATH_MAX);
				else
					realpath(urls[i], fullurl);
			
				if (fullurl[0])
				{
					Galeon_Browser_loadurl(galeon_browser_client,
							fullurl,
							!existing,
							&ev);
					bc++;
					existing = FALSE;
				}
			}
				
		if (bc == 0) {
			Galeon_Browser_loadurl(galeon_browser_client,
						"",
						!existing,
						&ev);
			bc++;
		}
		exit (0); /* more graceful exit needed? */
	}
	
//	client = gnome_master_client();  /* do we need this? */
	
	activate_corba_server(&ev, &orb, &galeon_servant, &root_poa, &pm);
	
#endif
	
	/* initialize gmodule */
	mod_self = g_module_open ( NULL, 0 );
		
	/* initialise GLADE */
	glade_gnome_init();

	/* Add a temp bookmark and exit if needed */
	if (bookmark_url) {
		temp_bookmarks_init ();
		add_temp_bookmark (SITE, NULL, bookmark_url, NULL);
		temp_bookmarks_save ();
		exit (0);
	}
	
	/* check if this is the first time this user has run Galeon */
	check_first_time();

	/* load Galeon setup */
	preferences_load();
	history_load();
	bookmarks_load();
	initialise_colours();

	/* set the profile dir for gtkmozembed */
#ifdef USE_PROFILE_API
	gtk_moz_embed_set_profile_path(g_concat_dir_and_file 
					       (g_get_home_dir (), ".galeon/mozilla/"), "galeon"); 	
#endif
	/* resume a crashed session, if any */
	if (session_autoresume())
	{
		bc++;
	}

	/* if this flag doesn't get cleared, then Galeon crashed */
	gnome_config_set_int("/galeon/General/crashed", TRUE);
	gnome_config_sync();

	
	if (urls)
		/* create a browser for each argument popt did not filter */
		for (i = 0; urls[i]; i++)
		{
			fullurl[0] = '\0';
			if (g_str_is_url(urls[i]))
				strncpy(fullurl, urls[i], PATH_MAX);
			else
				realpath(urls[i], fullurl);
			
			if (fullurl[0])
			{
				browser = browser_create_from_url(fullurl);
				bc++;
			}
		}
	

	if (bc == 0) { /* no browsers exist yet, create default one */
		browser_create_default(NULL);
		bc++;
	}
		
	if (ctx)
		poptFreeContext(ctx);

	/* set version in config -- this is for future use */
	gnome_config_set_string ("/galeon/General/version", VERSION);

	/* enter the main GTK event processing loop */
	gtk_main ();

	/* exit cleanly */
	return 0;
}

#ifdef CORBA_INTERFACE

static void
error_check(CORBA_Environment *ev)
{
	switch (ev->_major) {  /* check result code */
		case CORBA_NO_EXCEPTION:	break;
		case CORBA_SYSTEM_EXCEPTION:	g_print("CORBA System Exception: %s\n", CORBA_exception_id(ev));
				     		exit(3);
		case CORBA_USER_EXCEPTION:	g_print("CORBA User Exception: %s\n", CORBA_exception_id(ev));
						exit(3);
		default:	g_print("CORBA Unknown Exception: %s\n", CORBA_exception_id(ev));
	}
}	


/* activates the CORBA server interface */
static void activate_corba_server(CORBA_Environment *ev,
				CORBA_ORB *orb,
				CORBA_Object *galeon_servant,
				PortableServer_POA *root_poa,
				PortableServer_POAManager *pm)
{
	gint result;
	
	*root_poa = (PortableServer_POA)
	CORBA_ORB_resolve_initial_references(*orb, "RootPOA", ev);
	error_check(ev);
	
	*galeon_servant = impl_Galeon_Browser__create(*root_poa, ev);
	error_check(ev);
	
	*pm = PortableServer_POA__get_the_POAManager(*root_poa, ev);
	error_check(ev);
	
	PortableServer_POAManager_activate(*pm, ev);
	error_check(ev);
	
	result = goad_server_register(CORBA_OBJECT_NIL, *galeon_servant,
					"galeon_browser", "object", ev);
	error_check(ev);

	if (result) {    /* a galeon object already exists and active? */
		g_warning("Galeon object already active, server registration failed\n");
	}
}

#endif
					

/**
 * check_first_time: check if this is the first time the user
 * has run Galeon, and take the necessary steps if so
 */
static void 
check_first_time(void)
{
	gchar *conf_dir;
	gchar *netscape_dir;
	GtkWidget *w;

	/* configuration and cache directory is ${HOME}/.galeon */
 	conf_dir = g_strconcat (g_get_home_dir(), "/.galeon", NULL);
	
	/* if it exists, cancel any further processing */
	if (access (conf_dir, F_OK) == 0)
		return;

	/* no, try to create it */
	if (mkdir (conf_dir, 488) != 0)
	{
		g_error("couldn't make `%s' directory", conf_dir);
	}

	mozilla_prefs_import_language ();

	/* netscape directory would be... */
 	netscape_dir = g_strconcat (g_get_home_dir(), "/.netscape", NULL);

	/* does it exist? */
	if (access (netscape_dir, F_OK) == 0)
	{
		/* offer to import netscape preferences */
		w = gnome_question_dialog(_("This appears to be the first time "
					      "you have run Galeon\n\n"
					      "Would you like to import some of "
					      "your existing Netscape preferences?"),
					      first_time_prefs_cb, NULL);
		
		/* wait until question dialog is destroyed */
		gtk_signal_connect(GTK_OBJECT(w), "destroy", 
			   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &w);
		/* g_print("waiting for question"); */
		while (w)
		{
			if (gtk_events_pending()) gtk_main_iteration();
			usleep(1);
			/* g_print("o"); */
		}
	

		gnome_question_dialog(_("Galeon can also import your Netscape "
					"bookmarks.\n\n"
					"Would you like to do this now?"),
					first_time_bookmarks_cb, NULL);
	}
	
}

/**
 * first_time_cb: called if the user decides to import existing
 * netscape bookmarks or not
 */
static void
first_time_bookmarks_cb(gint reply, gpointer data)
{
        /* import if that was what was requested, otherwise ignore */
        if (reply == GNOME_YES)
	{
  	        /* FIXME: dont use this callback directly! */
	        bookmarks_import_netscape_activate_cb (NULL, NULL);
	}
}

static void
first_time_prefs_cb(gint reply, gpointer data)
{
	GtkWidget *w;
	
	/* import if that was what was requested, otherwise ignore */
	if (reply == GNOME_YES)
	{
		if (netscape_import_prefs() == FALSE)
		{
			w = gnome_error_dialog(_("Unable to import Netscape preferences"));
			gnome_message_box_set_modal(GNOME_MESSAGE_BOX(w));
			
			gtk_signal_connect(GTK_OBJECT(w), "destroy", 
				   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &w);
		
			/*g_print("waiting for ok");*/
			while (w)
			{
				if (gtk_events_pending()) gtk_main_iteration();
				usleep(1);
				/*g_print(".");*/
			}
		}
	}
}
