/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "libgnomeui/gnome-window-icon.h"

/* local function prototypes */
static void browser_init_data (GaleonBrowser *gb);
static void browser_add_accelerators (GaleonBrowser *browser);
static void browser_create_mozembed (GaleonBrowser *gb);
static GaleonBrowser *browser_create_window (void);
static GtkWidget *browser_notebook_label_new (GaleonBrowser *browser,
		const gchar *label);

/* the global list of all GaleonBrowser *windows* (visible stuff) */
GList *all_windows = NULL;

/* the global list of all GaleonBrowser *browsers* (containing mozembeds) */
GList *all_browsers = NULL;

/* the parent window, for tabbed browsing */
GaleonBrowser *master_browser = NULL;

/* notebook when doing tabbed browsing */
GtkWidget *master_notebook = NULL;

/* gtkmozembed type of drag and drop */
static const GtkTargetEntry drop_types[] =
{
	{ "GALEON_URL",    0, DND_TARGET_GALEON_URL   },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL },
	{ "STRING",        0, DND_TARGET_STRING       }
};
static const gint drop_types_num_items = (sizeof (drop_types) / 
					  sizeof (GtkTargetEntry));

/* url type of drag and drop */
const GtkTargetEntry url_drag_types[] = 
{
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint url_drag_types_num_items = (sizeof (url_drag_types) /
				       sizeof (GtkTargetEntry));

/**
 * browser_create_default: create a new browser pointing at the page
 * specified by configuration
 */
GaleonBrowser *
browser_create_default (GaleonBrowser *previous_browser)
{
	GaleonBrowser *browser;
	gchar *home_page_url;
	gchar *last_page_url = NULL;
	gboolean free_last_page = FALSE;
	gint page_type;

	/* get location of home page */
	home_page_url = gnome_config_get_string(CONF_GENERAL_HOMEPAGE);

	/* get location of last page: use previous browser if one,
	 * otherwise resort to fetching it from global history */
	if (previous_browser != NULL)
	{
		last_page_url = gtk_moz_embed_get_location 
			(previous_browser->mozEmbed);
	}
	else
	{
		last_page_url = history_get_last_url();
		free_last_page = TRUE;
	}
	
	/* find out where we're supposed to start */
	if (previous_browser == NULL)
		page_type = gnome_config_get_int(CONF_GENERAL_HOMEPAGE_TYPE);
	else
		page_type = gnome_config_get_int(CONF_GENERAL_NEWPAGE_TYPE);

	/* create a browser */
	browser = browser_create ();
	browser_set_visibility (browser, TRUE);

       	/* go to the appropriate page */
	if (page_type == HOME_PAGE && home_page_url != NULL)
	{
		/* load home page, if set */
		browser_load_url (browser, home_page_url);
	}
	else if (page_type == LAST_PAGE && last_page_url != NULL)
	{
		/* load page pointed at by last browser */
		browser_load_url (browser, last_page_url);

		/* copy history, if any */
		if (previous_browser != NULL)
		{
			mozilla_copy_session_history (previous_browser, 
						      browser);
		}
	}
	else
	{
		/* even in case of error, it's a good default */
		browser_load_url (browser, "about:blank");
	}

	/* free allocated strings */
	if (home_page_url) g_free(home_page_url);
	if (free_last_page == TRUE && last_page_url) g_free(last_page_url);
	
	return browser;
}

/**
 * browser_set_visibility: 
 */
void
browser_set_visibility (GaleonBrowser *browser, gboolean visibility)
{
	if (tabbed_mode)
	{
		browser = master_browser;
	}

	if (!visibility)
	{
		gtk_widget_hide (browser->WMain);
	}
	else
	{
		if (!GTK_WIDGET_VISIBLE(browser->WMain))
		{
			set_settings_menu_window (browser);
			gtk_widget_show(browser->WMain);
		}
	}
}

/**
 * browser_load_url: interpret and load a URL into a given browser
 */
void
browser_load_url (GaleonBrowser *browser, const gchar *url)
{
	/* check arguments */
	g_assert(browser != NULL);
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return;

	/* load the URL */
	if (strcmp(url, MYPORTAL_URL) == 0)
	{
		gtk_moz_embed_stop_load (browser->mozEmbed);
		portal_render_into_browser (browser);
	}
	else
	{
		gtk_container_focus (GTK_CONTAINER (browser->mozEmbed),
				     GTK_DIR_TAB_FORWARD);
		gtk_container_focus (GTK_CONTAINER (browser->mozEmbed),
				     GTK_DIR_TAB_FORWARD);
		/* load the url */
		gtk_moz_embed_load_url (browser->mozEmbed, url);
	}

	/* initialise embed whenever a document is first loaded */
	if (browser->embed == NULL)
	{
		browser_embed_init (browser);
	}
/*
	if (GTK_CHECK_MENU_ITEM(browser->view_toolbar)->active && !tabbed_mode)
	{
		gtk_widget_grab_focus(browser->toolbar_entry);
	}
*/
}

/**
 * browser_create_from_url: create a browser from a given url string
 */
GaleonBrowser *
browser_create_from_url (const gchar *url)
{
	GaleonBrowser *browser;

	/* check argument */
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return NULL;

	/* create a window */
	browser = browser_create();
	browser_set_visibility (browser, TRUE);

	/* load the url */
	browser_load_url(browser, url);

	return browser;
}

/**
 * browser_create_from_url_view_source: create a browser from a given 
 * url string  in view source mode
 */
GaleonBrowser *
browser_create_from_url_view_source (const gchar *url)
{
	GaleonBrowser *browser;
	GtkWidget *view_source_mode_menuitem;

	/* check argument */
	g_assert(url != NULL);

	/* error if we can't handle this URL */
	if (handle_foreign_protocols(url))
		return NULL;

	/* create a window */
	browser = browser_create();
	browser_set_visibility (browser, TRUE);

	/* load a blank url, needed to initialize the embed */
	browser_load_url(browser, "about:blank");

	/* set view source mode */
	mozilla_view_source (browser, TRUE);

	/* synchronise view source menu */
	if (!tabbed_mode)
	{
		view_source_mode_menuitem = 
			glade_lookup_widget(browser->WMain, 
					    "view_source_mode");
		gtk_check_menu_item_set_active 
			(GTK_CHECK_MENU_ITEM (view_source_mode_menuitem), 
			 TRUE);
	}

	/* load the url */
	browser_load_url(browser, url);

	/* return completed browser */
	return browser;
}

/**
 * browser_create: create a main window widget with a browser widget
 * inside it.
 */
GaleonBrowser *
browser_create (void)
{
	gint page, tab_position;
        GaleonBrowser *gb;

	/* build tabbed window if needed */
	if (tabbed_mode && master_browser == NULL)
	{
		/* make the toplevel browser window */
		master_browser = browser_create_window ();
		
		/* make the toplevel notebobok */
		master_notebook = gtk_notebook_new ();

		/* set some notebook properties */
		gtk_notebook_popup_enable (GTK_NOTEBOOK (master_notebook));
		gtk_notebook_set_scrollable (GTK_NOTEBOOK (master_notebook),
					     TRUE);
		tab_position = gnome_config_get_int 
			(CONF_APPEARANCE_TABBED_POSITION);
		gtk_notebook_set_tab_pos (GTK_NOTEBOOK (master_notebook),
					  tab_position);
		gtk_signal_connect (GTK_OBJECT (master_notebook), 
				    "switch_page",
				    browser_notebook_switch_page_cb, 
				    master_browser);

		/* insert notebook into toplevel window */
		gnome_app_set_contents (GNOME_APP (master_browser->WMain), 
					GTK_WIDGET (master_notebook));
	}

	/* build appropriate widgets */
	if (tabbed_mode)
	{
		/* build a browser structure */
		gb = g_new0 (GaleonBrowser, 1);

		/* FIXME: this is the most evil bit */
		/* this is going to be removed soon */
		memcpy (gb, master_browser, sizeof(GaleonBrowser));
#ifdef DRAG_LISTENER
		gb->has_listener = FALSE;
#endif
		gb->embed = NULL;

		/* build embedding widget */
		browser_create_mozembed (gb);

		/* add as a tab into the notebook */
		gtk_notebook_append_page 
			(GTK_NOTEBOOK (master_notebook),
			 GTK_WIDGET (gb->mozEmbed),
			 browser_notebook_label_new (gb, _("Untitled")));
		browser_set_notebook_label_status (gb, NEW);

		/* show it */
		gtk_widget_show (GTK_WIDGET (gb->mozEmbed));

		/* switch to page if set in config */
		if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_AUTOJUMP))
		{
			/* argh, this is laborious! */
			page = gtk_notebook_page_num 
				(GTK_NOTEBOOK (master_notebook),
				 GTK_WIDGET (gb->mozEmbed));
			gtk_notebook_set_page (GTK_NOTEBOOK (master_notebook),
					       page);
		}
	}
	else
	{
		/* build browser window */
		gb = browser_create_window ();

		/* build embedding widget */
		browser_create_mozembed (gb);

		/* put embedding widget into window */
		gnome_app_set_contents (GNOME_APP (gb->WMain), 
					GTK_WIDGET (gb->mozEmbed));
	}

	/* add browser to list */
	all_browsers = g_list_append (all_browsers, gb);

	/* return the completed GaleonBrowser */
	return gb;
}

/**
 * browser_create_window: create a browser structure and main window, 
 * but without an embedding widget.
 * NB: This should only be called from browser_create
 */
static GaleonBrowser *
browser_create_window (void)
{
	static int have_selection = FALSE;
        GaleonBrowser *gb;
        GtkWidget *window;
	GladeXML *gxml;
	GnomeEntry *ge;
	gint width, height;
	gboolean menubar_relief;

	/* use libglade to build the window */
	gxml = glade_xml_new (glade_file (), "WMain");
	window = glade_xml_get_widget (gxml, "WMain");
 
	/* set mini icon */
	gnome_window_icon_set_from_file (GTK_WINDOW(window), 
					 gnome_pixmap_file ("galeon.png"));

	/* don't know why libglade does not set this */
	GNOME_APP(window)->menubar = glade_xml_get_widget (gxml, "menubar1"); 

	/* allocate and initialise the GaleonBrowser structure */
	gb = g_new0 (GaleonBrowser, 1);
	gb->WMain = window;

  	browser_init_data (gb);

	/* so widgets can look the data up */
	gtk_object_set_data (GTK_OBJECT (gb->WMain), "mozilla", gb);

	/* set menubar relief according to gnome prefs (libglade does not)*/
	menubar_relief = gnome_preferences_get_menubar_relief ();

	gnome_dock_item_set_shadow_type 
		(GNOME_DOCK_ITEM (gb->menubar->parent), 
		 menubar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);

	/* connect the signals */
	glade_xml_signal_autoconnect_full (gxml, (GladeXMLConnectFunc) 
					   glade_signal_connect_func, gb);

	/* create the toolbar */
	toolbar_create (gb);

	/* find the default width and height */
	width = gnome_config_get_int (CONF_APPEARANCE_WINWIDTH);
	height = gnome_config_get_int (CONF_APPEARANCE_WINHEIGHT);

	/* set window default size */
	gtk_widget_set_usize (GTK_WIDGET(window), width, height);

	/* add it to the list of browser windows */
	all_windows = g_list_append(all_windows, gb);

	/* create the bookmarks menu */
	bookmarks_create_menu (gb);
	bookmarks_create_tb (gb);

	/* setup the history dropdown menu */
	ge = GNOME_ENTRY(gb->toolbar_gnomeentry);
	auto_completion_add_from_entry (GNOME_ENTRY(ge));

	/* set url entry drop destination */
	gtk_drag_dest_set 
		(GTK_WIDGET (gnome_entry_gtk_entry 
			     (GNOME_ENTRY (gb->toolbar_gnomeentry))),
		 GTK_DEST_DEFAULT_ALL,
		 drop_types, drop_types_num_items,
		 GDK_ACTION_COPY);

	/* set bookmark menuitem drop destination */
	gtk_drag_dest_set (glade_xml_get_widget (gxml, "bookmarks"), 
			   GTK_DEST_DEFAULT_ALL,
			   url_drag_types, url_drag_types_num_items,
			   GDK_ACTION_COPY);
	gtk_signal_connect (GTK_OBJECT 
			    (glade_xml_get_widget (gxml, "bookmarks")), 
			    "drag_data_received",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_menuitem_drag_data_received_cb),
			    gb);

	/* set selection signal */
	gtk_selection_add_target (gb->WMain, GDK_SELECTION_PRIMARY, 
				  GDK_SELECTION_TYPE_STRING, 1);
	gtk_signal_connect (GTK_OBJECT(gb->WMain), "selection_get",
			    GTK_SIGNAL_FUNC (window_selection_get_cb),
			    &have_selection);
	gtk_signal_connect (GTK_OBJECT(gb->WMain), "selection_received",
			    GTK_SIGNAL_FUNC (window_selection_received_cb),
			    gb);

	/* add additional accelerators */
	browser_add_accelerators (gb);

	/* return the part-completed GaleonBrowser */
	return gb;
}

/* signals to connect on each browser widget */
static const struct
{ 
	char *event; 
	void *func; /* should be a GtkSignalFunc or similar */
}
signal_connections[] =
{
	{ "location",        mozembed_location_changed_cb  },
	{ "title",           mozembed_title_changed_cb     },
	{ "net_start",       mozembed_load_started_cb      },
	{ "net_stop",        mozembed_load_finished_cb     },
	{ "net_state",       mozembed_net_status_change_cb },
	{ "progress",        mozembed_progress_change_cb   },
	{ "link_message",    mozembed_link_message_cb      },
	{ "js_status",       mozembed_js_status_cb         },
	{ "open_uri",        mozembed_open_uri_cb          },
	{ "visibility",      mozembed_visibility_cb        },
	{ "destroy_browser", mozembed_destroy_brsr_cb      },	
	{ "dom_mouse_click", mozembed_dom_mouse_click_cb   },
	{ "dom_key_press",   mozembed_dom_key_press_cb     },
	{ "size_to",         mozembed_size_to_cb           },
	/* terminator -- must be last in the list! */
	{ NULL, NULL } 
};

/**
 * browser_create_mozembed: create a GtkMozEmbed widget and set it up
 * NB: This should only be called from browser_create
 */
static void
browser_create_mozembed (GaleonBrowser *gb)
{
	static gboolean preferences_set = FALSE;
	gint i;

	/* make an embedding widget */
	gb->mozEmbed = (GtkMozEmbed *) gtk_moz_embed_new ();

	/* ref/unref widget -- is this really needed? */
	gtk_widget_ref (GTK_WIDGET (gb->mozEmbed));
	gtk_object_set_data_full (GTK_OBJECT (gb->mozEmbed), "mozEmbed",
				  GTK_WIDGET (gb->mozEmbed),
				  (GtkDestroyNotify) gtk_widget_unref);
	gtk_object_set_data (GTK_OBJECT (gb->mozEmbed), "browser", gb);

	/* some mozilla settings */
	gtk_moz_embed_set_chrome_mask (gb->mozEmbed, 
				       GTK_MOZ_EMBED_FLAG_ALLCHROME);
	gtk_signal_connect (GTK_OBJECT (gb->mozEmbed), "destroy",
			    GTK_SIGNAL_FUNC (mozembed_destroy_cb), gb);

	/* connect signals */
	for (i = 0; signal_connections[i].event != NULL; i++)
	{
		gtk_signal_connect_while_alive (GTK_OBJECT(gb->mozEmbed),
						signal_connections[i].event,
						signal_connections[i].func, 
						gb, GTK_OBJECT(gb->mozEmbed));
	}

	/* connect open window signal according to the new windows pref */
	if (gnome_config_get_int (CONF_GENERAL_NEWWINDOWS) == 0)
	{
		gtk_signal_connect_while_alive (GTK_OBJECT (gb->mozEmbed),
						"new_window",
						mozembed_new_window_cb, gb,
						GTK_OBJECT (gb->mozEmbed));
	}

	/* set gtkmozembed drag and drop destination */
	gtk_drag_dest_set (GTK_WIDGET(gb->mozEmbed), GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );

	/* set gtmozembed drag and drop signals */
	gtk_signal_connect (GTK_OBJECT(gb->mozEmbed), "drag_drop",
			    GTK_SIGNAL_FUNC(mozembed_drag_drop_cb), gb);
	gtk_signal_connect (GTK_OBJECT(gb->mozEmbed), "drag_data_received",
	GTK_SIGNAL_FUNC(window_drag_data_received), gb);

	/* set links drag signal */
	gtk_signal_connect (GTK_OBJECT 
			    (gb->mozEmbed), 
			    "drag_data_get",
			    GTK_SIGNAL_FUNC 
			    (window_drag_data_get_cb),
			    gb);

	/* set preferences */
	if (!preferences_set) 
	{
#ifndef USE_PROFILE_API
		mozilla_prefs_init (g_concat_dir_and_file 
				    (g_get_home_dir (), ".galeon/mozilla"));
#endif
		mozilla_prefs_set ();

		preferences_set = TRUE;
	}

}

/**
 * browser_embed_init: call it after the first page is loaded
 */
void browser_embed_init(GaleonBrowser *browser)
{
	gboolean event_listener = gnome_config_get_bool(CONF_MOUSE_LINKS_DRAG);
	browser->embed = mozilla_embed_init (browser, event_listener);
	if (tabbed_mode)
	{
		/* fix up for tabbed mode */
		if (browser->mozEmbed == master_browser->mozEmbed)
		{
			master_browser->embed = browser->embed;
		}
	}
}

/**
 * browser_show_error: show a browser error message
 */
void browser_show_error(GaleonBrowser *browser, char *errormsg)
{
	/* show the error message on status bar */
	browser->statusMessage = errormsg;
	browser_update_status_bar_text (browser);
}

/**
 * browser_progress_clear: clear all infos about download progress
 */
void
browser_progress_clear(GaleonBrowser *browser)
{
	/* set all progress values to 0 */ 
	browser->loadPercent = 0;
	browser->bytesLoaded = 0;
	browser->maxBytesLoaded = 0;
}

/**
 * browser_update_status_bar_test: update the status bar of the toplevel browser window
 */
void
browser_update_status_bar_text(GaleonBrowser *browser)
{
	gchar message[256];	

	gnome_appbar_pop(GNOME_APPBAR(browser->appbar1));

	if (browser->tempMessage) 
	{
		/* fill the status bar text with the current temporary message */
		gnome_appbar_push(GNOME_APPBAR(browser->appbar1),browser->tempMessage);
	}
	else if (browser->statusMessage)
	{

		if ((browser->loadPercent) 
		    && (browser->bytesLoaded <= browser->maxBytesLoaded))
		{
			/* fill the status bar text with progress info: percentual and kb */
			g_snprintf(message, 255, _("%s (%d%% complete, %d kB of %d kB loaded)"), 
				   browser->statusMessage, browser->loadPercent, 
				   (!browser->bytesLoaded) ? 0 : browser->bytesLoaded/1024,
				   (!browser->maxBytesLoaded) ? 0 : browser->maxBytesLoaded/1024);
		}
		else if (browser->bytesLoaded)
		{
			/* fill the status bar text with progress info: only kb */
			g_snprintf(message, 255, _("%s (%d kB loaded)"), browser->statusMessage, 
				   (!browser->bytesLoaded) ? 0 : browser->bytesLoaded/1024);
		}
		else
		{
			g_snprintf(message, 255, "%s", browser->statusMessage);
		}


		gnome_appbar_push(GNOME_APPBAR(browser->appbar1),message);
	}

}

/**
 * browser_update_temp_message: update the temporary message
 */
void
browser_update_temp_message(GaleonBrowser *browser, const char *message)
{
	/* free the previous message */
	if (browser->tempMessage) 
	{
		g_free(browser->tempMessage);
	}
	
	/* fill the new temporary message */
	if (message) 
	{
		browser->tempMessage = g_strdup(message);
	}	
	else
	{	
		browser->tempMessage = NULL;
	}
	browser_update_status_bar_text(browser);
}

/**
 * browser_update_nav_buttons: update back and forward toolbar buttons status
 */
void
browser_update_nav_buttons (GaleonBrowser *browser)
{
	gboolean can_go_back, can_go_forward;

	/* query mozilla */
	can_go_back = gtk_moz_embed_can_go_back (browser->mozEmbed);
	can_go_forward = gtk_moz_embed_can_go_forward (browser->mozEmbed);

	/* set widget status appropriately */
	gtk_widget_set_sensitive (browser->BBack, can_go_back);
	gtk_widget_set_sensitive (browser->BForward, can_go_forward);
}

/**
 * browser_progress_action: update the status bar progress
 */
gboolean
browser_progress_action (GaleonBrowser *browser)
{
	/* FIXME comment here */
	GtkWidget *progress;
	gfloat val;

	g_return_val_if_fail(browser->appbar1 != NULL, FALSE);
	progress = GTK_WIDGET(gnome_appbar_get_progress(GNOME_APPBAR(browser->appbar1)));

	/* update progressive loader "activity" indicator */
	val = gtk_progress_get_value(GTK_PROGRESS(progress));
	if (browser->loadPercent == 0)
	{
		if (++val > 100)
			val = 0;
	}
	else
		val = browser->loadPercent;

	gtk_progress_set_value(GTK_PROGRESS(progress), val);

	/* returns true always, must be explicitly destroyed */
	return TRUE;
}

/**
 * browser_init_data: init GaleonBrowser data structure
 */
static void
browser_init_data (GaleonBrowser *gb)
{
	gb->spinner = NULL;
	gb->statusMessage = NULL;
	gb->loadPercent = 0;
	gb->bytesLoaded = 0;
	gb->maxBytesLoaded = 0;
	gb->tempMessage = NULL;
	gb->timeout = 0;
	gb->bookmarks_toolbars = NULL;
	gb->bookmarks_tooltips = NULL;
	gb->mime_ignore_once = FALSE;
	gb->is_popup = FALSE;
	gb->embed = NULL;

	/* fill GUI components */
	gb->dock = glade_lookup_widget (gb->WMain, "dock");
	gb->appbar1 = glade_lookup_widget (gb->WMain, "appbar1");
	gb->menubar = glade_lookup_widget (gb->WMain, "menubar1");	
	gb->view_source_mode = glade_lookup_widget (gb->WMain, "view_source_mode");	
	gb->view_toolbar = glade_lookup_widget (gb->WMain, "view_toolbar");	
	gb->view_menubar = glade_lookup_widget (gb->WMain, "view_menubar");	
	gb->view_statusbar = glade_lookup_widget (gb->WMain, "view_statusbar");	
	gb->view_fullscreen = glade_lookup_widget (gb->WMain, "view_fullscreen");	
	gb->load_images_always = glade_lookup_widget (gb->WMain, "load_images_always");
	gb->load_images_from_current_server_only = 
		glade_lookup_widget (gb->WMain, "load_images_from_current_server_only");
	gb->load_images_never = glade_lookup_widget (gb->WMain, "load_images_never");
	gb->use_own_fonts = glade_lookup_widget (gb->WMain, "use_own_fonts");
	gb->use_own_colors = glade_lookup_widget (gb->WMain, "use_own_colors");
	gb->enable_java = glade_lookup_widget (gb->WMain, "enable_java");
	gb->enable_javascript = glade_lookup_widget (gb->WMain, "enable_javascript");
	gb->use_galeon_mime_handling = glade_lookup_widget (gb->WMain, 
							    "use_galeon_mime_handling");
}

/**
 * browser_show_find_dialog: show the find dialog
 */
void browser_show_find_dialog (GaleonBrowser *browser, gpointer target)
{
	GtkWidget *dFind;
	GladeXML *gxml;

	gxml = glade_xml_new(glade_file(), "dFind");
	glade_xml_signal_autoconnect_full (gxml, (GladeXMLConnectFunc) glade_signal_connect_func, browser);
	dFind = glade_xml_get_widget(gxml, "dFind");

	/* if the dialog has a target show it as modal. Opening a new page would segfault. */
	if (target)
	{
		gtk_window_set_modal (GTK_WINDOW(dFind), TRUE);
	}

	gtk_object_set_data (GTK_OBJECT(dFind), "target", target);

	gtk_widget_show(dFind);
	window_set_layer(dFind);

	browser->start_find = TRUE;
}

/**
 * browser_show_open_dialog: show the open dialog
 */
void browser_show_open_dialog(GaleonBrowser *browser)
{
	GtkWidget *fs;
	GladeXML *gxml;

	gxml = glade_xml_new(glade_file(), "fsopen");
	glade_xml_signal_autoconnect_full (gxml, (GladeXMLConnectFunc) glade_signal_connect_func, browser);
	fs = glade_xml_get_widget(gxml, "fsopen");

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs),gnome_config_get_string(CONF_DIR_OPEN));

	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * browser_show_openurl_dialog: show the open url dialog
 */
void browser_show_openurl_dialog (GaleonBrowser *browser)
{
	GtkWidget *dialog = NULL;
	GtkBox *vbox;
	GtkBoxChild *child;
	GList *children;

	dialog = gnome_request_dialog(
		FALSE, _("Enter the URL of the location to open:                    \n"),
		NULL, 100, (GnomeStringCallback) open_url_ok_button_clicked_cb,
		browser, NULL
		);
	gtk_window_set_title(GTK_WINDOW(dialog), _("Open URL"));

	window_set_layer(dialog);

	/* find the entry */
	vbox = GTK_BOX(GNOME_DIALOG(dialog)->vbox);
	children = vbox->children;

	while (children)
	{
		child = (GtkBoxChild *) children->data;
		if (GTK_IS_ENTRY(child->widget))
		{
			extern gchar *open_url_str;

			gtk_signal_connect_after(GTK_OBJECT(child->widget),
						 "key_press_event",
						 GTK_SIGNAL_FUNC(
						   window_location_entry_key_press_cb),
						 browser);
			if (open_url_str)
				gtk_entry_set_text(GTK_ENTRY(child->widget),
						   open_url_str);
				gtk_entry_select_region(GTK_ENTRY(child->widget),
							0, -1);
			break;
		}
		children = children->next;
	}

	gnome_dialog_run(GNOME_DIALOG (dialog));
}

/**
 * browser_edit_bookmarks: show the edit bookmarks dialog
 */
void browser_edit_bookmarks (GaleonBrowser *browser)
{
	gint pane_pos;

	if (!bookmarks_editor)
	{
		gint x, y, width, height;

		bookmarks_editor_init ();

		gnome_config_push_prefix("/galeon/State/");
		x = gnome_config_get_int("bookmarks_editor_x=-1");
		y = gnome_config_get_int("bookmarks_editor_y=-1");
		if (!(x == -1 && y == -1))
			gtk_widget_set_uposition(bookmarks_editor->dialog,
						 x, y);

		width = gnome_config_get_int("bookmarks_editor_width=-1");
		height = gnome_config_get_int("bookmarks_editor_height=-1");
		if (!(width == -1 && height == -1))
			gtk_window_set_default_size(
				     GTK_WINDOW(bookmarks_editor->dialog),
				     width,height);
		pane_pos = gnome_config_get_int("bookmarks_editor_vpane=-1");
		if (pane_pos != -1)
			gtk_paned_set_position(
					GTK_PANED(bookmarks_editor->vpane),
					pane_pos);
		gnome_config_pop_prefix ();
	}
	bookmarks_editor->browser = browser;
	gtk_widget_show (bookmarks_editor->dialog);

	window_set_layer(bookmarks_editor->dialog);
}

/**
 * browser_show_temporary_bookmarks: show the temporary bookmarks dialog
 */
void browser_show_temporary_bookmarks (GaleonBrowser *browser)
{
	if (!temp_bookmarks_window)
	{
		gint x, y, width, height;

		temp_bookmarks_init ();
		gnome_config_push_prefix("/galeon/State/");
		x = gnome_config_get_int("temp_bookmarks_x=-1");
		y = gnome_config_get_int("temp_bookmarks_y=-1");
		if (!(x == -1 && y == -1))
			gtk_widget_set_uposition(temp_bookmarks_window->dialog,
						 x, y);

		width = gnome_config_get_int("temp_bookmarks_width=-1");
		height = gnome_config_get_int("temp_bookmarks_height=-1");
		if (!(width == -1 && height == -1))
			gtk_window_set_default_size(
				     GTK_WINDOW(temp_bookmarks_window->dialog),
				     width,height);
		gnome_config_pop_prefix ();
	}
	gtk_widget_show (temp_bookmarks_window->dialog);
	window_set_layer(temp_bookmarks_window->dialog);

	temp_bookmarks_window->browser = browser;
}

/**
 * browser_show_history: show the history dialog
 */
void browser_show_history (GaleonBrowser *browser)
{
	history_show_dialog ();
	
	/* this is to know wich window last invoked the history dialog */
	gtk_object_set_data(GTK_OBJECT(dHistory), "WINDOW", (gpointer) browser->WMain);
	gtk_widget_show(dHistory);
	window_set_layer(dHistory);
}

/**
 * browser_add_temporary_bookmark: add a temporary bookmark
 */
void browser_add_temporary_bookmark (GaleonBrowser *browser)
{
	gchar *name = g_strdup (gtk_moz_embed_get_title (browser->mozEmbed));
	gchar *url = gtk_editable_get_chars (GTK_EDITABLE (browser->toolbar_entry), 0, -1);
	add_temp_bookmark (SITE, name, url, NULL);
	g_free (url);
	g_free (name);
}

/**
 * browser_show_save_dialog: show the save dialog
 */
void browser_show_save_dialog (GaleonBrowser *browser, gpointer target)
{
	GtkWidget *fs;
	GladeXML *gxml;

	gxml = glade_xml_new(glade_file(), "fssave");
	glade_xml_signal_autoconnect_full (gxml,(GladeXMLConnectFunc) glade_signal_connect_func, browser);
	fs = glade_xml_get_widget(gxml, "fssave");

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs),gnome_config_get_string(CONF_DIR_SAVE));

	gtk_object_set_data (GTK_OBJECT(fs), "target", target);
	
	gtk_widget_show(fs);
	window_set_layer(fs);
}

/**
 * browser_toolbar_show: show the toolbar
 */
void browser_toolbar_show (GaleonBrowser *browser)
{
	gtk_widget_show (browser->main_dockitem);	

	if (gnome_config_get_int ("/galeon/Appearance/url_location=1")==0)
		gtk_widget_show (browser->location_dockitem);
	bookmarks_toolbar_set_visible (browser, TRUE);

}

/**
 * browser_toolbar_hide: hide the toolbar
 */
void browser_toolbar_hide (GaleonBrowser *browser)
{
	gtk_widget_hide (browser->main_dockitem);
	if (browser->location_dockitem)
		gtk_widget_hide (browser->location_dockitem);
	bookmarks_toolbar_set_visible (browser, FALSE);

	if (tabbed_mode)
		gtk_widget_queue_resize(master_notebook);
	else
		gtk_widget_queue_resize(GTK_WIDGET(browser->mozEmbed));
}

/**
 * browser_menubar_show: show the menubar
 */
void browser_menubar_show (GaleonBrowser *browser)
{
	gtk_widget_show (browser->menubar->parent);	
}

/**
 * browser_menubar_hide: hide the menubar
 */
void browser_menubar_hide (GaleonBrowser *browser)
{
	gtk_widget_hide (browser->menubar->parent);
	if (tabbed_mode)
		gtk_widget_queue_resize(master_notebook);
	else
		gtk_widget_queue_resize(GTK_WIDGET(browser->mozEmbed));
}

/**
 * browser_statusbar_show: show the statusbar
 */
void browser_statusbar_show (GaleonBrowser *browser)
{
	gtk_widget_show(browser->appbar1->parent);
}

/**
 * browser_statusbar_hide: hide the statusbar
 */
void browser_statusbar_hide (GaleonBrowser *browser)
{
	gtk_widget_hide(browser->appbar1->parent);
	gtk_widget_queue_resize(GTK_WIDGET(browser->mozEmbed));
}

/**
 * browser_view_source: view web page source 
 */
void browser_view_source (GaleonBrowser *browser, gpointer target)
{
	if (gnome_config_get_bool (CONF_HANDLERS_USE_EXTERNAL_SOURCE_VIEWER)) {
		browser_view_source_external (browser, target);
	} 
	else 
	{
		gchar *url = NULL; 
		
		if (target)
		{
			url = mozilla_get_eventtarget_url (browser, target);
		}
		else 
		{
			url = gtk_moz_embed_get_location (browser->mozEmbed);
		}

		if (url != NULL) 
		{
			browser_create_from_url_view_source (url);
		} 
		else 
		{
			gnome_error_dialog (
				_("Can't get the url for current page.\n"
				  "No source avaible."));
		}
	}
}

/**
 * browser_view_source_external: view web page source with an external application 
 */
void 
browser_view_source_external (GaleonBrowser *browser, gpointer target)
{
	char *filename;
	pid_t pid;

	/* get a name for the temporary file */
	filename = tmpnam_ex (".html");
		
	if (!filename) {
		gnome_error_dialog ( _("Could not create a temporary file"));
		return;
	}
		
	/* save the file from mozilla to a temporary file */
	
	if (!mozilla_save (browser, filename, target)) {
		gnome_error_dialog (_("Something went wrong while saving the "
				      "file to a temporary location"));
		g_free (filename);
		return;
	}
		
		
	/* Fork and invoke the external program. */
	pid = fork ();
	if (pid == 0) {
		/* child process */
		gchar *prog = gnome_config_get_string
			(CONF_HANDLERS_EXTERNAL_SOURCE_VIEWER);
		execlp (prog, prog, filename, NULL);
		g_warning (_("Error executing external program"));
		_exit (0);
	} else if (pid < 0) {
		/* error */
		g_error ("Error creating subprocess");
		return;
	} else {
		/* parent process */
		/* fork again and give the external program time to start  
		   before removing the temp file */
		pid_t pid2 = fork ();
		if (pid2 == 0) {
				/* child */
			sleep (45); /* 45 secs should be enough */
			remove (filename);
			_exit (0);
		} else if (pid2 < 0) {
			g_error ("Error creating subprocess");
		}
		g_free (filename);
	}
}

/**
 * browser_close: close a browser
 */
void browser_close (GaleonBrowser *browser)
{
	/* stop this particular embedding widget */
	gtk_moz_embed_stop_load (browser->mozEmbed);

	/* save the history out to disk -- although this entails a bit
	 * of overhead hopefully it will mean we don't end up losing so
	 * much of the history on a crash... MattA 10/12/2000 */
	history_save ();

	/* tabbed mode? */
	if (tabbed_mode)
	{
		/* destroy the embedding widget -- this will also destroy 
		 * the notebook tab and it's label */
		gtk_widget_destroy (GTK_WIDGET (browser->mozEmbed));
	}
	else
	{	
		/* remove window from list */
		all_windows = g_list_remove (all_windows, browser);

		/* FIXME: temporary hack to keep Galeon from exiting when
			  it gets a Gdk error */
		gdk_error_trap_push();
		/* destroy the window -- this will also destroy the mozmbed */
		gtk_widget_destroy (browser->WMain);
		gdk_flush();
		gdk_error_trap_pop();
	}
}

/**
 * browser_open_frame: open the frame pointed by the event target 
 */
void 
browser_open_frame (GaleonBrowser *browser, gpointer target, gboolean same_window)
{
	gchar *url = NULL; 
	
	if (target) {
		url = mozilla_get_eventtarget_url (browser, target);
	} else { /* should not reach here never... */
		url = gtk_moz_embed_get_location (browser->mozEmbed);
	}

	if (url != NULL) {
		if (same_window) {
			gtk_moz_embed_load_url (browser->mozEmbed, url);
		} else {
			browser_create_from_url (url);
		}
	}
}


/**
 * browser_go_home:
 */
void
browser_go_home(GaleonBrowser *browser, gboolean newwin)
{
	gchar *startpage = gnome_config_get_string("/galeon/General/startpage=www.gnome.org");

        if (startpage != NULL && strlen(startpage) != 0)
        {
		if (newwin)
			browser_create_from_url(startpage);
		else
			browser_load_url(browser, startpage);
	}
        else
		gnome_error_dialog(_("You must specify a Start Page in the Preferences dialog!"));

       g_free(startpage);
}

/*
 * browser_reload: call gtk_moz_embed_reload but first check if 
 * we are not in MyPortal
 */
void
browser_reload (GaleonBrowser *browser)
{
	if (strcmp (gtk_entry_get_text (GTK_ENTRY (browser->toolbar_entry)),
		    MYPORTAL_URL) == 0)
	{
		gtk_moz_embed_stop_load (browser->mozEmbed);
		portal_render_into_browser (browser);
	}
	else
	{
		gtk_moz_embed_reload(browser->mozEmbed, 
				     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
	}
}

/**
 * Add aditional accelerators. Remember to keep in sync with mozembed_dom_key_press_cb
 */
static void
browser_add_accelerators (GaleonBrowser *browser)
{
	GladeXML *gxml = glade_get_widget_tree (browser->WMain);
	GtkAccelGroup *ag = GNOME_APP (browser->WMain)->accel_group;
	int i;
	struct {
		gchar *widget_name;
		gchar *signal_name;
		guint key;
		guint mods;
		GtkAccelFlags flags;
	} accelerators[] = {
		{ "new_window_menuitem", "activate", GDK_n, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ "reload_page_menuitem", "activate", GDK_r, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ "zoom_in_menuitem", "activate", GDK_plus, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ "zoom_out_menuitem", "activate", GDK_minus, GDK_MOD1_MASK, GTK_ACCEL_VISIBLE },
		{ NULL, NULL, 0, 0, 0 }
	};
	for (i = 0; accelerators[i].widget_name != NULL; i++) {
		GtkWidget *w = glade_xml_get_widget (gxml, accelerators[i].widget_name);
		gtk_widget_add_accelerator 
			(w, accelerators[i].signal_name, ag, accelerators[i].key,
			 accelerators[i].mods, accelerators[i].flags);
	}
}

/**
 *  Request that the widget be placed above or below the Gnome panels
 */
void
window_set_layer (GtkWidget *widget)
{
	/* fullscreen mode flag */
	extern gboolean fullscreen_active;
	gboolean raise_windows;

	raise_windows = gnome_config_get_bool(
				"/galeon/Appearance/fullscreen_stacking=TRUE");

	if (fullscreen_active && raise_windows)
		gnome_win_hints_set_layer(widget, WIN_LAYER_ABOVE_DOCK);
	else
		gnome_win_hints_set_layer(widget, WIN_LAYER_NORMAL);

	gdk_window_raise(widget->window);
}

#if 0
/**
 * browsing_style_set: handle switch between tabbed/non-tabbed style
 * FIXME: this code isn't used yet (mainly coz it doesn't work...)
 */
void
browsing_style_set (void)
{
	gboolean tabbed;
	GList *l;

	/* tabbed mode? */
	tabbed = gnome_config_get_bool (CONF_APPEARANCE_TABBED);

	/* check against current mode */
	if (tabbed && master_browser == NULL)
	{
		/* switch to tabbed mode */

		/* create the master browser */
		browser_create ();

		/* reparent children */
		for (l = all_browsers; l != NULL; l = g_list_next(l));
		{
			GaleonBrowser *browser = (GaleonBrowser *)(l->data);
			GtkMozEmbed *embed = browser->mozEmbed;

			/* ugh! */
			memcpy (browser, master_browser, 
				sizeof (GaleonBrowser));
			browser->mozEmbed = embed;

			/* fix up widget */
			gtk_widget_unparent (GTK_WIDGET (browser->mozEmbed));
			gtk_notebook_append_page 
				(GTK_NOTEBOOK (master_notebook),
				 GTK_WIDGET (browser->mozEmbed),
				 browser_notebook_label_new (browser, 
					 _("Untitled")));
			gtk_widget_destroy (browser->WMain);
		}
	}
	else if (!tabbed && master_browser != NULL)
	{
		master_browser = NULL;
		/* FIXME */
	}
	/* else no change */
}
#endif

void
browser_set_notebook_label (GaleonBrowser *browser, const gchar *text)
{
	GList *child;
	GtkWidget *label;
	gchar *shortened;
	gint length;

	g_return_if_fail (browser != NULL);
	g_return_if_fail (text != NULL);
	g_return_if_fail (browser->mozEmbed != NULL);

	/* get the label widget */
	label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (master_notebook),
					    GTK_WIDGET (browser->mozEmbed));

	/* get out if theres a problem */
	g_return_if_fail (label != NULL);

	/* abbreviate the text label */
	length = gnome_config_get_int(CONF_APPEARANCE_TABBED_SHORTEN);
	shortened = shorten_name (text, length);
	g_return_if_fail (shortened != NULL);

	/* find and set the text label */
	child = gtk_container_children (GTK_CONTAINER (label));
	while (child != NULL)
	{
		if (GTK_IS_LABEL (child->data))
		{
			gtk_label_set_text (GTK_LABEL (child->data), 
					    shortened);
			break;
		}
		child = child->next;
	}

	/* the menu text is the full text */
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (master_notebook),
					  GTK_WIDGET (browser->mozEmbed),
					  text);

	/* free allocated strings */
	g_free (shortened);
}

void
browser_set_notebook_label_status (GaleonBrowser *browser, TabbedStatus status)
{
	GtkWidget *label;
	GList *l;

	g_return_if_fail (browser != NULL);
	g_return_if_fail (browser->mozEmbed != NULL);
	
	/* get the label widget */
	label = gtk_notebook_get_tab_label (GTK_NOTEBOOK (master_notebook),
					    GTK_WIDGET (browser->mozEmbed));
	g_return_if_fail (label != NULL);

	l = gtk_container_children (GTK_CONTAINER (label));
	for ( ; l ; l = l->next)
	{
		/* skip anything that isn't a text label */
		if (!(GTK_IS_LABEL (l->data)))
			continue;

		switch (status)
		{
		case NORMAL:
			gtk_widget_restore_default_style 
				(GTK_WIDGET (l->data));
			break;
			
		case NEW:
			gtk_widget_set_style
				(GTK_WIDGET (l->data), blue_text_style);
			break;
			
		case LOADING:
			gtk_widget_set_style
				(GTK_WIDGET (l->data), red_text_style);
			break;
		}
		break;
	}
}

static GtkWidget *
browser_notebook_label_new (GaleonBrowser *browser, const gchar *label)
{
	static GnomePixmap *close_pix = NULL;
	GtkWidget *box = gtk_hbox_new (FALSE, 4);
	GtkWidget *lab = gtk_label_new (label);
	GtkWidget *but = gtk_button_new ();

	if (! close_pix) close_pix = GNOME_PIXMAP 
		(gnome_pixmap_new_from_file (SHARE_DIR "/small-close.xpm"));
	
	gtk_button_set_relief (GTK_BUTTON (but), GTK_RELIEF_NONE);
	gtk_container_add (GTK_CONTAINER (but), 
			gnome_pixmap_new_from_gnome_pixmap (close_pix));
	gtk_box_pack_start (GTK_BOX (box), lab, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (box), but, FALSE, FALSE, 0);
	gtk_widget_show_all (box);
	gtk_signal_connect (GTK_OBJECT (but), "clicked", 
			GTK_SIGNAL_FUNC (browser_notebook_close_clicked_cb),
			browser);
	gtk_drag_dest_set (box, GTK_DEST_DEFAULT_ALL,
			   drop_types, drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );
	gtk_signal_connect (GTK_OBJECT (box), "drag_data_received",
			GTK_SIGNAL_FUNC(window_drag_data_received), browser);

	return box;
}

