/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "prefs.h"
#include "glade.h"
#include "mozilla.h"
#include "mozilla_i18n.h"
#include "mozilla_prefs.h"
#include "window.h"
#include "toolbar.h"
#include "bookmarks.h"
#include "embed.h"
#include "misc.h"
#include "mime.h"
#include "state.h"
#include "eel-gconf-extensions.h"
#include "dialog.h"
#include "toolbar_editor.h"

#include <stdlib.h>
#include <string.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtklist.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkinvisible.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-color-picker.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

/* default MIME types */
extern struct
{
	gchar *name;
	MimeAction action;
} default_mimetypes[];

/* local types */
typedef struct
{ 
	gchar const *name;
	PreferencesPageId page;
	gchar const *icon;
} PreferencesPanelButton;

typedef struct
{ 
	gchar const *name;
	PreferencesPanelButton const *icons;
} PreferencesSidebarItem;

/* local function prototypes */
static void preferences_lookup_widgets (PreferencesDialog *pd, GladeXML *gxml);
static PreferencesDialog *preferences_build_dialog (void);
static void preferences_build_sidebar (PreferencesDialog *pd);
static void preferences_sidebar_select_panel_cb (GtkWidget *button,
						 PreferencesDialog *pd);
static void preferences_sidebar_select_page_cb (GtkWidget *button,
						PreferencesDialog *pd);
static void preferences_set_dialog_to_config (PreferencesDialog *pd);
static void preferences_set_config_to_dialog (PreferencesDialog *pd);
static void set_config_from_optionmenu (GtkWidget *optionmenu,
					gchar *config_name);
static void set_config_from_spin_button (GtkWidget *spinbutton,
					 gchar *config_name);
static void set_config_from_togglebutton (GtkWidget *togglebutton,
					  gchar *config_name);
static void set_config_from_int_entry (GtkWidget *entry, gchar *config_name);
static void set_config_from_color (GtkWidget *colorpicker, gchar *config_name);
static void set_optionmenu_from_config (GtkWidget *optionmenu,
					gchar *config_name);
static void set_spin_button_from_config (GtkWidget *spinbutton,
					 gchar *config_name);
static void set_int_entry_from_config (GtkWidget *entry, gchar *config_name);
static void set_togglebutton_from_config (GtkWidget *togglebutton,
					  gchar *config_name);
static void set_list_from_config (GtkWidget *list, gchar *config_name);
static void set_config_from_list (GtkWidget *list, gchar *config_name);
static void set_color_from_config (GtkWidget *colorpicker, gchar *config_name);
static void preferences_mime_set_config_to_dialog (PreferencesDialog *pd);
static void preferences_mime_set_dialog_to_config (PreferencesDialog *pd);
void prefs_mime_edit_cb (GtkWidget *w, GaleonWindow *window);
static void prefs_dialog_save_geometry (PreferencesDialog *pd);
void prefs_fonts_init (PreferencesDialog *pd);
void prefs_fonts_free (PreferencesDialog *pd, int enc);
void prefs_fonts_free_all (PreferencesDialog *pd);
void prefs_fonts_save (PreferencesDialog *pd);
static int get_int_from_optionmenu (GtkWidget *optionmenu);
static void preferences_theme_selector_init (PreferencesDialog *pd);
static void theme_selector_fill_themelist_from_dir (GtkCList *list, gchar *base);
static void free_theme_selector (GtkCList *list);

/* local variables */
static const char invalid_chars[] = "\t\r\n\"$&<>,+=#!()'|{}[]?~`;%\\" "/:";
static const char digits_base16[] = "0123456789abcdef";
static GtkWidget *page_buttons[PREFS_PAGE_COUNT];
static GtkWidget *current_panel = NULL;

/**
 * prefs_key_len: length of @str when converted to a gconf key
 */
guint prefs_key_len (gchar const * str)
{
        size_t len = 1;

        g_return_val_if_fail (str != NULL, 0);
        while (*str) len+= (strchr (invalid_chars, *str++) ? 3 : 1);

        return len;
}

/**
 * prefs_key_new: allocates a gconf key corresponding to @str
 */
gchar * prefs_key_new (gchar const * str)
{
        gchar const * in = str;
        gchar * key, * out;

        g_return_val_if_fail (str != NULL, NULL);
        out = key = g_new0 (gchar, prefs_key_len (in));

        while (*in)
                if (strchr (invalid_chars, *in))
                        *out++ = ':',
                        *out++ = digits_base16 [*in >> 4],
                        *out++ = digits_base16 [*in++ & 15];
                else
                        *out++ = *in++;

        return key;
}

/**
 * preferences_open: open the prefs dialog
 */
PreferencesDialog *
preferences_open (GaleonWindow *window)
{
	static PreferencesDialog *pd = NULL;

	/* dialog available? */
	if (pd == NULL)
	{
		/* nope, build it */
		pd = preferences_build_dialog ();
	}
	else if (GTK_WIDGET_VISIBLE (pd->dialog))
	{
		/* already visible, raise to top */
		gdk_window_raise (GTK_WIDGET (pd->dialog)->window);
		return pd;
	}
	
	/* get the saved geometry */
	state_load_window_state (GTK_WIDGET (pd->dialog), "prefs_dialog");

	/* initialize fonts data structure */
	prefs_fonts_init (pd);
	prefs_fonts_language_init (pd);

	/* set values in dialog to stored values in prefs */
	preferences_set_dialog_to_config (pd);

	/* display it */
	gtk_widget_show (GTK_WIDGET (pd->dialog));
	dialog_set_parent (GTK_WIDGET (pd->dialog), window->WMain);

	/* the dialog */
	return pd;
}

/**
 * preferences_show_page: open preferences at a particular page
 */
void
preferences_show_page (GaleonWindow *window, PreferencesPageId id)
{
	PreferencesDialog *pd;
	GtkWidget *panel;

	/* make sure the window is open */
	pd = preferences_open (window);

	/* switch to appropiate panel */
	panel = GTK_WIDGET (gtk_object_get_data 
			   (GTK_OBJECT (page_buttons[id]), "panel"));
	if (current_panel != NULL && current_panel != panel)
	{
		gtk_widget_hide (current_panel);
	}
	if (current_panel != panel)
	{
		gtk_widget_show (GTK_WIDGET (current_panel = panel));
	}

	/* click appropiate button in order to load page */
	gtk_button_clicked (GTK_BUTTON (page_buttons[id]));
}

/**
 * preferences_close: close the preferences dialog window (hides it)
 */
void
preferences_close (PreferencesDialog *pd)
{
	prefs_dialog_save_geometry (pd);
	gtk_widget_hide (GTK_WIDGET (pd->dialog));
	prefs_fonts_free_all (pd);
}

/**
 * preferences_load: load Galeon preferences needed at startup
 */
void
preferences_load (void)
{
	mime_db_load ();
}

/**
 * preferences_save: save all Galeon preferences
 */
void
preferences_save (void)
{
	mime_db_save ();
	eel_gconf_suggest_sync ();      
}

/**
 * preferences_apply: make any global changes required for system state
 * to match config state
 */
void
preferences_apply (PreferencesDialog *pd)
{
	preferences_set_config_to_dialog (pd);

	/* save changed fonts values */
	prefs_fonts_remember (pd);
	prefs_fonts_save (pd);
}

/**
 * preferences_help: show the prefs Galeon manual page   
 */
void
preferences_help (void)
{
	GnomeHelpMenuEntry pref_help = { "galeon", "preferences.html" };
	gchar *file = NULL;

	/* Try to find the preferences help page.  If we fail, use ghelp
	   to load the index, instead */
	file = gnome_help_file_find_file("galeon", "preferences.html");

	if (file)
	{
	        gnome_help_display (NULL, &pref_help);
	}
	else
	{
		gnome_help_goto (NULL, "ghelp:galeon");
	}

	g_free(file);
}

static PreferencesDialog *
preferences_build_dialog (void)
{
	PreferencesDialog *pd;
	GladeXML *gxml;
	GList *li;
	GtkWidget *menu;

	/* not been generated yet... */
	pd = g_new0 (PreferencesDialog, 1);
	
	/* build the widgets */
	gxml = glade_widget_new ("preferences.glade", 
				 "preferences_dialog", &(pd->dialog), pd);
		
	/* lookup all the widgets */
	preferences_lookup_widgets (pd, gxml);

	/* no longer needed */
	gtk_object_unref (GTK_OBJECT (gxml));
	
	/* build the tree */
	preferences_build_sidebar (pd);
	
	/* get the charsets from mozilla if they are still empty */
	if (sorted_charset_titles == NULL)
	{
		mozilla_get_charsets (&charsets, &sorted_charset_titles);
	}
	
	/* initialize the default charset combo */
	gtk_combo_set_popdown_strings (GTK_COMBO (pd->default_charset_combo), 
				       sorted_charset_titles);

	/* attach the font language optionmenu event */
	menu = GTK_OPTION_MENU(pd->lang_encoding_optionmenu)->menu;
	li = GTK_MENU_SHELL(menu)->children;
	for (; li != NULL; li = li->next)
	{
		gtk_signal_connect 
			(GTK_OBJECT (li->data), "activate",
			 GTK_SIGNAL_FUNC(prefs_encoding_activated_cb), pd);
	} 
	pd->last_language = 0;

	/* return completed dialog */
	return pd;
}

/**
 * preferences_build_sidebar: build the sidebar of the preferences dialog
 */
static void
preferences_build_sidebar (PreferencesDialog *pd)
{
	PreferencesPanelButton browsing_icons[] =
	{
		{ _("General"),         PREFS_PAGE_ID_BROWSING_GENERAL,
					SHARE_DIR "/general.png" },
		{ _("Bookmarks"),       PREFS_PAGE_ID_BROWSING_BOOKMARKS,
					SHARE_DIR "/bookmarks.png" },
		{ _("History"),         PREFS_PAGE_ID_BROWSING_HISTORY,
					SHARE_DIR "/history.png" },
		{ NULL } /* terminator */
        };
	PreferencesPanelButton user_interface_icons[] =
	{
		{ _("Tabs"),            PREFS_PAGE_ID_USER_INTERFACE_TABS,
					SHARE_DIR "/tabs.png" },
		{ _("Windows"),         PREFS_PAGE_ID_USER_INTERFACE_WINDOWS,
					SHARE_DIR "/windows.png" },
		{ _("Toolbars"),        PREFS_PAGE_ID_USER_INTERFACE_TOOLBARS,
					SHARE_DIR "/toolbars.png" },
		{ _("Mouse"),           PREFS_PAGE_ID_USER_INTERFACE_MOUSE,
					SHARE_DIR "/mouse.png" },
		{ NULL } /* terminator */
        };
	PreferencesPanelButton handlers_icons[] =
	{
		{ _("Downloading"),     PREFS_PAGE_ID_HANDLERS_DOWNLOADING,
					SHARE_DIR "/download.png" },
		{ _("Programs"),        PREFS_PAGE_ID_HANDLERS_PROGRAMS,
					SHARE_DIR "/programs.png" },
		{ _("MIME Types"),      PREFS_PAGE_ID_HANDLERS_MIME_TYPES, 
					SHARE_DIR "/mime.png" },
		{ NULL } /* terminator */
	};
	PreferencesPanelButton rendering_icons[] =
	{
		{ _("Fonts/Colors"),    PREFS_PAGE_ID_RENDERING_FONTS_COLORS,
					SHARE_DIR "/fonts.png" },
		{ _("Languages"),       PREFS_PAGE_ID_RENDERING_LANGUAGES,
					SHARE_DIR "/babelfish.png" },
		{ NULL } /* terminator */
	};
	PreferencesPanelButton advanced_icons[] =
	{
		{ _("Network"),         PREFS_PAGE_ID_ADVANCED_NETWORK,
					SHARE_DIR "/network.png" },
		{ _("Filtering"),       PREFS_PAGE_ID_ADVANCED_FILTERING,
					SHARE_DIR "/stop.png" },
		{ _("Persistent Data"), PREFS_PAGE_ID_ADVANCED_PERSISTENT_DATA,
					SHARE_DIR "/harddrive.png" },
		{ _("Crash Recovery"),  PREFS_PAGE_ID_ADVANCED_CRASH_RECOVERY,
					SHARE_DIR "/crash.png" },
		{ NULL } /* terminator */
	};
	PreferencesSidebarItem sidebar[] = 
	{
		{  _("Browsing"),        browsing_icons       },
		{  _("User Interface"),  user_interface_icons },
		{  _("Handlers"),        handlers_icons       },
		{  _("Rendering"),       rendering_icons      },
		{  _("Advanced"),        advanced_icons       },
		{ NULL } /* terminator */
	};

	PreferencesSidebarItem *item;

	g_return_if_fail (pd != NULL);
	g_return_if_fail (pd->sidebar != NULL);

	for (item = sidebar; NULL != item->name; ++item)
	{
		GtkWidget *button, *panel_button, *widget, *panel;
		PreferencesPanelButton const *icon;
		GSList *button_group = NULL;

		panel_button = gtk_button_new_with_label (item->name);
		panel = gtk_vbox_new (FALSE, 0);

		gtk_object_set_data (GTK_OBJECT (panel_button), "panel", 
				     panel);
		gtk_signal_connect (GTK_OBJECT (panel_button), "clicked",
				    preferences_sidebar_select_panel_cb, pd);
		gtk_box_pack_start (GTK_BOX (pd->sidebar), panel_button,
				    FALSE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (pd->sidebar), panel,
				    TRUE, TRUE, 2);

		if (sidebar == item) 
		{
			gtk_button_clicked (GTK_BUTTON (panel_button));
		}
		gtk_widget_show (panel_button);

		/* invisible dummy button -> nothing selected */
		button = gtk_radio_button_new (button_group);
		button_group = gtk_radio_button_group 
			(GTK_RADIO_BUTTON (button));
		gtk_box_pack_start (GTK_BOX (panel), button, FALSE, TRUE, 0);
		gtk_button_clicked (GTK_BUTTON (button));

		/* fill the panel */
		for (icon = item->icons; NULL != icon->name; ++icon)
		{
			GtkWidget *widget, *vbox;
			PixmapData *icon_data;
			gint id = icon->page;

			page_buttons[id] = gtk_radio_button_new (button_group);
			gtk_button_set_relief (GTK_BUTTON (page_buttons[id]),
					       GTK_RELIEF_NONE);
			gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON 
					    (page_buttons[id]), FALSE);
			button_group = gtk_radio_button_group
						(GTK_RADIO_BUTTON (
						page_buttons[id]));
			gtk_object_set_data (GTK_OBJECT (page_buttons[id]), 
					    "page_id", GINT_TO_POINTER (id));
			gtk_object_set_data (GTK_OBJECT (page_buttons[id]), 
					    "panel", panel);
			gtk_signal_connect (GTK_OBJECT (page_buttons[id]), 
					    "clicked",
					    preferences_sidebar_select_page_cb,
					    pd);
			gtk_box_pack_start (GTK_BOX (panel), page_buttons[id],
					    FALSE, TRUE, 0);
			gtk_widget_show (page_buttons[id]);


			vbox = gtk_vbox_new (FALSE, 0);
			gtk_container_add (GTK_CONTAINER (page_buttons[id]), 
					    vbox);
			gtk_widget_show (vbox);


			icon_data = pixmap_data_new_from_file (icon->icon);

			if (NULL != icon_data)
			{
				widget = gtk_pixmap_new (icon_data->pixmap,
							 icon_data->mask);
				gtk_box_pack_start (GTK_BOX (vbox), widget,
						    FALSE, FALSE, 0);
				gtk_widget_show (widget);

				g_free (icon_data);
			}

			widget = gtk_label_new (icon->name);
			gtk_misc_set_alignment (GTK_MISC (widget), .5, .5);
			gtk_label_set_line_wrap (GTK_LABEL (widget), FALSE);
			gtk_box_pack_start (GTK_BOX (vbox), widget,
					    FALSE, TRUE, 0);
			gtk_widget_show (widget);
		}

		gtk_object_set_data (GTK_OBJECT (panel_button), 
				     "group", button_group);

		/* force Gtk+ to align the panel buttons to the top */
		widget = gtk_invisible_new ();
		gtk_box_pack_end (GTK_BOX (panel), widget, TRUE, TRUE, 0);
		gtk_widget_show (widget);
	}
}

/**
 * preferences_sidebar_select_panel: show the selected (clicked) panel
 * or hide it when active already
 */
static void preferences_sidebar_select_panel_cb (GtkWidget *button,
						 PreferencesDialog *pd)
{
	GtkWidget *panel = GTK_WIDGET
		(gtk_object_get_data (GTK_OBJECT (button), "panel"));
	GSList *group;
	int page_id = 0;

	if (NULL != current_panel)
	{
		gtk_widget_hide (current_panel);
	}

	if (current_panel != panel)
	{
		gtk_widget_show (current_panel = panel);

		group = (GSList *)(gtk_object_get_data 
				   (GTK_OBJECT (button), "group"));

		for (; group != NULL ; group = g_slist_next(group))
		{
			if (gtk_toggle_button_get_active 
			    (GTK_TOGGLE_BUTTON (group->data)))
			{
				page_id = GPOINTER_TO_INT 
					(gtk_object_get_data
					 (GTK_OBJECT (group->data), 
					  "page_id"));
			}
		}
	}
	else
		current_panel = NULL;

	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), page_id);
}

/**
 * preferences_sidebar_select_page: activate the selected (clicked) page
 */
static void preferences_sidebar_select_page_cb (GtkWidget *button,
						PreferencesDialog *pd)
{
	int page_id = GPOINTER_TO_INT
		(gtk_object_get_data (GTK_OBJECT (button), "page_id"));
	gtk_notebook_set_page (GTK_NOTEBOOK (pd->notebook), page_id);
}
					     
/**
 * preferences_set_config_to_dialog: read all the preferences values 
 * out of the dialog box widget and store them in the configuration space
 */
static void
preferences_set_config_to_dialog (PreferencesDialog *pd)
{
	gchar *charset_title;
	gchar *cur_charset;

	/* set each of the general configuration options */
	set_config_from_editable (pd->startpage_entry, CONF_GENERAL_HOMEPAGE);
	set_config_from_radiobuttongroup (pd->startup_show_homepage,
				    CONF_GENERAL_STARTPAGE_TYPE);
	set_config_from_radiobuttongroup (pd->newpage_show_homepage, 
				    CONF_GENERAL_NEWPAGE_TYPE);
	set_config_from_radiobuttongroup (pd->bookmarks_use_page_title,
				    CONF_BOOKMARKS_TITLE);
	set_config_from_radiobuttongroup (pd->bookmarks_show_actions_top,
				   CONF_BOOKMARKS_XITEMS);
	set_config_from_togglebutton (pd->always_save_session_check,
				      CONF_GENERAL_ALWAYS_SAVE_SESSION);
	set_config_from_togglebutton (pd->favicons_enabled_check,
				     CONF_BOOKMARKS_FAVICONS_ENABLED);
	set_config_from_togglebutton (pd->allow_popups_checkbutton, 
				      CONF_FILTERING_ALLOW_POPUPS);

	/* autobookmarks preferences */
	set_config_from_togglebutton (pd->autobookmarks_enabled_check,
				      CONF_BOOKMARKS_AB_ENABLED);
	set_config_from_spin_button (pd->autobookmarks_count_spin, 
				     CONF_BOOKMARKS_AB_COUNT);
	set_config_from_spin_button (pd->autobookmarks_shorten_spin,
				     CONF_BOOKMARKS_AB_SHORTEN);
	
	/* process downloading preferences */
	set_config_from_togglebutton (pd->download_autodownload,
				     CONF_DOWNLOADING_AUTO_DOWNLOAD);
	set_config_from_togglebutton (pd->download_ask_dir,
				     CONF_DOWNLOADING_ASK_DIR);
	set_config_from_editable (pd->download_dir_entry, 
				  CONF_DOWNLOADING_DIR);
        set_config_from_radiobuttongroup (pd->download_use_builtin, 
					  CONF_DOWNLOADING_DOWNLOADER);
	set_config_from_editable (pd->download_command_entry,
				  CONF_DOWNLOADING_EXTERNAL_COMMAND);
	set_config_from_togglebutton (pd->download_run_in_terminal,
				      CONF_DOWNLOADING_EXTERNAL_TERMINAL);
	/* process appearance preferences */
	set_config_from_togglebutton (pd->tabbed_enable_check, 
				      CONF_TABS_TABBED);
	set_config_from_togglebutton (pd->tabbed_popups_enable_check, 
				     CONF_TABS_TABBED_POPUPS);
	set_config_from_togglebutton (pd->tabbed_autojump_check, 
				     CONF_TABS_TABBED_AUTOJUMP);
	set_config_from_togglebutton (pd->tabbed_insert_new_tabs_check, 
				     CONF_TABS_TABBED_INSERT_NEW_TABS);
	set_config_from_optionmenu (pd->tabbed_position_optionmenu, 
				   CONF_TABS_TABBED_POSITION);
        set_config_from_radiobuttongroup (pd->tabbed_style_radiobutton1, 
                                   CONF_TABS_TABBED_SHORTEN_STYLE);
	set_config_from_togglebutton (pd->tabbed_always_show_check, 
				     CONF_TABS_TABBED_ALWAYS_SHOW);
	set_config_from_togglebutton (pd->tabbed_closebutton_check, 
				     CONF_TABS_TABBED_CLOSEBUTTON);
	set_config_from_togglebutton (pd->tabbed_closebutton_insensitive_check, 
				     CONF_TABS_TABBED_CLOSEBUTTON_INSENSITIVE);
	set_config_from_spin_button (pd->tabbed_shorten_spin, 
				     CONF_TABS_TABBED_SHORTEN_CHARS);
        set_config_from_spin_button (pd->tabbed_shorten_spin2, 
                                    CONF_TABS_TABBED_SHORTEN_POINTS);
	set_config_from_togglebutton (pd->tabbed_separator_check,
				     CONF_TABS_TABBED_SEPARATOR);
	set_config_from_togglebutton (pd->tabbed_vowel_check,
				      CONF_TABS_TABBED_VOWEL);
	set_config_from_togglebutton (pd->tabbed_prefix_check,
				      CONF_TABS_TABBED_PREFIX);
	set_config_from_togglebutton (pd->toolbar_share_main_toolbar, 
				      CONF_TOOLBAR_URL_LOCATION);
	set_config_from_optionmenu (pd->toolbar_style_optionmenu, 
				    CONF_TOOLBAR_STYLE);
	set_config_from_togglebutton (pd->use_own_fonts_checkbutton,
				      CONF_RENDERING_USE_OWN_FONTS);
 	set_config_from_optionmenu (pd->default_font_optionmenu,
				    CONF_RENDERING_DEFAULT_FONT);
	set_config_from_togglebutton (pd->show_toolbars_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_TOOLBARS);
	set_config_from_togglebutton (pd->show_bookmarks_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_BOOKMARKS);
	set_config_from_togglebutton (pd->show_menubar_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_MENUBAR);
	set_config_from_togglebutton (pd->show_statusbar_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_STATUSBAR);
	set_config_from_editable (pd->windows_title_entry, 
				  CONF_WINDOWS_TITLE);
	set_config_from_togglebutton (pd->fullscreen_stacking_checkbutton,
				     CONF_WINDOWS_FULLSCREEN_STACKING);
	set_config_from_editable (pd->spinner_dir_entry, 
				  CONF_TOOLBAR_SPINNER_DIR);
	set_config_from_editable (pd->theme_dir_entry,
				  CONF_TOOLBAR_THEME_DIR);
	set_config_from_color (pd->background_color, 
			       CONF_RENDERING_BG_COLOR);
	set_config_from_color (pd->text_color, 
			       CONF_RENDERING_TEXT_COLOR);
	set_config_from_color (pd->unvisited_link_color,
			       CONF_RENDERING_UNVISITED_LINKS);
	set_config_from_color (pd->visited_link_color, 
			       CONF_RENDERING_VISITED_LINKS);
	set_config_from_togglebutton (pd->use_own_colors_checkbutton,
				      CONF_RENDERING_USE_OWN_COLORS);
	set_config_from_togglebutton (pd->underline_links_checkbutton, 
				      CONF_RENDERING_UNDERLINE_LINKS);
	set_config_from_spin_button (pd->smart_bm_entry_width_spin,
				     CONF_BOOKMARKS_ENTRY_WIDTH);
	set_config_from_togglebutton (pd->smart_bm_goto_button_checkbutton, 
				      CONF_BOOKMARKS_GOTO_BUTTON);
	set_config_from_togglebutton (pd->smart_bm_history_checkbutton,
				      CONF_BOOKMARKS_HISTORY);
	set_config_from_color (pd->tabbed_loading_color, 
			       CONF_TABS_TABBED_LOADING_COLOR);
	set_config_from_color (pd->tabbed_new_color, 
			       CONF_TABS_TABBED_NEW_COLOR);

	/* process mouse options */
	set_config_from_togglebutton (pd->history_popup_checkbutton,
				      CONF_MOUSE_HISTORY_POPUP);
	set_config_from_optionmenu (pd->middle_button_optionmenu,
				    CONF_MOUSE_MMB_ACTION);
	set_config_from_optionmenu (pd->right_button_optionmenu,
				    CONF_MOUSE_RMB_ACTION);

	/* process completion preferences */
	set_config_from_togglebutton (pd->autocompletion_enabled_checkbutton,
				     CONF_HISTORY_AUTOCOMP_ENABLE);
	set_config_from_radiobuttongroup (pd->autocompletion_sort_whole,
					  CONF_COMPLETION_SORT);

        /* process handlers preferences */
	set_config_from_radiobuttongroup (pd->download_ftp_use_builtin,
				    CONF_DOWNLOADING_EXTERNAL_FTP);

	set_config_from_togglebutton (pd->use_external_source_viewer_check,
				      CONF_PROGRAMS_USE_EXTERNAL_SOURCE_VIEWER);
	set_config_from_editable (pd->external_source_viewer_entry, 
				  CONF_PROGRAMS_EXTERNAL_SOURCE_VIEWER);
 
        /* process advanced preferences */
	set_config_from_togglebutton (pd->proxy_use_checkbutton, 
				      CONF_NETWORK_PROXY_ENABLED);
	set_config_from_editable (pd->http_proxy_entry, 
				  CONF_NETWORK_HTTP_PROXY);
	set_config_from_editable (pd->ssl_proxy_entry, 
				  CONF_NETWORK_SSL_PROXY);
	set_config_from_editable (pd->ftp_proxy_entry, 
				  CONF_NETWORK_FTP_PROXY);
	set_config_from_editable (pd->no_proxies_for_entry, 
				  CONF_NETWORK_NO_PROXIES_FOR);
	set_config_from_spin_button (pd->http_proxy_port_spin,
				     CONF_NETWORK_HTTP_PROXY_PORT);
	set_config_from_spin_button (pd->ssl_proxy_port_spin, 
				     CONF_NETWORK_SSL_PROXY_PORT);
	set_config_from_spin_button (pd->ftp_proxy_port_spin, 
				     CONF_NETWORK_FTP_PROXY_PORT);
	set_config_from_optionmenu (pd->cache_compare_optionmenu, 
				    CONF_NETWORK_CACHE_COMPARE);
	set_config_from_int_entry (pd->disk_cache_entry, 
				   CONF_NETWORK_DISK_CACHE);
	set_config_from_int_entry (pd->mem_cache_entry, 
				   CONF_NETWORK_MEMORY_CACHE);
	set_config_from_togglebutton (pd->java_checkbutton, 
				      CONF_FILTERING_JAVA_ENABLED);
	set_config_from_togglebutton (pd->javascript_checkbutton,
				      CONF_FILTERING_JAVASCRIPT_ENABLED);
	set_config_from_editable (pd->user_agent_entry,
				  CONF_NETWORK_USER_AGENT);
	set_config_from_radiobuttongroup (pd->images_always_load, 
					  CONF_FILTERING_IMAGE_LOADING_TYPE);
	set_config_from_togglebutton (pd->statusbar_rewrite_check,
				      CONF_FILTERING_STATUSBAR_REWRITE);
	set_config_from_optionmenu (pd->crash_optionmenu, 
				    CONF_CRASH_RECOVERY);
	set_config_from_optionmenu (pd->autodetect_charset_optionmenu,
				    CONF_LANGUAGE_AUTODETECT_CHARSET);
	set_config_from_list (pd->lang_list, CONF_RENDERING_LANGUAGE);
	set_config_from_optionmenu (pd->http_version_optionmenu, 
				    CONF_NETWORK_HTTP_VERSION); 
	set_config_from_togglebutton (pd->proxy_keepalive_checkbutton, 
				      CONF_NETWORK_PROXY_KA);
	set_config_from_togglebutton (pd->warn_cookie_checkbutton, 
				      CONF_PERSISTENT_COOKIE_WARN);
	set_config_from_radiobuttongroup (pd->cookies_always_accept,
					  CONF_PERSISTENT_COOKIES_BEHAVIOR);
	set_config_from_togglebutton (pd->passwords_save_checkbutton,
				      CONF_PERSISTENT_PASSWORDS_SAVE);
	set_config_from_optionmenu (pd->wheel_nokey_action_optionmenu,
				    CONF_MOUSE_WHEEL_NOKEY_ACTION); 
	set_config_from_optionmenu (pd->wheel_ctrl_action_optionmenu,
				    CONF_MOUSE_WHEEL_CTRL_ACTION); 
	set_config_from_optionmenu (pd->wheel_shift_action_optionmenu,
				    CONF_MOUSE_WHEEL_SHIFT_ACTION);
	set_config_from_optionmenu (pd->wheel_alt_action_optionmenu,
				    CONF_MOUSE_WHEEL_ALT_ACTION);  
	set_config_from_spin_button (pd->wheel_nokey_step_spinbutton,
				     CONF_MOUSE_WHEEL_NOKEY_STEP);
	set_config_from_spin_button (pd->wheel_ctrl_step_spinbutton,
				     CONF_MOUSE_WHEEL_CTRL_STEP);
	set_config_from_spin_button (pd->wheel_shift_step_spinbutton,
				     CONF_MOUSE_WHEEL_SHIFT_STEP);
	set_config_from_spin_button (pd->wheel_alt_step_spinbutton,
				     CONF_MOUSE_WHEEL_ALT_STEP);
	set_config_from_togglebutton (pd->wheel_nokey_default_step_checkbutton,
				      CONF_MOUSE_WHEEL_NOKEY_DEF_STEP);
	set_config_from_togglebutton (pd->wheel_ctrl_default_step_checkbutton,
				      CONF_MOUSE_WHEEL_CTRL_DEF_STEP);
	set_config_from_togglebutton (pd->wheel_shift_default_step_checkbutton,
				      CONF_MOUSE_WHEEL_SHIFT_DEF_STEP);
	set_config_from_togglebutton (pd->wheel_alt_default_step_checkbutton,
				      CONF_MOUSE_WHEEL_ALT_DEF_STEP);
	set_config_from_optionmenu (pd->crash_behaviour_optionmenu,
				    CONF_CRASH_BEHAVIOUR);
	set_config_from_togglebutton (pd->autozoom_checkbutton,
				      CONF_HISTORY_AUTOZOOM);

	/*handle the special charset case */
	charset_title = gtk_entry_get_text 
		(GTK_ENTRY (pd->default_charset_entry));
	cur_charset = g_hash_table_lookup (charsets, 
					   charset_title);
	if (cur_charset!=NULL)
		eel_gconf_set_string (CONF_LANGUAGE_DEFAULT_CHARSET,
				      cur_charset);
	eel_gconf_set_string (CONF_LANGUAGE_DEFAULT_CHARSET_TITLE
				    , charset_title);

	/* process history preferences */
       	set_config_from_spin_button (pd->history_expire_spin, 
				     CONF_HISTORY_EXPIRE);

	preferences_mime_set_config_to_dialog (pd);

	/* save the toolbar state */
 	toolbar_editor_save (pd);
	/* save all these preferences to disk */
	preferences_save ();
}

static void
preferences_set_dialog_to_config (PreferencesDialog *pd)
{
	gint smart_bm_entry_width;

	/* fill general */
	set_editable_from_config (pd->startpage_entry,
				  CONF_GENERAL_HOMEPAGE);
	set_radiobuttongroup_from_config (pd->startup_show_homepage,
					  CONF_GENERAL_STARTPAGE_TYPE);
	set_radiobuttongroup_from_config (pd->newpage_show_homepage, 
					  CONF_GENERAL_NEWPAGE_TYPE);
	set_radiobuttongroup_from_config (pd->bookmarks_use_page_title, 
				   CONF_BOOKMARKS_TITLE);
	set_radiobuttongroup_from_config (pd->bookmarks_show_actions_top,
				   CONF_BOOKMARKS_XITEMS);
	set_togglebutton_from_config (pd->always_save_session_check,
				      CONF_GENERAL_ALWAYS_SAVE_SESSION);
	set_togglebutton_from_config (pd->favicons_enabled_check,
				     CONF_BOOKMARKS_FAVICONS_ENABLED);
	set_togglebutton_from_config (pd->allow_popups_checkbutton, 
				      CONF_FILTERING_ALLOW_POPUPS);

	/* autobookmarks preferences */
	set_togglebutton_from_config (pd->autobookmarks_enabled_check,
				     CONF_BOOKMARKS_AB_ENABLED);
	set_spin_button_from_config (pd->autobookmarks_count_spin, 
				     CONF_BOOKMARKS_AB_COUNT);
	set_spin_button_from_config (pd->autobookmarks_shorten_spin, 
				    CONF_BOOKMARKS_AB_SHORTEN);
	
	/* process downloading preferences */
	set_togglebutton_from_config (pd->download_autodownload,
				     CONF_DOWNLOADING_AUTO_DOWNLOAD);
	set_togglebutton_from_config (pd->download_ask_dir,
				     CONF_DOWNLOADING_ASK_DIR);
	set_editable_from_config (pd->download_dir_entry, 
				  CONF_DOWNLOADING_DIR);
        set_radiobuttongroup_from_config (pd->download_use_builtin, 
					  CONF_DOWNLOADING_DOWNLOADER);
	set_editable_from_config (pd->download_command_entry,
				  CONF_DOWNLOADING_EXTERNAL_COMMAND);
	set_togglebutton_from_config (pd->download_run_in_terminal,
				      CONF_DOWNLOADING_EXTERNAL_TERMINAL);
	
	/* process appearance preferences */
	set_togglebutton_from_config (pd->tabbed_enable_check, 
				      CONF_TABS_TABBED);
	set_togglebutton_from_config (pd->tabbed_popups_enable_check, 
				     CONF_TABS_TABBED_POPUPS);
	set_togglebutton_from_config (pd->tabbed_autojump_check, 
				     CONF_TABS_TABBED_AUTOJUMP);
	set_togglebutton_from_config (pd->tabbed_insert_new_tabs_check, 
				     CONF_TABS_TABBED_INSERT_NEW_TABS);
	set_optionmenu_from_config (pd->tabbed_position_optionmenu, 
				   CONF_TABS_TABBED_POSITION);
        set_radiobuttongroup_from_config (pd->tabbed_style_radiobutton1, 
                                   CONF_TABS_TABBED_SHORTEN_STYLE);
	set_togglebutton_from_config (pd->tabbed_always_show_check, 
				   CONF_TABS_TABBED_ALWAYS_SHOW);
	set_togglebutton_from_config (pd->tabbed_closebutton_check, 
				   CONF_TABS_TABBED_CLOSEBUTTON);
	set_togglebutton_from_config (pd->tabbed_closebutton_insensitive_check, 
				   CONF_TABS_TABBED_CLOSEBUTTON_INSENSITIVE);
	set_spin_button_from_config (pd->tabbed_shorten_spin,
				    CONF_TABS_TABBED_SHORTEN_CHARS);
        set_spin_button_from_config (pd->tabbed_shorten_spin2,
                                    CONF_TABS_TABBED_SHORTEN_POINTS);
	set_togglebutton_from_config (pd->tabbed_separator_check,
				    CONF_TABS_TABBED_SEPARATOR);
	set_togglebutton_from_config (pd->tabbed_vowel_check,
				    CONF_TABS_TABBED_VOWEL);
	set_togglebutton_from_config (pd->tabbed_prefix_check,
				    CONF_TABS_TABBED_PREFIX);
	set_togglebutton_from_config (pd->toolbar_share_main_toolbar, 
				    CONF_TOOLBAR_URL_LOCATION);
	set_optionmenu_from_config (pd->toolbar_style_optionmenu, 
				    CONF_TOOLBAR_STYLE);
	set_togglebutton_from_config (pd->use_own_fonts_checkbutton, 
				     CONF_RENDERING_USE_OWN_FONTS);
 	set_optionmenu_from_config (pd->default_font_optionmenu,
				   CONF_RENDERING_DEFAULT_FONT);
	set_togglebutton_from_config (pd->show_toolbars_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_TOOLBARS);
	set_togglebutton_from_config (pd->show_bookmarks_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_BOOKMARKS);
	set_togglebutton_from_config (pd->show_menubar_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_MENUBAR);
	set_togglebutton_from_config (pd->show_statusbar_in_fullscreen_check,
				     CONF_WINDOWS_FS_SHOW_STATUSBAR);
	set_editable_from_config (pd->windows_title_entry,
				 CONF_WINDOWS_TITLE);
	set_togglebutton_from_config (pd->fullscreen_stacking_checkbutton,
				     CONF_WINDOWS_FULLSCREEN_STACKING);
	set_editable_from_config (pd->spinner_dir_entry,
				 CONF_TOOLBAR_SPINNER_DIR);
	set_editable_from_config (pd->theme_dir_entry,
				  CONF_TOOLBAR_THEME_DIR);
	set_color_from_config (pd->background_color, 
			       CONF_RENDERING_BG_COLOR);
	set_color_from_config (pd->text_color, 
			       CONF_RENDERING_TEXT_COLOR);
	set_color_from_config (pd->unvisited_link_color, 
			       CONF_RENDERING_UNVISITED_LINKS);
	set_color_from_config (pd->visited_link_color, 
			       CONF_RENDERING_VISITED_LINKS);
	set_togglebutton_from_config (pd->use_own_colors_checkbutton,
				      CONF_RENDERING_USE_OWN_COLORS);
	set_togglebutton_from_config (pd->underline_links_checkbutton, 
				      CONF_RENDERING_UNDERLINE_LINKS);
	set_spin_button_from_config (pd->smart_bm_entry_width_spin,
				     CONF_BOOKMARKS_ENTRY_WIDTH);
	set_togglebutton_from_config (pd->smart_bm_goto_button_checkbutton,
				      CONF_BOOKMARKS_GOTO_BUTTON);
	set_togglebutton_from_config (pd->smart_bm_history_checkbutton,
				      CONF_BOOKMARKS_HISTORY);
	smart_bm_entry_width = gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (pd->smart_bm_entry_width_spin));
	gtk_widget_set_usize (pd->smart_bm_entry_width_spin,
			      smart_bm_entry_width, -2);
	set_color_from_config (pd->tabbed_loading_color, 
			       CONF_TABS_TABBED_LOADING_COLOR);
	set_color_from_config (pd->tabbed_new_color, 
			       CONF_TABS_TABBED_NEW_COLOR);

	/* process mouse options */
	set_togglebutton_from_config (pd->history_popup_checkbutton,
				      CONF_MOUSE_HISTORY_POPUP);
	set_optionmenu_from_config (pd->middle_button_optionmenu,
				    CONF_MOUSE_MMB_ACTION);
	set_optionmenu_from_config (pd->right_button_optionmenu,
				    CONF_MOUSE_RMB_ACTION);

	/* process completion preferences */
	set_togglebutton_from_config (pd->autocompletion_enabled_checkbutton,
				     CONF_HISTORY_AUTOCOMP_ENABLE);
	set_radiobuttongroup_from_config (pd->autocompletion_sort_whole,
				   CONF_COMPLETION_SORT);
	
        /* process handlers preferences */
	set_radiobuttongroup_from_config (pd->download_ftp_use_builtin, 
					  CONF_DOWNLOADING_EXTERNAL_FTP);

	set_togglebutton_from_config (pd->use_external_source_viewer_check, 
				      CONF_PROGRAMS_USE_EXTERNAL_SOURCE_VIEWER);
	set_editable_from_config (pd->external_source_viewer_entry, 
				  CONF_PROGRAMS_EXTERNAL_SOURCE_VIEWER);
 
        /* process advanced preferences */
	set_togglebutton_from_config (pd->proxy_use_checkbutton,
				     CONF_NETWORK_PROXY_ENABLED);
	set_editable_from_config (pd->http_proxy_entry, 
				  CONF_NETWORK_HTTP_PROXY);
	set_editable_from_config (pd->ssl_proxy_entry, 
				  CONF_NETWORK_SSL_PROXY);
	set_editable_from_config (pd->ftp_proxy_entry, 
				  CONF_NETWORK_FTP_PROXY);
	set_editable_from_config (pd->no_proxies_for_entry, 
				  CONF_NETWORK_NO_PROXIES_FOR);
	set_spin_button_from_config (pd->http_proxy_port_spin,
				    CONF_NETWORK_HTTP_PROXY_PORT);
	set_spin_button_from_config (pd->ssl_proxy_port_spin,
				    CONF_NETWORK_SSL_PROXY_PORT);
	set_spin_button_from_config (pd->ftp_proxy_port_spin,
				    CONF_NETWORK_FTP_PROXY_PORT);
	set_optionmenu_from_config (pd->http_version_optionmenu,
				   CONF_NETWORK_HTTP_VERSION);
	set_togglebutton_from_config (pd->proxy_keepalive_checkbutton,
				     CONF_NETWORK_PROXY_KA);
	set_optionmenu_from_config (pd->cache_compare_optionmenu,
				    CONF_NETWORK_CACHE_COMPARE);
	set_int_entry_from_config (pd->disk_cache_entry,
				   CONF_NETWORK_DISK_CACHE);
	set_int_entry_from_config (pd->mem_cache_entry, 
				   CONF_NETWORK_MEMORY_CACHE);
	set_togglebutton_from_config (pd->java_checkbutton, 
				      CONF_FILTERING_JAVA_ENABLED);
	set_togglebutton_from_config (pd->javascript_checkbutton,
				     CONF_FILTERING_JAVASCRIPT_ENABLED);
	set_editable_from_config (pd->user_agent_entry,
				  CONF_NETWORK_USER_AGENT);
	set_radiobuttongroup_from_config (pd->images_always_load,
					  CONF_FILTERING_IMAGE_LOADING_TYPE);
	set_togglebutton_from_config (pd->statusbar_rewrite_check,
				  CONF_FILTERING_STATUSBAR_REWRITE);
	set_optionmenu_from_config (pd->crash_optionmenu, 
				    CONF_CRASH_RECOVERY);
	set_optionmenu_from_config (pd->autodetect_charset_optionmenu,
				   CONF_LANGUAGE_AUTODETECT_CHARSET);
 	set_editable_from_config (pd->default_charset_entry,
				  CONF_LANGUAGE_DEFAULT_CHARSET_TITLE); 
 	set_list_from_config (pd->lang_list,
			      CONF_RENDERING_LANGUAGE); 
	set_togglebutton_from_config (pd->warn_cookie_checkbutton,
				     CONF_PERSISTENT_COOKIE_WARN);
	set_radiobuttongroup_from_config (pd->cookies_always_accept,
				   CONF_PERSISTENT_COOKIES_BEHAVIOR);
	set_togglebutton_from_config (pd->passwords_save_checkbutton,
				     CONF_PERSISTENT_PASSWORDS_SAVE);
	set_optionmenu_from_config (pd->wheel_nokey_action_optionmenu,
				   CONF_MOUSE_WHEEL_NOKEY_ACTION); 
	set_optionmenu_from_config (pd->wheel_ctrl_action_optionmenu,
				   CONF_MOUSE_WHEEL_CTRL_ACTION); 
	set_optionmenu_from_config (pd->wheel_shift_action_optionmenu,
				   CONF_MOUSE_WHEEL_SHIFT_ACTION);
	set_optionmenu_from_config (pd->wheel_alt_action_optionmenu,
				   CONF_MOUSE_WHEEL_ALT_ACTION);  
	set_spin_button_from_config (pd->wheel_nokey_step_spinbutton,
				    CONF_MOUSE_WHEEL_NOKEY_STEP);
	set_spin_button_from_config (pd->wheel_ctrl_step_spinbutton,
				    CONF_MOUSE_WHEEL_CTRL_STEP);
	set_spin_button_from_config (pd->wheel_shift_step_spinbutton,
				    CONF_MOUSE_WHEEL_SHIFT_STEP);
	set_spin_button_from_config (pd->wheel_alt_step_spinbutton,
				    CONF_MOUSE_WHEEL_ALT_STEP);
	set_togglebutton_from_config (pd->wheel_nokey_default_step_checkbutton,
				     CONF_MOUSE_WHEEL_NOKEY_DEF_STEP);
	set_togglebutton_from_config (pd->wheel_ctrl_default_step_checkbutton,
				     CONF_MOUSE_WHEEL_CTRL_DEF_STEP);
	set_togglebutton_from_config (pd->wheel_shift_default_step_checkbutton,
				     CONF_MOUSE_WHEEL_SHIFT_DEF_STEP);
	set_togglebutton_from_config (pd->wheel_alt_default_step_checkbutton,
				     CONF_MOUSE_WHEEL_ALT_DEF_STEP);
	set_optionmenu_from_config (pd->crash_behaviour_optionmenu,
				   CONF_CRASH_BEHAVIOUR);
	set_togglebutton_from_config (pd->autozoom_checkbutton,
				     CONF_HISTORY_AUTOZOOM);

	/* process history preferences */
       	set_spin_button_from_config (pd->history_expire_spin, 
				     CONF_HISTORY_EXPIRE);

	/* initialise toolbar editor */
 	toolbar_editor_init (pd);

	/* initialise theme selector */
	preferences_theme_selector_init (pd);

        /* process mime types */
	preferences_mime_set_dialog_to_config (pd);
}

static void
preferences_mime_set_config_to_dialog (PreferencesDialog *pd)
{
	int i;
	MimeItem *item;

	/* free up any existing list of MIME types */
	if (mime_types != NULL)
	{
		g_list_foreach (mime_types, (GFunc)mime_item_free, NULL);
		g_list_free (mime_types);
		mime_types = NULL;
	}

	for (i = 0; i < GTK_CLIST (pd->mime_list)->rows; i++)
	{
		item = gtk_clist_get_row_data (GTK_CLIST (pd->mime_list), i);
		mime_types = g_list_append (mime_types, 
					    mime_item_clone (item));
	}
}

static void
preferences_mime_set_dialog_to_config (PreferencesDialog *pd)
{
	gchar *text[5] = { NULL, NULL, NULL, NULL, NULL };
	GtkWidget *clist = pd->mime_list;
	gint row, i;
	GList *l;

	/* freeze and empty the list */
	for (i = 0; i < GTK_CLIST (clist)->rows; i++)
	{
		MimeItem *item = NULL;
		item = gtk_clist_get_row_data (GTK_CLIST (clist), i);
		mime_item_free (item);
	}
	gtk_clist_freeze (GTK_CLIST(clist));
	gtk_clist_clear (GTK_CLIST(clist));

	/* re-insert every element */
	for (l = mime_types; l != NULL; l = g_list_next(l))
	{
		MimeItem *m = (MimeItem *)(l->data);
		MimeItem *clone;

		/* insert the row */
		text[0] = g_strdup (m->name);
		switch (m->action) {
		case MIME_SAVE_TO_DISK:
			text[1] = _("Save to disk");
			break;
		case MIME_RUN_PROGRAM:  
			text[1] = _("Run with Helper App");
			break;
		case MIME_UNKNOWN:
			text[1] = _("Unknown");
			break;
		case MIME_IGNORE:
			text[1] = _("Ignored");
			break;
		case MIME_ASK_ACTION:
			text[1] = _("Ask the user");
			break;
		}
		text[2] = g_strdup(m->helperApp);
		if (m->alwaysUse)
			text[3] = _("True");
		else
			text[3] = _("False");

		row = gtk_clist_append (GTK_CLIST(clist), text);
		clone = mime_item_clone (m);
		gtk_clist_set_row_data (GTK_CLIST(clist), row, clone);
		/* gtk copies these and its safe to free them right after they 
		   are appended */
		g_free (text[0]);
		g_free (text[2]);
	}

	/* thaw the list -- make changes visible */
	gtk_clist_thaw (GTK_CLIST(clist));	
}

static void
preferences_lookup_widgets (PreferencesDialog *pd, GladeXML *gxml)
{
	WidgetLookup lookup_table[] =
	{
{ "notebook",                       &(pd->notebook)                          },
{ "sidebar",                        &(pd->sidebar)                           },
{ "themes_page",                    &(pd->themes_page)                       },
{ "spinner_iconlist",               &(pd->iconlist)                          },
{ "theme_list",                     &(pd->themelist)                         },
{ "theme_dir_entry",                &(pd->theme_dir_entry)                   },
{ "smart_bm_example_entry",         &(pd->smart_bm_example_entry)            },
{ "lang_entry",                     &(pd->lang_entry)                        },
{ "toolbar_style_optionmenu",       &(pd->toolbar_style_optionmenu)          },
{ "toolbar_share_main_toolbar",     &(pd->toolbar_share_main_toolbar)        },
{ "startpage_entry",                &(pd->startpage_entry)                   },
{ "startup_show_homepage",          &(pd->startup_show_homepage)             },
{ "newpage_show_homepage",          &(pd->newpage_show_homepage)             },
{ "always_save_session_check",      &(pd->always_save_session_check)         },
{ "bookmarks_use_page_title",       &(pd->bookmarks_use_page_title)          },
{ "bookmarks_show_actions_top",     &(pd->bookmarks_show_actions_top)        },
{ "favicons_enabled_check",         &(pd->favicons_enabled_check)            },
{ "autobookmarks_enabled_check",    &(pd->autobookmarks_enabled_check)       },
{ "autobookmarks_count_spin",       &(pd->autobookmarks_count_spin)          },
{ "autobookmarks_shorten_spin",     &(pd->autobookmarks_shorten_spin)        },
{ "smart_bm_entry_width_spin",      &(pd->smart_bm_entry_width_spin)         },
{ "smart_bm_goto_button_check",     &(pd->smart_bm_goto_button_checkbutton)  },
{ "smart_bm_history_check",         &(pd->smart_bm_history_checkbutton)      },
{ "java_checkbutton",               &(pd->java_checkbutton)                  },
{ "javascript_checkbutton",         &(pd->javascript_checkbutton)            },
{ "user_agent_entry",               &(pd->user_agent_entry)                  },

/* toolbars editor */
{ "available_clist",                &(pd->available_clist)                   },
{ "current_clist",                  &(pd->current_clist)                     },
{ "edit_toolbars_add_button",       &(pd->add_button)                        },
{ "edit_toolbars_remove_button",    &(pd->remove_button)                     },
{ "edit_toolbars_up_button",        &(pd->up_button)                         },
{ "edit_toolbars_down_button",      &(pd->down_button)                       },

/* mime page */
{ "mime_list",                      &(pd->mime_list)                         },
{ "mime_name_entry",                &(pd->mime_name_entry)                   },
{ "mime_save_to_disk",	            &(pd->mime_save_to_disk)		     },
{ "mime_helper_frame",	            &(pd->mime_helper_frame)		     },
{ "mime_helper_entry",	            &(pd->mime_helper_entry)		     },
{ "mime_always_use",	            &(pd->mime_always_use)		     },
{ "mime_terminal",	            &(pd->mime_terminal)		     },
{ "mime_url_helper",	            &(pd->mime_url_helper)		     },

/* downloader page */
{ "download_use_builtin",           &(pd->download_use_builtin)              },
{ "download_command_entry",         &(pd->download_command_entry)            },
{ "download_command_combo",         &(pd->download_command_combo)            },
{ "download_run_in_terminal",       &(pd->download_run_in_terminal)          },
{ "download_autodownload",          &(pd->download_autodownload)             },
{ "download_ask_dir",               &(pd->download_ask_dir)                  },
{ "download_dir_entry",             &(pd->download_dir_entry)                },
{ "download_ftp_use_builtin",       &(pd->download_ftp_use_builtin)          },

/* tabbed browsing page */
{ "tabbed_enable_check",            &(pd->tabbed_enable_check)               },
{ "tabbed_popups_enable_check",     &(pd->tabbed_popups_enable_check)        },
{ "tabbed_autojump_check",          &(pd->tabbed_autojump_check)             },
{ "tabbed_insert_new_tabs_check",   &(pd->tabbed_insert_new_tabs_check)      },
{ "tabbed_position_optionmenu",     &(pd->tabbed_position_optionmenu)        },
{ "radiobutton1", /* FIXME */       &(pd->tabbed_style_radiobutton1)         },
{ "tabbed_always_show_check",       &(pd->tabbed_always_show_check)          },
{ "tabbed_closebutton_check",       &(pd->tabbed_closebutton_check)          },
{ "tabbed_closebutton_insensitive_check",
				 &(pd->tabbed_closebutton_insensitive_check) },
{ "tabbed_shorten_spin",            &(pd->tabbed_shorten_spin)               },
{ "tabbed_shorten_spin2",/* FIXME */&(pd->tabbed_shorten_spin2)              },
{ "tabbed_separator_check",         &(pd->tabbed_separator_check)            },
{ "tabbed_vowel_check",             &(pd->tabbed_vowel_check)                },
{ "tabbed_prefix_check",            &(pd->tabbed_prefix_check)               },
{ "tabbed_loading_color",	    &(pd->tabbed_loading_color)		     },
{ "tabbed_new_color",		    &(pd->tabbed_new_color)		     },
{ "autocompletion_enabled_checkbutton", 
                                   &(pd->autocompletion_enabled_checkbutton) },
{ "autocompletion_sort_whole",      &(pd->autocompletion_sort_whole)         },
{ "http_proxy_entry",               &(pd->http_proxy_entry)                  },
{ "ssl_proxy_entry",                &(pd->ssl_proxy_entry)                   },
{ "ftp_proxy_entry",                &(pd->ftp_proxy_entry)                   },
{ "no_proxies_for_entry",           &(pd->no_proxies_for_entry)              },
{ "http_proxy_port_spin",           &(pd->http_proxy_port_spin)              },
{ "ssl_proxy_port_spin",            &(pd->ssl_proxy_port_spin)               },
{ "ftp_proxy_port_spin",            &(pd->ftp_proxy_port_spin)               },
{ "disk_cache_entry",               &(pd->disk_cache_entry)                  },
{ "mem_cache_entry",                &(pd->mem_cache_entry)                   },
{ "serif_font_fpicker",             &(pd->serif_font_fpicker)                },
{ "sansserif_font_fpicker",         &(pd->sansserif_font_fpicker)            },
{ "cursive_font_fpicker",           &(pd->cursive_font_fpicker)              },
{ "fantasy_font_fpicker",           &(pd->fantasy_font_fpicker)              },
{ "monospace_font_fpicker",         &(pd->monospace_font_fpicker)            },
{ "minimum_font_size_spin",         &(pd->minimum_font_size_spin)            },
{ "use_own_fonts_checkbutton",      &(pd->use_own_fonts_checkbutton)         },
{ "default_font_optionmenu",        &(pd->default_font_optionmenu)           },
{ "history_expire_spin",            &(pd->history_expire_spin)               },
{ "images_always_load",             &(pd->images_always_load)                },
{ "show_toolbars_in_fullscreen_check",
                                  &(pd->show_toolbars_in_fullscreen_check)   },
{ "show_bookmarks_in_fullscreen_check",
                                  &(pd->show_bookmarks_in_fullscreen_check)  },
{ "show_menubar_in_fullscreen_check", 
                                  &(pd->show_menubar_in_fullscreen_check)    },
{ "show_statusbar_in_fullscreen_check",
                                  &(pd->show_statusbar_in_fullscreen_check)  },
{ "use_external_source_viewer_check",
                                  &(pd->use_external_source_viewer_check)    },
{ "external_source_viewer_entry",   &(pd->external_source_viewer_entry)      },
{ "crash_optionmenu",               &(pd->crash_optionmenu)                  },
{ "allow_popups_checkbutton",       &(pd->allow_popups_checkbutton)          },
{ "windows_title_entry",            &(pd->windows_title_entry)               },
{ "fullscreen_stacking_checkbutton",&(pd->fullscreen_stacking_checkbutton)   },
{ "spinner_dir_entry",              &(pd->spinner_dir_entry)                 },
{ "autodetect_charset_optionmenu",  &(pd->autodetect_charset_optionmenu)     },
{ "default_charset_combo",          &(pd->default_charset_combo)             },
{ "default_charset_entry",          &(pd->default_charset_entry)             },
{ "lang_list",                      &(pd->lang_list)                         },
{ "background_color",               &(pd->background_color)                  },
{ "text_color",                     &(pd->text_color)                        },
{ "unvisited_link_color",           &(pd->unvisited_link_color)              },
{ "visited_link_color",             &(pd->visited_link_color)                },
{ "use_own_colors_checkbutton",     &(pd->use_own_colors_checkbutton)        },
{ "underline_links_checkbutton",    &(pd->underline_links_checkbutton)       },
{ "proxy_keepalive_checkbutton",    &(pd->proxy_keepalive_checkbutton)       },
{ "proxy_use_checkbutton",          &(pd->proxy_use_checkbutton)             },
{ "http_version_optionmenu",        &(pd->http_version_optionmenu)           },
{ "history_popup_checkbutton",      &(pd->history_popup_checkbutton)         },
{ "middle_button_optionmenu",       &(pd->middle_button_optionmenu)          },
{ "right_button_optionmenu",        &(pd->right_button_optionmenu)           },
{ "cookies_always_accept",          &(pd->cookies_always_accept)             },
{ "cookie_manager_button",          &(pd->cookie_manager_button)             },
{ "warn_cookie_checkbutton",        &(pd->warn_cookie_checkbutton)           },
{ "passwords_save_checkbutton",     &(pd->passwords_save_checkbutton)        },
{ "wheel_nokey_action_optionmenu",  &(pd->wheel_nokey_action_optionmenu)     },
{ "wheel_nokey_step_spinbutton",    &(pd->wheel_nokey_step_spinbutton)       },
{ "wheel_nokey_default_step_checkbutton",
                                 &(pd->wheel_nokey_default_step_checkbutton) },
{ "wheel_alt_action_optionmenu",    &(pd->wheel_alt_action_optionmenu)       },
{ "wheel_alt_step_spinbutton",      &(pd->wheel_alt_step_spinbutton)         },
{ "wheel_alt_default_step_checkbutton",
                                 &(pd->wheel_alt_default_step_checkbutton)   },
{ "wheel_ctrl_action_optionmenu",   &(pd->wheel_ctrl_action_optionmenu)      },
{ "wheel_ctrl_step_spinbutton",     &(pd->wheel_ctrl_step_spinbutton)        },
{ "wheel_ctrl_default_step_checkbutton",
                                 &(pd->wheel_ctrl_default_step_checkbutton)  },
{ "wheel_shift_action_optionmenu",  &(pd->wheel_shift_action_optionmenu)     },
{ "wheel_shift_step_spinbutton",    &(pd->wheel_shift_step_spinbutton)       },
{ "wheel_shift_default_step_checkbutton", 
                                 &(pd->wheel_shift_default_step_checkbutton) },
{ "crash_behaviour_optionmenu",     &(pd->crash_behaviour_optionmenu)        },
{ "links_drag_checkbutton",         &(pd->links_drag_checkbutton)            },
{ "cache_compare_optionmenu",       &(pd->cache_compare_optionmenu)          },
{ "statusbar_rewrite_check",        &(pd->statusbar_rewrite_check)           },
{ "lang_encoding",                  &(pd->lang_encoding_optionmenu)          },
{ "autozoom_checkbutton",           &(pd->autozoom_checkbutton)              },

{ NULL, NULL } /* terminator, must be last */
        };

        /* look them all up */
        lookup_widgets (gxml, lookup_table);
}

void
set_config_from_editable (GtkWidget *editable, gchar *config_name)
{
	gchar *value;
	gchar *old_value = NULL;

	old_value = eel_gconf_get_string(config_name);
	/* get all the text into a new string */	
	value = gtk_editable_get_chars (GTK_EDITABLE(editable), 0, -1);
	
	if ((old_value == NULL)||(strcmp(old_value, value) != 0)) {
		eel_gconf_set_string (config_name, value);
	}

	/* free the allocated strings */
	g_free (old_value);
	g_free (value);
}

static void
set_config_from_optionmenu (GtkWidget *optionmenu, gchar *config_name)
{
	int index = get_int_from_optionmenu (optionmenu);
	
	if (eel_gconf_get_integer(config_name)!=index)
		eel_gconf_set_integer (config_name, index);
}

void
set_config_from_radiobuttongroup (GtkWidget *radiobutton, gchar *config_name)
{
	gint index;

	/* get value from radio button group */
	index = gtk_radio_button_get (GTK_RADIO_BUTTON (radiobutton));

	if (eel_gconf_get_integer(config_name)!=index)
		eel_gconf_set_integer (config_name, index);
}

static void
set_config_from_spin_button (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* read the value as an integer */
	value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(spinbutton));

	if (eel_gconf_get_integer(config_name)!=value)
		eel_gconf_set_integer (config_name, value);
}

static void
set_config_from_togglebutton (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* read the value */
	value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(togglebutton));

	if (eel_gconf_get_boolean(config_name)!=value)
		eel_gconf_set_boolean (config_name, value);
}

static void
set_config_from_int_entry (GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the string */
	value_string = gtk_entry_get_text (GTK_ENTRY(entry));

	/* convert it into an integer */
	/* FIXME: atoi is a rubbish function to use, it
	 * doesn't detect any errors. We should fix this -
	 * perhaps by using spin buttons for these things? -- MattA */
	value = atoi(value_string);

	if (eel_gconf_get_integer(config_name)!=value)
		eel_gconf_set_integer (config_name, value);
}

static void
set_config_from_list (GtkWidget *list, gchar *config_name)
{
	GList *children = g_list_copy (GTK_LIST(list)->children);
	GSList *strings = NULL;

	while (children) 
	{
		GtkObject       *list_item;
		gchar           *item_data_string;
        
		list_item = GTK_OBJECT(children->data);
		item_data_string = gtk_object_get_data(list_item, "string");
		strings = g_slist_append(strings, item_data_string);
		children=children->next;
	}
	
	/* set the configuration value */
	eel_gconf_set_string_list (config_name, strings);
	/* FIXME: free those lists properly */
	g_list_free (children);
	g_slist_free (strings);
}

static void
set_config_from_color (GtkWidget *colorpicker, gchar *config_name)
{
	guint8 r, g, b, a;
	gchar color_string[9];

	/* get color values from color picker */
	gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (colorpicker), 
				   &r, &g, &b, &a);

	/* write into string (bounded size) */
	snprintf (color_string, 9, "#%02X%02X%02X", r, g, b);

	/* set the configuration value */
	eel_gconf_set_string (config_name, color_string);
}

void
set_editable_from_config (GtkWidget *editable, gchar *config_name)
{
	gchar *value;

	/* get the current value from the configuration space */
	value = eel_gconf_get_string (config_name);
	
	/* set this string value in the widget */
	if (value)
		gtk_entry_set_text(GTK_ENTRY(editable), value);

	/* free the allocated string */
	g_free (value);
}

static void
set_optionmenu_from_config (GtkWidget *optionmenu, gchar *config_name)
{
	gint index;

	/* get the current value from the configuration space */
	index = eel_gconf_get_integer (config_name);

	/* set this option value in the widget */
	gtk_option_menu_set_history (GTK_OPTION_MENU (optionmenu), index);
}

void
set_radiobuttongroup_from_config (GtkWidget *radiobutton, gchar *config_name)
{
	gint index;

        /* get the current value from the configuration space */
        index = eel_gconf_get_integer (config_name);

	/* set it (finds the group for us) */
	gtk_radio_button_set (GTK_RADIO_BUTTON (radiobutton), index);
}

static void
set_spin_button_from_config (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* get the current value from the configuration space */
	value = eel_gconf_get_integer (config_name);

	/* set this option value in the widget */
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinbutton), value);
}

static void
set_togglebutton_from_config (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* get the current value from the configuration space */
	value = eel_gconf_get_boolean (config_name);

	/* set this option value in the widget */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (togglebutton), value);
}

static void
set_int_entry_from_config (GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the current value from the configuration space */
	value = eel_gconf_get_integer (config_name);

	/* convert to a string */
	value_string = g_strdup_printf("%d", value);

	/* set this string value in the widget */
	gtk_entry_set_text(GTK_ENTRY(entry), value_string);

	/* free the allocated string */
	g_free (value_string);
}

static void
set_list_from_config (GtkWidget *list, gchar *config_name)
{
	GSList *strings, *tmp;
	GList *items = NULL;

	/* empty the list */
	gtk_list_clear_items (GTK_LIST (list), 0, -1);

	/* get the array of strings */
        strings = eel_gconf_get_string_list (config_name);
	tmp = strings;
	if (tmp == NULL) 
	{
		return;
	}
	while (tmp!=NULL)
	{	
		GtkWidget *list_item;
		char *label;
		
		label = tmp->data;	
		list_item = gtk_list_item_new_with_label (label);
		gtk_object_set_data (GTK_OBJECT (list_item), "string", label);
		items = g_list_append (items, list_item);
		gtk_widget_show (list_item);
		tmp = tmp->next;		
	}

	if (items != NULL)
	{
		gtk_list_append_items (GTK_LIST (list), items);
	}

	/* free the vector and list */
	g_slist_free (strings);
}

static void
set_color_from_config (GtkWidget *colorpicker, gchar *config_name)
{
	gchar *color_string;
	guint r, g, b;

	/* get the string from config */
	color_string = eel_gconf_get_string (config_name);

	if (color_string)
	{
		/* parse it and setup the color picker */
		sscanf (color_string, "#%2X%2X%2X", &r, &g, &b);
		gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (colorpicker), 
					   r, g, b, 0);
		/* free the string */
		g_free (color_string);
	}
}

void
mouse_wheel_controls_set_sensitive (PreferencesDialog *pd, WheelModKey key,
				    gboolean sensitive)
{
	GtkWidget *spinbutton = NULL, *checkbutton = NULL;
	
	switch (key)
	{
	case WHEEL_MOD_NOKEY:
		spinbutton = pd->wheel_nokey_step_spinbutton;
		checkbutton = pd->wheel_nokey_default_step_checkbutton;
		break;
	case WHEEL_MOD_ALT:
		spinbutton = pd->wheel_alt_step_spinbutton;
		checkbutton = pd->wheel_alt_default_step_checkbutton;
		break;
	case WHEEL_MOD_CTRL:
		spinbutton = pd->wheel_ctrl_step_spinbutton;
		checkbutton = pd->wheel_ctrl_default_step_checkbutton;
		break;
	case WHEEL_MOD_SHIFT:
		spinbutton = pd->wheel_shift_step_spinbutton;
		checkbutton = pd->wheel_shift_default_step_checkbutton;
		break;
	default:
		break;
	}

	if (spinbutton && checkbutton)
	{
		gtk_widget_set_sensitive (spinbutton, sensitive);
		gtk_widget_set_sensitive (checkbutton, sensitive);
	}
}

static void
prefs_dialog_save_geometry (PreferencesDialog *pd)
{
	state_save_window_state (GTK_WIDGET (pd->dialog)->window, 
				"prefs_dialog");
}

gboolean
preferences_galeon_used_by_gnome_for_ftp (void)
{
	gchar *handler;
	gboolean uses_galeon;

	handler = gnome_config_get_string ("/Gnome/URL Handlers/ftp-show");
	if (handler == NULL)
	{
		handler = gnome_config_get_string 
			("/Gnome/URL Handlers/default-show");
	}
	if (handler == NULL)
	{
		uses_galeon = FALSE;
	}
	else
	{
		uses_galeon = (strstr (handler, "galeon") != NULL);
		free (handler);
	}

	return uses_galeon;
}

void
prefs_fonts_init (PreferencesDialog *pd)
{
	int i;
	pd->fonts = g_new (LanguageFontInfo, LANG_ENC_NUM);

	for (i=0; i<LANG_ENC_NUM; i++)
	{
		pd->fonts[i].serif = NULL;
		pd->fonts[i].sansserif = NULL;
		pd->fonts[i].cursive = NULL;
		pd->fonts[i].fantasy = NULL;
		pd->fonts[i].monospace = NULL;
		pd->fonts[i].min_size = -1;
		pd->fonts[i].changed = FALSE;
	}
}

void
prefs_fonts_free (PreferencesDialog *pd, int enc)
{
	g_free (pd->fonts[enc].serif);
	g_free (pd->fonts[enc].sansserif);
	g_free (pd->fonts[enc].cursive);
	g_free (pd->fonts[enc].fantasy);
	g_free (pd->fonts[enc].monospace);
}
void
prefs_fonts_free_all (PreferencesDialog *pd)
{
	int i;

	for (i=0; i<LANG_ENC_NUM; i++)
	{
		if (pd->fonts[i].changed)
		{
			prefs_fonts_free (pd, i);
		}
	}

	g_free (pd->fonts);
}

void
prefs_fonts_save (PreferencesDialog *pd)
{
	int i;

	for (i=0; i<LANG_ENC_NUM; i++)
	{
		if (pd->fonts[i].changed)
		{
			mozilla_prefs_set_fonts (&pd->fonts[i], i);
		}
	}
}

void
prefs_fonts_language_init (PreferencesDialog *pd)
{
	int i = get_int_from_optionmenu (pd->lang_encoding_optionmenu);

	if (!pd->fonts[i].changed)
	{
		mozilla_prefs_get_fonts (&pd->fonts[i], i);
	}
		
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->serif_font_fpicker),
		 pd->fonts[i].serif);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->sansserif_font_fpicker),
		 pd->fonts[i].sansserif);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->cursive_font_fpicker),
		 pd->fonts[i].cursive);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->fantasy_font_fpicker),
		 pd->fonts[i].fantasy);
	gnome_font_picker_set_font_name 
		(GNOME_FONT_PICKER(pd->monospace_font_fpicker),
		 pd->fonts[i].monospace);
	gtk_spin_button_set_value 
		(GTK_SPIN_BUTTON(pd->minimum_font_size_spin),
		 pd->fonts[i].min_size);

	pd->fonts[i].changed = TRUE;
	pd->last_language = i;
}

void
prefs_fonts_remember (PreferencesDialog *pd)
{
	int i = pd->last_language;

	if (pd->fonts[i].changed)
	{
		prefs_fonts_free (pd, i);
	}

	pd->fonts[i].serif = g_strdup(gnome_font_picker_get_font_name 
		(GNOME_FONT_PICKER(pd->serif_font_fpicker)));
	pd->fonts[i].sansserif = g_strdup(gnome_font_picker_get_font_name 
		(GNOME_FONT_PICKER(pd->sansserif_font_fpicker)));
	pd->fonts[i].cursive = g_strdup(gnome_font_picker_get_font_name 
		(GNOME_FONT_PICKER(pd->cursive_font_fpicker)));
	pd->fonts[i].fantasy = g_strdup(gnome_font_picker_get_font_name 
		(GNOME_FONT_PICKER(pd->fantasy_font_fpicker)));
	pd->fonts[i].monospace = g_strdup(gnome_font_picker_get_font_name 
		(GNOME_FONT_PICKER(pd->monospace_font_fpicker)));
	pd->fonts[i].min_size = gtk_spin_button_get_value_as_int 
		(GTK_SPIN_BUTTON(pd->minimum_font_size_spin));

	pd->fonts[i].changed = TRUE;
}

static 
int get_int_from_optionmenu (GtkWidget *optionmenu)
{
	GtkWidget *menu;
	GList *list;
	gpointer item;
	gint index;

	/* extract the selection */
	menu = GTK_OPTION_MENU(optionmenu)->menu;
	list = GTK_MENU_SHELL(menu)->children;
	item = gtk_menu_get_active (GTK_MENU(menu));
	index = g_list_index (list, item);
	
	return index;
}

/**
 * preferences_theme_selector_init: Initialize theme selector
 */
static void
preferences_theme_selector_init (PreferencesDialog *pd)
{
	GtkCList *list = GTK_CLIST (pd->themelist);
	gchar *home_theme_dir, *home_nautilus_theme_dir;
	gchar *row[1];
	gchar *pref_theme_path = NULL;

	gtk_clist_freeze (list);

	gtk_clist_set_sort_column (list, 0);
	gtk_clist_set_sort_type (list, GTK_SORT_ASCENDING);

	/* clear the list out */
	free_theme_selector (list);
	gtk_clist_clear (list);

	/* fill list */
	row[0] = g_strdup (_("GNOME default"));
	gtk_clist_append (list, row);
	g_free (row[0]);
	gtk_clist_set_row_data (list, 0, g_strdup (""));

	row[0] = g_strdup (_("Nautilus theme"));
	gtk_clist_append (list, row);
	g_free (row[0]);
	gtk_clist_set_row_data (list, 1, 
				g_strdup ("NAUTILUS_THEME"));

	/* select default row */
	gtk_clist_select_row (list, 0, 0);

	/* select theme */
	pref_theme_path = eel_gconf_get_string (CONF_TOOLBAR_THEME_DIR);
	if (!pref_theme_path || strcmp (pref_theme_path, "") == 0)
	{
		gtk_clist_select_row (list, 0, 0);
	}
	if (pref_theme_path && strcmp (pref_theme_path, "NAUTILUS_THEME") == 0)
	{
		gtk_clist_select_row (list, 1, 0);
	}
	if (pref_theme_path) g_free (pref_theme_path);

	/* load all the galeon themes */
	theme_selector_fill_themelist_from_dir (GTK_CLIST (pd->themelist),
		SHARE_DIR "/themes");
	home_theme_dir =
		g_strconcat (g_get_home_dir (), "/.galeon/themes", NULL);
	theme_selector_fill_themelist_from_dir (GTK_CLIST (pd->themelist),
		home_theme_dir);
	g_free (home_theme_dir);

	/* load all the nautilus themes */
	theme_selector_fill_themelist_from_dir (GTK_CLIST (pd->themelist),
		SHARE_DIR "/../pixmaps/nautilus");
	home_nautilus_theme_dir =
		g_strconcat (g_get_home_dir (), "/.nautilus/themes", NULL);
	theme_selector_fill_themelist_from_dir (GTK_CLIST (pd->themelist),
		home_nautilus_theme_dir);
	g_free (home_nautilus_theme_dir);

	gtk_clist_sort (list);
	gtk_clist_thaw (list);
}

/**
 * theme_selector_fill_themelist_from_dir: Fill themelist with available themes
 * starting from base
 */
static void
theme_selector_fill_themelist_from_dir (GtkCList *theme_list, gchar *base)
{
	gchar *icon;
	gchar *pref_theme_path; 	
	GList *node;
	GList *list = NULL;
	GnomeVFSResult rc;

	pref_theme_path = eel_gconf_get_string (CONF_TOOLBAR_THEME_DIR);	

	rc = gnome_vfs_directory_list_load
		(&list, base, (GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
			       GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE |
			       GNOME_VFS_FILE_INFO_FOLLOW_LINKS), NULL);

	for (node = list; node; node = node->next)
	{
		GnomeVFSFileInfo *info;
		gchar *path;

		info = node->data;
		if (info->name[0] == '.')
			continue;
		if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)
			continue;

		path = g_concat_dir_and_file (base, info->name);

		/* Check to see if there is a Refresh.png */
		icon = g_concat_dir_and_file (path, "Refresh.png");
		if (g_file_exists (icon))
		{
			gchar *row[1] = { g_basename (path) };
			gint index = gtk_clist_append (theme_list, row);
			gtk_clist_set_row_data (theme_list, index,
						g_strdup (path));

			/* Select the icon configured in prefs */
			if (pref_theme_path && 
			    strcmp (pref_theme_path, path) == 0)
			{
				gtk_clist_select_row (theme_list, index, 0);
			}
		}

		g_free (path);
		g_free (icon);
	}

	gnome_vfs_file_info_list_free (list);
	g_free (pref_theme_path);
}

/**
 * free_themes_list: Free any existing themes list.
 */
static void
free_theme_selector (GtkCList *list)
{
	gint i;

	for (i = 0; i < list->rows; i++)
		g_free (gtk_clist_get_row_data (list, i));
}
