/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla_prefs.h"
#include "mozilla_notifiers.h"
#include "mozilla_i18n.h"
#include "mozilla.h"
#include "prefs.h"

#include <string.h>
#include <stdlib.h>
#include <libgnome/gnome-defs.h>
#include <locale.h>
#include <libgnome/gnome-i18n.h>


void mozilla_prefs_set_default_font (void);
static int
mozilla_prefs_set_font (gchar *name, gchar *key, int lang_enc);
static void
mozilla_prefs_set_font_size (gchar *key, int size, int lang_enc);
static int
mozilla_prefs_get_font_size (gchar *key, int lang_enc);
static gchar*
mozilla_prefs_get_font (gchar *name_key, gchar *size_key, int lang_enc);

extern const gchar *lang_encode_name[LANG_ENC_NUM];
extern const gchar *lang_encode_item[LANG_ENC_NUM];

/**
 * mozilla_prefs_set: set mozilla preferences
 */
void
mozilla_prefs_set (void)
{
	mozilla_notifiers_set_defaults();

	mozilla_preference_set_boolean ("mozilla.widget.raise-on-setfocus",
					FALSE);
	mozilla_preference_set_boolean ("browser.display.use_system_colors",
					FALSE);

	/* set default search engine */
	mozilla_preference_set ("keyword.URL",_("http://www.google.com/search?q="));
	mozilla_preference_set_boolean ("security.checkloaduri", FALSE); 

	mozilla_save_prefs ();
}

/**
 * mozilla_prefs_set_fonts: set fonts preferences
 */
void 
mozilla_prefs_set_fonts (LanguageFontInfo *info, int lang_enc)
{
	int variable_size, fixed_size;

	g_return_if_fail (info != NULL);

	/* set serif font */
	variable_size = mozilla_prefs_set_font 
		(info->serif, "font.name.serif", 
		lang_enc);

	/* set sans serif font */
	mozilla_prefs_set_font 
		(info->sansserif, "font.name.sans-serif", 
		 lang_enc);

	/* set cursive font */
	mozilla_prefs_set_font 
		(info->cursive, "font.name.cursive", 
		 lang_enc);

	/* set fantasy font */
	mozilla_prefs_set_font 
		(info->fantasy, "font.name.fantasy", 
		 lang_enc);

	/* set monospace font */
	fixed_size = mozilla_prefs_set_font 
		(info->monospace, "font.name.monospace", 
		 lang_enc);

	/* set font size */
	mozilla_prefs_set_font_size ("font.size.variable", 
				     variable_size, lang_enc);
	mozilla_prefs_set_font_size ("font.size.fixed",
				    fixed_size, lang_enc);
	mozilla_prefs_set_font_size ("font.min-size.variable", 
				    info->min_size, lang_enc);
	mozilla_prefs_set_font_size ("font.min-size.fixed", 
				    info->min_size, lang_enc);
}

/**
 * mozilla_prefs_get_fonts: get fonts preferences
 */
void 
mozilla_prefs_get_fonts (LanguageFontInfo *info, int lang_enc)
{
	g_return_if_fail (info != NULL);

	/* get serif font */
	info->serif = mozilla_prefs_get_font 
		("font.name.serif","font.size.variable",
		 lang_enc);

	/* get sans serif font */
	info->sansserif = mozilla_prefs_get_font 
		("font.name.sans-serif","font.size.variable", 
		 lang_enc);

	/* get cursive font */
	info->cursive = mozilla_prefs_get_font
		("font.name.cursive", "font.size.variable", 
		 lang_enc);

	/* get fantasy font */
	info->fantasy = mozilla_prefs_get_font 
		("font.name.fantasy", "font.size.variable", 
		 lang_enc);

	/* get monospace font */
	info->monospace = mozilla_prefs_get_font 
		("font.name.monospace", "font.size.fixed", 
		 lang_enc);

	/* get font size */
	info->min_size = mozilla_prefs_get_font_size 
		("font.min-size.variable", lang_enc);
}

static int
mozilla_prefs_get_font_size (gchar *key, int lang_enc)
{
	gchar key_buf[128];

	g_snprintf (key_buf, 128, "%s.%s",
		    key, lang_encode_item [lang_enc]);
	return mozilla_preference_get_int (key_buf);
}

static gchar*
mozilla_prefs_get_font (gchar *name_key, gchar *size_key, int lang_enc)
{
	gchar key_buf[128];
	gchar *font_name;
	gchar **font;
	gchar *result;
	int font_size;

	g_snprintf (key_buf, 128, "%s.%s",
		    name_key, lang_encode_item [lang_enc]);
	font_name = mozilla_preference_get (key_buf);
	g_snprintf (key_buf, 128, "%s.%s",
		    size_key, lang_encode_item [lang_enc]);
	font_size = mozilla_preference_get_int (key_buf);

	if (!font_name)
	{
		font_name = g_strdup ("*-*-*-*");
	}

	font = g_strsplit (font_name,"-",-1);
	result = g_strdup_printf("-%s-%s-*-*-*-*-%d-*-*-*-*-*-%s-%s",
				 font[0], font[1], font_size,
				 font[2], font [3]);

	g_strfreev (font);
	g_free (font_name);

	return result;
}

static int
mozilla_prefs_set_font (gchar *name, gchar *key, int lang_enc)
{
	gchar *font_name;
	gchar **font;
	gchar key_buf[128];
	int size;

	font = g_strsplit (name,"-",-1);
	font_name = g_strconcat(font[1],"-",font[2],"-",
				font[13],"-",font[14], NULL);
	g_snprintf (key_buf, 128, "%s.%s",
		    key, lang_encode_item [lang_enc]);

	if (strcmp(font_name, "*-*-*-*")!=0)
	{
		mozilla_preference_set (key_buf, font_name);
	}

	size = atoi (font[7]);

	g_strfreev (font);
	g_free (font_name);

	return size;
}

static void
mozilla_prefs_set_font_size (gchar *key, int size, int lang_enc)
{
	gchar key_buf[128];
	g_snprintf (key_buf, 128, "%s.%s",
		    key, lang_encode_item [lang_enc]);
	mozilla_preference_set_int (key_buf, size);
}

