/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozcallbacks.h"
#include "misc.h"
#include "eel-gconf-extensions.h"

#include <stdlib.h>
#include <string.h>
#include <libgnome/gnome-i18n.h>

#include "nsIAboutModule.h"
#include "nsIIOService.h"
#include "nsIServiceManager.h"
#include "nsIFactory.h"
#include "nsXPComFactory.h"
#include "nsCOMPtr.h"
#include "nsIURI.h"
#include "nsNetCID.h"
#include "nsIStorageStream.h"
#include "nsNetUtil.h"

/* set this to use a file in /tmp rather than generating on the fly  */
#define USE_INTERMEDIATE_FILE 0

/* output stream type */
#if USE_INTERMEDIATE_FILE
typedef FILE OutStream;
#else
typedef nsIOutputStream OutStream;
#endif

/* local function prototypes */
static void render_from_path (OutStream *stream, const char *path);
static void render_from_item (OutStream *stream, BookmarkItem *b,
					const gchar *prefix, gint depth);
static void render_from_list (OutStream *stream, GList *l,
					const gchar *prefix, gint depth);
static void render_url (OutStream *stream, BookmarkItem *b);
static void render_search_form (OutStream *stream, BookmarkItem *b);

/* hackish shorthand to render HTML into the browser */
#if USE_INTERMEDIATE_FILE
#define RENDER(str) \
        fwrite (str, strlen (str), 1, stream);
#else
static PRUint32 bytesWriten;
#define RENDER(str) \
        stream->Write (str, strlen(str), &bytesWriten);
#endif

static NS_DEFINE_CID(kSimpleURICID,     NS_SIMPLEURI_CID);

class GMyportalProtocolHandler : public nsIProtocolHandler,
				 public nsIAboutModule
{
  public:
	NS_DECL_ISUPPORTS
	NS_DECL_NSIPROTOCOLHANDLER

	GMyportalProtocolHandler (void);
	virtual ~GMyportalProtocolHandler();

	/* additional members */
	NS_METHOD CreateMyportalPage (const gchar *path);
	nsCOMPtr<nsIChannel> mChannel;
	nsCOMPtr<nsIURI> mURI;
};

/* Implementation file */
NS_IMPL_ISUPPORTS2 (GMyportalProtocolHandler, nsIProtocolHandler, nsIAboutModule)

GMyportalProtocolHandler::GMyportalProtocolHandler (void)
{
	NS_INIT_ISUPPORTS();
}

GMyportalProtocolHandler::~GMyportalProtocolHandler()
{
	/* destructor code */
}

/* readonly attribute string scheme; */
NS_IMETHODIMP GMyportalProtocolHandler::GetScheme(char * *aScheme)
{
	nsresult rv = NS_OK;
	if (aScheme)
		*aScheme = PL_strdup("myportal");
	else
		rv = NS_ERROR_NULL_POINTER;
	return rv;
}

/* readonly attribute long defaultPort; */
NS_IMETHODIMP GMyportalProtocolHandler::GetDefaultPort(PRInt32 *aDefaultPort)
{
	nsresult rv = NS_OK;
	if (aDefaultPort)
		*aDefaultPort = -1;
	else
		rv = NS_ERROR_NULL_POINTER;
	return rv;
}

#if MOZILLA_VERSION > VERSION3(0,9,3)

/* readonly attribute short URIType; */
NS_IMETHODIMP GMyportalProtocolHandler::GetURIType(PRInt16 *aURIType)
{
	if (aURIType)
		*aURIType = nsIProtocolHandler::URI_STD;
	else
		return NS_ERROR_NULL_POINTER;
	return NS_OK;
} 

#endif

/* nsIURI newURI (in string aSpec, in nsIURI aBaseURI); */
NS_IMETHODIMP GMyportalProtocolHandler::NewURI(const char *aSpec,
					       nsIURI *aBaseURI,
					       nsIURI **_retval)
{
	nsresult rv = NS_OK;
	nsCOMPtr <nsIURI> newUri;

	rv = nsComponentManager::CreateInstance(kSimpleURICID, NULL,
						NS_GET_IID(nsIURI),
						getter_AddRefs(newUri));

        if (NS_SUCCEEDED(rv)) 
        {
		newUri->SetSpec(aSpec);
		rv = newUri->QueryInterface(NS_GET_IID(nsIURI),
					    (void **) _retval);
        }
	return rv;
}

/* nsIChannel newChannel (in nsIURI aURI); */
NS_IMETHODIMP GMyportalProtocolHandler::NewChannel(nsIURI *aURI,
					      nsIChannel **_retval)
{
	nsresult rv;
	PRBool isabout;
	mURI = aURI;
	rv =  aURI->SchemeIs ("about", &isabout);
	if (NS_FAILED(rv)) return rv;
	if (isabout)
		rv = CreateMyportalPage ("/");
	else
	{
		char *path;
		rv = aURI->GetPath (&path);
		if (NS_FAILED(rv)) return rv;
		rv = CreateMyportalPage (path);
		nsMemory::Free (path);
	}
	NS_IF_ADDREF (*_retval = mChannel);
	return rv;
}

/* boolean allowPort (in long port, in string scheme); */
NS_IMETHODIMP GMyportalProtocolHandler::AllowPort(PRInt32 port, const char *scheme,
					     PRBool *_retval)
{
	*_retval = PR_FALSE;
	return NS_OK;
}

NS_METHOD GMyportalProtocolHandler::CreateMyportalPage (const gchar *path)
{
	nsresult rv;
	char *charset;

	/* check bookmarks are loaded */
	g_return_val_if_fail (bookmarks_root != NULL,NS_ERROR_FAILURE);
	
	/* get the stylesheet filename */
	gchar *stylesheet_filename = user_file ("myportal.css", FALSE);
	
	/* open the rendering stream */
#if USE_INTERMEDIATE_FILE
	gchar *filename = g_strconcat (g_get_home_dir (),
				       "/.galeon/myportal.html",
				       NULL);
	gchar *myportalURI = g_strconcat ("file://",filename, NULL);
	FILE *stream = fopen (filename, "w");
	if (stream == NULL)
	{
		g_warning ("unable to create `%s'", filename);
		g_free (stylesheet_filename);
		g_free (filename);
		return NS_ERROR_FAILURE;
	}
#else
	nsCOMPtr<nsIStorageStream> sStream;
	nsCOMPtr<nsIOutputStream> stream;

	rv = NS_NewStorageStream(256, (PRUint32)-1, getter_AddRefs(sStream));
	if (NS_FAILED(rv)) return rv;

	rv = sStream->GetOutputStream(0, getter_AddRefs(stream));
	if (NS_FAILED(rv)) return rv;

#endif

	/* render the complete portal */
	gchar full_path[PATH_MAX];
	RENDER ("<html><head>\n");
	RENDER ("<link rel=\"stylesheet\" href=\"file:");
	RENDER (realpath (stylesheet_filename, full_path));
	RENDER ("\" type=\"text/css\">\n");
	RENDER ("<title>");
	RENDER (_("My Portal"));
	RENDER ("</title>");

	charset = eel_gconf_get_string(CONF_LANGUAGE_DEFAULT_CHARSET);
	if (charset)
	{
		RENDER ("<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=");
		RENDER (eel_gconf_get_string(CONF_LANGUAGE_DEFAULT_CHARSET));
		RENDER ("\">");
	}

	RENDER("</head>\n");
	RENDER ("<body>\n");
	RENDER ("<a href=\"" GALEON_HOMEPAGE_URL "\">"
	       "<img class=\"logo\" src=\"file:" SHARE_DIR
	       "/logo.png\" alt=\"\"></a>");
	render_from_path (stream, path);
	RENDER ("</body></html>\n");

	/* finish the rendering */
#if USE_INTERMEDIATE_FILE
	fclose (stream);
	g_free (filename);

	static NS_DEFINE_CID(kIOServiceCID, NS_IOSERVICE_CID);
	nsCOMPtr<nsIIOService> ioService = do_GetService (kIOServiceCID, &rv);
	if (NS_FAILED(rv) || !ioService) return rv;
	rv = ioService->NewChannel(myportalURI, nsnull,
				   getter_AddRefs(mChannel));

	g_free (myportalURI);
#else
	nsCOMPtr<nsIInputStream> iStream;
	PRUint32 size;  
    
	rv = sStream->GetLength(&size);
	if (NS_FAILED(rv)) return rv;

	rv = sStream->NewInputStream(0, getter_AddRefs(iStream));
	if (NS_FAILED(rv)) return rv;

	rv = NS_NewInputStreamChannel(getter_AddRefs(mChannel), mURI,
				      iStream, "text/html", size);
	if (NS_FAILED(rv)) return rv;
#endif
	g_free (stylesheet_filename);
	return rv;
}

NS_DEF_FACTORY (GMyportalProtocolHandler, GMyportalProtocolHandler);

/**
 * NS_NewMyportalProtocolHandlerFactory:
 */ 
nsresult NS_NewMyportalHandlerFactory(nsIFactory** aFactory)
{
	NS_ENSURE_ARG_POINTER(aFactory);
	*aFactory = nsnull;

	nsGMyportalProtocolHandlerFactory *result = new nsGMyportalProtocolHandlerFactory;
	if (result == NULL)
	{
		return NS_ERROR_OUT_OF_MEMORY;
	}
    
	NS_ADDREF(result);
	*aFactory = result;

	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////

static gint 
bookmark_name_compare (const BookmarkItem *a, const gchar *name)
{
	if (a->name == NULL)
		return -1;
	gchar *stripped = strip_uline_accel (a->name);
	int rv = strcmp(stripped, name);
	g_free (stripped);
	return rv;
}

static void
render_from_path (OutStream *stream, const char *path)
{
	BookmarkItem *root = bookmarks_root;
	GString *clippedpath = g_string_new (path);
	GString *prefix = g_string_new ("myportal://");
	gboolean blockquote_started = FALSE;
	gchar *anch = strchr (path, '#');
	if (anch)
		g_string_truncate (clippedpath, anch-path);
	gchar **paths = g_strsplit (clippedpath->str, "/", 0);

	for (int i=0; paths[i] != NULL; ++i)
	{
		if (!*paths[i]) //ignore empty components
			continue;
		gchar *name = unescape_hexed_string (paths[i]);
		GList *match = g_list_find_custom (root->list, name,
					(GCompareFunc)bookmark_name_compare);
		g_free (name);
		if (match == NULL)
		{
			gchar *buf = g_strdup_printf(_("%s not found\n"), 
						      clippedpath->str);
			RENDER (buf);
			g_free (buf);
			return;
		}

		if (!blockquote_started)
		{
			RENDER ("<blockquote>\n");
			RENDER ("<strong>");
			blockquote_started = TRUE;
		}
		else
			RENDER ("/");
		gchar *strippedname = strip_uline_accel (root->name);
		RENDER ("<a href=\"");
		RENDER (prefix->str);
		RENDER ("\">");
		RENDER (strippedname);
		g_free (strippedname);
		RENDER ("</a>");
		
		g_string_sprintfa (prefix, "%s/", paths[i]);
		root = (BookmarkItem*)match->data;
	}
	g_strfreev (paths);
	if (blockquote_started)
		RENDER ("</strong>\n");
	render_from_item (stream, root, prefix->str, 0);
	g_string_free (prefix, TRUE);
	g_string_free (clippedpath, TRUE);
	if (blockquote_started)
		RENDER ("</blockquote>\n");
}

/**
 * render_from_item: render a single bookmark item
 */
static void
render_from_item (OutStream *stream, BookmarkItem *b, const gchar *prefix, 
		  gint depth)
{
	gchar *strippedname;
	gchar *escapedname, *newprefix, *id;
	/* could be the case if there are no bookmarks */
	if (b == NULL)
		return;

	/* otherwise do the appropriate thing */
	switch (b->type)
	{
	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		RENDER ("<blockquote>\n");
		strippedname = strip_uline_accel (b->name);
		RENDER ("<a class=\"heading\" ");
		id = g_strdup_printf ("%li", b->id);
		if (!b->alias_of)
		{
			RENDER ("name=\"");
			RENDER (id);
			RENDER ("\" ");
		}
		RENDER ("href=\"");
		if (depth == 0)
		{
			newprefix = g_strdup (prefix);
		}
		else
		{
			escapedname = escape_path  (strippedname);
			newprefix = g_strconcat (prefix, escapedname, "/", NULL);
			g_free(escapedname);
		}
		RENDER (newprefix);
		RENDER ("\">");
		RENDER ("<strong>");
		RENDER (strippedname);
		RENDER ("</strong>");
		g_free (strippedname);
		RENDER ("</a>");
		if (b->alias_of)
		{
		/* we point to the root portal since 1) the real bm might not
		 * be in this same level we are viewing, and 2) relative
		 * anchors don't seem to work with myportal://  ;) */
			RENDER (" -> <a href=\"myportal://#");
			RENDER (id);
			RENDER ("\">#");
			RENDER (id);
			RENDER ("</a>");
		}
		g_free (id);
		if (!b->alias_of || depth==0)
			render_from_list (stream, b->list, newprefix, depth+1);
		g_free (newprefix);
		RENDER ("</blockquote>\n");
		break;
		
	case BM_SITE:
		if (strstr (b->url, "%s"))
		{
			render_search_form (stream, b);
		}
		else
		{
			render_url (stream, b);
		}
		break;
		
	case BM_SEPARATOR:
		RENDER ("<br>\n");
		break;
	}
}

/**
 * render_from_list: render a list of bookmark items
 */
static void
render_from_list (OutStream *stream, GList *l, const gchar *prefix, gint depth)
{
	for (; l != NULL; l = l->next)
	{
		render_from_item (stream, (BookmarkItem *)(l->data), prefix,
				  depth);
	}
}

/**
 * render_url: render a bookmark url
 */
static void
render_url (OutStream *stream, BookmarkItem *b)
{

	RENDER ("<a href=\"");

	/* If url has no preceding "http://" or "ftp://" presume "http://" */
	if (!strstr (b->url, "://") && !strstr (b->url, "javascript:")) {
		RENDER ("http://");
	}
	
	RENDER (b->url);
	RENDER ("\">");
	if (b->pixmap_data == NULL)
	{
		gchar *strippedname = strip_uline_accel (b->name);
		RENDER (strippedname);
		g_free (strippedname);
	}
	else
	{
		RENDER ("<img src=\"file://");
		RENDER (b->pixmap_file);
		RENDER ("\">");
	}
	RENDER ("</a>\n");
}

/**
 * render_search_form: render a search form based on a smart bookmark
 * FIXME: tidy
 */
static void 
render_search_form (OutStream *stream, BookmarkItem *b)
{
	gchar *path;    /* static part of URL */
	gchar **query;  /* query string, broken into parameters */
	int num_inputs = 0, pathlen;
	gchar *tmp;
	int i;

	/* parse URL to separate static parts from parts needing user input */
	tmp = strchr (b->url, '?');
	if (!tmp || ((int)strlen (b->url)) < (tmp - b->url + 2))
		return; /* no query in URL, can't render it */
	pathlen = tmp - b->url;
	path = g_strndup (b->url, pathlen);
	RENDER ("<form action=\"");
	if (!strstr (b->url, "://"))
	{
		RENDER ("http://");
	}
	RENDER (path);
	RENDER ("\" method=get>");
	g_free (path);

	/* count occurrences of %s in b->url */
	tmp = b->url;
	while ((tmp = strstr (tmp+1, "%s")))
	       ++num_inputs;

	query = g_strsplit (b->url + pathlen + 1, "&", -1);

	for (i = 0; query[i]; ++i) {
		gchar **param, *smtpos;
		param = g_strsplit (query[i], "=", 2);
		if (param[0] && param[1] && (smtpos = strstr (param[1], "%s")))
		{
			/* fill in the field with the text minus the %s */
			gchar *val=g_strdup_printf ("%.*s%s", smtpos-param[1],
					param[1],param[1]+(smtpos-param[1]+2));
			/* add a form input */
			RENDER (num_inputs>1?param[0]:"");
			RENDER ("<input type=text name=\"");
			RENDER (param[0]);
			RENDER ("\" value=\"");
			RENDER (val);
			RENDER ("\" size=");
			RENDER (num_inputs>1?"10>":"30>");
			g_free (val);
		}
		else
		{
			/* add the static parts of URL as hidden form inputs */
			RENDER ("<input type=hidden name=\"");
			RENDER (param[0]);
			RENDER ("\" value=\"");
			if (param[1])
				RENDER (param[1]);
			RENDER ("\">");
		}
		g_strfreev (param);
	}
	g_strfreev (query);
	
	if (b->pixmap_data == NULL)
	{
		gchar *strippedname = strip_uline_accel (b->name);
		RENDER ("<input type=submit value=\"");
		RENDER (strippedname);
		g_free (strippedname);
	}
	else
	{
		RENDER ("<input type=image src=\"file://");
			RENDER (b->pixmap_file);
	}
	RENDER ("\"></form>\n");
}

#undef RENDER
