/*
 *  Copyright (C) 2000-2001 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "glade.h"
#include "misc.h"
#include "prefs.h"
#include "print.h"
#include "mozilla.h"
#include "dialog.h"

#include <stdlib.h>
#include <string.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtktogglebutton.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-uidefs.h>

typedef struct
{
	GtkWidget *print_dialog;
	GtkWidget *print_entry;
	GtkWidget *file_entry;
	GtkWidget *print_rb;
	GtkWidget *letter_rb;
	GtkWidget *top_entry;
	GtkWidget *bottom_entry;
	GtkWidget *left_entry;
	GtkWidget *right_entry;
	GtkWidget *all_pages_rb;
	GtkWidget *from_spinbutton;
	GtkWidget *to_spinbutton;
	GtkWidget *first_rb;
	GaleonWindow *window;
}
PrintWidgets;

static void
print_lookup_widgets (GladeXML *gxml, PrintWidgets *pw);

static PrintInfo *
print_get_info (PrintWidgets *pw);

static float
get_float_from_entry (GtkWidget *entry);

static int
get_int_from_entry (GtkWidget *entry);

void
print_ok_button_cb (GtkWidget *button, PrintWidgets *widgets);

void
print_cancel_button_cb (GtkWidget *button, PrintWidgets *widgets);

void
print_preview_button_cb (GtkWidget *button, PrintWidgets *widgets);

void
print_destroy_cb (GtkWidget *dialog, PrintWidgets *widgets);

void
print_info_free (PrintInfo *info);

static void print_options_get (PrintWidgets *widgets);
static void print_options_set (PrintWidgets *widgets);

void print_show_dialog (GaleonWindow *window)
{	
	GladeXML *gxml;
	PrintWidgets *pw;

	if (window->print_dialog) return;

	pw = g_new0 (PrintWidgets, 1);

	gxml = glade_widget_new ("galeon.glade", "print_dialog", &pw->print_dialog, pw);
	
	print_lookup_widgets (gxml, pw);

	gtk_window_set_modal (GTK_WINDOW(pw->print_dialog), TRUE);
	dialog_set_parent (pw->print_dialog, window->WMain);

	window->print_dialog = pw->print_dialog;
	pw->window = window;
	print_options_get (pw);

	/* show the print dialog */
	gtk_widget_show (window->print_dialog);
	gtk_object_unref (GTK_OBJECT (gxml));	
}

static void
print_lookup_widgets (GladeXML *gxml, PrintWidgets *pw)
{
	pw->print_entry = glade_xml_get_widget(gxml, "printer_entry");
	pw->file_entry = glade_xml_get_widget(gxml, "file_entry");
	pw->print_rb = glade_xml_get_widget(gxml, "printer_radiobutton");
	pw->letter_rb = glade_xml_get_widget(gxml, "letter_radiobutton");
	pw->top_entry = glade_xml_get_widget(gxml, "top_entry");
	pw->bottom_entry = glade_xml_get_widget(gxml, "bottom_entry");
	pw->left_entry = glade_xml_get_widget(gxml, "left_entry");
	pw->right_entry = glade_xml_get_widget(gxml, "right_entry");
	pw->all_pages_rb = glade_xml_get_widget(gxml, "all_pages_radiobutton");
	pw->first_rb = glade_xml_get_widget(gxml, "first_radiobutton");
	pw->to_spinbutton = glade_xml_get_widget(gxml, "to_spinbutton");
	pw->from_spinbutton = glade_xml_get_widget(gxml, "from_spinbutton");
}

static PrintInfo *
print_get_info (PrintWidgets *pw)
{
	PrintInfo *info;

	info = g_new0 (PrintInfo, 1);

      	info->print_to_file = 
		gtk_radio_button_get (GTK_RADIO_BUTTON (pw->print_rb));
	info->printer = gtk_editable_get_chars (
		GTK_EDITABLE (pw->print_entry), 0, -1);
	info->file = gtk_editable_get_chars (
		GTK_EDITABLE (pw->file_entry), 0, -1);

	info->top_margin = get_float_from_entry (pw->top_entry);
	info->bottom_margin = get_float_from_entry (pw->bottom_entry);
	info->left_margin = get_float_from_entry (pw->left_entry);
	info->right_margin = get_float_from_entry (pw->right_entry);

	info->from_page = get_int_from_entry (pw->from_spinbutton);
	info->to_page = get_int_from_entry (pw->to_spinbutton);

	info->paper = gtk_radio_button_get (GTK_RADIO_BUTTON (pw->letter_rb));
	info->pages = gtk_radio_button_get 
		(GTK_RADIO_BUTTON (pw->all_pages_rb));
	info->reversed = gtk_radio_button_get 
		(GTK_RADIO_BUTTON (pw->first_rb));

	return info;
}

void
print_info_free (PrintInfo *info)
{
	g_free (info->printer);
	g_free (info->file);
	g_free (info);
}

static
void print_options_get (PrintWidgets *widgets)
{
	set_editable_from_config (widgets->top_entry, 
				  CONF_PRINT_TOP_MARGIN);
	set_editable_from_config (widgets->bottom_entry,
				  CONF_PRINT_BOTTOM_MARGIN);
	set_editable_from_config (widgets->left_entry, 
				  CONF_PRINT_LEFT_MARGIN);
	set_editable_from_config (widgets->right_entry, 
				  CONF_PRINT_TIGHT_MARGIN);
	set_editable_from_config (widgets->print_entry, 
				  CONF_PRINT_PRINTER);
	set_editable_from_config (widgets->file_entry, 
				  CONF_PRINT_FILE);
	set_radiobuttongroup_from_config (widgets->print_rb, 
					  CONF_PRINT_PRINTON); 
	set_radiobuttongroup_from_config (widgets->letter_rb, 
					  CONF_PRINT_PAPER); 
}

static
void print_options_set (PrintWidgets *widgets)
{
	set_config_from_editable (widgets->top_entry, 
				  CONF_PRINT_TOP_MARGIN);
	set_config_from_editable (widgets->bottom_entry, 
				  CONF_PRINT_BOTTOM_MARGIN);
	set_config_from_editable (widgets->left_entry, 
				  CONF_PRINT_LEFT_MARGIN);
	set_config_from_editable (widgets->right_entry, 
				  CONF_PRINT_TIGHT_MARGIN);
	set_config_from_editable (widgets->print_entry, 
				  CONF_PRINT_PRINTER);
	set_config_from_editable (widgets->file_entry, 
				  CONF_PRINT_FILE);
	set_config_from_radiobuttongroup (widgets->print_rb, 
					  CONF_PRINT_PRINTON); 
	set_config_from_radiobuttongroup (widgets->letter_rb, 
					  CONF_PRINT_PAPER);        
}

void
print_ok_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	PrintInfo *info;

	info = print_get_info (widgets);

	if (!mozilla_print (widgets->window->active_embed, info, FALSE))
	{
		gnome_error_dialog ("Not yet implemented.");
	}

	print_options_set (widgets);
	
	print_info_free (info);

	gtk_widget_destroy (widgets->print_dialog);
}

void
print_preview_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	PrintInfo *info;

	info = print_get_info (widgets);
	mozilla_print (widgets->window->active_embed, info, TRUE);
	print_info_free (info);
}

void
print_cancel_button_cb (GtkWidget *button, PrintWidgets *widgets)
{
	gtk_widget_destroy (widgets->print_dialog);
}

void
print_destroy_cb (GtkWidget *dialog, PrintWidgets *widgets)
{
	widgets->window->print_dialog = NULL;
	g_free (widgets);
}

static float
get_float_from_entry (GtkWidget *entry)
{
	double f;
	char *temp;

	temp = gtk_editable_get_chars (
		GTK_EDITABLE (entry), 0, -1);

	f = strtod(temp, (char**)NULL);

	g_free (temp);

	return f;
}

static int
get_int_from_entry (GtkWidget *entry)
{
	int i;
	char *temp;

	temp = gtk_editable_get_chars (
		GTK_EDITABLE (entry), 0, -1);
	i = strtol(temp, (char **)NULL, 10);
	g_free (temp);

	return i;
}
