/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

#if MOZILLA_VERSION > VERSION3(0,9,4)

#include "ProgressListener.h"

#include "mozilla.h"
#include "misc.h"
#include "glade.h"
#include "dialog.h"

#include <gtk/gtkentry.h>
#include <gtk/gtkprogress.h>
#include <gtk/gtkoptionmenu.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-i18n.h>

#include "nsXPIDLString.h"
#include "nsIChannel.h"
#include "nsIFTPChannel.h"

NS_IMPL_ISUPPORTS3(GPersistProgressListener, nsIWebProgressListener,
		   GIProgressCallbacks, nsISupportsWeakReference)

//------------------------------------------------------------------------------

GPersistProgressListener::
	GPersistProgressListener(nsIWebBrowserPersist *aPersist,
				  nsIDOMWindow *aParent, nsIURI *aURI,
				  nsIFile *aFile, PRInt32 aAction,
				  PRInt64 aTimeDownloadStarted)
{
	NS_INIT_ISUPPORTS();
	
	/* member initializers and constructor code */
	nsresult rv;
	
	mInterval = 500000; //in microsecs == 500ms == 0.5s
	mPriorKRate = 0;
	mRateChanges = 0;
	mRateChangeLimit = 2; //Only update rate every second
	
	mPersist = aPersist;
	mParent = aParent;
	mUri = aURI;
	mFile = aFile;
	mAction = aAction;
	mTimeDownloadStarted = aTimeDownloadStarted;

	GtkWidget *aParentWidget = mozilla_find_gtk_parent (mParent);

	nsCOMPtr<nsIWebProgressListener> progress =
			NS_STATIC_CAST(nsIWebProgressListener *, this);
	nsCOMPtr<GIProgressCallbacks> callbacks =
			do_QueryInterface (progress);
	GladeXML *gxml = glade_widget_new ("galeon.glade", 
					   "download_progress_dialog",
					   &mProgressDialog, callbacks);

	mProgressBar = glade_xml_get_widget (gxml, "progressbar");
	mLocation = glade_xml_get_widget (gxml, "location_entry");
	mFileName = glade_xml_get_widget (gxml, "filename_entry");
	mStatus = glade_xml_get_widget (gxml, "status_entry");
	mTimeElapsed = glade_xml_get_widget (gxml, "elapsed_entry");
	mTimeRemaining = glade_xml_get_widget (gxml, "remaining_entry");
	mPauseButton = glade_xml_get_widget (gxml, "pause_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	mCheckedCanPause = PR_FALSE;
	mCanPause = PR_FALSE;
	mIsPaused = PR_FALSE;

	PRInt64 now = PR_Now ();
	mLastUpdate = now;
	if (mTimeDownloadStarted != 0)
		mStartTime = mTimeDownloadStarted;
	else
		mStartTime = now;
	mElapsed = now - mStartTime;

	char *text;

	rv = mUri->GetSpec (&text);
	gtk_entry_set_text (GTK_ENTRY(mLocation),text);
	g_free (text);
	
	rv = mFile->GetPath (&text);
	gtk_entry_set_text (GTK_ENTRY(mFileName),text);
	g_free (text);
	
	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));
	gtk_label_set_text (GTK_LABEL(mTimeRemaining),FormatTime(0));

	dialog_set_parent (mProgressDialog, aParentWidget);

	aPersist->SetProgressListener (this);

	// Increment window count while progress is under way.
	window_count++;
}

GPersistProgressListener::~GPersistProgressListener()
{
	/* destructor code */
}

/* void onStateChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aStateFlags, in unsigned long aStatus); */
NS_IMETHODIMP GPersistProgressListener::
			OnStateChange(nsIWebProgress *aWebProgress,
				      nsIRequest *aRequest,PRInt32 aStateFlags,
				      PRUint32 aStatus)
{
	if (aStateFlags & nsIWebProgressListener::STATE_STOP)
	{
		if (GTK_IS_WIDGET(mProgressDialog))
			gtk_widget_destroy (mProgressDialog);

		nsXPIDLCString filename;
		mFile->GetPath(getter_Copies(filename));

		switch (mAction)
		{
		case ACTION_VIEWSOURCE:
			launch_external_viewer (filename.get());
			break;
		case ACTION_SETBACKGROUND:
			gchar *command;

			/* build command */
			command = g_strconcat ("background-properties-capplet "
					       "--init-session-settings --ignore "
					       "--background-image=",
					       filename.get(), NULL);

			/* execute it synchronously */
			gnome_execute_shell (NULL, command);

			/* free */
			g_free (command);
			break;
		}

		// Decrement window count as this progess is finished.
		window_count--;

		// If there are no more windows left, exit galeon.
		if (g_list_length (all_windows) == 0 && !galeon_server_mode
		    && window_count ==0)
		{
			galeon_exit ();
		}
	}

	return NS_OK;
}

/* void onProgressChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aCurSelfProgress, in long aMaxSelfProgress, in long aCurTotalProgress, in long aMaxTotalProgress); */
NS_IMETHODIMP GPersistProgressListener::
			OnProgressChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest,
					 PRInt32 aCurSelfProgress,
					 PRInt32 aMaxSelfProgress,
					 PRInt32 aCurTotalProgress,
					 PRInt32 aMaxTotalProgress)
{
	char *text, *rateStr, *totalStr;

	if (!mCheckedCanPause)
	{
		mCheckedCanPause = PR_TRUE;

		nsresult rv;
		nsCOMPtr<nsIFTPChannel> aChannel = do_QueryInterface (aRequest,
								      &rv);
		if (NS_SUCCEEDED(rv))
		{
			mCanPause = PR_TRUE;
			gtk_widget_show (mPauseButton);
		}
		else
		{
			mCanPause = PR_FALSE;
			gtk_widget_hide (mPauseButton);
		}
	}
	mRequest = aRequest;

	PRInt64 now = PR_Now ();

	if ((now - mLastUpdate < mInterval) && 
	     (aMaxTotalProgress != -1) &&  
	     (aCurTotalProgress < aMaxTotalProgress))
		return NS_OK;

	mLastUpdate = now;
	mElapsed = now - mStartTime;

	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));

	PRInt32 currentKBytes = (PRInt32)(aCurTotalProgress/1024 +.5);

	PRInt32 totalKBytes = (PRInt32)(aMaxTotalProgress/1024 +.5);
	if (aMaxTotalProgress != -1)
	{
		gfloat progress = (gfloat)aCurTotalProgress/
				  (gfloat)aMaxTotalProgress;
		gchar *strper = g_strdup_printf (_("%.2f%% - Downloading"), 
						 progress*100);

		if (progress > 0 && progress < 1)
			gtk_window_set_title (GTK_WINDOW(mProgressDialog), 
					      strper);

		if (progress >= 0 && progress <= 1.0)
		{
			gtk_progress_set_percentage (GTK_PROGRESS(mProgressBar),
						     progress);
		}
		else
		{
			gtk_progress_set_format_string
					(GTK_PROGRESS(mProgressBar), "?? %%");
		}
		totalStr = g_strdup_printf ("%d",totalKBytes);
		g_free (strper);
	}
	else
	{
		gtk_progress_set_format_string (GTK_PROGRESS(mProgressBar),
						"?? %%");
		totalStr = g_strdup ("??");
	}

	PRInt64 currentRate;
	if (mElapsed)
		currentRate = ((PRInt64)(aCurTotalProgress))*1000000 / mElapsed;
	else
		currentRate = 0;
		
	if (currentRate)
	{
		PRFloat64 currentKRate = ((PRFloat64)currentRate)/1024;
		if (currentKRate != mPriorKRate)
		{
			if (mRateChanges++ == mRateChangeLimit)
			{
				mPriorKRate = currentKRate;
				mRateChanges = 0;
			}
			else
			{
				currentKRate = mPriorKRate;
			}
		}
		else
		{
			mRateChanges = 0;
		}
		rateStr = g_strdup_printf("%.2f",currentKRate);
	}
	else
	{
		rateStr = g_strdup ("??.??");
	}

	if (mIsPaused)
	{
		text = g_strdup (_("Download Paused"));
	}
	else
	{
		text = g_strdup_printf (_("%dK of %sK bytes at %sK bytes/sec"),
					currentKBytes, totalStr, rateStr);
	}
	gtk_label_set_text (GTK_LABEL(mStatus),text);
	g_free (text);
	g_free (totalStr);
	g_free (rateStr);

	if (currentRate && (aMaxTotalProgress != -1))
	{
		PRInt32 remaining = 
			(PRInt32)((aMaxTotalProgress - aCurTotalProgress)
				   /currentRate +.5);
		gtk_label_set_text (GTK_LABEL(mTimeRemaining),
				    FormatTime(remaining));
	}
	else
	{
		gtk_label_set_text (GTK_LABEL(mTimeRemaining), _("Unknown"));
	}

	return NS_OK;
}

/* void onLocationChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsIURI location); */
NS_IMETHODIMP GPersistProgressListener::
			OnLocationChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, nsIURI *location)
{
    return NS_OK;
}

/* void onStatusChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsresult aStatus, in wstring aMessage); */
NS_IMETHODIMP GPersistProgressListener::
			OnStatusChange(nsIWebProgress *aWebProgress,
				       nsIRequest *aRequest, nsresult aStatus,
				       const PRUnichar *aMessage)
{
    return NS_OK;
}

/* void onSecurityChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long state); */
NS_IMETHODIMP GPersistProgressListener::
			OnSecurityChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, PRInt32 state)
{
    return NS_OK;
}

NS_IMETHODIMP GPersistProgressListener::CancelHelperProgress (void)
{
	if (mIsPaused)
		TogglePause ();
	return mPersist->CancelSave ();
}

NS_IMETHODIMP GPersistProgressListener::TogglePause (void)
{
	if (!mCheckedCanPause) return NS_ERROR_FAILURE;

	nsresult rv;
	if (mIsPaused)
	{
		rv = mRequest->Resume ();
		if (NS_SUCCEEDED(rv))
		{
			gtk_label_set_text (GTK_LABEL(GTK_BIN(mPauseButton)
					    ->child),
					    _("Pause"));
		}
	}
	else
	{
		rv = mRequest->Suspend ();
		if (NS_SUCCEEDED(rv))
		{
			gtk_label_set_text (GTK_LABEL(mStatus),
					    _("Download Paused"));
			gtk_label_set_text (GTK_LABEL(GTK_BIN(mPauseButton)
					    ->child),
					    _("Resume"));
		}
	}
	mIsPaused = !mIsPaused;
	return rv;
}

char *GPersistProgressListener::FormatTime (PRUint32 aTime)
{
	PRUint32 secs = (PRUint32)(aTime+.5);
	PRUint32 hours = secs/3600;
	secs -= hours*3600;
	PRUint32 mins = secs/60;
	secs -= mins*60;
	char *result;
	if (hours)
		result = g_strdup_printf ("%u:%02u.%02u",hours,mins,secs);
	else
		result = g_strdup_printf ("%02u.%02u",mins,secs);
	return result;
}

#endif

