/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla.h"
#include "GaleonWrapper.h"
#include "PrintListener.h"
#include "GlobalHistory.h"

#include <gtkmozembed_internal.h>
#include <unistd.h>

#include "nsIContentViewer.h"
#include "nsIPermissionManager.h"
#include "nsIGlobalHistory.h"
#include "nsIDocShellHistory.h"
#include "nsIWebBrowserFind.h"
#include "nsIWebBrowserFocus.h"
#include "nsIDocument.h"
#include "nsISHEntry.h"
#include "nsISHistoryInternal.h"
#include "nsIHistoryEntry.h"
#include "nsIWebBrowserPrint.h"
#include "nsIURI.h"
#include "nsIPresShell.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIComponentManager.h"
#include "nsIDOMDocument.h"
#include "nsIDOMElement.h"
#include "nsIDOMNodeList.h"
#include "nsIScriptGlobalObject.h"
#include "nsIDOMWindowInternal.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIInterfaceRequestor.h"
#include "nsIFocusController.h"
#include "nsIWebBrowserPersist.h"
#include "nsIStreamTransfer.h"
#include "nsNetUtil.h"
#include "nsIChromeEventHandler.h"
#include "nsIClipboardCommands.h"
#include "nsIDOMDocumentStyle.h"
#include "nsIDocShellTreeItem.h"
#include "nsIDocShellTreeOwner.h"
#include "nsIHTMLContentContainer.h"
#include "nsICSSLoader.h"
#include "nsICSSStyleSheet.h"
#include "nsICSSLoaderObserver.h"
#include "nsIStyleSet.h"
#include "nsIDocumentObserver.h"
#if MOZILLA_VERSION > VERSION3(0,9,4)
#include "ProgressListener.h"
#include "nsCWebBrowser.h"
#include "nsReadableUtils.h"
#else
#include "PersistProgress.h"
#include "gStringHelper.h"
#endif

GaleonWrapper::GaleonWrapper ()
{
}

GaleonWrapper::~GaleonWrapper ()
{
}

nsresult GaleonWrapper::Init (GaleonEmbed *galeon_embed)
{
	nsresult result;

	return_val_if_not_embed (galeon_embed, NS_ERROR_FAILURE);

	embed = galeon_embed;
	mGtkMozEmbed = GTK_MOZ_EMBED(embed->mozEmbed);

	gtk_moz_embed_get_nsIWebBrowser (mGtkMozEmbed,
					 getter_AddRefs(mWebBrowser));
	if (!mWebBrowser) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShellHistory> dsHistory = do_QueryInterface (DocShell);
	if (!dsHistory) return NS_ERROR_FAILURE;

	static NS_DEFINE_CID(kGlobalHistoryCID, GALEON_GLOBALHISTORY_CID);

	nsCOMPtr<nsIFactory> GHFactory;
	result = NS_NewGlobalHistoryFactory(getter_AddRefs(GHFactory));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	result = nsComponentManager::RegisterFactory(kGlobalHistoryCID,
						     "Global history",
						     NS_GLOBALHISTORY_CONTRACTID,
						     GHFactory,
						     PR_TRUE);

	nsCOMPtr<nsIGlobalHistory> inst =  
		do_GetService(NS_GLOBALHISTORY_CONTRACTID, &result);
	dsHistory->SetGlobalHistory(inst);

	nsCOMPtr<nsPIDOMWindow> piWin;
	GetPIDOMWindow(getter_AddRefs(piWin));
	if (!piWin) return NS_ERROR_FAILURE;
  
        nsCOMPtr<nsIChromeEventHandler> chromeHandler;
        piWin->GetChromeEventHandler(getter_AddRefs(chromeHandler));
  
	mEventReceiver = do_QueryInterface(chromeHandler);

        mEventListener = new GaleonEventListener();
        mEventListener->Init(this);

	return AddListener ();

	return NS_OK;
}

nsresult GaleonWrapper::GetDocShell (nsIDocShell **aDocShell)
{
        nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
        browserAsItem = do_QueryInterface(mWebBrowser);
        if (!browserAsItem) return NS_ERROR_FAILURE;

        // get the owner for that item
        nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
        browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
        if (!treeOwner) return NS_ERROR_FAILURE;

        // get the primary content shell as an item
        nsCOMPtr<nsIDocShellTreeItem> contentItem;
        treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
        if (!contentItem) return NS_ERROR_FAILURE;

        // QI that back to a docshell
        nsCOMPtr<nsIDocShell> DocShell;
        DocShell = do_QueryInterface(contentItem);
        if (!DocShell) return NS_ERROR_FAILURE;

        *aDocShell = DocShell.get();

        NS_IF_ADDREF(*aDocShell);
        
        return NS_OK;
}

nsresult GaleonWrapper::GetPIDOMWindow(nsPIDOMWindow **aPIWin)
{
  *aPIWin = nsnull;

  nsCOMPtr<nsIDOMWindow> domWindow;
  mWebBrowser->GetContentDOMWindow(getter_AddRefs(domWindow));
  if (!domWindow)
    return NS_ERROR_FAILURE;

  nsCOMPtr<nsPIDOMWindow> domWindowPrivate = do_QueryInterface(domWindow);
  nsCOMPtr<nsIDOMWindowInternal> rootWindow;
  domWindowPrivate->GetPrivateRoot(getter_AddRefs(rootWindow));
  
  nsCOMPtr<nsIChromeEventHandler> chromeHandler;
  nsCOMPtr<nsPIDOMWindow> piWin(do_QueryInterface(rootWindow));

  *aPIWin = piWin.get();

  if (*aPIWin) {
    NS_ADDREF(*aPIWin);
    return NS_OK;
  }

  return NS_ERROR_FAILURE;
}

/* This is how gtkmozembed is focusing childs,
   I'm not using nsIWebBrowserFocus because gtkmozembed
   doesnt, so I guess something could be buggy there */

nsresult GaleonWrapper::Activate ()
{
	nsCOMPtr<nsPIDOMWindow> piWin;
	GetPIDOMWindow(getter_AddRefs(piWin));
	if (!piWin) return NS_ERROR_FAILURE;

	return piWin->Activate();
}

nsresult GaleonWrapper::Deactivate ()
{
	nsCOMPtr<nsPIDOMWindow> piWin;
	GetPIDOMWindow(getter_AddRefs(piWin));
	if (!piWin) return NS_ERROR_FAILURE;

	return piWin->Deactivate();
}

nsresult GaleonWrapper::SetOffline(PRBool aOffline)
{
	nsresult rv;
	nsCOMPtr<nsIIOService> io = do_GetService(NS_IOSERVICE_CONTRACTID, &rv);
	if (NS_FAILED(rv))
		return NS_ERROR_FAILURE;

	return io->SetOffline(aOffline);
}

nsresult GaleonWrapper::Print (nsIPrintOptions *options, PRBool preview)
{
	nsresult result;

	nsCOMPtr<nsIWebBrowserPrint> print(do_GetInterface(mWebBrowser, &result));
	if (NS_FAILED(result) || !print) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetMainDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	if (!preview)
	{
		result = print->Print (DOMWindow, options, NULL);
	}
	else
	{
		char *filename;

		filename = g_strconcat(g_get_tmp_dir(),"/galeon-print-XXXXXX", NULL);
		/* get a name for the temporary file */
		int res;
		res = mkstemp (filename);
		if (res == -1)
			return NS_ERROR_FAILURE;
		close (res);
		mozilla_delete_temp_file_on_exit (filename);

		nsCString fileString(filename);
		options->SetToFileName (ToNewUnicode(fileString));
		options->SetPrintToFile (TRUE);

		PrintListener *listener = 
			new PrintListener(filename);
		result = print->Print (DOMWindow, options, listener); 
	}

	return result;
}

nsresult GaleonWrapper::GetSHistory (nsISHistory **aSHistory)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = ContentNav->GetSessionHistory (getter_AddRefs (SessionHistory));
	if (!SessionHistory) return NS_ERROR_FAILURE;

	*aSHistory = SessionHistory.get();
	NS_IF_ADDREF (*aSHistory);

	return NS_OK;
}

nsresult GaleonWrapper::Destroy ()
{
  	RemoveListener ();
      	mWebBrowser = nsnull;
	mChromeNav = nsnull;
	
	return NS_OK;
}

nsresult GaleonWrapper::AddListener ()
{
	nsresult rv = NS_OK;

        if (!mEventReceiver || !mEventListener) return NS_ERROR_FAILURE;

        nsCOMPtr<nsIDOMEventListener> eventListener =
                NS_STATIC_CAST(nsIDOMEventListener *,
                               NS_STATIC_CAST(nsIDOMDragListener *, mEventListener));

        rv = mEventReceiver->AddEventListenerByIID(eventListener,
                                                 NS_GET_IID(nsIDOMDragListener));
        if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
	
	return NS_OK;
}

nsresult GaleonWrapper::RemoveListener ()
{
	nsresult rv = NS_OK;

	if (!mEventReceiver || !mEventListener) return NS_ERROR_FAILURE;
        
        nsCOMPtr<nsIDOMEventListener> eventListener =
                NS_STATIC_CAST(nsIDOMEventListener *,
                               NS_STATIC_CAST(nsIDOMDragListener *, mEventListener));

        rv = mEventReceiver->RemoveEventListenerByIID(eventListener,
                                                      NS_GET_IID(nsIDOMDragListener));

	return rv;
}

nsresult GaleonWrapper::GoToHistoryIndex (PRInt16 index)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	return  ContentNav->GotoIndex (index);
}

nsresult GaleonWrapper::SetZoom (float aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->SetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetZoom (float *aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->GetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetDOMWindow (nsIDOMWindow **aDOMWindow)
{
	nsresult rv;

	nsCOMPtr<nsIWebBrowserFocus> focus = do_GetInterface(mWebBrowser, &rv);
	if (NS_FAILED(rv) || !focus) return NS_ERROR_FAILURE;

	rv = focus->GetFocusedWindow (aDOMWindow);
	if (NS_FAILED(rv))
		rv = GetMainDOMWindow (aDOMWindow);
	return rv;
}

nsresult GaleonWrapper::GetMainDOMWindow (nsIDOMWindow **aDOMWindow)
{
	return mWebBrowser->GetContentDOMWindow (aDOMWindow);
}

nsresult GaleonWrapper::GetDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow)
	{
		return GetMainDOMDocument (aDOMDocument);
	}

	return DOMWindow->GetDocument (aDOMDocument);
}

nsresult GaleonWrapper::GetMainDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	return contentViewer->GetDOMDocument (aDOMDocument);
}

nsresult GaleonWrapper::SaveURI (const char *url, char *filename, int action)
{
        nsresult rv;

	nsString s;
	s.AssignWithConversion(url);
	nsCOMPtr<nsIURI> linkURI;
	rv = NS_NewURI(getter_AddRefs(linkURI), s);
	if (NS_FAILED(rv) || !linkURI) return NS_ERROR_FAILURE;

#if MOZILLA_VERSION > VERSION3(0,9,4)
	nsCOMPtr<nsIWebBrowserPersist> persist = 
			do_CreateInstance(NS_WEBBROWSERPERSIST_CONTRACTID, &rv);
	if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

	nsCOMPtr<nsILocalFile> file;
	NS_NewLocalFile(filename, PR_TRUE, getter_AddRefs(file)); 
	if (NS_FAILED(rv) || !file) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindow> parent;
	rv = GetDOMWindow (getter_AddRefs(parent));

	mProgress =
		new GPersistProgressListener (persist, parent,
					       linkURI, file, action);
	return persist->SaveURI(linkURI, nsnull, file);
#else
	nsCOMPtr<nsIWebBrowserPersist> persist = do_GetInterface(mWebBrowser, &rv);
	if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebBrowserPersistProgress> progress;

	if (action)
	{
		progress = new PersistProgress(filename, action);
		persist->SetProgressListener (progress);
	}
	return persist->SaveURI(linkURI, nsnull, filename);
#endif
}

nsresult GaleonWrapper::SaveLink (const char *url)
{
	nsresult rv;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	rv = mWebBrowser->GetContentDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(rv) || !DOMWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindowInternal> DOMWInternal =
						do_QueryInterface (DOMWindow);
	if (!DOMWInternal) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIStreamTransfer> xfer = 
				do_GetService (NS_ISTREAMTRANSFER_CONTRACTID);
	if (!xfer) return NS_ERROR_FAILURE;

	return xfer->SelectFileAndTransferLocationSpec 
		(url, DOMWInternal, "", "", PR_TRUE, nsnull, nsnull);
}

nsresult GaleonWrapper::SaveDocument (char *filename, const char *datapath,
				      int action, PRBool mainDoc)
{
	nsresult rv;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	if (mainDoc)
		rv = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	else
		rv = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(rv) || !aDOMDocument) return NS_ERROR_FAILURE;

#if MOZILLA_VERSION > VERSION3(0,9,4)
	nsCOMPtr<nsIWebBrowserPersist> persist = 
			do_CreateInstance(NS_WEBBROWSERPERSIST_CONTRACTID, &rv);
	if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

	nsCOMPtr<nsILocalFile> file;
	rv = NS_NewLocalFile(filename, PR_TRUE, getter_AddRefs(file)); 
	if (NS_FAILED(rv) || !file) return NS_ERROR_FAILURE;

	nsCOMPtr<nsILocalFile> path;
	if (datapath)
		NS_NewLocalFile(datapath, PR_TRUE, getter_AddRefs(path));
	else
		path = nsnull;

	if (action)
	{

		nsCOMPtr<nsIDocument> aDocument =
				do_QueryInterface (aDOMDocument, &rv);
		if (NS_FAILED(rv) || !aDocument) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIURI> aURI;
		rv = aDocument->GetDocumentURL (getter_AddRefs(aURI));
		if (NS_FAILED(rv) || !aURI) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIDOMWindow> parent;
		GetDOMWindow (getter_AddRefs(parent));

		mProgress =
			new GPersistProgressListener (persist, parent,
						       aURI, file, action);
	}
	return persist->SaveDocument (aDOMDocument, file, path);
#else
	nsCOMPtr<nsIWebBrowserPersist> persist =
			do_GetInterface(mWebBrowser, &rv);
	if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebBrowserPersistProgress> progress;

	if (action)
	{
		progress = new PersistProgress(filename, action);
		persist->SetProgressListener (progress);
	}
	return persist->SaveDocument (aDOMDocument, filename, datapath);
#endif
}

nsresult GaleonWrapper::GetSHInfo (PRInt32 *count, PRInt32 *index)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	SessionHistory->GetCount (count);
	SessionHistory->GetIndex (index);	

	return NS_OK;
}

nsresult GaleonWrapper::GetSHTitleAtIndex (PRInt32 index, PRUnichar **title)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIHistoryEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	result = he->GetTitle (title);
	if (!NS_SUCCEEDED(result) || (!title)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::GetSHUrlAtIndex (PRInt32 index, char **url)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIHistoryEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIURI> uri;
	result = he->GetURI (getter_AddRefs(uri));
	if (!NS_SUCCEEDED(result) || (!uri)) return NS_ERROR_FAILURE;

	result = uri->GetSpec(url);
	if (!NS_SUCCEEDED(result) || (!*url)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::Find (const PRUnichar *search_string, 
			      PRBool matchcase, PRBool search_backwards,
			      PRBool search_wrap_around,
			      PRBool search_for_entire_word,
			      PRBool search_in_frames,
			      PRBool *didFind)
{
	nsCOMPtr<nsIWebBrowserFind> finder (do_GetInterface(mWebBrowser));

	finder->SetSearchString (search_string);
	finder->SetFindBackwards (search_backwards);
	finder->SetWrapFind (search_wrap_around);
	finder->SetEntireWord (search_for_entire_word);
	finder->SetMatchCase (matchcase);
	finder->SetSearchFrames (search_in_frames);
	return finder->FindNext(didFind);
}

nsresult GaleonWrapper::ReloadDocument ()
{
	nsresult result;

	nsCOMPtr<nsIWebBrowserFocus> focus(do_GetInterface(mWebBrowser));

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = focus->GetFocusedWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIScriptGlobalObject> scriptGlobal = do_QueryInterface(DOMWindow);
	if (!scriptGlobal) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> docshell;
	if (NS_FAILED(scriptGlobal->GetDocShell(getter_AddRefs(docshell))))
        return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn = do_QueryInterface (docshell, &result);
	if (!wn || !NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	result = wn->Reload (nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE | 
			     nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::GetMainDocumentUrl (char **url)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));

	return uri->GetSpec (url);
}

nsresult GaleonWrapper::GetDocumentUrl (char **url)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));

	uri->GetSpec (url);

	return NS_OK;
}

nsresult  GaleonWrapper::CopyHistoryTo (GaleonWrapper *dest)
{
	nsresult result;
	int count,index;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_src = do_QueryInterface (DocShell,
							       &result);
	if (!wn_src) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_src;
	result = wn_src->GetSessionHistory (getter_AddRefs (h_src));
	if (!NS_SUCCEEDED(result) || (!h_src)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> destDocShell;
	result = dest->GetDocShell (getter_AddRefs(destDocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_dest = do_QueryInterface (destDocShell,
								&result);
	if (!wn_dest) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_dest;
	result = wn_dest->GetSessionHistory (getter_AddRefs (h_dest));
	if (!NS_SUCCEEDED (result) || (!h_dest)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistoryInternal> hi_dest = do_QueryInterface (h_dest);
	if (!hi_dest) return NS_ERROR_FAILURE;

	h_src->GetCount (&count);
	h_src->GetIndex (&index);

	if (count) {
		nsCOMPtr<nsIHistoryEntry> he;
		nsCOMPtr<nsISHEntry> she;

		for (PRInt32 i = 0; i < count; i++) {

			result = h_src->GetEntryAtIndex (i, PR_FALSE,
							 getter_AddRefs (he));
			if (!NS_SUCCEEDED(result) || (!he))
				return NS_ERROR_FAILURE;

			she = do_QueryInterface (he);
			if (!she) return NS_ERROR_FAILURE;

			result = hi_dest->AddEntry (she, PR_TRUE);
			if (!NS_SUCCEEDED(result) || (!she))
				return NS_ERROR_FAILURE;
		}

		result = wn_dest->GotoIndex(index);
		if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
	}

	return NS_OK;
}


nsresult GaleonWrapper::SetSitePermission (gboolean block, PRInt32 type)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> DOMDocument;
	result = GetDOMDocument (getter_AddRefs(DOMDocument));
	if (NS_FAILED(result) || !DOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface (DOMDocument);
	if (!doc) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));

	char *url;
	uri->GetSpec (&url);
	
	nsCOMPtr<nsIPermissionManager> permissionManager =
			do_CreateInstance (NS_PERMISSIONMANAGER_CONTRACTID);
	
	result = permissionManager->Add (url,block ? PR_TRUE : PR_FALSE, type);

	return result;
}

nsresult GaleonWrapper::ForceCharacterSet (char *charset) 
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	nsAutoString charset_str;
	charset_str.AssignWithConversion (charset);
	result = mdv->SetForceCharacterSet (ToNewUnicode(charset_str));

	return result;
}

nsresult GaleonWrapper::CanCutSelection(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanCutSelection (result);
}

nsresult GaleonWrapper::CanCopySelection(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanCopySelection (result);
}

nsresult GaleonWrapper::CanPaste(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanPaste (result);
}

nsresult GaleonWrapper::CutSelection(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CutSelection ();
}

nsresult GaleonWrapper::CopySelection(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CopySelection ();
}

nsresult GaleonWrapper::Paste(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->Paste ();
}

nsresult GaleonWrapper::GetFaviconURL (char **favicon_url)
{
	nsresult result;
	PRUint32 links_count;
	char *relstr;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsString strname;
	strname.AssignWithConversion("LINK");

	nsCOMPtr<nsIDOMNodeList> aLinks;
	result = aDOMDocument->GetElementsByTagName (strname, 
						     getter_AddRefs (aLinks));
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	result = aLinks->GetLength (&links_count);
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	for (PRUint32 i = 0; i < links_count; i++)
	{
		nsCOMPtr<nsIDOMNode> aLink;
		result = aLinks->Item (i, getter_AddRefs (aLink));
		if (NS_FAILED (result)) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIDOMElement> linkElement;
		linkElement = do_QueryInterface (aLink);
		if (!linkElement) return NS_ERROR_FAILURE;

		nsAutoString attr;
		attr.AssignWithConversion("rel");
		nsAutoString value;
		linkElement->GetAttribute (attr, value);
		relstr = ToNewCString (value);
		g_strup (relstr);

		if (strcmp(relstr, "SHORTCUT ICON") == 0)
		{
			char *link = NULL;
			
			nsMemory::Free (relstr);

			nsAutoString attr;
			attr.AssignWithConversion("href");
			nsAutoString value;
			linkElement->GetAttribute (attr, value);
			if (value.IsEmpty()) continue;

			link = ToNewCString (value);

			if (!link)
			{
				*favicon_url = NULL;
				return NS_OK;
			}
			
			nsCOMPtr<nsIDocument> doc = 
				do_QueryInterface (aDOMDocument);
			if(!doc) return NS_ERROR_FAILURE;
			
			nsCOMPtr<nsIURI> uri;
			doc->GetDocumentURL(getter_AddRefs(uri));

			result = uri->Resolve (link, favicon_url);
			if (NS_FAILED (result)) return NS_ERROR_FAILURE;
			
			nsMemory::Free (link);
			return NS_OK;
		}
	}

	return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::GetStyleSheets(nsIDOMStyleSheetList** list)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> doc;
	result = GetDOMDocument(getter_AddRefs(doc));
	if (NS_FAILED(result) || !doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMDocumentStyle> docstyle(do_QueryInterface(doc));
	if (!docstyle) return NS_ERROR_FAILURE;

	result = docstyle->GetStyleSheets(list);
	if (NS_FAILED(result) || !list) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::LoadOverrideStyleSheet(char *css,
					       nsIStyleSheet **return_sheet)
{
	nsresult result;
	PRBool completed;

	/* catch necessary stuff */
	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> ps;
	result = DocShell->GetPresShell (getter_AddRefs(ps));
	if (NS_FAILED(result) || !ps) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc;
	ps->GetDocument(getter_AddRefs(doc));

	nsCOMPtr<nsIHTMLContentContainer> container = do_QueryInterface(doc);
	if(!container) return NS_ERROR_FAILURE;

	nsICSSLoader *loader;
	result = container->GetCSSLoader (loader);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	/* load sheet */
	nsString s;
	s.AssignWithConversion(css);
	
	nsCOMPtr<nsIURI> uri;
	result = NS_NewURI(getter_AddRefs(uri), s);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsICSSStyleSheet *sheet;
	result = loader->LoadAgentSheet(uri, sheet, completed, nsnull);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	/* catch stylesheet stuff and apply by appending it as a override
	 * sheet and enabling stylesheets */
	nsCOMPtr<nsIStyleSheet> styleSheet = do_QueryInterface(sheet);
	if (!styleSheet) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsIStyleSet> styleSet;
	result = ps->GetStyleSet(getter_AddRefs(styleSet));
	if (NS_FAILED(result) || !styleSet) return NS_ERROR_FAILURE;

	styleSet->AppendOverrideStyleSheet(styleSheet);
	styleSheet->SetOwningDocument(doc);

	/* hack to notify mozilla that stylesheets have changed */
	styleSheet->SetEnabled(PR_FALSE);
	styleSheet->SetEnabled(PR_TRUE);

	*return_sheet = (nsIStyleSheet*)styleSheet;
	
	return NS_OK;
}

nsresult GaleonWrapper::RemoveOverrideStyleSheet(nsIStyleSheet *remove)
{
	nsresult result;

	/* get all necessary stuff */
	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	/* catch presshell */
	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> ps;
	result = DocShell->GetPresShell (getter_AddRefs(ps));
	if (NS_FAILED(result) || !ps) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIStyleSet> styleSet;
	result = ps->GetStyleSet(getter_AddRefs(styleSet));
	if (NS_FAILED(result) || !styleSet) return NS_ERROR_FAILURE;

	/* remove it */
	styleSet->RemoveOverrideStyleSheet(remove);
	remove->SetEnabled(PR_FALSE);

	return NS_OK;
}
