/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "EventContext.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDocument.h"
#include "nsIDOMDocument.h"
#include "nsIInterfaceRequestor.h"
#include "nsIDOMHTMLImageElement.h"
#include "nsIDOMElement.h"
#include "nsIURI.h"

#if MOZILLA_VERSION > VERSION3(0,9,4)
#include "nsIDOMNSDocument.h"
#include "nsReadableUtils.h"
#else
#include "gStringHelper.h"
#endif

EventContext::EventContext ()
{
}

EventContext::~EventContext ()
{
}

nsresult EventContext::Init (nsIDOMEvent *event, GaleonWrapper *wrapper)
{
	mEvent = event;
	mWrapper = wrapper;

	return NS_OK;
}

nsresult EventContext::GetKeyEventInfo (WrapperKeyEventInfo *info)
{
	PRBool mod_key;
	nsresult res;
	nsIDOMKeyEvent *keyEvent = (nsIDOMKeyEvent*)mEvent;

	/* if we are in a text area do not process keys */
	nsCOMPtr<nsIDOMEventTarget>  targetNode;
	res = mEvent->GetTarget(getter_AddRefs(targetNode));
	if (NS_FAILED(res) || !targetNode) return NS_ERROR_FAILURE;
	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(targetNode);
	/* don't grab alt combos, thus you can still access the menus and
	 *  other alt functions.  */
	keyEvent->GetAltKey (&mod_key);
	if (node && !mod_key)
	{
		nsCOMPtr<nsIDOMHTMLElement> element;

		element = do_QueryInterface(node);
		if (element)
		{
			nsAutoString tag;
			element->GetTagName(tag);

			nsCOMPtr<nsIDOMHTMLInputElement> inputelement;

			inputelement = do_QueryInterface(element);
			if (inputelement) {
				nsAutoString type;
				inputelement->GetType(type);
				if (type.EqualsWithConversion 
						("text", PR_TRUE) ||
				    type.EqualsWithConversion 
						("password", PR_TRUE) ||
				    type.EqualsWithConversion 
						("file", PR_TRUE))
				{
					return NS_ERROR_FAILURE;
				}
			}
			else if (!tag.EqualsWithConversion ("a", PR_TRUE) &&
				 !tag.EqualsWithConversion ("html", PR_TRUE))
			{
				return NS_ERROR_FAILURE;
			}

		}
	}

	/* Get the context */

	res = GetEventContext (targetNode, &info->ctx);
	if (NS_FAILED(res)) return res;
	
	/* Get the key/modifier */

	info->modifier = 0;

	keyEvent->GetCharCode((PRUint32*)&info->key);

	if (info->key==0) 
	{
		keyEvent->GetKeyCode((PRUint32*)&info->key);
		info->modifier |= KEY_CODE;
	}

	keyEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	keyEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	keyEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	keyEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;

	return NS_OK;
}

nsresult EventContext::GetEventContext (nsIDOMEventTarget *EventTarget,
					WrapperContextInfo *info)
{
	nsresult rv;

	info->context = CONTEXT_DOCUMENT;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(EventTarget, &rv);
	if (NS_FAILED(rv) || !node) return NS_ERROR_FAILURE;

        /* Is page xul ? then do not display context menus
	 * FIXME I guess there is an easier way ... */
	/* From philipl: This test needs to be here otherwise we
	 * arrogantly assume we can QI to a HTMLElement, which is
	 * not true for xul content. */ 

	nsCOMPtr<nsIDOMDocument> domDoc;
	rv = node->GetOwnerDocument(getter_AddRefs(domDoc));
	if (NS_FAILED(rv) || !domDoc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc, &rv);
	if (NS_FAILED(rv) || !doc) return NS_ERROR_FAILURE;

#if MOZILLA_VERSION > VERSION3(0,9,4)
	nsCOMPtr<nsIDOMNSDocument> nsDoc = do_QueryInterface(domDoc, &rv);
	if (NS_FAILED(rv) || !nsDoc) return NS_ERROR_FAILURE;
#else
#define nsDoc doc
#endif

	nsString mime;  
	rv = nsDoc->GetContentType(mime);
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	// Return NS_OK for xul as this is the correct point to exit for
	// the xul case. Nothing had failed.

	if (mime.EqualsWithConversion ("text/xul"))
		return NS_OK;

	// Now we know that the page isn't a xul window, we can try and
	// do something useful with it.

	PRUint16 type;
	rv = node->GetNodeType(&type);
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	if (nsIDOMNode::ELEMENT_NODE == type)
	{
		nsCOMPtr<nsIDOMHTMLElement> element = do_QueryInterface(node,
									&rv);
		if (NS_FAILED(rv) || !element) return NS_ERROR_FAILURE;

		nsAutoString tag;
		rv = element->GetTagName(tag);
		if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

		if (tag.EqualsWithConversion("img", PR_TRUE))
		{
			info->context |= CONTEXT_IMAGE;

			nsAutoString img;
			nsCOMPtr <nsIDOMHTMLImageElement> image = 
						do_QueryInterface(node, &rv);
			if (NS_FAILED(rv) || !image) return NS_ERROR_FAILURE;			

			rv = image->GetSrc (img);
			if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

			info->img = ToNewCString(img);
		}
		else if (tag.EqualsWithConversion("input", PR_TRUE))
		{
			nsCOMPtr<nsIDOMElement> element;
			element = do_QueryInterface (node);
			if (!element) return NS_ERROR_FAILURE;

			nsAutoString attr;
			attr.AssignWithConversion("type");
			nsAutoString value;
			element->GetAttribute (attr, value);

			if (value.EqualsWithConversion("image", PR_TRUE))
			{
				info->context |= CONTEXT_IMAGE;
				nsCOMPtr<nsIDOMHTMLInputElement> input;
				input = do_QueryInterface (node);
				if (!input) return NS_ERROR_FAILURE;

				nsAutoString img;
				rv = input->GetSrc (img);
				if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

				nsCOMPtr<nsIURI> uri;
				doc->GetDocumentURL(getter_AddRefs(uri));

				char *src = ToNewCString (img);

				rv = uri->Resolve (src, &info->img);
				nsMemory::Free (src);
				if (NS_FAILED (rv)) return NS_ERROR_FAILURE;
			}
			    else
			{
				info->context |= CONTEXT_INPUT;
			}
		}
		else if (tag.EqualsWithConversion("textarea", PR_TRUE))
		{
			info->context |= CONTEXT_INPUT;
		}
	}


	/* Is page framed ? */
	PRBool framed;
	IsPageFramed (node, &framed);
	info->framed_page = (gboolean)framed;

	/* Bubble out, looking for items of interest */
	while (node)
	{
		PRUint16 type;
		rv = node->GetNodeType(&type);
		if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

		if (nsIDOMNode::ELEMENT_NODE == type)
		{
			nsCOMPtr<nsIDOMHTMLElement> element = 
					do_QueryInterface(node, &rv);
			if (NS_FAILED(rv) || !element) return NS_ERROR_FAILURE;

			nsAutoString tag;
			rv = element->GetTagName(tag);
			if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

			/* Link */
			if (tag.EqualsWithConversion("a", PR_TRUE))
			{
				PRBool isHref;
				rv = element->HasAttribute (NS_LITERAL_STRING("href"),
							    &isHref);
				if (NS_FAILED(rv))
					return NS_ERROR_FAILURE;
				if (isHref == PR_FALSE)
					return NS_OK;

				info->context |= CONTEXT_LINK;

				nsCOMPtr <nsIDOMHTMLAnchorElement> anchor =
					do_QueryInterface(node, &rv);
				if (NS_SUCCEEDED(rv) && anchor) 
				{
					nsAutoString href;
					rv = anchor->GetHref (href);
					if (NS_FAILED(rv))
						return NS_ERROR_FAILURE;
					
					info->link = ToNewCString(href);
					
					/* Get the text of the link */
					nsCOMPtr<nsIDOMNSHTMLElement> nsElement;
					nsElement = do_QueryInterface (element,
								       &rv);
					if (NS_SUCCEEDED(rv) && nsElement)
					{
						nsAutoString linkhtml;
						rv = nsElement->GetInnerHTML
								(linkhtml);
						if (NS_SUCCEEDED(rv))
						{
							info->linktext =
								mozilla_unicode_to_locale
							(linkhtml.get());
						}
					}
				}
			
			}			
			if (tag.EqualsWithConversion("area", PR_TRUE))
			{
				info->context |= CONTEXT_LINK;
				nsCOMPtr <nsIDOMHTMLAreaElement> area =
						do_QueryInterface(node, &rv);
				if (NS_SUCCEEDED(rv) && area)
				{
					nsAutoString href;
					rv = area->GetHref (href);
					if (NS_FAILED(rv))
						return NS_ERROR_FAILURE;
					
					info->link = ToNewCString(href);
				}
			}
			else if (tag.EqualsWithConversion("textarea", PR_TRUE))
			{
				info->context |= CONTEXT_INPUT;
			}

			if (info->bgimg == NULL)
			{
				nsCOMPtr<nsIDOMElement> domelement;
				domelement = do_QueryInterface (node);
				if (!domelement) return NS_ERROR_FAILURE;

				nsAutoString attr;
				attr.AssignWithConversion("background");
				nsAutoString value;
				domelement->GetAttribute (attr, value);
				
				if (!value.IsEmpty())
				{
					gchar *tmp = ToNewCString(value);

					nsIURI *uri;
					doc->GetBaseURL(uri);
					rv = uri->Resolve 
						(tmp, &info->bgimg);
					if (NS_FAILED (rv))
						return NS_ERROR_FAILURE;
					nsMemory::Free (tmp);
				}
				else
				{
					info->bgimg = NULL;
				}
			}
			else if (info->bgimg == NULL)
			{
				nsCOMPtr<nsIDOMHTMLBodyElement> bgelement;
				bgelement = do_QueryInterface (node);
				if (bgelement)
				{
					nsAutoString value;
					bgelement->GetBackground (value);

					if (!value.IsEmpty())
					{
						gchar *tmp = ToNewCString(value);
						nsIURI *uri;
						doc->GetBaseURL(uri);
						rv = uri->Resolve 
							(tmp, &info->bgimg);
						if (NS_FAILED (rv))
							return NS_ERROR_FAILURE;
						nsMemory::Free (tmp);

					}
					else
					{
						info->bgimg = NULL;
					}
				}
			}

			if (info->bgimg == NULL)
			{
				nsAutoString cssurl;
				rv = GetCSSBackground (node, cssurl);
				if (NS_SUCCEEDED (rv))
				{
					char *tmp;
					tmp = ToNewCString(cssurl);
					nsIURI *uri;
					doc->GetBaseURL(uri);
					rv = uri->Resolve (tmp,
							   &info->bgimg);
					nsMemory::Free (tmp);
					if (NS_FAILED (rv)) 
						return NS_ERROR_FAILURE;
				}
			}
		}
		
		nsCOMPtr<nsIDOMNode> parentNode;
		node->GetParentNode (getter_AddRefs(parentNode));
		node = parentNode;
	}
	
	return NS_OK;
}

nsresult EventContext::GetCSSBackground (nsIDOMNode *node, nsAutoString& url)
{
	nsresult result;

	nsCOMPtr<nsIDOMElementCSSInlineStyle> style;
	style = do_QueryInterface (node);
	if (!style) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMCSSStyleDeclaration> decl;
	result = style->GetStyle (getter_AddRefs(decl));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsAutoString prop;
	prop.AssignWithConversion("background-image");
	nsAutoString value;
	decl->GetPropertyValue (prop, value);
	if (value.IsEmpty())
	{
		prop.AssignWithConversion("background");
		decl->GetPropertyValue (prop, value);
		if (value.IsEmpty())
		{
			prop.AssignWithConversion("background-repeat");
			decl->GetPropertyValue (prop, value);
			if (value.IsEmpty())
				return NS_ERROR_FAILURE;
		}
	}

	PRInt32 start, end;
	nsAutoString startsub;
	nsAutoString cssurl;
	startsub.AssignWithConversion("url(");
	nsAutoString endsub;
	endsub.AssignWithConversion(")");
	start = value.Find (startsub) + 4;
	end = value.Find (endsub);

	if (start == -1 || end == -1)
		return NS_ERROR_FAILURE;

	url = Substring (value, start, end - start);

	return NS_OK;
}

nsresult EventContext::GetMouseEventInfo (WrapperMouseEventInfo *info)
{
	nsresult result;
	DOMTimeStamp ts;
	nsIDOMMouseEvent *aMouseEvent = (nsIDOMMouseEvent*)mEvent;

	aMouseEvent->GetButton ((PRUint16*)&info->button);

	aMouseEvent->GetTimeStamp(&ts);
	info->timestamp = ts;

	/* be sure we are not clicking on the scroolbars */
	/* FIXME this need to be moved in GetEventContext and cleaned */

	nsCOMPtr<nsIDOMEventTarget> OriginalTarget;
	result = aMouseEvent->GetOriginalTarget(getter_AddRefs(OriginalTarget));
	if (NS_FAILED(result) || !OriginalTarget) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNode> OriginalNode = do_QueryInterface(OriginalTarget);
	if (!OriginalNode) return NS_ERROR_FAILURE;

	nsString nodename;
	OriginalNode->GetNodeName(nodename);

	if (nodename.EqualsWithConversion ("xul:thumb") ||
	    nodename.EqualsWithConversion ("xul:slider"))
		return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventTarget> EventTarget;
	result = aMouseEvent->GetTarget(getter_AddRefs(EventTarget));
	if (NS_FAILED(result) || !EventTarget) return NS_ERROR_FAILURE;

	result = GetEventContext (EventTarget, &info->ctx);
	if (NS_FAILED(result)) return result;

	/* Get the modifier */

	PRBool mod_key;

	info->modifier = 0;

	aMouseEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	aMouseEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	aMouseEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	aMouseEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;

	return NS_OK;
}

nsresult EventContext::IsPageFramed (nsIDOMNode *node, PRBool *Framed)
{
	nsresult result;
	
	nsCOMPtr<nsIDOMDocument> mainDocument;
	result = mWrapper->GetMainDOMDocument (getter_AddRefs(mainDocument));
	if (NS_FAILED(result) || !mainDocument) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsIDOMDocument> nodeDocument;
	result = node->GetOwnerDocument (getter_AddRefs(nodeDocument));
	if (NS_FAILED(result) || !nodeDocument) return NS_ERROR_FAILURE;
 
	*Framed = (mainDocument != nodeDocument);

        return NS_OK;
}

