/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "Galeon.h"
#include "prefs.h"
#include "history.h"
#include "embed.h"
#include "session.h"
#include "mozilla_prefs.h"
#include "misc.h"
#include "bookmarks.h"
#include "window.h"
#include "dialog.h"

#include <gtkmozembed.h>
#include <glade/glade.h>

/* GNOME includes */
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-init.h>
#include <libgnomeui/gnome-geometry.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <liboaf/liboaf.h>
#ifdef ENABLE_APPLET
#include <applet-widget.h>
#endif
#ifdef ENABLE_GNOME_FILE_SELECTOR
#include <bonobo/bonobo-main.h>
#endif /* !ENABLE_GNOME_FILE_SELECTOR */

#include <errno.h>

/* local function prototypes */
static gint translate_url_arguments (poptContext context, gchar ***url);
static CORBA_ORB corba_init (int *argc, char *argv[]);
static gboolean  new_view_on_running_shell (gint n_urls, gchar **url);
static void new_shell (CORBA_ORB orb);
static gboolean galeon_init (int argc, char *argv[]);
static void create_default_browser (void);
static inline ViewState get_viewstate (void);
static gint read_mozilla_version (void);
static gint check_mozilla_version (void);

/* import from Galeon-impl.c */
Galeon_Browser impl_Galeon_Browser__create (PortableServer_POA poa, 
					    CORBA_Environment *corba_env);

/* global variables */
gboolean galeon_server_mode = FALSE;
gboolean galeon_panel_mode = FALSE;

/* file local variables */
CORBA_Environment corba_env;                 /* corba environment for errors */
static gboolean open_in_existing   = FALSE;  /* load in existing window?     */
static gboolean open_in_new_tab    = FALSE;  /* force open in a new tab?     */
static gboolean noraise            = FALSE;  /* no raise                     */
static gboolean open_in_new_window = FALSE;  /* force open in a new window?  */
static gboolean open_fullscreen    = FALSE;  /* open galeon in full screen ? */
static gchar *geometry_string      = NULL;   /* the geometry string          */
static gchar *bookmark_url         = NULL;   /* the temp bookmark to add     */
static gboolean close_option       = FALSE;  /* --close                      */
static gboolean quit_option        = FALSE;  /* --quit                       */

/* mozilla version detected at runtime */
int mozilla_version=0;

/* command line argument parsing structure */
static struct poptOption popt_options[] =
{
	{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &oaf_popt_options, 0, NULL,
	  NULL },
	{ "new-tab", 'n', POPT_ARG_NONE, &open_in_new_tab, 0,
	  N_("Open a new tab in an existing Galeon window"),
	  NULL },
	{ "new-window", 'w', POPT_ARG_NONE, &open_in_new_window, 0,
	  N_("Open a new window in an existing Galeon process"),
	  NULL },
	{ "noraise", '\0', POPT_ARG_NONE, &noraise, 0,
	  N_("Do not raise the window when opening a page in an existing Galeon process"),
	  NULL },
	{ "fullscreen", 'f', POPT_ARG_NONE, &open_fullscreen, 0,
	  N_("Run Galeon in full screen mode"),
	  NULL },
	{ "existing", 'x', POPT_ARG_NONE, &open_in_existing, 0,
	  N_("Attempt to load URL in existing Galeon window"),
	  NULL },
#ifdef ENABLE_APPLET
	{ "panel", 'p', POPT_ARG_NONE, &galeon_panel_mode, 0,
	  N_("Open as a panel applet"),
	  NULL },		  
#endif
	{ "server", 's', POPT_ARG_NONE, &galeon_server_mode, 0,
	  N_("Don't open any windows; instead act as a server "
	     "for quick startup of new Galeon instances"),
	  NULL },		  
	{ "add-bookmark", 't', POPT_ARG_STRING, &bookmark_url,
	  0, N_("Add a bookmark (don't open any window)"), 
	  N_("URL")},
	{ "geometry", 'g', POPT_ARG_STRING, &geometry_string,
	  0, N_("Create the initial window with the given geometry.\n"
		"see X(1) for the GEOMETRY format"),
	  N_("GEOMETRY")},
	{ "close", 'c', POPT_ARG_NONE, &close_option, 0,
	  N_("Close all Galeon windows"),
	  NULL },
	{ "quit", 'q', POPT_ARG_NONE, &quit_option, 0,
	  N_("Same as --close, but exits server mode too"),
	  NULL },

	/* terminator, must be last */
	{ NULL, 0, 0, NULL, 0, NULL, NULL }
};

/**
 * main: main entry point
 *
 * Set values to standard values, and change later by reading the 
 * options defined at the command line.
 * 
 * Initialize localization, glade, gnome, etc.
 */
int
main (int argc, char *argv[])
{
	GaleonEmbed *last_embed = NULL;
	gboolean tabbed_mode;
	poptContext context;
	gboolean no_default;
	CORBA_ORB orb;
	gint i, n_urls;
	gchar **url;

	 /* make stdout line buffered - we only use it for debug info */
	setvbuf (stdout, NULL, _IOLBF, 0);

#ifdef ENABLE_NLS
	/* initialise localisation */
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);
#endif

#ifdef ENABLE_APPLET
	/* panel mode? this check has to be done here since we
	 * don't use gnome_init_* when in panel mode */
	for (i = 1; i < argc; i++)
	{
		if (strcmp (argv[i], "-p") == 0 || 
		    strcmp (argv[i], "--panel") == 0)
		{
			galeon_panel_mode = TRUE;
		}
	}

	/* initialise GNOME */
	if (galeon_panel_mode)
	{
		applet_widget_init (PACKAGE, VERSION, argc, argv, 
				    popt_options, 0, &context);
	}
	else
#endif
	{
		gnome_init_with_popt_table (PACKAGE, VERSION, argc, argv,
					    popt_options, 0, &context);
	}

	if (galeon_panel_mode && galeon_server_mode)
	{
		galeon_server_mode = FALSE;
		g_message (_("Panel mode. "
			     "Ignoring server mode option."));
	}

	/* load arguments that aren't regular options (urls to load) */
	n_urls = translate_url_arguments (context, &url);

	/* initialise CORBA */
	orb = corba_init (&argc, argv);
	
	/* check if galeon is already running and use that instead */
	if (!galeon_panel_mode && !galeon_server_mode)
	{
		if (new_view_on_running_shell (n_urls, url))
		{
			/* done */
			return 0;
		} 
	}

	if (close_option || quit_option)
	{
		g_message (_("Nothing to close: "
			     "no opened Galeon windows found."));
		return 0;
	}

	/* start a new CORBA shell */
	new_shell (orb);

	/* initialise required external and internal Galeon services */
	no_default = galeon_init (argc, argv);

	/* add a temp bookmark and exit if needed */
	if (bookmark_url != NULL)
	{
		bookmarks_load ();
		/* this will save the bookmarks too -- Matthew */
		add_bookmark_default (BM_SITE, NULL, bookmark_url, NULL);
		return 0;
	}
	
	/* if we have command-line supplied URLs */
	tabbed_mode = eel_gconf_get_boolean (CONF_TABS_TABBED) ? 1 : 0;
	for (i = 0; i < n_urls; i++)
	{
		last_embed = embed_create_from_url (last_embed, url[i],
						    FALSE, !tabbed_mode);
	}
	g_strfreev (url);

	/* check if browser exists, or create default one */
	if (!no_default && !galeon_server_mode && !galeon_panel_mode)
	{
		create_default_browser ();
	}

	/* start main loop */
#ifdef ENABLE_APPLET
	if (galeon_panel_mode)
	{
		/* build panel and run special main loop */
		panel_main ();
	}
	else
#endif
	{
		/* enter the main GTK event processing loop */
		gtk_main ();
	}

	eel_gconf_monitor_remove ("/apps/galeon");
	eel_gconf_monitor_remove ("/apps/nautilus/preferences");

	/* exit cleanly */
	return 0;
}

/**
 * read_mozilla_version: extract mozilla version from mozilla prefs file
 */
static gint
read_mozilla_version (void)
{
	gchar agentstr[160], *p, *endptr;
	gint ver= 0, v;
	gchar *preffile;
	FILE *f;

	preffile = g_strconcat (getenv ("MOZILLA_FIVE_HOME"),
				"/defaults/pref/all.js", NULL);
	f = fopen (preffile, "r");
	
	if (f == NULL)
	{
		g_warning ("error opening %s: %s", preffile, strerror (errno));
		g_free (preffile);
		return 0;
	}

	while ((p = fgets (agentstr, 160, f)) != NULL)
	{
		if ((p = strstr (agentstr, "useragent.misc")) != NULL &&
			(p = strstr (p, "rv:")) != NULL)
			break;
	}
	fclose(f);

	if (p == NULL)
	{
		g_warning ("Mozilla version not found in %s", preffile);
	}
	else
	{
		p += 3;
		v = strtol (p, &endptr, 10);
		ver = v << 24;
		if (*endptr == '.')
		{
			p = endptr + 1;
			v = strtol (p, &endptr, 10);
			ver += v << 16;
			if (*endptr == '.')
			{
				p = endptr + 1;
				v = strtol (p, &endptr, 10);
				ver += v << 8;
			}
		}
		if (*endptr == '+')
		{
			ver++;
		}
	}

	g_free (preffile);
	return ver;
}

/**
 * check_mozilla_version: verify that the runtime mozilla is the same version
 * as the compile time.  If not print a warning and display a confirmation
 * dialog.
 */
static gint
check_mozilla_version (void)
{
	mozilla_version = read_mozilla_version ();

	if (MOZILLA_VERSION != mozilla_version)
	{
		gchar *questionStr, *rememberStr, *verMatchStr;
		gboolean continuev=0, rememberv=0;
		g_warning ("compiled MOZILLA_VERSION ("VERSIONFMT") !="
			" detected mozilla_version ("VERSIONFMT")",
			VERSIONFMTARGS (MOZILLA_VERSION),
			VERSIONFMTARGS (mozilla_version));

		verMatchStr = g_strdup_printf ("%xv%x", MOZILLA_VERSION,
						mozilla_version);
		rememberStr = eel_gconf_get_string 
				(CONF_MOZ_MISMATCH);

		if (rememberStr != NULL)
		{
			int ignore = strcmp (rememberStr, verMatchStr);
			g_free (rememberStr);
			if (ignore == 0)
			{
				g_free (verMatchStr);
				return eel_gconf_get_integer
				    (CONF_MOZ_MISMATCH_ACTION);
			}
		}

		questionStr = g_strdup_printf (_(
"The version of Mozilla Galeon was compiled against ("VERSIONFMT") "
"does not match the detected version ("VERSIONFMT").\n\n"
"You may experience segfaults and other oddities, please test "
"with a matching version of Mozilla before reporting any bugs.\n\n"
"Continue anyway?"),
			VERSIONFMTARGS (MOZILLA_VERSION),
			VERSIONFMTARGS (mozilla_version));


		continuev = dialog_confirm_check (NULL, _("Version mismatch!"),
					questionStr, _("Don't ask me again"),
					&rememberv);
	
		g_free (questionStr);
		if (rememberv)
		{
			eel_gconf_set_string
				(CONF_MOZ_MISMATCH,
				verMatchStr);
			eel_gconf_set_integer
				(CONF_MOZ_MISMATCH_ACTION,
				 !continuev);
		}
		g_free (verMatchStr);
		return !continuev;
	}
	return 0;
}

/**
 * translate_url_arguments: gather URL arguments and expand them fully
 * with realpath if they're filenames
 */
static gint
translate_url_arguments (poptContext context, gchar ***urls)
{
	gchar buffer[PATH_MAX];
	gchar **args;
	gint i, n;

	/* any context remaining? */
	if (context == NULL)
	{
		*urls = NULL;
		return 0;
	}

	/* get the args and check */
	args = (gchar **) poptGetArgs (context);
	if (args == NULL)
	{
		poptFreeContext (context);
		*urls = NULL;
		return 0;
	}

	/* count args */
	for (n = 0; args[n] != NULL; n++)
		/* nothing */;

	/* allocate pointer array */
	*urls = g_new0 (gchar *, n + 1);
	
	/* translate each one */
	for (i = 0; i < n; i++)
	{
		/* try to expand as files */
		if (realpath (args[i], buffer) != NULL)
		{
			(*urls)[i] = g_strconcat ("file://", buffer, NULL);
		}
		else
		{
			(*urls)[i] = g_strdup (args[i]);
		}
	}
	poptFreeContext (context);
	(*urls)[i] = NULL;

	/* return the number of urls */
	return n;
}

/**
 * corba_init: initialise Gnorba (for gtm) and OAF (for remote
 * Galeon startups).
 */
static CORBA_ORB
corba_init (int *argc, char *argv[])
{
	CORBA_ORB orb;

	/* catch exceptions */
	CORBA_exception_init (&corba_env);

	/* initialise OAF */
	orb = oaf_init (*argc, argv);

#ifdef ENABLE_GNOME_FILE_SELECTOR
        if ( !bonobo_init (orb, NULL, NULL)) {
                g_warning ("Cannot initialize Bonobo component framework !");
        }
#endif /* !ENABLE_GNOME_FILE_SELECTOR */
        
	/* return handle */
	return orb;
}

/**
 * new_view_on_running_shell: 
 *
 * Create new view on existing shell. 
 * Returns false if no shell is currently running.
 */
static gboolean 
new_view_on_running_shell (gint n_urls, gchar **url)
{
	CORBA_Object client;
	ViewState viewstate;
	gboolean result = TRUE;
	gint i;
	
	/* compute target viewstate */
	viewstate = get_viewstate ();

	/* try to find an existing shell to run on */
	client = oaf_activate ("repo_ids.has('IDL:galeon/browser:1.0')", NULL,
			       OAF_FLAG_EXISTING_ONLY, NULL, &corba_env);

	/* check if we found one */
	if (CORBA_Object_is_nil (client, &corba_env))
	{
		return FALSE;
	}

	/* if found and we're given a bookmark to add... */
	if (bookmark_url != NULL)
	{
		Galeon_Browser_addTempBookmark (client, bookmark_url, 
						bookmark_url, &corba_env);
		return TRUE;
	}

	if (close_option || quit_option)
	{
		Galeon_Browser_quit (client, quit_option, &corba_env);
		return TRUE;
	}

	/* otherwise we'll be opening windows, so warn */
	g_message (_("Galeon already running, using existing process"));

	/* provided with urls? */
	if (n_urls == 0)
	{
		/* no, open a default window */
		result = Galeon_Browser_loadurl (client, "", viewstate, 
						 &corba_env);
	}

	/* open all of the urls */
	for (i = 0; i < n_urls; i++)
	{
		result = Galeon_Browser_loadurl (client, url[i], viewstate, 
						 &corba_env);
		if (!result)
		{
			break;
		}
	}

#if 0
	/* check */
	if (result)
	{
		GtkWidget *dialog = gnome_error_dialog 
			(_("Galeon remote startup failed.\n\n"
			   "Try exiting other OAF applications\n"
			   "(e.g. Evolution and Nautilus), and running\n"
			   "\"oaf-slay\" from the command line."));
			 
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	}
#endif

	/* done */
	return TRUE;
}

/**
 * new_shell: create a new CORBA shell
 */
static void 
new_shell (CORBA_ORB orb)
{
	PortableServer_POA poa;
	Galeon_Browser client;
	gchar *reg_id;

	/* make a client */
	poa = (PortableServer_POA) CORBA_ORB_resolve_initial_references 
		(orb, "RootPOA", &corba_env);
	client = impl_Galeon_Browser__create (poa, &corba_env);

	/* check */
	if (client == NULL)
	{
		g_warning (_("Galeon object already active, "
			     "server registration failed\n"));
		return;
	}

	/* register server with oaf */
	reg_id = oaf_make_registration_id ("OAFIID:GNOME_Galeon_Automation",
					   g_getenv ("DISPLAY"));
	oaf_active_server_register (reg_id, client);

	PortableServer_POAManager_activate
		(PortableServer_POA__get_the_POAManager (poa, &corba_env), 
		 &corba_env);

	g_free (reg_id);
}

/**
 * galeon_init: initialise Galeon and the services on which it depends
 */
static gboolean
galeon_init (int argc, char *argv[])
{
	static gchar *restart_argv[] = { "galeon", "--server", NULL };
	GnomeClient *session_client;
	gboolean session_recovery;
	gboolean vfs_success;
	gboolean new_user;
	gchar *profile;
	extern gboolean pushed_startup;
	gboolean moz_version_checked = FALSE;
/*	GdkAtom message_type;*/
	gchar *teststr;
	gchar *mozprefs;
	gboolean set_mozilla_prefs = FALSE;

	/* init mozilla home */
	gtk_moz_embed_set_comp_path (g_getenv ("MOZILLA_FIVE_HOME"));

	/* set a path for the profile */
	profile = g_concat_dir_and_file (g_get_home_dir (), ".galeon/mozilla");

	/* no mozilla prefs ? */
	mozprefs = g_concat_dir_and_file (profile, "/galeon/prefs.js"); 
	if (!g_file_exists(mozprefs))
	{
		set_mozilla_prefs = TRUE;
	}
	g_free (mozprefs);

	/* initialize profile */
	gtk_moz_embed_set_profile_path (profile, "galeon");
	g_free (profile);

	/* initialise embed */
	if (!pushed_startup)
	{
		embed_startup_init ();
	}

	/* set mozilla prefs if necessary */
	if (set_mozilla_prefs)
	{
		mozilla_prefs_set ();
	}

	/* ensure mozilla matches (and initialise mozilla_version var) */
	if (g_getenv ("MOZILLA_FIVE_HOME") != NULL)
	{
		if (check_mozilla_version ())
		{
			exit (1);
		}
		moz_version_checked = TRUE;
	}

	/* set the user agent, this is always neccesary because the galeon
	   version may change */
	mozilla_prefs_set_user_agent ();

	/* initialise GNOME VFS */
	vfs_success = gnome_vfs_init ();
	g_assert (vfs_success);

	/* initialise GLADE */
	glade_gnome_init ();

	/* preload the prefs */
	/* it also enables notifiers support */
	eel_gconf_monitor_add ("/apps/galeon");
	eel_gconf_monitor_add ("/apps/nautilus/preferences");

	/* check if gconf schemas are working */
	teststr = gconf_client_get_string (eel_gconf_client_get_global(),
					  "/apps/galeon/gconf_test", NULL);
	if (!teststr)
	{
		GtkWidget *dialog;
		dialog = gnome_error_dialog ("Cannot find a schema for galeon preferences. \nCheck your gconf setup, look at galeon FAQ for more info");
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		exit (0);
	}
	else
	{
		g_free (teststr);
	}

	/* check if this is the first time this user has run Galeon */
	new_user = newuser_check ();

	/* initialise Galeon itself */
	initialise_colours ();
	preferences_load ();
	history_init ();
	bookmarks_init ();

	/* if we couldn't check the first time, try now since initializing moz
	 * will set the MOZILLA_FIVE_HOME in some configurations */
	if (!moz_version_checked)
	{
		if (check_mozilla_version ())
		{
			exit (1);
		}
	}

	/* resume a crashed session, if any */
	session_recovery = session_autoresume ();

	/* connect our session recovery to GNOME session */
	session_client = gnome_master_client ();
	gnome_client_set_restart_command 
		(session_client, galeon_server_mode ? 2 : 1, restart_argv);
	gnome_client_set_restart_style (GNOME_CLIENT (session_client),
					GNOME_RESTART_IF_RUNNING);
	gtk_signal_connect (GTK_OBJECT (session_client), "save_yourself",
			    (GtkSignalFunc)client_save_yourself_cb, NULL);
		
	/* update tab styles and aa pixmaps when theme changed */
	/* FIXME theme change is caught too many times, disabling for now
	 *    -- Jorn 10/10/2001 */
#if 0
	message_type = gdk_atom_intern ("_GTK_READ_RCFILES", FALSE);
	gdk_add_client_message_filter (message_type, (GdkFilterFunc)
				       gtk_theme_changed_cb, NULL);
#endif

	/* FIXME: catch GDK errors which happen on theme changes
	 * (until we work out what causes this) */
	gdk_error_trap_push ();

	/* if this flag doesn't get cleared, then Galeon crashed */
	eel_gconf_set_integer (CONF_CRASH_CRASHED, TRUE);

	/* return saying whether or not to override default window creation */
	return (new_user || session_recovery);
}

/**
 * create_default_browser: called at the end to make sure at least one
 * browser is created in normal modes.
 */
static void
create_default_browser (void)
{
	GaleonEmbed *embed;
	GaleonWindow *window;
	gint xpos, ypos;
	gint width, height;
	gchar *url;
	
	/* don't bother if there's already at least one */
	if (g_list_length (all_embeds) != 0)
	{
		return;
	}

	/* create an empty window */
	window = window_create (GTK_MOZ_EMBED_FLAG_ALLCHROME);
	
	/* show as fullscreen? */
	/* THE WINDOW NEEDS TO BE VISIBLE HERE */
	if (open_fullscreen)
	{
		gtk_widget_show (GTK_WIDGET (window->WMain));
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->view_fullscreen),
			 TRUE);
	}
	/* collect geometry information */
	else if (gnome_parse_geometry (geometry_string, &xpos, &ypos,
				       &width, &height))
	{
		/* setup size and show it */
		gtk_window_set_default_size (GTK_WINDOW (window->WMain), 
					     width, height);
		gtk_widget_set_uposition (GTK_WIDGET (window->WMain),
					  xpos, ypos);
		window->set_size = TRUE;
	}

	/* build the default embed inside it */
	embed = embed_create_in_window (window);
	embed_set_visibility (embed, TRUE);
	window_grab_location_focus (window);
	url = embed_get_default_url (NULL);
	embed_load_url (embed, url);
	g_free (url);
}

/**
 * get_viewstate:
 *
 * Determine the viewstate. A viewstate defines how the URI should be opened.
 * 
 * VIEWSTATE_EXISTING:    Open up in existing window if there is one
 * VIEWSTATE_WINDOW:      Open up in window
 * VIEWSTATE_TAB:         Open up in tab
 * VIEWSTATE_NORAISE_TAB: Open up in tab without raising the window
 */
static inline ViewState
get_viewstate (void)
{
	ViewState result;
	gboolean tabbed_mode;

	tabbed_mode = eel_gconf_get_boolean (CONF_TABS_TABBED);

	result = tabbed_mode ? VIEWSTATE_TAB : VIEWSTATE_WINDOW;

	if ((noraise && open_in_new_tab) ||
	    (noraise && tabbed_mode))
	{
		result = VIEWSTATE_NORAISE_TAB;
	}
	else if (open_in_new_window)
	{
		result = VIEWSTATE_WINDOW;
	}
	else if (open_in_new_tab)
	{
		result = VIEWSTATE_TAB;
	}
	else if (open_in_existing)
	{
		result = VIEWSTATE_EXISTING;
	}

	return result;
}
