/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla.h"
#include "embed.h"
#include "misc.h"
#include "history.h"
#include "eel-gconf-extensions.h"

#include <string.h>
#include <gtkmozembed.h>

/* for nautilus dnd desktop items */
#include <libgnome/gnome-mime.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnomevfs/gnome-vfs-mime.h>

/** 
 * embed_drag_data_received_cb:
 */
void
embed_drag_data_received_cb (GtkWidget *widget, GdkDragContext *context,
			     gint x, gint y, GtkSelectionData *selection_data,
			     guint info, guint time, GaleonEmbed *embed)
{
	gchar *data = selection_data->data;
	gboolean tabbed;
	GList *uris, *l;	
	gchar *drop_uri;
	gchar *parse_uri;
	gchar *real_uri;
	gchar **tmp;

	/* get tab status */
	tabbed = eel_gconf_get_boolean (CONF_TABS_TABBED);

	switch (info)
	{
	case DND_TARGET_NETSCAPE_URL:
		/* netscape format is: url \n title */
		tmp = g_strsplit(data, "\n", 2);
		if (tmp)
		{
			embed_load_url (embed, tmp[0]);
		}
		else
		{
			embed_load_url (embed, data);
		}
		g_strfreev (tmp);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_GALEON_URL:
		/* standard types are just strings */
		embed_load_url (embed, data);
		break;

	case DND_TARGET_TEXT_URI_LIST:
		/* list as passed by Nautilus */
		uris = gnome_uri_list_extract_uris (data);
		for (l = uris; l != NULL; l = g_list_next (l))
		{
			/* find the uri to open */
			drop_uri = (gchar *)(l->data);
			parse_uri = misc_parse_uri (drop_uri);
			real_uri = (parse_uri == NULL ? drop_uri : parse_uri);

			/* open in first or next embed */
			if (l == uris)
			{
				embed_load_url (embed, real_uri);
			}
			else
			{
				embed_create_from_url (embed, real_uri,
						       FALSE, !tabbed);
			}
			
			/* free allocated string */
			g_free (drop_uri);
			if (parse_uri != NULL)
			{
				g_free (parse_uri);
			}
		}
		/* free parsed list */
		g_list_free (uris);
		break;

       
	default:
		/* shouldn't happen */
		g_warning ("unexpected drop type %ud\n", info);
		break;
	}
}

/**
 * embed_tab_close_clicked_cb: close a tab
 */
void 
embed_tab_close_button_clicked_cb (GtkWidget *button, GaleonEmbed *embed)
{
	return_if_not_embed (embed);
	embed_close (embed);
}

/**
 * save_site_location: used when switching to a new tab, save the old
 * (possibly edited) site location in the old embed and hack up the
 * selection to preserve it (if any).
 */
void
save_site_location (GaleonEmbed *embed)
{
	GaleonWindow *window;
	guchar *text, *stored;
	int start, end, tmp;

	/* get window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* if no location entry then don't change anything */
	if (window->location_entry == NULL)
	{
		return;
	}

	/* free old value */
	if (embed->site_location != NULL)
	{
		g_free (embed->site_location);
	}
	
	/* get location text */
	embed->site_location = gtk_editable_get_chars 
		(GTK_EDITABLE (window->location_entry), 0, -1);
			
	/* did we have the selection? */
	if (!(GTK_EDITABLE (window->location_entry)->has_selection))
	{
		/* no, give up */
		return;
	}

	/* get selected text */
	start = GTK_EDITABLE (window->location_entry)->selection_start_pos;
	end = GTK_EDITABLE (window->location_entry)->selection_end_pos;

	/* switch them when they are in the opposite order,
	   it happens if you select from right to left */
	if (start > end)
	{
		tmp = start;
		start = end;
		end = tmp;
	}

	text = gtk_editable_get_chars 
		(GTK_EDITABLE (window->location_entry), 
		 start, end);

	/* free up stored per-window selection */
	stored = gtk_object_get_data (GTK_OBJECT (window->WMain), "selection");
	gtk_object_remove_data (GTK_OBJECT (window->WMain), "selection");
	g_free (stored);

        /* store in parent window */
	gtk_object_set_data (GTK_OBJECT (window->WMain), "selection", text);
	
	/* window takes ownership of primary selection */
	gtk_selection_owner_set (window->WMain, GDK_SELECTION_PRIMARY,
				 GDK_CURRENT_TIME);
}

void
embed_change_stylesheet_cb (GtkWidget *mi, AlternateStyleSheet *css)
{
	GaleonEmbed *embed =
		(GaleonEmbed*) gtk_object_get_data (GTK_OBJECT (mi), "embed");
	gint index =
	       GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (mi), "index"));
	return_if_not_embed (embed);

	history_set_selected_authorcss (embed->site_location, index);
	mozilla_set_alternate_stylesheet (embed, css);
}
