/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"
#include "toolbar.h"
#include "toolbar_editor.h"
#include "glade.h"

#include <string.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-stock.h>

/**
 * toolbar_editor_init: initialize toolbar editor 
 */
void
toolbar_editor_init (PreferencesDialog *pd)
{
	GtkCList *clist, *alist;
	ToolbarItemType type;
	gint i, count;
	gchar *key;

	/* get lists from dialog */
	clist = GTK_CLIST (pd->current_clist);
	alist = GTK_CLIST (pd->available_clist);
	g_return_if_fail (clist != NULL);
	g_return_if_fail (alist != NULL);

	/* clear them out */
	gtk_clist_clear (alist);
	gtk_clist_clear (clist);

	/* set them up */
	gtk_clist_set_sort_column (alist, 0);
	gtk_clist_set_sort_type (alist, GTK_SORT_ASCENDING);
	gtk_clist_set_use_drag_icons (clist, TRUE);
	gtk_clist_set_use_drag_icons (alist, TRUE);
	gtk_clist_set_reorderable (clist, TRUE);
	gtk_clist_freeze (alist);
	gtk_clist_freeze (clist);

        /* set button relief (glade is broken) */
	gtk_button_set_relief (GTK_BUTTON(pd->add_button),    GTK_RELIEF_NONE);
	gtk_button_set_relief (GTK_BUTTON(pd->remove_button), GTK_RELIEF_NONE);
	gtk_button_set_relief (GTK_BUTTON(pd->up_button),     GTK_RELIEF_NONE);
	gtk_button_set_relief (GTK_BUTTON(pd->down_button),   GTK_RELIEF_NONE);

	/* prepare config */
	gnome_config_push_prefix ("/galeon/Toolbar/");

	/* fill in available list with all possible items */
	for (i = 0; i < TOOLBAR_ITEM_COUNT; i++)
	{
		toolbar_editor_insert_item (alist, i);
	}

	/* get current count of used items */
	count = gnome_config_get_int ("count=-1");
	g_assert (count != -1);

	/* fill in current list and remove current items from available list */
	for (i = 0; i < count; i++)
	{
		/* get item type */
		key = g_strdup_printf ("item%d=-1", i);
		type = gnome_config_get_int (key);
		g_assert (type != -1);
		g_free (key);

		/* check for illegal values */
		if (type < 0 || type >= TOOLBAR_ITEM_COUNT)
		{
			g_warning (_("illegal toolbar item type\n"));
			continue;
		}

		/* remove it from available, add it to current */
		if (type != TOOLBAR_ITEM_SEPARATOR)
		{
			toolbar_editor_remove_item (alist, type);
		}
		toolbar_editor_insert_item (clist, type);
	}

	/* done with this set */
	gnome_config_pop_prefix ();

	/* show list updates */
	gtk_clist_sort (alist);
	gtk_clist_thaw (alist);
	gtk_clist_thaw (clist);
}

/**
 * toolbar_editor_save: save toolbar editor settings
 */
void
toolbar_editor_save (PreferencesDialog *pd)
{
	GtkCList *clist = GTK_CLIST (pd->current_clist);
	ToolbarItemType type;
	gchar *key;
	gint i;

	/* working in config area Toolbar */
	gnome_config_push_prefix ("/galeon/Toolbar/");

	/* store number of items */
	gnome_config_set_int ("count", clist->rows);

	/* save used items */
	for (i = 0; i < clist->rows; i++)
	{
		/* build key, get type from list */
		key = g_strdup_printf ("item%d", i);
		type = GPOINTER_TO_INT (gtk_clist_get_row_data (clist, i));
		gnome_config_set_int (key, type);
		g_free (key);
	}

	/* synchronise store */
	gnome_config_pop_prefix ();
	gnome_config_sync ();
}

/**
 * toolbar_editor_insert_item: insert a given toolbar item into a clist
 */
gint
toolbar_editor_insert_item (GtkCList *clist, ToolbarItemType type)
{
	gchar *text[1] = { NULL };
	GtkWidget *pixmap;
	gint row;
	
	/* get a row number */
	row = gtk_clist_append (GTK_CLIST (clist), text);

	/* get stock item if available */
	if (toolbar_items[type].stock_icon)
	{
		/* build shrunken pixmap */
		pixmap = gnome_stock_pixmap_widget_at_size 
			(NULL, toolbar_items[type].stock_icon,
			 16, 16);

		/* use this to make a nice entry */
		gtk_clist_set_pixtext (clist, row, 0, 
				       _(toolbar_items[type].label), 2, 
				       GNOME_STOCK (pixmap)->pixmap.pixmap, 
				       GNOME_STOCK (pixmap)->pixmap.mask);
		
		/* no longer need the widget itself */
		gtk_object_unref (GTK_OBJECT (pixmap));
	}
	else
	{
		/* otherwise just insert dull old text */
		gtk_clist_set_text (clist, row, 0, 
				    _(toolbar_items[type].label));
		gtk_clist_set_shift (clist, row, 0, 0, 18);
	}

	/* label it so we can find it again later */
	gtk_clist_set_row_data (clist, row, GINT_TO_POINTER (type));

	/* return row */
	return row;
}

/**
 * toolbar_editor_remove_item: remove a given toolbar item from a clist
 */
void
toolbar_editor_remove_item (GtkCList *clist, ToolbarItemType type)
{
	gint row;

	/* find it by the row data we set */
	row = gtk_clist_find_row_from_data (GTK_CLIST (clist),
					    GINT_TO_POINTER (type));

	/* remove it */
	gtk_clist_remove (GTK_CLIST (clist), row);
}
